
//------------------------------------------------------------//
//                                                            //
//   Class for storage of company specific data               //
//                                                            //
//------------------------------------------------------------//

class CompanyData
{
	_company_id = null; // company id
	_goal_list = null; // index = GOLD, SILVER or BRONZE - contains the GSGoal id for current goal
	_cargo_goal_list = null; // index = cargo goal 0, 1, 2 - contains the GSGoal id for cargo goal
	_transported_list = null; // index = 0, 1, 2 for the three cargoes (with lowest cargo ID first)
	_completed_list = null; // index = 0, 1,  2 for the three cargoes. Set to 1 when a cargo is found to be completed and news/events are sent.
	_hq_sign = null; // sign id of the sign sitting ontop of the HQ
	_ai_monthly_report = null; // boolean flag that AIs can toggle to enable/disable monthly progress reports.

	_goal_ptr = null;

	constructor(company_id, goal_ptr)
	{
		this._company_id = company_id;
		this._goal_list = [-1, -1, -1];
		this._cargo_goal_list = [-1, -1, -1];
		this._transported_list = [0, 0, 0];
		this._completed_list = [0, 0, 0];
		this._hq_sign = -1;
		this._ai_monthly_report = false;

		this._goal_ptr = goal_ptr;
	}

	// Creates goal in the GUI
	function CreateGoals();
	function UpdateGoals();

	// Send information about goal to AI or Human players
	function InformAboutGoal();

	// Adds a company league table to the book (called from main.nut)
	function AddCompanyLeaguePage(league_table, year, show_page);

	// Track goal progress
	function TrackProgress();

	// Getter/Setter for enable/disable monthly reports to AI
	function SetAIMonthlyReport(new_value);
	function GetAIMonthlyReport();

	// Called when the end of game has been reached
	function GameEnd();

	// Called at end of year (but not when the game end has been reached)
	function EndOfYear(year);

	// For save/load
	function SaveToTable();
	static function CreateFromTable(table, version, goal_ptr);
}

function CompanyData::CreateGoals()
{
	this.UpdateGoals();
}
function CompanyData::UpdateGoals()
{
	if (Story.IsStoryBookAvailable()) {
		this.UpdateGoals_Trunk();
	} else {
		this.UpdateGoals_Stable(); // 1.3 
	}
}
function CompanyData::GetPercentTransported()
{
	local target = GSController.GetSetting("transport_target");
	local percent = [0, 0, 0];
	for(local i = 0; i < 3; i++)
	{
		percent[i] = (this._transported_list[i] * 100 / target).tointeger();
	}
	return percent;
}
function CompanyData::UpdateGoals_Stable()
{
	local medal = this.GetMedal();
	local reached_medal_news_shown = false;
	local medal_debug_name = ["Gold", "Silver", "Bronze"];

	// Percent accomplished
	local target = GSController.GetSetting("transport_target");
	local percent = [0, 0, 0];
	local best_non_finished_percent = 0; // best percent value of non-finished cargoes
	for(local i = 0; i < 3; i++)
	{
		percent[i] = (this._transported_list[i] * 100 / target).tointeger();
		if(percent[i] < 100 && percent[i] > best_non_finished_percent)
		{
			best_non_finished_percent = percent[i];
		}

		Log.Info("percent[" + i + "]:" + percent[i] + "  completed: " + this._completed_list[i], Log.LVL_DEBUG);
		if( (percent[i] >= 100) && (this._completed_list[i] != 1) )
		{
			// Found a completed cargo that haven't been communicated yet
			Log.Info("Cargo " + GSCargo.GetCargoLabel(this._goal_ptr._cargo_list[i]) + " was found to be completed, but not yet communicated", Log.LVL_SUB_DECISIONS);

			// Send one CargoCompleted command to AIs per completed cargo.
			g_scp_manager.SendCargoCompleted(this._company_id, this._goal_ptr._cargo_list[i]);

			// Human is not told about individual completed cargos, instead it is informed about completed medals

			// Update _completed_list
			this._completed_list[i] = 1;
		}
	}

	// Create goals for the goal GUI
	for(local i = 0; i < 3; i++)
	{
		local should_exist = i == medal - 1;
		local exists = GSGoal.IsValidGoal(this._goal_list[i]);

		if(exists && !should_exist)
		{
			// Reached a medal
			if(!reached_medal_news_shown)
			{
				// Show news about the new medal 
				// (if two medals have been reached within a month, only the best one is shown)
				this.ShowReachedMedalNews();
				reached_medal_news_shown = true;
			}
		}

		Log.Info(GSCompany.GetName(this._company_id) + " - medal: " + medal_debug_name[i] + "  exist: " + exists + "  should exist: " + should_exist, Log.LVL_DEBUG);

		// Exist, but should be removed?
		//if(exists && !should_exist)
		// always remove -> update progress
		{
			Log.Info("remove goal " + medal_debug_name[i]);
			GSGoal.Remove(this._goal_list[i]);
			this._goal_list[i] = -1; // set to a goal ID that OpenTTD will never pick
		}

		// Doesn't exist, but should exist?
		if(/*!exists &&*/ should_exist)
		{
			Log.Info("add goal " + medal_debug_name[i]);
			this._goal_list[i] =
				GSGoal.New(this._company_id, GSText(GSText.STR_GOAL_COMPANY_GOLD + i, best_non_finished_percent), GSGoal.GT_NONE, 0);
		}
	}
}
function CompanyData::UpdateGoals_Trunk()
{
	local medal = this.GetMedal();
	local reached_medal = false;
	local medal_debug_name = ["Gold", "Silver", "Bronze"];

	// Percent accomplished
	local target = GSController.GetSetting("transport_target");
	local percent = this.GetPercentTransported();

	// Create/update the goals for the Goal GUI
	for(local i = 0; i < 3; i++)
	{
		Log.Info("percent[" + i + "]:" + percent[i] + "  completed: " + this._completed_list[i], Log.LVL_DEBUG);
		if(!GSGoal.IsValidGoal(this._cargo_goal_list[i]))
		{
			this._cargo_goal_list[i] =
				GSGoal.New(this._company_id, GSText(GSText.STR_CARGO_GOAL, this._goal_ptr._cargo_list[i], target, this._goal_ptr.GetEndYear()), GSGoal.GT_NONE, 0);
		}

		GSGoal.SetProgress(this._cargo_goal_list[i], GSText(GSText.STR_GOAL_PROGRESS, percent[i]));

		if(!GSGoal.IsCompleted(this._cargo_goal_list[i]) && percent[i] >= 100) {
			GSGoal.SetCompleted(this._cargo_goal_list[i], true);

			// Found a completed cargo that haven't been communicated yet
			Log.Info("Cargo " + GSCargo.GetCargoLabel(this._goal_ptr._cargo_list[i]) + " was found to be completed, but not yet communicated", Log.LVL_SUB_DECISIONS);

			// Send one CargoCompleted command to AIs per completed cargo.
			g_scp_manager.SendCargoCompleted(this._company_id, this._goal_ptr._cargo_list[i]);

			// Human is not told about individual completed cargos, instead it is informed about completed medals
			reached_medal = true;

			// Update _completed_list
			this._completed_list[i] = 1;
		}
	}

	if(reached_medal) {
		// Show news to humans once if one or more medals have been reached.
		this.ShowReachedMedalNews();
	}
}

/* returns GOLD, SILVER, BRONZE or NO_MEDAL */
function CompanyData::GetMedal()
{
	local num_achived = 0;
	for(local i = 0; i < 3; i++)
	{
		if(this._transported_list[i] > GSController.GetSetting("transport_target"))
		{
			num_achived++;
		}
	}

	switch(num_achived)
	{
		case 3: return GOLD;
		case 2: return SILVER;
		case 1: return BRONZE;
		default: return NO_MEDAL;
	}
}

function CompanyData::InformAboutGoal()
{
	g_scp_manager.SendCurrentGoal(this._company_id, this._company_id);

	// AIs can't read news, but humans can be on the same company as AIs,
	// so always send news even if the company has registered with the
	// SCP protocol.
	this.AddInfoStoryPages();
}

function CompanyData::AddInfoStoryPages(auto_show = true)
{
	local target = GSController.GetSetting("transport_target");

	local text = GSText(GSText.STR_INTRO);
	for (local i = 0; i < 3; i++) {
		text.AddParam(this._goal_ptr._cargo_list[i]);
		text.AddParam(target);
	}
	text.AddParam(this._goal_ptr.GetEndYear() - 1);
	if (auto_show) {
		Story.ShowMessage(this._company_id, text, GSText(GSText.STR_INTRO_TITLE));
	} else {
		// When auto_show is false, don't show any message, just silently create the story page
		// (if story API exist, otherwise just don't show anything)
		if (Story.IsStoryBookAvailable()) {
			Story.NewStoryPage(this._company_id, GSText(GSText.STR_INTRO_TITLE), [
					[GSStoryPage.SPET_TEXT, 0, text]
			]);
		}
	}

	// On newer OpenTTD versions, show a second page in the story book
	// with the medals explained.
	if (Story.IsStoryBookAvailable()) {
		local p = Story.NewStoryPage(this._company_id, GSText(GSText.STR_GOAL_PAGE_TITLE), [
				[GSStoryPage.SPET_TEXT, 0, GSText(GSText.STR_GOAL_PAGE_P1)],
				[GSStoryPage.SPET_GOAL, this._cargo_goal_list[0], null],
				[GSStoryPage.SPET_GOAL, this._cargo_goal_list[1], null],
				[GSStoryPage.SPET_GOAL, this._cargo_goal_list[2], null],
				[GSStoryPage.SPET_TEXT, 0, GSText(GSText.STR_GOAL_PAGE_P2)],
		]);
		Log.Info("Second page: " + p);
	}
}

// Helper for AddCompanyLeaguePage
function NumColorText(percent)
{ 
	local string = percent >= 100 ? GSText.STR_GREEN : GSText.STR_BLACK;
	return GSText(string);
}

// Updates the league table in the story book
function CompanyData::AddCompanyLeaguePage(league_table, year, show_page)
{
	local elements = [];
	Log.Info("elements array start len: " + elements.len(), Log.LVL_DEBUG);
	// company items:
	for(local i = 0; i < league_table.len(); i++)
	{
		local item = league_table[i];
		assert(item.medal >= GOLD && item.medal <= NO_MEDAL);
		elements.append([GSStoryPage.SPET_TEXT, 0, GSText(GSText.STR_LEAGUE_PAGE_P_COMPANY,
					item.company_id,
					GSText(GSText.STR_STORY_MEDAL_GOLD + item.medal),
					NumColorText(item.percent[0]), item.percent[0],
					NumColorText(item.percent[1]), item.percent[1],
					NumColorText(item.percent[2]), item.percent[2]
		)]);
		Log.Info("company element: " + GSCompany.GetName(item.company_id) + " medal: " + item.medal + " percent: " + item.percent[0] + ", " + item.percent[1] + ", " + item.percent[2], Log.LVL_DEBUG);
	}
	// page element for cargo key:
	elements.append([GSStoryPage.SPET_TEXT, 0, GSText(GSText.STR_LEAGUE_PAGE_P_INFO,
				1 << this._goal_ptr._cargo_list[0], 1 << this._goal_ptr._cargo_list[1], 1 << this._goal_ptr._cargo_list[2])]);

	local page = Story.NewStoryPage(this._company_id, GSText(GSText.STR_LEAGUE_PAGE_TITLE, year), elements);
	if(show_page) GSStoryPage.Show(page);
}

function CompanyData::TrackProgress()
{
	local town_list = GSTownList();
	local industry_list = GSIndustryList();

	for(local i = 0; i < 3; i++)
	{
		local cargo = this._goal_ptr._cargo_list[i];
		local transported = 0;

		// Scan all towns and industries for delivery of cargo since last check

		// Todo: do some optimization and only monitor towns/industries that can accept the given cargo
		foreach(town_id, _ in town_list)
		{
			transported += GSCargoMonitor.GetTownDeliveryAmount(this._company_id, cargo, town_id, true);
		}
		// Fix: Don't monitor both towns and industries. Even cargoes like Coal are registered by towns.
		/*foreach(industry_id, _ in industry_list)
		{
			transported += GSCargoMonitor.GetIndustryDeliveryAmount(this._company_id, cargo, industry_id, true);
		}
		*/

		Log.Info("transported: " + transported + " (" + GSCargo.GetCargoLabel(cargo) + ")", Log.LVL_DEBUG);

		this._transported_list[i] += transported;
	}

	// Update goals
	this.UpdateGoals();
	this.UpdateHQSign();

	// Show monthly progress
	if(GSController.GetSetting("monthly_progress") == 1) // should humans get monthly progress?
	{
		this.ShowProgressNews();
	}
	Log.Info("Monthly report for AI " + GSCompany.GetName(this._company_id) + " = " + this._ai_monthly_report, Log.LVL_DEBUG);
	if(this._ai_monthly_report == true || this._ai_monthly_report == 1) // do AI want a monthly report?
	{
		Log.Info("Send monthly report to AI " + GSCompany.GetName(this._company_id), Log.LVL_SUB_DECISIONS);
		g_scp_manager.SendCurrentGoal(this._company_id, this._company_id);
	}
}

function CompanyData::SetAIMonthlyReport(new_value)
{
	// Allow both 1 and true or 0 and false.
	if(new_value == 1) new_value = true;
	if(new_value == 0) new_value = false;
	if(new_value == true || new_value == false)
	{
		this._ai_monthly_report = new_value;
	}
}
function CompanyData::GetAIMonthlyReport()
{
	return this._ai_monthly_report;
}

function CompanyData::UpdateHQSign()
{
	// Todo, once basic functionality is done
	local hq_tile = GSCompany.GetCompanyHQ(this._company_id);

	// Put the sign on the south tile of the HQ
	local hq_sign_tile = Direction.GetAdjacentTileInDirection(
			GSCompany.GetCompanyHQ(this._company_id),
			Direction.DIR_S);

	local hq_exist = GSMap.IsValidTile(hq_tile) && GSMap.IsValidTile(hq_sign_tile);
	if(this._hq_sign != null && !hq_exist)
	{
		// Sign exist, but no HQ => remove sign
		GSSign.RemoveSign(this._hq_sign);
		this._hq_sign = null;
	}
	else if(hq_exist)
	{
		// The HQ exist
		local medal = this.GetMedal();
		assert(medal >= GOLD && medal <= NO_MEDAL);
		local sign_text = GSText(GSText.STR_HQ_MEDAL_GOLD + medal);
		if(this._hq_sign == null)
		{
			// HQ exist, but no sign yet => create new sign
			this._hq_sign = GSSign.BuildSign(hq_sign_tile, sign_text);
		}
		else
		{
			// HQ exist as well as a sign
			if(GSSign.GetLocation(this._hq_sign) == hq_sign_tile)
			{
				// The sign is at the right location => update only the text contents
				GSSign.SetName(this._hq_sign, sign_text);
			}
			else
			{
				// The sign exist, but at the wrong tile
				GSSign.RemoveSign(this._hq_sign);
				this._hq_sign = GSSign.BuildSign(hq_sign_tile, sign_text);
			}
		}
	}
	else
	{
		// No HQ and no old sign
	}
}

function CompanyData::GameEnd()
{
	local medal = this.GetMedal();
	assert(medal >= GOLD && medal <= NO_MEDAL);
	local text = GSText(GSText.STR_GAME_END_GOLD + medal, 
			this._goal_ptr._cargo_list[0],
			this._transported_list[0],
			this._goal_ptr._cargo_list[1],
			this._transported_list[1],
			this._goal_ptr._cargo_list[2]
			this._transported_list[2]
	);
	local page = Story.ShowMessage(this._company_id, text, GSText(GSText.STR_GAME_END_TITLE));
	if (Story.IsStoryBookAvailable()) GSStoryPage.Show(page);
}

function CompanyData::EndOfYear(year)
{
	// yearly progress?
	if(GSController.GetSetting("monthly_progress") == 0)
		this.ShowProgressNews();
}
function CompanyData::GetTimeLeftText()
{
	// Time left is displayed as 'years'+'days' if it is more than one or 'days' if it is the last year.
	local time_left = this._goal_ptr.GetTimeLeft();
	local time_left_string_id = time_left.years >= 1? GSText.STR_YEARS_LEFT : GSText.STR_DAYS_LEFT;
	local time_left_text = GSText(time_left_string_id);
	if (time_left.years >= 1) time_left_text.AddParam(time_left.years);
	time_left_text.AddParam(time_left.days);
	return time_left_text;
}
function CompanyData::ShowProgressNews()
{
	local target = GSController.GetSetting("transport_target")
	local percent = [0, 0, 0];
	for(local i = 0; i < 3; i++)
		percent[i] = (this._transported_list[i] * 100 / target).tointeger();

	// Create news string and add parameters
	local news_text = GSText(GSText.STR_GOAL_PROGRESS_NEWS);
	for(local i = 0; i < 3; i++)
	{
		news_text.AddParam(1 << this._goal_ptr._cargo_list[i]);
		news_text.AddParam(this._goal_ptr._cargo_list[i]);
		news_text.AddParam(this._transported_list[i]);
		news_text.AddParam(percent[i]);
	}

	// Time left is displayed as 'years'+'days' if it is more than one or 'days' if it is the last year.
	local time_left_text = this.GetTimeLeftText();
	news_text.AddParam(time_left_text);

	// Show news
	GSNews.Create(GSNews.NT_GENERAL, news_text, this._company_id);
}

function CompanyData::GetProgressReportTable()
{
	local table = { medal = MedalToStr(this.GetMedal()), cargo = [] };

	local target = GSController.GetSetting("transport_target")
	for(local i = 0; i < 3; i++)
	{
		local cargo_tbl = {
			cargo_id = this._goal_ptr._cargo_list[i],
			transported = this._transported_list[i],
			target = target
		};
		
		table.cargo.append(cargo_tbl);
	}

	return table;
}

function CompanyData::ShowReachedMedalNews()
{
	local medal = this.GetMedal();
	assert(medal != NO_MEDAL);
	local text = GSText(GSText.STR_GOAL_REACHED_GOLD + medal, this._goal_ptr.GetEndYear());
	local page = Story.ShowMessage(this._company_id, text, GSText(GSText.STR_GOAL_REACHED_TITLE));
	if (Story.IsStoryBookAvailable()) GSStoryPage.Show(page);
}

function CompanyData::SaveToTable()
{
	return {
		company_id = this._company_id,
		goal_list = this._goal_list,
		cargo_goal_list = this._cargo_goal_list,
		transported_list = this._transported_list,
		completed_list = this._completed_list
		hq_sign = this._hq_sign,
	};
}

/* static */ function CompanyData::CreateFromTable(table, version, goal_ptr, create_story_pages)
{
	local result = CompanyData(table.company_id, goal_ptr);

	result._goal_list = table.goal_list;
	result._transported_list = table.transported_list;
	result._hq_sign = table.hq_sign;

	// Remove the old goals if they exist and Story Book is available
	if(Story.IsStoryBookAvailable()) {
		for(local i = 0; i < 3; i++) {
			GSGoal.Remove(result._goal_list[i]);
			result._goal_list[i] = -1;
		}
	}

	if(table.rawin("completed_list"))
	{
		result._completed_list = table.completed_list;
	}
	else
	{
		// if completed_list doesn't exist in the save, assume that all accomplished
		// goals have been communicated.
		local target = GSController.GetSetting("transport_target");
		for(local i = 0; i < 3; i++)
		{
			result._completed_list[i] = result._transported_list[i] >= target;
			Log.Info("completed cargo " + i + ": " + result._completed_list[i], Log.LVL_INFO);
		}
	}

	if(table.rawin("cargo_goal_list"))
	{
		result._cargo_goal_list = table.cargo_goal_list;
	}

	if(Story.IsStoryBookAvailable())
	{
		// Remove old non-story goals
		for(local i = 0; i < 3; i++) {
			GSGoal.Remove(result._goal_list[i]);
			result._goal_list[i] = -1;
		}

		// Create StoryBook edition of goals (if they don't exist)
		result.UpdateGoals();
	}

	if(create_story_pages) result.AddInfoStoryPages(false);

	return result;
}
