
//------------------------------------------------------------//
//                                                            //
//   Class for storage of global goal data                    //
//                                                            //
//------------------------------------------------------------//

class GoalData
{
	_cargo_list = null; // index = 0, 1, 2 for the three cargoes (with lowest cargo ID first)
	_min_transported = 0;
	_start_year = 0;
	_goal_list = null; // index = 0, 1, 2 for the three cargoes. Contains the global (non company specific GSGoal GoalIDs)

	constructor()
	{
		this._cargo_list = [-1, -1, -1];
		this._min_transported = 0;
		this._start_year = 0;
		this._goal_list = [-1, -1, -1];
	}

	// pick cargoes
	function InitalizeGoals();
	function EndOfYear();

	function GetEndYear();
	function GetDaysLeft(); // give or take

	// If end year is 1957, and curr year is 1956, the result is 0
	// in 1957 the result will remain at 0
	function GetYearsLeft();

	// return {years, days}
	function GetTimeLeft();

	// For save/load
	function SaveToTable();
	static function CreateFromTable(table, version);
}

// Returns a GSList containing the three goal cargoes

function GoalData::PickGoalCargos()
{
	local raw_cargo_list = GSList();
	local processed_cargo_list = GSList();

	local industry_type_list = GSIndustryTypeList();
	foreach(ind_type, _ in industry_type_list)
	{
		if (GSIndustryType.IsRawIndustry(ind_type)) {
			raw_cargo_list.AddList(GSIndustryType.GetProducedCargo(ind_type));
		}

		if (GSIndustryType.IsProcessingIndustry(ind_type)) {
			processed_cargo_list.AddList(GSIndustryType.GetProducedCargo(ind_type));
		}
	}

	// Remove the raw cargoes from the producing list, unless that cause
	// all cargoes to be removed.
	{
		local tmp_list = GSList();
		tmp_list.AddList(processed_cargo_list);
		tmp_list.RemoveList(raw_cargo_list);
		if (!tmp_list.IsEmpty()) {
			processed_cargo_list = tmp_list;
		}
	}

	// add town cargo to raw_cargo_list
	raw_cargo_list.AddList(Helper.GetTownProducedCargoList())

	Log.Info("All raw cargoes: ", Log.LVL_INFO);
	foreach(c, _ in raw_cargo_list)
		Log.Info("    " + GSCargo.GetCargoLabel(c), Log.LVL_INFO);
	Log.Info("All processed cargoes: ", Log.LVL_INFO);
	foreach(c, _ in processed_cargo_list)
		Log.Info("    " + GSCargo.GetCargoLabel(c), Log.LVL_INFO);


	// Pick 3 cargoes
	local result = GSList();

	raw_cargo_list.Valuate(GSBase.RandItem);
	raw_cargo_list.Sort(GSList.SORT_BY_VALUE, GSList.SORT_ASCENDING);
	processed_cargo_list.Valuate(GSBase.RandItem);
	processed_cargo_list.Sort(GSList.SORT_BY_VALUE, GSList.SORT_ASCENDING);

	// Add a random raw + processing cargo 
	local log_str = "";
	if (!raw_cargo_list.IsEmpty()) {
		local raw_cargo = raw_cargo_list.Begin();
		result.AddItem(raw_cargo, 0);
		processed_cargo_list.RemoveItem(raw_cargo);

		log_str += GSCargo.GetCargoLabel(raw_cargo) + " (raw)";
	}
	if (!processed_cargo_list.IsEmpty()) {
		local processed_cargo = processed_cargo_list.Begin();
		result.AddItem(processed_cargo, 0);

		if (log_str.len() > 0) log_str += ", ";
		log_str += GSCargo.GetCargoLabel(processed_cargo) + " (processed)";
	}

	// Fill up with random cargos to a total of three cargos. (the method above doesn't guarantee that
	// two cargoes are picked)
	local cargo_list = GSCargoList();
	cargo_list.RemoveList(result);
	cargo_list.Valuate(GSBase.RandItem);
	cargo_list.KeepTop(3 - result.Count());
	result.AddList(cargo_list);
	foreach(c, _ in cargo_list) {
		if (log_str.len() > 0) log_str += ", ";
		log_str += GSCargo.GetCargoLabel(c) + " (random)";
	}
	Log.Info("Picked goal cargoes: " + log_str, Log.LVL_INFO);

	return result;
}

function GoalData::InitalizeGoals()
{
	local cargo_list = this.PickGoalCargos();

	// Sort them by cargo ID
	cargo_list.Sort(GSList.SORT_BY_ITEM, GSList.SORT_ASCENDING);

	// Store the cargo IDs in this._cargo_list
	local i = 0;
	local cargo_list_str = "";
	foreach(cargo, _ in cargo_list)
	{
		this._cargo_list[i] = cargo;

		if(cargo_list_str != "") cargo_list_str += ", ";
		cargo_list_str += GSCargo.GetCargoLabel(cargo);
		i++;
	}

	// save the start year of the game
	this._start_year = GSDate.GetYear(GSDate.GetCurrentDate());

	// create global goals for the GUI
	this.CreateGUIGoals();


	// log some info about goal target
	Log.Info("Target value: " + GSController.GetSetting("transport_target"), Log.LVL_INFO);
	Log.Info("By year: " + this.GetEndYear());
}

function GoalData::CreateGUIGoals()
{
	local transport_target = GSController.GetSetting("transport_target");
	local ALL_COMPANIES = GSCompany.COMPANY_INVALID;

	if(Story.IsStoryBookAvailable()) {
		/* Use slimed global goal when the story book is available */
		local text = GSText(GSText.STR_GOAL_GLOBAL, 
				1 << this._cargo_list[0],
				1 << this._cargo_list[1],
				1 << this._cargo_list[2],
				this.GetEndYear() - 1);
		this._goal_list[0] = GSGoal.New(ALL_COMPANIES, text, GSGoal.GT_NONE, 0);
		this._goal_list[1] = -1;
		this._goal_list[2] = -1;
	} else {
		for(local i = 0; i < 3; i++)
		{
			// The first goal contain parameters for the cargo goals
			local text = GSText(GSText.STR_GOAL_GLOBAL_GOLD + i);
			if(i == 0)
			{
				for(local j = 0; j < 3; j++)
				{
					text.AddParam(this._cargo_list[j]);
					text.AddParam(transport_target);
				}
			}
			// and all have the game end year at the end.
			text.AddParam(this.GetEndYear() - 1);
			this._goal_list[i] = GSGoal.New(ALL_COMPANIES, text, GSGoal.GT_NONE, 0);
		}
	}
}

function GoalData::GetEndYear()
{
	return this._start_year + GSController.GetSetting("play_years");
}

function GoalData::GetDaysLeft() 
{
	local date = GSDate.GetCurrentDate();
	local end_year = this.GetEndYear();

	local end_date = GSDate.GetDate(end_year, 1, 1);

	return max(0, end_date - date);
}

function GoalData::GetYearsLeft()
{
	local date = GSDate.GetCurrentDate();
	local curr_year = GSDate.GetYear(date);
	local end_year = this.GetEndYear();

	return max(0, end_year - curr_year - 1);
}

function GoalData::GetTimeLeft()
{
	local date = GSDate.GetCurrentDate();
	local curr_year = GSDate.GetYear(date);
	local end_year = this.GetEndYear();
	local curr_year_end_date = GSDate.GetDate(curr_year + 1, 1, 1) - 1;

	local result = {
		years = max(0, end_year - curr_year - 1),
		days  = max(0, curr_year_end_date - date + 1)
	};

	if (result.days >= 365) { result.years++; result.days -= 365; }

	return result;
}

function GoalData::SaveToTable()
{
	return {
		cargo_list = this._cargo_list,
		start_year = this._start_year,
		goal_list = this._goal_list
	};
}

/*static*/ function GoalData::CreateFromTable(table, version)
{
	local result = GoalData();
	result._cargo_list = table.cargo_list;
	result._start_year = table.start_year;

	if(table.rawin("goal_list")) { // not available in version < 3.
		result._goal_list = table.goal_list;

		/* Has old global goals, but story book is available? */
		if (GSGoal.IsValidGoal(result._goal_list[2]) && Story.IsStoryBookAvailable()) {
			/* Migrate from old goals to new slimed global goals */
			for (local i = 0; i < 3; i++) {
				GSGoal.Remove(result._goal_list[i]);
				result._goal_list[i] = -1;
			}
			result.CreateGUIGoals();
		}
	} else {
		result.CreateGUIGoals(); // create the global goals that were not available in < 3
	}

	return result;
}
