/*
 * This file is part of MinimalGS, which is a GameScript for OpenTTD
 * Copyright (C) 2012  Leif Linse
 *
 * MinimalGS is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License
 *
 * MinimalGS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MinimalGS; If not, see <http://www.gnu.org/licenses/> or
 * write to the Free Software Foundation, Inc., 51 Franklin Street, 
 * Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/*** Libraries ***/

/* Import SuperLib for GameScript */
import("util.superlib", "SuperLib", 35);
Result <- SuperLib.Result;
Log <- SuperLib.Log;
Helper <- SuperLib.Helper;
ScoreList <- SuperLib.ScoreList;
Tile <- SuperLib.Tile;
Direction <- SuperLib.Direction;
Town <- SuperLib.Town;
Industry <- SuperLib.Industry;
Story <- SuperLib.Story;

/* Import SCP */
import("Library.SCPLib", "SCPLib", 45);

/* Import other libraries */
import("queue.fibonacci_heap", "FibonacciHeap", 2);

/*** GS includes ***/
require("version.nut"); // get SELF_VERSION

require("goal.nut");
require("companydata.nut");
require("scp.nut");


// Goals
GOLD <- 0;
SILVER <- 1;
BRONZE <- 2;

NO_MEDAL <- 3;

// Use this pointer to access the SCPManager instance that wraps up the SCP library
// with high level communication methods.
g_scp_manager <- null;


function MedalToStr(medal)
{
	switch(medal)
	{
		case GOLD: return "gold";
		case SILVER: return "silver";
		case BRONZE: return "bronze";
		case NO_MEDAL: return "no medal";
	}

	return "error";
}

class MainClass extends GSController 
{
	_loaded_data = null;
	_loaded_from_version = null;

	_company_list = null;
	_goal_data = null;

	constructor()
	{
		this._loaded_data = null;
		this._loaded_from_version = null;

		this._company_list = [];
		this._goal_data = null; // created by Init
	}

	// Return the company data class instance for a company or null if the company doesn't exist
	// in our in-memory list of companies.
	function GetCompanyData(company_id);
}

function MainClass::Start()
{
	this.Init();

	// If we load a game which has been continued past the end year,
	// don't run past this point. NoCarGoal has not been written to
	// send useful news etc. after the end date.
	{
		local year = GSDate.GetYear(GSDate.GetCurrentDate());
		if(year >= this._goal_data.GetEndYear()) {
			while (true) { GSController.Sleep(5); }
		}
	}

	Log.Info("Setup done", Log.LVL_INFO);

	// Wait for the game to start
	this.Sleep(1);

	local last_goal_check = GSDate.GetCurrentDate();
	local last_loop_year = GSDate.GetYear(GSDate.GetCurrentDate());
	while (true) {
		local loop_start_tick = GSController.GetTick();

		// Handle events
		this.HandleEvents();

		// Reached end of year?
		local current_date = GSDate.GetCurrentDate();
		local year = GSDate.GetYear(current_date);
		if(year != last_loop_year)
		{
			// reached end of challenge?
			if(year == this._goal_data.GetEndYear())
			{
				this.GameEnd();
			}
			else if(year < this._goal_data.GetEndYear())
			{
				this.EndOfYear(year - 1);
			}

			last_loop_year = year;
		}
	
		// Once a month
		if(current_date > last_goal_check + 30)
		{
			last_goal_check = current_date;

			// Check goals setup once a month to fix problems when events has been lost
			this.UpdateCompanyList();

			// Has any goals been fulfilled?
			GSLog.Info("Scan goals START");
			this.ScanGoals();
			GSLog.Info("Scan goals END");

			// Send a monthly report to Admin port
			this.SendReportToAdminPort();
		}


		// Check for incoming SCP messages
		Log.Info("SCP.Check", Log.LVL_SUB_DECISIONS);
		for(local s=0; s<20 && g_scp_manager.Check(); s++) {};

		// Loop with a frequency of five days
		local ticks_used = GSController.GetTick() - loop_start_tick;
		this.Sleep(Helper.Max(1, 5 * 74 - ticks_used));
	}
}

function MainClass::Init()
{
	// Setup SCPManager
	g_scp_manager = SCPManager(this);
	

	if (this._loaded_data != null)
	{
		// Copy loaded data from this._loaded_data to this.*
		// or do whatever with the loaded data

		// in the event that current OpenTTD version support story book, but the save was
		// saved by a version which didn't, we need to detect this and create pages.
		local create_pages_for_loaded_companies = 
				Story.IsStoryBookAvailable() && 
				this._loaded_data.company_list.len() > 0 &&
				!GSStoryPage.IsValidStoryPage(0);
		if (create_pages_for_loaded_companies) Log.Info("Loaded game have no story pages, so create pages for existing companies.", Log.LVL_INFO);

		Log.Info((this._loaded_data.company_list.len() > 0) + " " + GSStoryPage.IsValidStoryPage(0), Log.LVL_INFO);

		// load goal data
		this._goal_data = GoalData.CreateFromTable(this._loaded_data.goal_data, this._loaded_from_version);

		// Load company data from loaded data
		foreach(_, company_table in this._loaded_data.company_list)
		{
			if(company_table != null)
			{
				Log.Info("Loading data for company " + GSCompany.GetName(company_table.company_id), Log.LVL_INFO);
				this._company_list.append(
					CompanyData.CreateFromTable(company_table,
						this._loaded_from_version,
						this._goal_data,
						create_pages_for_loaded_companies
					)
				);
			}
		}

		this._loaded_data = null; // don't attempt to load again
	}
	else
	{
		// New game
		this._goal_data = GoalData();
		this._goal_data.InitalizeGoals(); // select goal cargoes
	}

	// Add possible new companies
	this.UpdateCompanyList();

	// Start up cargo monitoring by reading from
	// all cargo monitors.
	this.ScanGoals();
}

function MainClass::HandleEvents()
{
	if(GSEventController.IsEventWaiting())
	{
		local ev = GSEventController.GetNextEvent();

		if(ev == null)
			return;

		local ev_type = ev.GetEventType();
		if(ev_type == GSEvent.ET_COMPANY_NEW ||
				ev_type == GSEvent.ET_COMPANY_BANKRUPT ||
				ev_type == GSEvent.ET_COMPANY_MERGER)
		{
			Log.Info("A company was created/bankrupt/merged => update company list", Log.LVL_INFO);

			// Update the goal list when:
			// - a new company has been created
			// - a company has gone bankrupt
			// - a company has been bought by another company
			this.UpdateCompanyList();
		}
	}
}

function MainClass::UpdateCompanyList()
{
	// Loop over all possible company IDs
	for(local c = GSCompany.COMPANY_FIRST; c <= GSCompany.COMPANY_LAST; c++)
	{
		// has goals already been set up for this company?
		local existing = null;
		local existing_idx = 0;
		foreach(company_data in this._company_list)
		{
			if(company_data._company_id == c)
			{
				existing = company_data;
				break;
			}
			existing_idx++;
		}

		// does the company exist in the game
		if(GSCompany.ResolveCompanyID(c) == GSCompany.COMPANY_INVALID)
		{
			if(existing != null)
			{
				// Remove data for no longer existing company
				this._company_list.remove(existing_idx);
			}
			continue;
		}
	
		// If the company can be resolved and exist (goals has been setup) => don't do anything
		if(existing != null) continue;

		// Company goals has not yet been setup for this company
		local company_data = CompanyData(c, this._goal_data);
		company_data.CreateGoals();
		company_data.InformAboutGoal();
		company_data.TrackProgress();

		this._company_list.append(company_data);
	}
}

function MainClass::ScanGoals()
{
	foreach(company_data in this._company_list)
	{
		company_data.TrackProgress();
	}
}

/*
 * Adds a company league table to the story book of each company
 */
function MainClass::AddCompanyLeaguePages(show_page, year)
{
	if(!Story.IsStoryBookAvailable()) {
		Log.Info("Company league table is not shown because your OpenTTD version doesn't seem to support Story Pages", Log.LVL_INFO);
		return;
	}

	local heap = FibonacciHeap();

	foreach(company_data in this._company_list)
	{
		local percent = company_data.GetPercentTransported();
		local item = {
			company_id = company_data._company_id,
			medal = company_data.GetMedal(),
			percent = percent,
		};
		local secondary_order = 0;
		if (item.medal == GOLD) {
			// This overflows the 100 range allocated for secondary order, however
			// GOLD should have the lowest sort value. So by inverting the score it
			// will not alter the medal sort.
			secondary_order = -1 * (percent[0] + percent[1] + percent[2]);
		} else {
			// Get best non-completed percent
			local best_percent = 0;
			for(local i = 0; i < 3; i++) {
				if(percent[i] < 100 && percent[i] > best_percent) {
					best_percent = percent[i];
				}
			}
			secondary_order = 100 - best_percent;
		}
			
		// Order by how near gold medal the company is. If there is a tie, order by company id.
		local order_value = item.medal * 3000 + secondary_order * 20 + item.company_id;
		heap.Insert(item, order_value);
	}

	local league_table = [];
	while(heap.Count() > 0)
	{
		local item = heap.Pop();
		league_table.append(item);
	}

	Log.Info("Add league table page for all companies", Log.LVL_INFO);
	foreach(company_data in this._company_list)
	{
		company_data.AddCompanyLeaguePage(league_table, year, show_page);
	}
}

// Called when the end of game has been reached
function MainClass::GameEnd()
{
	Log.Info("Game End", Log.LVL_INFO);

	// Show End of Game message for each company
	foreach(company_data in this._company_list)
	{
		company_data.GameEnd();
	}

	// Add the final league table after the end of game page
	local year = this._goal_data.GetEndYear() - 1;
	this.AddCompanyLeaguePages(false, year);

	// Pause the game 
	if (GSController.GetSetting("pause_at_game_end") == 1) {
		GSGame.Pause();

		// Unpause when the pause_at_game_end is toggled off
		while (GSController.GetSetting("pause_at_game_end") != 0) { GSController.Sleep(5); }
		GSGame.Unpause();
	}

	// Disable ourself as NoCarGoal has not been designed to handle time past the game end.
	while (true) { GSController.Sleep(5); }
}

// Called at end of year (but not when the game end has been reached)
// @param year The year that just ended
function MainClass::EndOfYear(year)
{
	Log.Info("End Of Year", Log.LVL_INFO);

	this.AddCompanyLeaguePages(true, year);

	foreach(company_data in this._company_list)
	{
		company_data.EndOfYear(year);
	}
}

function MainClass::Save()
{
	Log.Info("Saving data to savegame", Log.LVL_INFO);

	// If Init() has not finished loading data from save,
	// then save the data that was loaded from save.
	if (this._loaded_data != null) return this._loaded_data;

	local company_save_list = [];
	foreach(company_data in this._company_list)
	{
		company_save_list.append(company_data.SaveToTable());
	}

	return { 
		goal_data = this._goal_data.SaveToTable(),
		company_list = company_save_list
	};
}

function MainClass::Load(version, tbl)
{
	Log.Info("Loading data from savegame made with version " + version + " of the game script", Log.LVL_INFO);

	if(version > SELF_VERSION)
	{
		Log.Warning("Warning: Loading from a newer version of TransportGoals", Log.LVL_INFO);
	}

	// Store a copy of the table from the save game
	// but do not process the loaded data yet. Wait with that to Init
	// so that OpenTTD doesn't kick us for taking too long to load.
	this._loaded_data = {}
   	foreach(key, val in tbl)
	{
		this._loaded_data.rawset(key, val);
	}	
	this._loaded_from_version = version;
}

function MainClass::SendReportToAdminPort()
{
	local table = { 
		days_left = this._goal_data.GetDaysLeft(), 
		date_sent = GSDate.GetCurrentDate(),
		company_list = []
	};

	foreach(company in this._company_list)
	{
		table.company_list.append(company.GetProgressReportTable());
	}

	if (GSAdmin.Send(table))
	{
		Log.Info("Sent monthly report to admin port", Log.LVL_SUB_DECISIONS);
	}
}

// public function
function MainClass::GetCompanyData(company_id)
{
	foreach(company in this._company_list)
	{
		if(company._company_id == company_id) return company;
	}

	return null;
}

