/* $Id: core.cpp 8459 2007-01-30 17:12:46Z rubidium $ */

#ifdef ENABLE_NETWORK

#include "../../stdafx.h"
#include "../../debug.h"
#include "os_abstraction.h"
#include "core.h"
#include "packet.h"

/**
 * @file core.cpp Functions used to initialize/shut down the core network
 */

#ifdef __MORPHOS__
/* the library base is required here */
struct Library *SocketBase = NULL;
#endif

/**
 * Initializes the network core (as that is needed for some platforms
 * @return true if the core has been initialized, false otherwise
 */
bool NetworkCoreInitialize(void)
{
#if defined(__MORPHOS__) || defined(__AMIGA__)
	/*
	 *  IMPORTANT NOTE: SocketBase needs to be initialized before we use _any_
	 *  network related function, else: crash.
	 */
	DEBUG(net, 3, "[core] loading bsd socket library");
	SocketBase = OpenLibrary("bsdsocket.library", 4);
	if (SocketBase == NULL) {
		DEBUG(net, 0, "[core] can't open bsdsocket.library version 4, network unavailable");
		return false;
	}

#if defined(__AMIGA__)
	/* for usleep() implementation (only required for legacy AmigaOS builds) */
	TimerPort = CreateMsgPort();
	if (TimerPort != NULL) {
		TimerRequest = (struct timerequest*)CreateIORequest(TimerPort, sizeof(struct timerequest);
		if (TimerRequest != NULL) {
			if (OpenDevice("timer.device", UNIT_MICROHZ, (struct IORequest*)TimerRequest, 0) == 0) {
				TimerBase = TimerRequest->tr_node.io_Device;
				if (TimerBase == NULL) {
					/* free ressources... */
					DEBUG(net, 0, "[core] can't initialize timer, network unavailable");
					return false;
				}
			}
		}
	}
#endif // __AMIGA__
#endif // __MORPHOS__ / __AMIGA__

/* Let's load the network in windows */
#ifdef WIN32
	{
		WSADATA wsa;
		DEBUG(net, 3, "[core] loading windows socket library");
		if (WSAStartup(MAKEWORD(2, 0), &wsa) != 0) {
			DEBUG(net, 0, "[core] WSAStartup failed, network unavailable");
			return false;
		}
	}
#endif /* WIN32 */

	return true;
}

/**
 * Shuts down the network core (as that is needed for some platforms
 */
void NetworkCoreShutdown(void)
{
#if defined(__MORPHOS__) || defined(__AMIGA__)
	/* free allocated resources */
#if defined(__AMIGA__)
	if (TimerBase    != NULL) CloseDevice((struct IORequest*)TimerRequest); // XXX This smells wrong
	if (TimerRequest != NULL) DeleteIORequest(TimerRequest);
	if (TimerPort    != NULL) DeleteMsgPort(TimerPort);
#endif

	if (SocketBase != NULL) CloseLibrary(SocketBase);
#endif

#if defined(WIN32)
	WSACleanup();
#endif
}


/**
 * Serializes the GRFIdentifier (GRF ID and MD5 checksum) to the packet
 * @param p   the packet to write the data to
 * @param grf the GRFIdentifier to serialize
 */
void NetworkSocketHandler::Send_GRFIdentifier(Packet *p, const GRFIdentifier *grf)
{
	uint j;
	NetworkSend_uint32(p, grf->grfid);
	for (j = 0; j < sizeof(grf->md5sum); j++) {
		NetworkSend_uint8 (p, grf->md5sum[j]);
	}
}

/**
 * Deserializes the GRFIdentifier (GRF ID and MD5 checksum) from the packet
 * @param p   the packet to read the data from
 * @param grf the GRFIdentifier to deserialize
 */
void NetworkSocketHandler::Recv_GRFIdentifier(Packet *p, GRFIdentifier *grf)
{
	uint j;
	grf->grfid = NetworkRecv_uint32(this, p);
	for (j = 0; j < sizeof(grf->md5sum); j++) {
		grf->md5sum[j] = NetworkRecv_uint8(this, p);
	}
}

#endif /* ENABLE_NETWORK */
