﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/dynamodb/DynamoDB_EXPORTS.h>
#include <aws/dynamodb/model/AttributeValue.h>
#include <aws/dynamodb/model/ReturnValuesOnConditionCheckFailure.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DynamoDB {
namespace Model {

/**
 * <p> Represents a PartiQL statement that uses parameters. </p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ParameterizedStatement">AWS
 * API Reference</a></p>
 */
class ParameterizedStatement {
 public:
  AWS_DYNAMODB_API ParameterizedStatement() = default;
  AWS_DYNAMODB_API ParameterizedStatement(Aws::Utils::Json::JsonView jsonValue);
  AWS_DYNAMODB_API ParameterizedStatement& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DYNAMODB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p> A PartiQL statement that uses parameters. </p>
   */
  inline const Aws::String& GetStatement() const { return m_statement; }
  inline bool StatementHasBeenSet() const { return m_statementHasBeenSet; }
  template <typename StatementT = Aws::String>
  void SetStatement(StatementT&& value) {
    m_statementHasBeenSet = true;
    m_statement = std::forward<StatementT>(value);
  }
  template <typename StatementT = Aws::String>
  ParameterizedStatement& WithStatement(StatementT&& value) {
    SetStatement(std::forward<StatementT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The parameter values. </p>
   */
  inline const Aws::Vector<AttributeValue>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Vector<AttributeValue>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Vector<AttributeValue>>
  ParameterizedStatement& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersT = AttributeValue>
  ParameterizedStatement& AddParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace_back(std::forward<ParametersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional parameter that returns the item attributes for a PartiQL
   * <code>ParameterizedStatement</code> operation that failed a condition check.</p>
   * <p>There is no additional cost associated with requesting a return value aside
   * from the small network and processing overhead of receiving a larger response.
   * No read capacity units are consumed.</p>
   */
  inline ReturnValuesOnConditionCheckFailure GetReturnValuesOnConditionCheckFailure() const {
    return m_returnValuesOnConditionCheckFailure;
  }
  inline bool ReturnValuesOnConditionCheckFailureHasBeenSet() const { return m_returnValuesOnConditionCheckFailureHasBeenSet; }
  inline void SetReturnValuesOnConditionCheckFailure(ReturnValuesOnConditionCheckFailure value) {
    m_returnValuesOnConditionCheckFailureHasBeenSet = true;
    m_returnValuesOnConditionCheckFailure = value;
  }
  inline ParameterizedStatement& WithReturnValuesOnConditionCheckFailure(ReturnValuesOnConditionCheckFailure value) {
    SetReturnValuesOnConditionCheckFailure(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_statement;
  bool m_statementHasBeenSet = false;

  Aws::Vector<AttributeValue> m_parameters;
  bool m_parametersHasBeenSet = false;

  ReturnValuesOnConditionCheckFailure m_returnValuesOnConditionCheckFailure{ReturnValuesOnConditionCheckFailure::NOT_SET};
  bool m_returnValuesOnConditionCheckFailureHasBeenSet = false;
};

}  // namespace Model
}  // namespace DynamoDB
}  // namespace Aws
