﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/panorama/PanoramaRequest.h>
#include <aws/panorama/Panorama_EXPORTS.h>
#include <aws/panorama/model/JobResourceTags.h>
#include <aws/panorama/model/PackageImportJobInputConfig.h>
#include <aws/panorama/model/PackageImportJobOutputConfig.h>
#include <aws/panorama/model/PackageImportJobType.h>

#include <utility>

namespace Aws {
namespace Panorama {
namespace Model {

/**
 */
class CreatePackageImportJobRequest : public PanoramaRequest {
 public:
  AWS_PANORAMA_API CreatePackageImportJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreatePackageImportJob"; }

  AWS_PANORAMA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A client token for the package import job.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreatePackageImportJobRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An input config for the package import job.</p>
   */
  inline const PackageImportJobInputConfig& GetInputConfig() const { return m_inputConfig; }
  inline bool InputConfigHasBeenSet() const { return m_inputConfigHasBeenSet; }
  template <typename InputConfigT = PackageImportJobInputConfig>
  void SetInputConfig(InputConfigT&& value) {
    m_inputConfigHasBeenSet = true;
    m_inputConfig = std::forward<InputConfigT>(value);
  }
  template <typename InputConfigT = PackageImportJobInputConfig>
  CreatePackageImportJobRequest& WithInputConfig(InputConfigT&& value) {
    SetInputConfig(std::forward<InputConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags for the package import job.</p>
   */
  inline const Aws::Vector<JobResourceTags>& GetJobTags() const { return m_jobTags; }
  inline bool JobTagsHasBeenSet() const { return m_jobTagsHasBeenSet; }
  template <typename JobTagsT = Aws::Vector<JobResourceTags>>
  void SetJobTags(JobTagsT&& value) {
    m_jobTagsHasBeenSet = true;
    m_jobTags = std::forward<JobTagsT>(value);
  }
  template <typename JobTagsT = Aws::Vector<JobResourceTags>>
  CreatePackageImportJobRequest& WithJobTags(JobTagsT&& value) {
    SetJobTags(std::forward<JobTagsT>(value));
    return *this;
  }
  template <typename JobTagsT = JobResourceTags>
  CreatePackageImportJobRequest& AddJobTags(JobTagsT&& value) {
    m_jobTagsHasBeenSet = true;
    m_jobTags.emplace_back(std::forward<JobTagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A job type for the package import job.</p>
   */
  inline PackageImportJobType GetJobType() const { return m_jobType; }
  inline bool JobTypeHasBeenSet() const { return m_jobTypeHasBeenSet; }
  inline void SetJobType(PackageImportJobType value) {
    m_jobTypeHasBeenSet = true;
    m_jobType = value;
  }
  inline CreatePackageImportJobRequest& WithJobType(PackageImportJobType value) {
    SetJobType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An output config for the package import job.</p>
   */
  inline const PackageImportJobOutputConfig& GetOutputConfig() const { return m_outputConfig; }
  inline bool OutputConfigHasBeenSet() const { return m_outputConfigHasBeenSet; }
  template <typename OutputConfigT = PackageImportJobOutputConfig>
  void SetOutputConfig(OutputConfigT&& value) {
    m_outputConfigHasBeenSet = true;
    m_outputConfig = std::forward<OutputConfigT>(value);
  }
  template <typename OutputConfigT = PackageImportJobOutputConfig>
  CreatePackageImportJobRequest& WithOutputConfig(OutputConfigT&& value) {
    SetOutputConfig(std::forward<OutputConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken;
  bool m_clientTokenHasBeenSet = false;

  PackageImportJobInputConfig m_inputConfig;
  bool m_inputConfigHasBeenSet = false;

  Aws::Vector<JobResourceTags> m_jobTags;
  bool m_jobTagsHasBeenSet = false;

  PackageImportJobType m_jobType{PackageImportJobType::NOT_SET};
  bool m_jobTypeHasBeenSet = false;

  PackageImportJobOutputConfig m_outputConfig;
  bool m_outputConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace Panorama
}  // namespace Aws
