﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/RecommendationJobStatus.h>
#include <aws/sagemaker/model/RecommendationStepType.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class ListInferenceRecommendationsJobStepsRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API ListInferenceRecommendationsJobStepsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListInferenceRecommendationsJobSteps"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name for the Inference Recommender job.</p>
   */
  inline const Aws::String& GetJobName() const { return m_jobName; }
  inline bool JobNameHasBeenSet() const { return m_jobNameHasBeenSet; }
  template <typename JobNameT = Aws::String>
  void SetJobName(JobNameT&& value) {
    m_jobNameHasBeenSet = true;
    m_jobName = std::forward<JobNameT>(value);
  }
  template <typename JobNameT = Aws::String>
  ListInferenceRecommendationsJobStepsRequest& WithJobName(JobNameT&& value) {
    SetJobName(std::forward<JobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter to return benchmarks of a specified status. If this field is left
   * empty, then all benchmarks are returned.</p>
   */
  inline RecommendationJobStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(RecommendationJobStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline ListInferenceRecommendationsJobStepsRequest& WithStatus(RecommendationJobStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter to return details about the specified type of subtask.</p> <p>
   * <code>BENCHMARK</code>: Evaluate the performance of your model on different
   * instance types.</p>
   */
  inline RecommendationStepType GetStepType() const { return m_stepType; }
  inline bool StepTypeHasBeenSet() const { return m_stepTypeHasBeenSet; }
  inline void SetStepType(RecommendationStepType value) {
    m_stepTypeHasBeenSet = true;
    m_stepType = value;
  }
  inline ListInferenceRecommendationsJobStepsRequest& WithStepType(RecommendationStepType value) {
    SetStepType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListInferenceRecommendationsJobStepsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that you can specify to return more results from the list. Specify
   * this field if you have a token that was returned from a previous request.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListInferenceRecommendationsJobStepsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobName;
  bool m_jobNameHasBeenSet = false;

  RecommendationJobStatus m_status{RecommendationJobStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  RecommendationStepType m_stepType{RecommendationStepType::NOT_SET};
  bool m_stepTypeHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
