﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/sesv2/SESV2_EXPORTS.h>
#include <aws/sesv2/model/Attachment.h>
#include <aws/sesv2/model/Body.h>
#include <aws/sesv2/model/Content.h>
#include <aws/sesv2/model/MessageHeader.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SESV2 {
namespace Model {

/**
 * <p>Represents the email message that you're sending. The <code>Message</code>
 * object consists of a subject line and a message body.</p><p><h3>See Also:</h3>
 * <a href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/Message">AWS
 * API Reference</a></p>
 */
class Message {
 public:
  AWS_SESV2_API Message() = default;
  AWS_SESV2_API Message(Aws::Utils::Json::JsonView jsonValue);
  AWS_SESV2_API Message& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SESV2_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The subject line of the email. The subject line can only contain 7-bit ASCII
   * characters. However, you can specify non-ASCII characters in the subject line by
   * using encoded-word syntax, as described in <a
   * href="https://tools.ietf.org/html/rfc2047">RFC 2047</a>.</p>
   */
  inline const Content& GetSubject() const { return m_subject; }
  inline bool SubjectHasBeenSet() const { return m_subjectHasBeenSet; }
  template <typename SubjectT = Content>
  void SetSubject(SubjectT&& value) {
    m_subjectHasBeenSet = true;
    m_subject = std::forward<SubjectT>(value);
  }
  template <typename SubjectT = Content>
  Message& WithSubject(SubjectT&& value) {
    SetSubject(std::forward<SubjectT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The body of the message. You can specify an HTML version of the message, a
   * text-only version of the message, or both.</p>
   */
  inline const Body& GetBody() const { return m_body; }
  inline bool BodyHasBeenSet() const { return m_bodyHasBeenSet; }
  template <typename BodyT = Body>
  void SetBody(BodyT&& value) {
    m_bodyHasBeenSet = true;
    m_body = std::forward<BodyT>(value);
  }
  template <typename BodyT = Body>
  Message& WithBody(BodyT&& value) {
    SetBody(std::forward<BodyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of message headers that will be added to the email message.</p>
   */
  inline const Aws::Vector<MessageHeader>& GetHeaders() const { return m_headers; }
  inline bool HeadersHasBeenSet() const { return m_headersHasBeenSet; }
  template <typename HeadersT = Aws::Vector<MessageHeader>>
  void SetHeaders(HeadersT&& value) {
    m_headersHasBeenSet = true;
    m_headers = std::forward<HeadersT>(value);
  }
  template <typename HeadersT = Aws::Vector<MessageHeader>>
  Message& WithHeaders(HeadersT&& value) {
    SetHeaders(std::forward<HeadersT>(value));
    return *this;
  }
  template <typename HeadersT = MessageHeader>
  Message& AddHeaders(HeadersT&& value) {
    m_headersHasBeenSet = true;
    m_headers.emplace_back(std::forward<HeadersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The List of attachments to include in your email. All recipients will
   * receive the same attachments.</p>
   */
  inline const Aws::Vector<Attachment>& GetAttachments() const { return m_attachments; }
  inline bool AttachmentsHasBeenSet() const { return m_attachmentsHasBeenSet; }
  template <typename AttachmentsT = Aws::Vector<Attachment>>
  void SetAttachments(AttachmentsT&& value) {
    m_attachmentsHasBeenSet = true;
    m_attachments = std::forward<AttachmentsT>(value);
  }
  template <typename AttachmentsT = Aws::Vector<Attachment>>
  Message& WithAttachments(AttachmentsT&& value) {
    SetAttachments(std::forward<AttachmentsT>(value));
    return *this;
  }
  template <typename AttachmentsT = Attachment>
  Message& AddAttachments(AttachmentsT&& value) {
    m_attachmentsHasBeenSet = true;
    m_attachments.emplace_back(std::forward<AttachmentsT>(value));
    return *this;
  }
  ///@}
 private:
  Content m_subject;
  bool m_subjectHasBeenSet = false;

  Body m_body;
  bool m_bodyHasBeenSet = false;

  Aws::Vector<MessageHeader> m_headers;
  bool m_headersHasBeenSet = false;

  Aws::Vector<Attachment> m_attachments;
  bool m_attachmentsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SESV2
}  // namespace Aws
