/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.protocol.oid4vc.issuance.keybinding;

import com.fasterxml.jackson.core.type.TypeReference;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.jboss.logging.Logger;
import org.keycloak.common.VerificationException;
import org.keycloak.crypto.SignatureVerifierContext;
import org.keycloak.jose.jwk.JWK;
import org.keycloak.jose.jws.JWSHeader;
import org.keycloak.jose.jws.JWSInput;
import org.keycloak.jose.jws.JWSInputException;
import org.keycloak.models.KeycloakContext;
import org.keycloak.models.KeycloakSession;
import org.keycloak.protocol.oid4vc.issuance.OID4VCIssuerWellKnownProvider;
import org.keycloak.protocol.oid4vc.issuance.VCIssuanceContext;
import org.keycloak.protocol.oid4vc.issuance.VCIssuerException;
import org.keycloak.protocol.oid4vc.issuance.keybinding.AbstractProofValidator;
import org.keycloak.protocol.oid4vc.issuance.keybinding.AttestationKeyResolver;
import org.keycloak.protocol.oid4vc.issuance.keybinding.AttestationValidatorUtil;
import org.keycloak.protocol.oid4vc.issuance.keybinding.CNonceHandler;
import org.keycloak.protocol.oid4vc.model.ErrorType;
import org.keycloak.protocol.oid4vc.model.ProofTypesSupported;
import org.keycloak.protocol.oid4vc.model.Proofs;
import org.keycloak.protocol.oid4vc.model.SupportedCredentialConfiguration;
import org.keycloak.protocol.oid4vc.model.SupportedProofTypeData;
import org.keycloak.representations.AccessToken;
import org.keycloak.util.JsonSerialization;

public class JwtProofValidator
extends AbstractProofValidator {
    private static final Logger LOGGER = Logger.getLogger(JwtProofValidator.class);
    public static final String PROOF_JWT_TYP = "openid4vci-proof+jwt";
    private static final String CRYPTOGRAPHIC_BINDING_METHOD_JWK = "jwk";
    private static final String KEY_ATTESTATION_CLAIM = "key_attestation";
    private final AttestationKeyResolver keyResolver;

    public JwtProofValidator(KeycloakSession keycloakSession, AttestationKeyResolver keyResolver) {
        super(keycloakSession);
        this.keyResolver = keyResolver;
    }

    @Override
    public String getProofType() {
        return "jwt";
    }

    @Override
    public List<JWK> validateProof(VCIssuanceContext vcIssuanceContext) throws VCIssuerException {
        try {
            return this.validateJwtProof(vcIssuanceContext);
        }
        catch (IOException | VerificationException | JWSInputException e) {
            throw new VCIssuerException("Could not validate JWT proof", e);
        }
    }

    private List<JWK> validateJwtProof(VCIssuanceContext vcIssuanceContext) throws VCIssuerException, JWSInputException, VerificationException, IOException {
        Optional<List<String>> optionalProof = this.getProofFromContext(vcIssuanceContext);
        if (optionalProof.isEmpty() || optionalProof.get().isEmpty()) {
            return null;
        }
        List<String> jwtProofs = optionalProof.get();
        this.checkCryptographicKeyBinding(vcIssuanceContext);
        ArrayList<JWK> validJwks = new ArrayList<JWK>();
        for (int i = 0; i < jwtProofs.size(); ++i) {
            String jwt = jwtProofs.get(i);
            try {
                JWK jwk = this.validateSingleJwtProof(vcIssuanceContext, jwt);
                validJwks.add(jwk);
                LOGGER.debugf("Successfully validated JWT proof at index %d", i);
                continue;
            }
            catch (VCIssuerException e) {
                throw new VCIssuerException(String.format("Failed to validate JWT proof at index %d: %s", i, e.getMessage()), e);
            }
        }
        if (validJwks.isEmpty()) {
            throw new VCIssuerException("No valid JWT proof found in the proofs array");
        }
        LOGGER.debugf("Successfully validated %d JWT proofs", validJwks.size());
        return validJwks;
    }

    private JWK validateSingleJwtProof(VCIssuanceContext vcIssuanceContext, String jwt) throws VCIssuerException, JWSInputException, VerificationException, IOException {
        JWK jwk;
        JWSInput jwsInput = this.getJwsInput(jwt);
        JWSHeader jwsHeader = jwsInput.getHeader();
        this.validateJwsHeader(vcIssuanceContext, jwsHeader);
        if (jwsHeader.getKey() != null) {
            jwk = jwsHeader.getKey();
        } else if (jwsHeader.getKeyId() != null) {
            Map headerClaims = (Map)JsonSerialization.mapper.convertValue((Object)jwsHeader, (TypeReference)new TypeReference<Map<String, Object>>(){});
            if (!headerClaims.containsKey(KEY_ATTESTATION_CLAIM)) {
                throw new VCIssuerException("Key ID provided but no key_attestation in header to resolve it");
            }
            Object keyAttestation = headerClaims.get(KEY_ATTESTATION_CLAIM);
            if (keyAttestation == null) {
                throw new VCIssuerException("The 'key_attestation' claim is present in JWT header but is null.");
            }
            List<JWK> attestedKeys = AttestationValidatorUtil.validateAttestationJwt(keyAttestation.toString(), this.keycloakSession, vcIssuanceContext, this.keyResolver).getAttestedKeys();
            jwk = attestedKeys.stream().filter(k -> jwsHeader.getKeyId().equals(k.getKeyId())).findFirst().orElseThrow(() -> new VCIssuerException("No attested key found matching kid: " + jwsHeader.getKeyId()));
        } else {
            throw new VCIssuerException("Missing binding key. JWT must contain either jwk or kid in header.");
        }
        AccessToken proofPayload = (AccessToken)JsonSerialization.readValue((byte[])jwsInput.getContent(), AccessToken.class);
        this.validateProofPayload(vcIssuanceContext, proofPayload);
        SignatureVerifierContext signatureVerifierContext = this.getVerifier(jwk, jwsHeader.getAlgorithm().name());
        if (signatureVerifierContext == null) {
            throw new VCIssuerException("No verifier configured for " + String.valueOf(jwsHeader.getAlgorithm()));
        }
        if (!signatureVerifierContext.verify(jwsInput.getEncodedSignatureInput().getBytes(StandardCharsets.UTF_8), jwsInput.getSignature())) {
            throw new VCIssuerException("Could not verify signature of provided proof");
        }
        return jwk;
    }

    private void checkCryptographicKeyBinding(VCIssuanceContext vcIssuanceContext) {
        if (vcIssuanceContext.getCredentialConfig().getCryptographicBindingMethodsSupported() == null || !vcIssuanceContext.getCredentialConfig().getCryptographicBindingMethodsSupported().contains(CRYPTOGRAPHIC_BINDING_METHOD_JWK)) {
            throw new IllegalStateException("This SD-JWT implementation only supports jwk as cryptographic binding method");
        }
    }

    private Optional<List<String>> getProofFromContext(VCIssuanceContext vcIssuanceContext) throws VCIssuerException {
        return Optional.ofNullable(vcIssuanceContext.getCredentialConfig()).map(SupportedCredentialConfiguration::getProofTypesSupported).flatMap(proofTypesSupported -> {
            Optional.ofNullable(proofTypesSupported.getSupportedProofTypes().get("jwt")).orElseThrow(() -> new VCIssuerException("SD-JWT supports only jwt proof type."));
            Proofs proofs = vcIssuanceContext.getCredentialRequest().getProofs();
            if (proofs == null || proofs.getJwt() == null || proofs.getJwt().isEmpty()) {
                throw new VCIssuerException("Credential configuration requires a proof of type: jwt");
            }
            return Optional.of(proofs.getJwt());
        });
    }

    private JWSInput getJwsInput(String jwt) throws JWSInputException {
        return new JWSInput(jwt);
    }

    private void validateJwsHeader(VCIssuanceContext vcIssuanceContext, JWSHeader jwsHeader) throws VCIssuerException {
        Optional.ofNullable(jwsHeader.getAlgorithm()).orElseThrow(() -> new VCIssuerException("Missing jwsHeader claim alg"));
        Optional.ofNullable(vcIssuanceContext.getCredentialConfig()).map(SupportedCredentialConfiguration::getProofTypesSupported).map(ProofTypesSupported::getSupportedProofTypes).map(proofTypeData -> (SupportedProofTypeData)proofTypeData.get("jwt")).map(SupportedProofTypeData::getSigningAlgorithmsSupported).filter(supportedAlgs -> supportedAlgs.contains(jwsHeader.getAlgorithm().name())).orElseThrow(() -> new VCIssuerException("Proof signature algorithm not supported: " + jwsHeader.getAlgorithm().name()));
        Optional.ofNullable(jwsHeader.getType()).filter(type -> Objects.equals(PROOF_JWT_TYP, type)).orElseThrow(() -> new VCIssuerException("JWT type must be: openid4vci-proof+jwt"));
        Optional.ofNullable(jwsHeader.getKeyId()).ifPresent(keyId -> {
            throw new VCIssuerException("KeyId not expected in this JWT. Use the jwk claim instead.");
        });
    }

    private void validateProofPayload(VCIssuanceContext vcIssuanceContext, AccessToken proofPayload) throws VCIssuerException, VerificationException {
        String credentialIssuer = OID4VCIssuerWellKnownProvider.getIssuer(this.keycloakSession.getContext());
        Optional.ofNullable(proofPayload.getAudience()).map(Arrays::asList).filter(audiences -> audiences.contains(credentialIssuer)).orElseThrow(() -> new VCIssuerException("Proof not produced for this audience. Audience claim must be: " + credentialIssuer + " but are " + String.valueOf(Arrays.asList(proofPayload.getAudience()))));
        Optional.ofNullable(proofPayload.getIat()).orElseThrow(() -> new VCIssuerException("Missing proof issuing time. iat claim must be provided."));
        KeycloakContext keycloakContext = this.keycloakSession.getContext();
        CNonceHandler cNonceHandler = (CNonceHandler)this.keycloakSession.getProvider(CNonceHandler.class);
        try {
            cNonceHandler.verifyCNonce(proofPayload.getNonce(), List.of(OID4VCIssuerWellKnownProvider.getCredentialsEndpoint(keycloakContext)), Map.of("source_endpoint", OID4VCIssuerWellKnownProvider.getNonceEndpoint(keycloakContext)));
        }
        catch (VerificationException e) {
            throw new VCIssuerException(ErrorType.INVALID_NONCE, "The proofs parameter in the Credential Request uses an invalid nonce", e);
        }
    }
}

