﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/bedrock-agent/BedrockAgent_EXPORTS.h>
#include <aws/bedrock-agent/model/GuardrailConfiguration.h>
#include <aws/bedrock-agent/model/PromptFlowNodeSourceConfiguration.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace BedrockAgent {
namespace Model {

/**
 * <p>Contains configurations for a prompt node in the flow. You can use a prompt
 * from Prompt management or you can define one in this node. If the prompt
 * contains variables, the inputs into this node will fill in the variables. The
 * output from this node is the response generated by the model. For more
 * information, see <a
 * href="https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html">Node
 * types in a flow</a> in the Amazon Bedrock User Guide.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-2023-06-05/PromptFlowNodeConfiguration">AWS
 * API Reference</a></p>
 */
class PromptFlowNodeConfiguration {
 public:
  AWS_BEDROCKAGENT_API PromptFlowNodeConfiguration() = default;
  AWS_BEDROCKAGENT_API PromptFlowNodeConfiguration(Aws::Utils::Json::JsonView jsonValue);
  AWS_BEDROCKAGENT_API PromptFlowNodeConfiguration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_BEDROCKAGENT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Specifies whether the prompt is from Prompt management or defined inline.</p>
   */
  inline const PromptFlowNodeSourceConfiguration& GetSourceConfiguration() const { return m_sourceConfiguration; }
  inline bool SourceConfigurationHasBeenSet() const { return m_sourceConfigurationHasBeenSet; }
  template <typename SourceConfigurationT = PromptFlowNodeSourceConfiguration>
  void SetSourceConfiguration(SourceConfigurationT&& value) {
    m_sourceConfigurationHasBeenSet = true;
    m_sourceConfiguration = std::forward<SourceConfigurationT>(value);
  }
  template <typename SourceConfigurationT = PromptFlowNodeSourceConfiguration>
  PromptFlowNodeConfiguration& WithSourceConfiguration(SourceConfigurationT&& value) {
    SetSourceConfiguration(std::forward<SourceConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains configurations for a guardrail to apply to the prompt in this node
   * and the response generated from it.</p>
   */
  inline const GuardrailConfiguration& GetGuardrailConfiguration() const { return m_guardrailConfiguration; }
  inline bool GuardrailConfigurationHasBeenSet() const { return m_guardrailConfigurationHasBeenSet; }
  template <typename GuardrailConfigurationT = GuardrailConfiguration>
  void SetGuardrailConfiguration(GuardrailConfigurationT&& value) {
    m_guardrailConfigurationHasBeenSet = true;
    m_guardrailConfiguration = std::forward<GuardrailConfigurationT>(value);
  }
  template <typename GuardrailConfigurationT = GuardrailConfiguration>
  PromptFlowNodeConfiguration& WithGuardrailConfiguration(GuardrailConfigurationT&& value) {
    SetGuardrailConfiguration(std::forward<GuardrailConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  PromptFlowNodeSourceConfiguration m_sourceConfiguration;
  bool m_sourceConfigurationHasBeenSet = false;

  GuardrailConfiguration m_guardrailConfiguration;
  bool m_guardrailConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace BedrockAgent
}  // namespace Aws
