﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-idp/CognitoIdentityProvider_EXPORTS.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CognitoIdentityProvider {
namespace Model {

/**
 * <p>A user pool group. Contains details about the group and the way that it
 * contributes to IAM role decisions with identity pools. Identity pools can make
 * decisions about the IAM role to assign based on groups: users get credentials
 * for the role associated with their highest-priority group.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cognito-idp-2016-04-18/GroupType">AWS
 * API Reference</a></p>
 */
class GroupType {
 public:
  AWS_COGNITOIDENTITYPROVIDER_API GroupType() = default;
  AWS_COGNITOIDENTITYPROVIDER_API GroupType(Aws::Utils::Json::JsonView jsonValue);
  AWS_COGNITOIDENTITYPROVIDER_API GroupType& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_COGNITOIDENTITYPROVIDER_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the group.</p>
   */
  inline const Aws::String& GetGroupName() const { return m_groupName; }
  inline bool GroupNameHasBeenSet() const { return m_groupNameHasBeenSet; }
  template <typename GroupNameT = Aws::String>
  void SetGroupName(GroupNameT&& value) {
    m_groupNameHasBeenSet = true;
    m_groupName = std::forward<GroupNameT>(value);
  }
  template <typename GroupNameT = Aws::String>
  GroupType& WithGroupName(GroupNameT&& value) {
    SetGroupName(std::forward<GroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the user pool that contains the group.</p>
   */
  inline const Aws::String& GetUserPoolId() const { return m_userPoolId; }
  inline bool UserPoolIdHasBeenSet() const { return m_userPoolIdHasBeenSet; }
  template <typename UserPoolIdT = Aws::String>
  void SetUserPoolId(UserPoolIdT&& value) {
    m_userPoolIdHasBeenSet = true;
    m_userPoolId = std::forward<UserPoolIdT>(value);
  }
  template <typename UserPoolIdT = Aws::String>
  GroupType& WithUserPoolId(UserPoolIdT&& value) {
    SetUserPoolId(std::forward<UserPoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A friendly description of the group.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  GroupType& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the IAM role associated with the group. If a group has the highest
   * priority of a user's groups, users who authenticate with an identity pool get
   * credentials for the <code>RoleArn</code> that's associated with the group.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  GroupType& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A non-negative integer value that specifies the precedence of this group
   * relative to the other groups that a user can belong to in the user pool. Zero is
   * the highest precedence value. Groups with lower <code>Precedence</code> values
   * take precedence over groups with higher ornull <code>Precedence</code> values.
   * If a user belongs to two or more groups, it is the group with the lowest
   * precedence value whose role ARN is given in the user's tokens for the
   * <code>cognito:roles</code> and <code>cognito:preferred_role</code> claims.</p>
   * <p>Two groups can have the same <code>Precedence</code> value. If this happens,
   * neither group takes precedence over the other. If two groups with the same
   * <code>Precedence</code> have the same role ARN, that role is used in the
   * <code>cognito:preferred_role</code> claim in tokens for users in each group. If
   * the two groups have different role ARNs, the <code>cognito:preferred_role</code>
   * claim isn't set in users' tokens.</p> <p>The default <code>Precedence</code>
   * value is <code>null</code>.</p>
   */
  inline int GetPrecedence() const { return m_precedence; }
  inline bool PrecedenceHasBeenSet() const { return m_precedenceHasBeenSet; }
  inline void SetPrecedence(int value) {
    m_precedenceHasBeenSet = true;
    m_precedence = value;
  }
  inline GroupType& WithPrecedence(int value) {
    SetPrecedence(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the item was modified. Amazon Cognito returns this
   * timestamp in UNIX epoch time format. Your SDK might render the output in a
   * human-readable format like ISO 8601 or a Java <code>Date</code> object.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedDate() const { return m_lastModifiedDate; }
  inline bool LastModifiedDateHasBeenSet() const { return m_lastModifiedDateHasBeenSet; }
  template <typename LastModifiedDateT = Aws::Utils::DateTime>
  void SetLastModifiedDate(LastModifiedDateT&& value) {
    m_lastModifiedDateHasBeenSet = true;
    m_lastModifiedDate = std::forward<LastModifiedDateT>(value);
  }
  template <typename LastModifiedDateT = Aws::Utils::DateTime>
  GroupType& WithLastModifiedDate(LastModifiedDateT&& value) {
    SetLastModifiedDate(std::forward<LastModifiedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when the item was created. Amazon Cognito returns this
   * timestamp in UNIX epoch time format. Your SDK might render the output in a
   * human-readable format like ISO 8601 or a Java <code>Date</code> object.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationDate() const { return m_creationDate; }
  inline bool CreationDateHasBeenSet() const { return m_creationDateHasBeenSet; }
  template <typename CreationDateT = Aws::Utils::DateTime>
  void SetCreationDate(CreationDateT&& value) {
    m_creationDateHasBeenSet = true;
    m_creationDate = std::forward<CreationDateT>(value);
  }
  template <typename CreationDateT = Aws::Utils::DateTime>
  GroupType& WithCreationDate(CreationDateT&& value) {
    SetCreationDate(std::forward<CreationDateT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_groupName;
  bool m_groupNameHasBeenSet = false;

  Aws::String m_userPoolId;
  bool m_userPoolIdHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  int m_precedence{0};
  bool m_precedenceHasBeenSet = false;

  Aws::Utils::DateTime m_lastModifiedDate{};
  bool m_lastModifiedDateHasBeenSet = false;

  Aws::Utils::DateTime m_creationDate{};
  bool m_creationDateHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoIdentityProvider
}  // namespace Aws
