﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/docdb-elastic/DocDBElasticRequest.h>
#include <aws/docdb-elastic/DocDBElastic_EXPORTS.h>

#include <utility>

namespace Aws {
namespace DocDBElastic {
namespace Model {

/**
 */
class RestoreClusterFromSnapshotRequest : public DocDBElasticRequest {
 public:
  AWS_DOCDBELASTIC_API RestoreClusterFromSnapshotRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RestoreClusterFromSnapshot"; }

  AWS_DOCDBELASTIC_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the elastic cluster.</p>
   */
  inline const Aws::String& GetClusterName() const { return m_clusterName; }
  inline bool ClusterNameHasBeenSet() const { return m_clusterNameHasBeenSet; }
  template <typename ClusterNameT = Aws::String>
  void SetClusterName(ClusterNameT&& value) {
    m_clusterNameHasBeenSet = true;
    m_clusterName = std::forward<ClusterNameT>(value);
  }
  template <typename ClusterNameT = Aws::String>
  RestoreClusterFromSnapshotRequest& WithClusterName(ClusterNameT&& value) {
    SetClusterName(std::forward<ClusterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The KMS key identifier to use to encrypt the new Amazon DocumentDB elastic
   * clusters cluster.</p> <p>The KMS key identifier is the Amazon Resource Name
   * (ARN) for the KMS encryption key. If you are creating a cluster using the same
   * Amazon account that owns this KMS encryption key, you can use the KMS key alias
   * instead of the ARN as the KMS encryption key.</p> <p>If an encryption key is not
   * specified here, Amazon DocumentDB uses the default encryption key that KMS
   * creates for your account. Your account has a different default encryption key
   * for each Amazon Region.</p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  inline bool KmsKeyIdHasBeenSet() const { return m_kmsKeyIdHasBeenSet; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  RestoreClusterFromSnapshotRequest& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The capacity of each shard in the new restored elastic cluster.</p>
   */
  inline int GetShardCapacity() const { return m_shardCapacity; }
  inline bool ShardCapacityHasBeenSet() const { return m_shardCapacityHasBeenSet; }
  inline void SetShardCapacity(int value) {
    m_shardCapacityHasBeenSet = true;
    m_shardCapacity = value;
  }
  inline RestoreClusterFromSnapshotRequest& WithShardCapacity(int value) {
    SetShardCapacity(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of replica instances applying to all shards in the elastic
   * cluster. A <code>shardInstanceCount</code> value of 1 means there is one writer
   * instance, and any additional instances are replicas that can be used for reads
   * and to improve availability.</p>
   */
  inline int GetShardInstanceCount() const { return m_shardInstanceCount; }
  inline bool ShardInstanceCountHasBeenSet() const { return m_shardInstanceCountHasBeenSet; }
  inline void SetShardInstanceCount(int value) {
    m_shardInstanceCountHasBeenSet = true;
    m_shardInstanceCount = value;
  }
  inline RestoreClusterFromSnapshotRequest& WithShardInstanceCount(int value) {
    SetShardInstanceCount(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN identifier of the elastic cluster snapshot.</p>
   */
  inline const Aws::String& GetSnapshotArn() const { return m_snapshotArn; }
  inline bool SnapshotArnHasBeenSet() const { return m_snapshotArnHasBeenSet; }
  template <typename SnapshotArnT = Aws::String>
  void SetSnapshotArn(SnapshotArnT&& value) {
    m_snapshotArnHasBeenSet = true;
    m_snapshotArn = std::forward<SnapshotArnT>(value);
  }
  template <typename SnapshotArnT = Aws::String>
  RestoreClusterFromSnapshotRequest& WithSnapshotArn(SnapshotArnT&& value) {
    SetSnapshotArn(std::forward<SnapshotArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon EC2 subnet IDs for the elastic cluster.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSubnetIds() const { return m_subnetIds; }
  inline bool SubnetIdsHasBeenSet() const { return m_subnetIdsHasBeenSet; }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  void SetSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds = std::forward<SubnetIdsT>(value);
  }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  RestoreClusterFromSnapshotRequest& WithSubnetIds(SubnetIdsT&& value) {
    SetSubnetIds(std::forward<SubnetIdsT>(value));
    return *this;
  }
  template <typename SubnetIdsT = Aws::String>
  RestoreClusterFromSnapshotRequest& AddSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds.emplace_back(std::forward<SubnetIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of the tag names to be assigned to the restored elastic cluster, in
   * the form of an array of key-value pairs in which the key is the tag name and the
   * value is the key value.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  RestoreClusterFromSnapshotRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  RestoreClusterFromSnapshotRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of EC2 VPC security groups to associate with the elastic cluster.</p>
   */
  inline const Aws::Vector<Aws::String>& GetVpcSecurityGroupIds() const { return m_vpcSecurityGroupIds; }
  inline bool VpcSecurityGroupIdsHasBeenSet() const { return m_vpcSecurityGroupIdsHasBeenSet; }
  template <typename VpcSecurityGroupIdsT = Aws::Vector<Aws::String>>
  void SetVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    m_vpcSecurityGroupIdsHasBeenSet = true;
    m_vpcSecurityGroupIds = std::forward<VpcSecurityGroupIdsT>(value);
  }
  template <typename VpcSecurityGroupIdsT = Aws::Vector<Aws::String>>
  RestoreClusterFromSnapshotRequest& WithVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    SetVpcSecurityGroupIds(std::forward<VpcSecurityGroupIdsT>(value));
    return *this;
  }
  template <typename VpcSecurityGroupIdsT = Aws::String>
  RestoreClusterFromSnapshotRequest& AddVpcSecurityGroupIds(VpcSecurityGroupIdsT&& value) {
    m_vpcSecurityGroupIdsHasBeenSet = true;
    m_vpcSecurityGroupIds.emplace_back(std::forward<VpcSecurityGroupIdsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterName;
  bool m_clusterNameHasBeenSet = false;

  Aws::String m_kmsKeyId;
  bool m_kmsKeyIdHasBeenSet = false;

  int m_shardCapacity{0};
  bool m_shardCapacityHasBeenSet = false;

  int m_shardInstanceCount{0};
  bool m_shardInstanceCountHasBeenSet = false;

  Aws::String m_snapshotArn;
  bool m_snapshotArnHasBeenSet = false;

  Aws::Vector<Aws::String> m_subnetIds;
  bool m_subnetIdsHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::Vector<Aws::String> m_vpcSecurityGroupIds;
  bool m_vpcSecurityGroupIdsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DocDBElastic
}  // namespace Aws
