﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/healthlake/HealthLakeRequest.h>
#include <aws/healthlake/HealthLake_EXPORTS.h>
#include <aws/healthlake/model/InputDataConfig.h>
#include <aws/healthlake/model/OutputDataConfig.h>
#include <aws/healthlake/model/ValidationLevel.h>

#include <utility>

namespace Aws {
namespace HealthLake {
namespace Model {

/**
 */
class StartFHIRImportJobRequest : public HealthLakeRequest {
 public:
  AWS_HEALTHLAKE_API StartFHIRImportJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartFHIRImportJob"; }

  AWS_HEALTHLAKE_API Aws::String SerializePayload() const override;

  AWS_HEALTHLAKE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The import job name.</p>
   */
  inline const Aws::String& GetJobName() const { return m_jobName; }
  inline bool JobNameHasBeenSet() const { return m_jobNameHasBeenSet; }
  template <typename JobNameT = Aws::String>
  void SetJobName(JobNameT&& value) {
    m_jobNameHasBeenSet = true;
    m_jobName = std::forward<JobNameT>(value);
  }
  template <typename JobNameT = Aws::String>
  StartFHIRImportJobRequest& WithJobName(JobNameT&& value) {
    SetJobName(std::forward<JobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The input properties for the import job request.</p>
   */
  inline const InputDataConfig& GetInputDataConfig() const { return m_inputDataConfig; }
  inline bool InputDataConfigHasBeenSet() const { return m_inputDataConfigHasBeenSet; }
  template <typename InputDataConfigT = InputDataConfig>
  void SetInputDataConfig(InputDataConfigT&& value) {
    m_inputDataConfigHasBeenSet = true;
    m_inputDataConfig = std::forward<InputDataConfigT>(value);
  }
  template <typename InputDataConfigT = InputDataConfig>
  StartFHIRImportJobRequest& WithInputDataConfig(InputDataConfigT&& value) {
    SetInputDataConfig(std::forward<InputDataConfigT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const OutputDataConfig& GetJobOutputDataConfig() const { return m_jobOutputDataConfig; }
  inline bool JobOutputDataConfigHasBeenSet() const { return m_jobOutputDataConfigHasBeenSet; }
  template <typename JobOutputDataConfigT = OutputDataConfig>
  void SetJobOutputDataConfig(JobOutputDataConfigT&& value) {
    m_jobOutputDataConfigHasBeenSet = true;
    m_jobOutputDataConfig = std::forward<JobOutputDataConfigT>(value);
  }
  template <typename JobOutputDataConfigT = OutputDataConfig>
  StartFHIRImportJobRequest& WithJobOutputDataConfig(JobOutputDataConfigT&& value) {
    SetJobOutputDataConfig(std::forward<JobOutputDataConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The data store identifier.</p>
   */
  inline const Aws::String& GetDatastoreId() const { return m_datastoreId; }
  inline bool DatastoreIdHasBeenSet() const { return m_datastoreIdHasBeenSet; }
  template <typename DatastoreIdT = Aws::String>
  void SetDatastoreId(DatastoreIdT&& value) {
    m_datastoreIdHasBeenSet = true;
    m_datastoreId = std::forward<DatastoreIdT>(value);
  }
  template <typename DatastoreIdT = Aws::String>
  StartFHIRImportJobRequest& WithDatastoreId(DatastoreIdT&& value) {
    SetDatastoreId(std::forward<DatastoreIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that grants access permission to AWS
   * HealthLake.</p>
   */
  inline const Aws::String& GetDataAccessRoleArn() const { return m_dataAccessRoleArn; }
  inline bool DataAccessRoleArnHasBeenSet() const { return m_dataAccessRoleArnHasBeenSet; }
  template <typename DataAccessRoleArnT = Aws::String>
  void SetDataAccessRoleArn(DataAccessRoleArnT&& value) {
    m_dataAccessRoleArnHasBeenSet = true;
    m_dataAccessRoleArn = std::forward<DataAccessRoleArnT>(value);
  }
  template <typename DataAccessRoleArnT = Aws::String>
  StartFHIRImportJobRequest& WithDataAccessRoleArn(DataAccessRoleArnT&& value) {
    SetDataAccessRoleArn(std::forward<DataAccessRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The optional user-provided token used for ensuring API idempotency.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartFHIRImportJobRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The validation level of the import job.</p>
   */
  inline ValidationLevel GetValidationLevel() const { return m_validationLevel; }
  inline bool ValidationLevelHasBeenSet() const { return m_validationLevelHasBeenSet; }
  inline void SetValidationLevel(ValidationLevel value) {
    m_validationLevelHasBeenSet = true;
    m_validationLevel = value;
  }
  inline StartFHIRImportJobRequest& WithValidationLevel(ValidationLevel value) {
    SetValidationLevel(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobName;
  bool m_jobNameHasBeenSet = false;

  InputDataConfig m_inputDataConfig;
  bool m_inputDataConfigHasBeenSet = false;

  OutputDataConfig m_jobOutputDataConfig;
  bool m_jobOutputDataConfigHasBeenSet = false;

  Aws::String m_datastoreId;
  bool m_datastoreIdHasBeenSet = false;

  Aws::String m_dataAccessRoleArn;
  bool m_dataAccessRoleArnHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  ValidationLevel m_validationLevel{ValidationLevel::NOT_SET};
  bool m_validationLevelHasBeenSet = false;
};

}  // namespace Model
}  // namespace HealthLake
}  // namespace Aws
