﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/network-firewall/NetworkFirewallRequest.h>
#include <aws/network-firewall/NetworkFirewall_EXPORTS.h>

#include <utility>

namespace Aws {
namespace NetworkFirewall {
namespace Model {

/**
 */
class ListFlowOperationResultsRequest : public NetworkFirewallRequest {
 public:
  AWS_NETWORKFIREWALL_API ListFlowOperationResultsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListFlowOperationResults"; }

  AWS_NETWORKFIREWALL_API Aws::String SerializePayload() const override;

  AWS_NETWORKFIREWALL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the firewall.</p>
   */
  inline const Aws::String& GetFirewallArn() const { return m_firewallArn; }
  inline bool FirewallArnHasBeenSet() const { return m_firewallArnHasBeenSet; }
  template <typename FirewallArnT = Aws::String>
  void SetFirewallArn(FirewallArnT&& value) {
    m_firewallArnHasBeenSet = true;
    m_firewallArn = std::forward<FirewallArnT>(value);
  }
  template <typename FirewallArnT = Aws::String>
  ListFlowOperationResultsRequest& WithFirewallArn(FirewallArnT&& value) {
    SetFirewallArn(std::forward<FirewallArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the flow operation. This ID is returned in the
   * responses to start and list commands. You provide to describe commands.</p>
   */
  inline const Aws::String& GetFlowOperationId() const { return m_flowOperationId; }
  inline bool FlowOperationIdHasBeenSet() const { return m_flowOperationIdHasBeenSet; }
  template <typename FlowOperationIdT = Aws::String>
  void SetFlowOperationId(FlowOperationIdT&& value) {
    m_flowOperationIdHasBeenSet = true;
    m_flowOperationId = std::forward<FlowOperationIdT>(value);
  }
  template <typename FlowOperationIdT = Aws::String>
  ListFlowOperationResultsRequest& WithFlowOperationId(FlowOperationIdT&& value) {
    SetFlowOperationId(std::forward<FlowOperationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>When you request a list of objects with a <code>MaxResults</code> setting, if
   * the number of objects that are still available for retrieval exceeds the maximum
   * you requested, Network Firewall returns a <code>NextToken</code> value in the
   * response. To retrieve the next batch of objects, use the token returned from the
   * prior request in your next request.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListFlowOperationResultsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of objects that you want Network Firewall to return for
   * this request. If more objects are available, in the response, Network Firewall
   * provides a <code>NextToken</code> value that you can use in a subsequent call to
   * get the next batch of objects.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListFlowOperationResultsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Availability Zone where the firewall is located. For example,
   * <code>us-east-2a</code>.</p> <p>Defines the scope a flow operation. You can use
   * up to 20 filters to configure a single flow operation.</p>
   */
  inline const Aws::String& GetAvailabilityZone() const { return m_availabilityZone; }
  inline bool AvailabilityZoneHasBeenSet() const { return m_availabilityZoneHasBeenSet; }
  template <typename AvailabilityZoneT = Aws::String>
  void SetAvailabilityZone(AvailabilityZoneT&& value) {
    m_availabilityZoneHasBeenSet = true;
    m_availabilityZone = std::forward<AvailabilityZoneT>(value);
  }
  template <typename AvailabilityZoneT = Aws::String>
  ListFlowOperationResultsRequest& WithAvailabilityZone(AvailabilityZoneT&& value) {
    SetAvailabilityZone(std::forward<AvailabilityZoneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the primary endpoint associated with a firewall.</p>
   */
  inline const Aws::String& GetVpcEndpointId() const { return m_vpcEndpointId; }
  inline bool VpcEndpointIdHasBeenSet() const { return m_vpcEndpointIdHasBeenSet; }
  template <typename VpcEndpointIdT = Aws::String>
  void SetVpcEndpointId(VpcEndpointIdT&& value) {
    m_vpcEndpointIdHasBeenSet = true;
    m_vpcEndpointId = std::forward<VpcEndpointIdT>(value);
  }
  template <typename VpcEndpointIdT = Aws::String>
  ListFlowOperationResultsRequest& WithVpcEndpointId(VpcEndpointIdT&& value) {
    SetVpcEndpointId(std::forward<VpcEndpointIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of a VPC endpoint association.</p>
   */
  inline const Aws::String& GetVpcEndpointAssociationArn() const { return m_vpcEndpointAssociationArn; }
  inline bool VpcEndpointAssociationArnHasBeenSet() const { return m_vpcEndpointAssociationArnHasBeenSet; }
  template <typename VpcEndpointAssociationArnT = Aws::String>
  void SetVpcEndpointAssociationArn(VpcEndpointAssociationArnT&& value) {
    m_vpcEndpointAssociationArnHasBeenSet = true;
    m_vpcEndpointAssociationArn = std::forward<VpcEndpointAssociationArnT>(value);
  }
  template <typename VpcEndpointAssociationArnT = Aws::String>
  ListFlowOperationResultsRequest& WithVpcEndpointAssociationArn(VpcEndpointAssociationArnT&& value) {
    SetVpcEndpointAssociationArn(std::forward<VpcEndpointAssociationArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_firewallArn;
  bool m_firewallArnHasBeenSet = false;

  Aws::String m_flowOperationId;
  bool m_flowOperationIdHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_availabilityZone;
  bool m_availabilityZoneHasBeenSet = false;

  Aws::String m_vpcEndpointId;
  bool m_vpcEndpointIdHasBeenSet = false;

  Aws::String m_vpcEndpointAssociationArn;
  bool m_vpcEndpointAssociationArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace NetworkFirewall
}  // namespace Aws
