﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/firehose/Firehose_EXPORTS.h>
#include <aws/firehose/model/AmazonOpenSearchServerlessBufferingHints.h>
#include <aws/firehose/model/AmazonOpenSearchServerlessRetryOptions.h>
#include <aws/firehose/model/AmazonOpenSearchServerlessS3BackupMode.h>
#include <aws/firehose/model/CloudWatchLoggingOptions.h>
#include <aws/firehose/model/ProcessingConfiguration.h>
#include <aws/firehose/model/S3DestinationConfiguration.h>
#include <aws/firehose/model/VpcConfiguration.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace Firehose {
namespace Model {

/**
 * <p>Describes the configuration of a destination in the Serverless offering for
 * Amazon OpenSearch Service.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/AmazonOpenSearchServerlessDestinationConfiguration">AWS
 * API Reference</a></p>
 */
class AmazonOpenSearchServerlessDestinationConfiguration {
 public:
  AWS_FIREHOSE_API AmazonOpenSearchServerlessDestinationConfiguration() = default;
  AWS_FIREHOSE_API AmazonOpenSearchServerlessDestinationConfiguration(Aws::Utils::Json::JsonView jsonValue);
  AWS_FIREHOSE_API AmazonOpenSearchServerlessDestinationConfiguration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_FIREHOSE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role to be assumed by Firehose for
   * calling the Serverless offering for Amazon OpenSearch Service Configuration API
   * and for indexing documents.</p>
   */
  inline const Aws::String& GetRoleARN() const { return m_roleARN; }
  inline bool RoleARNHasBeenSet() const { return m_roleARNHasBeenSet; }
  template <typename RoleARNT = Aws::String>
  void SetRoleARN(RoleARNT&& value) {
    m_roleARNHasBeenSet = true;
    m_roleARN = std::forward<RoleARNT>(value);
  }
  template <typename RoleARNT = Aws::String>
  AmazonOpenSearchServerlessDestinationConfiguration& WithRoleARN(RoleARNT&& value) {
    SetRoleARN(std::forward<RoleARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The endpoint to use when communicating with the collection in the Serverless
   * offering for Amazon OpenSearch Service.</p>
   */
  inline const Aws::String& GetCollectionEndpoint() const { return m_collectionEndpoint; }
  inline bool CollectionEndpointHasBeenSet() const { return m_collectionEndpointHasBeenSet; }
  template <typename CollectionEndpointT = Aws::String>
  void SetCollectionEndpoint(CollectionEndpointT&& value) {
    m_collectionEndpointHasBeenSet = true;
    m_collectionEndpoint = std::forward<CollectionEndpointT>(value);
  }
  template <typename CollectionEndpointT = Aws::String>
  AmazonOpenSearchServerlessDestinationConfiguration& WithCollectionEndpoint(CollectionEndpointT&& value) {
    SetCollectionEndpoint(std::forward<CollectionEndpointT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Serverless offering for Amazon OpenSearch Service index name.</p>
   */
  inline const Aws::String& GetIndexName() const { return m_indexName; }
  inline bool IndexNameHasBeenSet() const { return m_indexNameHasBeenSet; }
  template <typename IndexNameT = Aws::String>
  void SetIndexName(IndexNameT&& value) {
    m_indexNameHasBeenSet = true;
    m_indexName = std::forward<IndexNameT>(value);
  }
  template <typename IndexNameT = Aws::String>
  AmazonOpenSearchServerlessDestinationConfiguration& WithIndexName(IndexNameT&& value) {
    SetIndexName(std::forward<IndexNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The buffering options. If no value is specified, the default values for
   * AmazonopensearchserviceBufferingHints are used.</p>
   */
  inline const AmazonOpenSearchServerlessBufferingHints& GetBufferingHints() const { return m_bufferingHints; }
  inline bool BufferingHintsHasBeenSet() const { return m_bufferingHintsHasBeenSet; }
  template <typename BufferingHintsT = AmazonOpenSearchServerlessBufferingHints>
  void SetBufferingHints(BufferingHintsT&& value) {
    m_bufferingHintsHasBeenSet = true;
    m_bufferingHints = std::forward<BufferingHintsT>(value);
  }
  template <typename BufferingHintsT = AmazonOpenSearchServerlessBufferingHints>
  AmazonOpenSearchServerlessDestinationConfiguration& WithBufferingHints(BufferingHintsT&& value) {
    SetBufferingHints(std::forward<BufferingHintsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The retry behavior in case Firehose is unable to deliver documents to the
   * Serverless offering for Amazon OpenSearch Service. The default value is 300 (5
   * minutes).</p>
   */
  inline const AmazonOpenSearchServerlessRetryOptions& GetRetryOptions() const { return m_retryOptions; }
  inline bool RetryOptionsHasBeenSet() const { return m_retryOptionsHasBeenSet; }
  template <typename RetryOptionsT = AmazonOpenSearchServerlessRetryOptions>
  void SetRetryOptions(RetryOptionsT&& value) {
    m_retryOptionsHasBeenSet = true;
    m_retryOptions = std::forward<RetryOptionsT>(value);
  }
  template <typename RetryOptionsT = AmazonOpenSearchServerlessRetryOptions>
  AmazonOpenSearchServerlessDestinationConfiguration& WithRetryOptions(RetryOptionsT&& value) {
    SetRetryOptions(std::forward<RetryOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Defines how documents should be delivered to Amazon S3. When it is set to
   * FailedDocumentsOnly, Firehose writes any documents that could not be indexed to
   * the configured Amazon S3 destination, with AmazonOpenSearchService-failed/
   * appended to the key prefix. When set to AllDocuments, Firehose delivers all
   * incoming records to Amazon S3, and also writes failed documents with
   * AmazonOpenSearchService-failed/ appended to the prefix.</p>
   */
  inline AmazonOpenSearchServerlessS3BackupMode GetS3BackupMode() const { return m_s3BackupMode; }
  inline bool S3BackupModeHasBeenSet() const { return m_s3BackupModeHasBeenSet; }
  inline void SetS3BackupMode(AmazonOpenSearchServerlessS3BackupMode value) {
    m_s3BackupModeHasBeenSet = true;
    m_s3BackupMode = value;
  }
  inline AmazonOpenSearchServerlessDestinationConfiguration& WithS3BackupMode(AmazonOpenSearchServerlessS3BackupMode value) {
    SetS3BackupMode(value);
    return *this;
  }
  ///@}

  ///@{

  inline const S3DestinationConfiguration& GetS3Configuration() const { return m_s3Configuration; }
  inline bool S3ConfigurationHasBeenSet() const { return m_s3ConfigurationHasBeenSet; }
  template <typename S3ConfigurationT = S3DestinationConfiguration>
  void SetS3Configuration(S3ConfigurationT&& value) {
    m_s3ConfigurationHasBeenSet = true;
    m_s3Configuration = std::forward<S3ConfigurationT>(value);
  }
  template <typename S3ConfigurationT = S3DestinationConfiguration>
  AmazonOpenSearchServerlessDestinationConfiguration& WithS3Configuration(S3ConfigurationT&& value) {
    SetS3Configuration(std::forward<S3ConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const ProcessingConfiguration& GetProcessingConfiguration() const { return m_processingConfiguration; }
  inline bool ProcessingConfigurationHasBeenSet() const { return m_processingConfigurationHasBeenSet; }
  template <typename ProcessingConfigurationT = ProcessingConfiguration>
  void SetProcessingConfiguration(ProcessingConfigurationT&& value) {
    m_processingConfigurationHasBeenSet = true;
    m_processingConfiguration = std::forward<ProcessingConfigurationT>(value);
  }
  template <typename ProcessingConfigurationT = ProcessingConfiguration>
  AmazonOpenSearchServerlessDestinationConfiguration& WithProcessingConfiguration(ProcessingConfigurationT&& value) {
    SetProcessingConfiguration(std::forward<ProcessingConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const CloudWatchLoggingOptions& GetCloudWatchLoggingOptions() const { return m_cloudWatchLoggingOptions; }
  inline bool CloudWatchLoggingOptionsHasBeenSet() const { return m_cloudWatchLoggingOptionsHasBeenSet; }
  template <typename CloudWatchLoggingOptionsT = CloudWatchLoggingOptions>
  void SetCloudWatchLoggingOptions(CloudWatchLoggingOptionsT&& value) {
    m_cloudWatchLoggingOptionsHasBeenSet = true;
    m_cloudWatchLoggingOptions = std::forward<CloudWatchLoggingOptionsT>(value);
  }
  template <typename CloudWatchLoggingOptionsT = CloudWatchLoggingOptions>
  AmazonOpenSearchServerlessDestinationConfiguration& WithCloudWatchLoggingOptions(CloudWatchLoggingOptionsT&& value) {
    SetCloudWatchLoggingOptions(std::forward<CloudWatchLoggingOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const VpcConfiguration& GetVpcConfiguration() const { return m_vpcConfiguration; }
  inline bool VpcConfigurationHasBeenSet() const { return m_vpcConfigurationHasBeenSet; }
  template <typename VpcConfigurationT = VpcConfiguration>
  void SetVpcConfiguration(VpcConfigurationT&& value) {
    m_vpcConfigurationHasBeenSet = true;
    m_vpcConfiguration = std::forward<VpcConfigurationT>(value);
  }
  template <typename VpcConfigurationT = VpcConfiguration>
  AmazonOpenSearchServerlessDestinationConfiguration& WithVpcConfiguration(VpcConfigurationT&& value) {
    SetVpcConfiguration(std::forward<VpcConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roleARN;
  bool m_roleARNHasBeenSet = false;

  Aws::String m_collectionEndpoint;
  bool m_collectionEndpointHasBeenSet = false;

  Aws::String m_indexName;
  bool m_indexNameHasBeenSet = false;

  AmazonOpenSearchServerlessBufferingHints m_bufferingHints;
  bool m_bufferingHintsHasBeenSet = false;

  AmazonOpenSearchServerlessRetryOptions m_retryOptions;
  bool m_retryOptionsHasBeenSet = false;

  AmazonOpenSearchServerlessS3BackupMode m_s3BackupMode{AmazonOpenSearchServerlessS3BackupMode::NOT_SET};
  bool m_s3BackupModeHasBeenSet = false;

  S3DestinationConfiguration m_s3Configuration;
  bool m_s3ConfigurationHasBeenSet = false;

  ProcessingConfiguration m_processingConfiguration;
  bool m_processingConfigurationHasBeenSet = false;

  CloudWatchLoggingOptions m_cloudWatchLoggingOptions;
  bool m_cloudWatchLoggingOptionsHasBeenSet = false;

  VpcConfiguration m_vpcConfiguration;
  bool m_vpcConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace Firehose
}  // namespace Aws
