/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.models.workflow;

import jakarta.ws.rs.BadRequestException;
import java.time.Duration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jboss.logging.Logger;
import org.keycloak.common.util.DurationConverter;
import org.keycloak.common.util.MultivaluedHashMap;
import org.keycloak.component.ComponentFactory;
import org.keycloak.component.ComponentModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.ModelValidationException;
import org.keycloak.models.RealmModel;
import org.keycloak.models.workflow.AdhocWorkflowEvent;
import org.keycloak.models.workflow.DefaultWorkflowExecutionContext;
import org.keycloak.models.workflow.EventBasedWorkflow;
import org.keycloak.models.workflow.ResourceType;
import org.keycloak.models.workflow.ResourceTypeSelector;
import org.keycloak.models.workflow.RunWorkflowTask;
import org.keycloak.models.workflow.ScheduleWorkflowTask;
import org.keycloak.models.workflow.ScheduledWorkflowRunner;
import org.keycloak.models.workflow.UserResourceTypeWorkflowProvider;
import org.keycloak.models.workflow.Workflow;
import org.keycloak.models.workflow.WorkflowEvent;
import org.keycloak.models.workflow.WorkflowExecutionContext;
import org.keycloak.models.workflow.WorkflowExecutor;
import org.keycloak.models.workflow.WorkflowInvalidStateException;
import org.keycloak.models.workflow.WorkflowProvider;
import org.keycloak.models.workflow.WorkflowStateProvider;
import org.keycloak.models.workflow.WorkflowStep;
import org.keycloak.models.workflow.WorkflowStepProvider;
import org.keycloak.models.workflow.WorkflowValidator;
import org.keycloak.representations.workflows.WorkflowRepresentation;
import org.keycloak.representations.workflows.WorkflowStepRepresentation;
import org.keycloak.services.scheduled.ClusterAwareScheduledTaskRunner;
import org.keycloak.timer.ScheduledTask;
import org.keycloak.timer.TaskRunner;
import org.keycloak.timer.TimerProvider;

public class DefaultWorkflowProvider
implements WorkflowProvider {
    private static final Logger log = Logger.getLogger(DefaultWorkflowProvider.class);
    private final KeycloakSession session;
    private final WorkflowStateProvider stateProvider;
    private final WorkflowExecutor executor;
    private final KeycloakSessionFactory sessionFactory;
    private final RealmModel realm;

    DefaultWorkflowProvider(KeycloakSession session, WorkflowExecutor executor) {
        this.session = session;
        this.executor = executor;
        this.sessionFactory = session.getKeycloakSessionFactory();
        this.stateProvider = (WorkflowStateProvider)this.sessionFactory.getProviderFactory(WorkflowStateProvider.class).create(session);
        this.realm = session.getContext().getRealm();
    }

    public ResourceTypeSelector getResourceTypeSelector(ResourceType type) {
        Objects.requireNonNull(type, "type");
        switch (type) {
            default: {
                throw new IncompatibleClassChangeError();
            }
            case USERS: 
        }
        return new UserResourceTypeWorkflowProvider(this.session);
    }

    public void updateWorkflow(Workflow workflow, WorkflowRepresentation representation) {
        WorkflowValidator.validateWorkflow(this.session, this, representation);
        if (!this.stateProvider.hasScheduledSteps(workflow.getId())) {
            this.removeWorkflow(workflow);
            representation.setId(workflow.getId());
            this.toModel(representation);
        } else {
            WorkflowRepresentation currentRepresentation = this.toRepresentation(workflow);
            if (!Objects.equals(currentRepresentation.getOn(), representation.getOn())) {
                throw new ModelValidationException("Cannot update 'on' configuration when there are scheduled resources for the workflow.");
            }
            List currentSteps = currentRepresentation.getSteps();
            List newSteps = Optional.ofNullable(representation.getSteps()).orElse(List.of());
            if (currentSteps.size() != newSteps.size()) {
                throw new ModelValidationException("Cannot change the number or order of steps when there are scheduled resources for the workflow.");
            }
            for (int i = 0; i < currentSteps.size(); ++i) {
                WorkflowStepRepresentation currentStep = (WorkflowStepRepresentation)currentSteps.get(i);
                WorkflowStepRepresentation newStep = (WorkflowStepRepresentation)newSteps.get(i);
                if (!Objects.equals(currentStep.getUses(), newStep.getUses())) {
                    throw new ModelValidationException("Cannot change the number or order of steps when there are scheduled resources for the workflow.");
                }
                newStep.setId(currentStep.getId());
                newStep.setPriority(Long.parseLong(currentStep.getPriority()));
            }
            workflow.updateConfig(representation.getConfig(), newSteps);
        }
        this.cancelScheduledWorkflow(workflow);
        this.scheduleWorkflow(workflow);
    }

    public void removeWorkflow(Workflow workflow) {
        Objects.requireNonNull(workflow, "workflow");
        ComponentModel component = this.getWorkflowComponent(workflow.getId());
        this.realm.getComponentsStream(workflow.getId(), WorkflowStepProvider.class.getName()).forEach(arg_0 -> ((RealmModel)this.realm).removeComponent(arg_0));
        this.realm.removeComponent(component);
        this.stateProvider.removeByWorkflow(workflow.getId());
        this.cancelScheduledWorkflow(workflow);
    }

    public Workflow getWorkflow(String id) {
        return new Workflow(this.session, this.getWorkflowComponent(id));
    }

    public Stream<Workflow> getWorkflows() {
        return this.realm.getComponentsStream(this.realm.getId(), WorkflowProvider.class.getName()).map(c -> new Workflow(this.session, c));
    }

    public Stream<WorkflowRepresentation> getScheduledWorkflowsByResource(String resourceId) {
        return this.stateProvider.getScheduledStepsByResource(resourceId).map(scheduledStep -> {
            Workflow workflow = this.getWorkflow(scheduledStep.workflowId());
            List<WorkflowStepRepresentation> steps = workflow.getSteps(scheduledStep.stepId()).map(this::toRepresentation).toList();
            Long scheduledAt = null;
            for (WorkflowStepRepresentation step : steps) {
                if (scheduledAt == null) {
                    scheduledAt = scheduledStep.scheduledAt();
                } else if (step.getAfter() != null) {
                    scheduledAt = scheduledAt + DurationConverter.parseDuration((String)step.getAfter()).toMillis();
                }
                step.setScheduledAt(scheduledAt);
            }
            return new WorkflowRepresentation(workflow.getId(), workflow.getName(), workflow.getConfig(), steps);
        });
    }

    public void submit(WorkflowEvent event) {
        this.processEvent(this.getWorkflows(), event);
    }

    public void runScheduledSteps() {
        this.getWorkflows().forEach(workflow -> {
            if (!workflow.isEnabled()) {
                log.debugf("Skipping workflow %s as it is disabled", (Object)workflow.getName());
                return;
            }
            this.stateProvider.getDueScheduledSteps(workflow).forEach(scheduled -> {
                DefaultWorkflowExecutionContext context = new DefaultWorkflowExecutionContext(this.session, (Workflow)workflow, (WorkflowStateProvider.ScheduledStep)scheduled);
                EventBasedWorkflow provider = new EventBasedWorkflow(this.session, this.getWorkflowComponent(workflow.getId()));
                if (!provider.validateResourceConditions(context)) {
                    log.debugf("Resource %s is no longer eligible for workflow %s. Cancelling execution of the workflow.", (Object)scheduled.resourceId(), (Object)scheduled.workflowId());
                    this.stateProvider.remove(scheduled.executionId());
                } else {
                    WorkflowStep step = context.getStep();
                    if (step == null) {
                        log.warnf("Could not find step %s in workflow %s for resource %s. Cancelling execution of the workflow.", (Object)scheduled.stepId(), (Object)scheduled.workflowId(), (Object)scheduled.resourceId());
                        this.stateProvider.remove(scheduled.executionId());
                    } else {
                        this.runWorkflow(context);
                    }
                }
            });
        });
    }

    public void activate(Workflow workflow, ResourceType type, String resourceId) {
        this.processEvent(Stream.of(workflow), (WorkflowEvent)new AdhocWorkflowEvent(type, resourceId));
    }

    public void deactivate(Workflow workflow, String resourceId) {
        this.stateProvider.removeByWorkflowAndResource(workflow.getId(), resourceId);
    }

    public void activateForAllEligibleResources(Workflow workflow) {
        if (workflow.isEnabled()) {
            WorkflowProvider provider = this.getWorkflowProvider(workflow);
            ResourceTypeSelector selector = provider.getResourceTypeSelector(ResourceType.USERS);
            selector.getResourceIds(workflow).forEach(resourceId -> this.processEvent(Stream.of(workflow), (WorkflowEvent)new AdhocWorkflowEvent(ResourceType.USERS, resourceId)));
        }
    }

    public WorkflowRepresentation toRepresentation(Workflow workflow) {
        List<WorkflowStepRepresentation> steps = workflow.getSteps().map(this::toRepresentation).toList();
        return new WorkflowRepresentation(workflow.getId(), workflow.getName(), workflow.getConfig(), steps);
    }

    public Workflow toModel(WorkflowRepresentation rep) {
        WorkflowValidator.validateWorkflow(this.session, this, rep);
        MultivaluedHashMap config = Optional.ofNullable(rep.getConfig()).orElse(new MultivaluedHashMap());
        if (rep.getCancelInProgress() != null) {
            config.putSingle((Object)"cancel-in-progress", (Object)rep.getCancelInProgress());
        }
        if (rep.getRestartInProgress() != null) {
            config.putSingle((Object)"restart-in-progress", (Object)rep.getRestartInProgress());
        }
        Workflow workflow = this.addWorkflow(new Workflow(this.session, rep.getId(), (Map)config));
        workflow.addSteps(rep.getSteps());
        return workflow;
    }

    public void close() {
    }

    private ComponentModel getWorkflowComponent(String id) {
        ComponentModel component = this.realm.getComponent(id);
        if (component == null || !WorkflowProvider.class.getName().equals(component.getProviderType())) {
            throw new BadRequestException("Not a valid resource workflow: " + id);
        }
        return component;
    }

    private WorkflowProvider getWorkflowProvider(Workflow workflow) {
        ComponentFactory factory = (ComponentFactory)this.sessionFactory.getProviderFactory(WorkflowProvider.class, "default");
        return (WorkflowProvider)factory.create(this.session, this.realm.getComponent(workflow.getId()));
    }

    private void processEvent(Stream<Workflow> workflows, WorkflowEvent event) {
        Map[] scheduledSteps = new Map[]{null};
        workflows.forEach(workflow -> {
            if (!workflow.isEnabled()) {
                log.debugf("Skipping workflow %s as it is disabled or has no steps", (Object)workflow.getName());
                return;
            }
            EventBasedWorkflow provider = new EventBasedWorkflow(this.session, this.getWorkflowComponent(workflow.getId()));
            try {
                WorkflowStateProvider.ScheduledStep scheduledStep;
                if (!provider.supports(event.getResourceType())) {
                    return;
                }
                DefaultWorkflowExecutionContext context = new DefaultWorkflowExecutionContext(this.session, (Workflow)workflow, event);
                if (scheduledSteps[0] == null) {
                    scheduledSteps[0] = this.stateProvider.getScheduledStepsByResource(event.getResourceId()).collect(Collectors.toMap(WorkflowStateProvider.ScheduledStep::workflowId, Function.identity()));
                }
                if ((scheduledStep = (WorkflowStateProvider.ScheduledStep)scheduledSteps[0].get(workflow.getId())) == null) {
                    if (provider.activate(context)) {
                        if (this.isAlreadyScheduledInSession(event, (Workflow)workflow)) {
                            return;
                        }
                        if (DurationConverter.isPositiveDuration((String)workflow.getNotBefore())) {
                            this.scheduleWorkflow(context);
                        } else {
                            this.runWorkflow(context);
                        }
                    }
                } else {
                    String executionId = scheduledStep.executionId();
                    String resourceId = scheduledStep.resourceId();
                    if (provider.restart(context)) {
                        new DefaultWorkflowExecutionContext(this.session, (Workflow)workflow, event, scheduledStep).restart(0);
                    } else if (provider.deactivate(context)) {
                        log.debugf("Workflow '%s' cancelled for resource %s (execution id: %s)", (Object)workflow.getName(), (Object)resourceId, (Object)executionId);
                        this.stateProvider.remove(executionId);
                    }
                }
            }
            catch (WorkflowInvalidStateException e) {
                workflow.setEnabled(false);
                workflow.setError(e.getMessage());
                workflow.updateConfig(workflow.getConfig(), null);
                log.warnf("Workflow %s was disabled due to: %s", (Object)workflow.getId(), (Object)e.getMessage());
            }
        });
    }

    private boolean isAlreadyScheduledInSession(WorkflowEvent event, Workflow workflow) {
        String resourceId;
        boolean isAlreadyScheduled;
        HashMap<String, Set> scheduled = (HashMap<String, Set>)this.session.getAttribute("kc.workflow.scheduled");
        if (scheduled == null) {
            scheduled = new HashMap<String, Set>();
            this.session.setAttribute("kc.workflow.scheduled", scheduled);
        }
        boolean bl = isAlreadyScheduled = !scheduled.computeIfAbsent(resourceId = event.getResourceId(), k -> new HashSet()).add(workflow.getId());
        if (isAlreadyScheduled) {
            log.debugf("Event %s for workflow %s and resource %s was previously processed for the resource", (Object)workflow.getName(), (Object)resourceId);
        }
        return isAlreadyScheduled;
    }

    private void scheduleWorkflow(WorkflowExecutionContext context) {
        this.executor.runTask(this.session, (Runnable)((Object)new ScheduleWorkflowTask((DefaultWorkflowExecutionContext)context)));
    }

    private void runWorkflow(DefaultWorkflowExecutionContext context) {
        this.executor.runTask(this.session, (Runnable)((Object)new RunWorkflowTask(context)));
    }

    private WorkflowStepRepresentation toRepresentation(WorkflowStep step) {
        return new WorkflowStepRepresentation(step.getId(), step.getProviderId(), step.getConfig());
    }

    private Workflow addWorkflow(Workflow workflow) {
        ComponentModel model = new ComponentModel();
        model.setId(workflow.getId());
        model.setParentId(this.realm.getId());
        model.setProviderId("default");
        model.setProviderType(WorkflowProvider.class.getName());
        MultivaluedHashMap config = workflow.getConfig();
        if (config != null) {
            model.setConfig(config);
        }
        workflow = new Workflow(this.session, this.realm.addComponentModel(model));
        this.scheduleWorkflow(workflow);
        return workflow;
    }

    private void scheduleWorkflow(Workflow workflow) {
        String scheduled = (String)workflow.getConfig().getFirst((Object)"schedule.after");
        if (scheduled != null) {
            Duration duration = DurationConverter.parseDuration((String)scheduled);
            TimerProvider timer = (TimerProvider)this.session.getProvider(TimerProvider.class);
            timer.schedule((TaskRunner)new ClusterAwareScheduledTaskRunner(this.sessionFactory, (ScheduledTask)new ScheduledWorkflowRunner(workflow.getId(), this.realm.getId()), duration.toMillis()), duration.toMillis());
        }
    }

    void cancelScheduledWorkflow(Workflow workflow) {
        ((TimerProvider)this.session.getProvider(TimerProvider.class)).cancelTask(new ScheduledWorkflowRunner(workflow.getId(), this.realm.getId()).getTaskName());
    }

    void rescheduleWorkflow(Workflow workflow) {
        this.cancelScheduledWorkflow(workflow);
        this.scheduleWorkflow(workflow);
    }
}

