// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/*
 * This public interface for constants is provided for
 * the use of the Encryption SDK for JavaScript only.  It can be used
 * as a reference but is not intended to be use by any packages other
 * than the Encryption SDK for JavaScript.
 *
 * See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/reference.html
 *
 * https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-aad (algorithms with signing)
 * https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-version
 * https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-content-type
 * https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/body-aad-reference.html (Body AAD Content)
 * https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-type
 * https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/body-aad-reference.html#body-aad-sequence-number
 */
export const ENCODED_SIGNER_KEY = 'aws-crypto-public-key';
/** @deprecated use import { MessageFormat } from '@aws-crypto/material-management' */
export { MessageFormat as SerializationVersion } from '@aws-crypto/material-management';
export var ContentType;
(function (ContentType) {
    ContentType[ContentType["NO_FRAMING"] = 1] = "NO_FRAMING";
    ContentType[ContentType["FRAMED_DATA"] = 2] = "FRAMED_DATA";
})(ContentType || (ContentType = {}));
Object.freeze(ContentType);
export var ContentAADString;
(function (ContentAADString) {
    ContentAADString["FRAME_STRING_ID"] = "AWSKMSEncryptionClient Frame";
    ContentAADString["FINAL_FRAME_STRING_ID"] = "AWSKMSEncryptionClient Final Frame";
    ContentAADString["NON_FRAMED_STRING_ID"] = "AWSKMSEncryptionClient Single Block";
})(ContentAADString || (ContentAADString = {}));
Object.freeze(ContentAADString);
export var ObjectType;
(function (ObjectType) {
    ObjectType[ObjectType["CUSTOMER_AE_DATA"] = 128] = "CUSTOMER_AE_DATA";
})(ObjectType || (ObjectType = {}));
Object.freeze(ObjectType);
export var SequenceIdentifier;
(function (SequenceIdentifier) {
    SequenceIdentifier[SequenceIdentifier["SEQUENCE_NUMBER_END"] = 4294967295] = "SEQUENCE_NUMBER_END";
})(SequenceIdentifier || (SequenceIdentifier = {}));
Object.freeze(SequenceIdentifier);
export var Maximum;
(function (Maximum) {
    // Maximum number of messages which are allowed to be encrypted under a single cached data key
    Maximum[Maximum["MESSAGES_PER_CACHED_KEY_LIMIT"] = 4294967296] = "MESSAGES_PER_CACHED_KEY_LIMIT";
    /* Maximum number of bytes that are allowed to be encrypted
     * under a single cached data key across messages.
     * The maximum value defined in the AWS Encryption SDK specification is 2 ** 63 - 1.
     * However Javascript can only perform safe operations on values
     * up to Number.MAX_SAFE_INTEGER === 9007199254740991 === 2 ** 53 - 1.
     * e.g
     * Number.MAX_SAFE_INTEGER + 1 === Number.MAX_SAFE_INTEGER + 2 => true
     * Number.MAX_SAFE_INTEGER + 1 > Number.MAX_SAFE_INTEGER + 2 => false
     * Number.MAX_SAFE_INTEGER + 1 < Number.MAX_SAFE_INTEGER + 2 => false
     *
     * This means that after 2 ** 53 - 1 the process of accumulating a byte count
     * will never yield an accurate comparison and so, never halt.
     *
     * The choice here to use 2 ** 53 - 1 instead of Number.MAX_SAFE_INTEGER is deliberate.
     * This is because in the future Number.MAX_SAFE_INTEGER could be raised to 2 ** 66
     * or some value larger 2 ** 63.
     */
    Maximum[Maximum["BYTES_PER_CACHED_KEY_LIMIT"] = 9007199254740991] = "BYTES_PER_CACHED_KEY_LIMIT";
    /* This value should be Maximum.FRAME_COUNT * Maximum.FRAME_SIZE.
     * However this would be ~ 2 ** 64, much larger than Number.MAX_SAFE_INTEGER.
     * For the same reasons outlined above in BYTES_PER_CACHED_KEY_LIMIT
     * this value is set to 2 ** 53 - 1.
     */
    Maximum[Maximum["BYTES_PER_MESSAGE"] = 9007199254740991] = "BYTES_PER_MESSAGE";
    /* Maximum number of bytes for a single AES-GCM "operation."
     * This is related to the GHASH block size,
     * and can be thought of as the maximum bytes
     * that can be encrypted with a single key/IV pair.
     * The AWS Encryption SDK for Javascript
     * does not support non-framed encrypt
     * https://github.com/awslabs/aws-encryption-sdk-specification/blob/master/data-format/message-body.md#non-framed-data
     * So this value is only needed to ensure
     * that messages submitted for decrypt
     * are well formed.
     */
    Maximum[Maximum["BYTES_PER_AES_GCM_NONCE"] = 68719476704] = "BYTES_PER_AES_GCM_NONCE";
    // Maximum number of frames allowed in one message as defined in specification
    Maximum[Maximum["FRAME_COUNT"] = 4294967295] = "FRAME_COUNT";
    // Maximum bytes allowed in a single frame as defined in specification
    Maximum[Maximum["FRAME_SIZE"] = 4294967295] = "FRAME_SIZE";
    // Maximum bytes allowed in a non-framed message ciphertext as defined in specification
    Maximum[Maximum["GCM_CONTENT_SIZE"] = 4294967295] = "GCM_CONTENT_SIZE";
    Maximum[Maximum["NON_FRAMED_SIZE"] = 4294967295] = "NON_FRAMED_SIZE";
    // Maximum number of AAD bytes allowed as defined in specification
    Maximum[Maximum["AAD_BYTE_SIZE"] = 65535] = "AAD_BYTE_SIZE";
})(Maximum || (Maximum = {}));
Object.freeze(Maximum);
// Default frame length when using framing
export const FRAME_LENGTH = 4096;
// Message ID length as defined in specification
export var MessageIdLength;
(function (MessageIdLength) {
    MessageIdLength[MessageIdLength["V1"] = 16] = "V1";
    MessageIdLength[MessageIdLength["V2"] = 32] = "V2";
})(MessageIdLength || (MessageIdLength = {}));
Object.freeze(MessageIdLength);
/** @deprecated use MessageIdLength */
export const MESSAGE_ID_LENGTH = MessageIdLength.V1;
//# sourceMappingURL=data:application/json;base64,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