// -*- C++ -*-

#ifndef _HEAPMANAGER_H_
#define _HEAPMANAGER_H_

#include <stdlib.h>

#include "guard.h"
#include "cpuinfo.h"

template <typename LockType,
	  typename HeapType>
class HeapManager : public HeapType {
public:

  HeapManager (void)
  {
    HL::Guard<LockType> g (heapLock);

    /// Initialize all heap maps (nothing yet assigned).
    int i;
    for (i = 0; i < MaxThreads; i++) {
      setTidMap (i, 0);
    }
    for (i = 0; i < MaxHeaps; i++) {
      setInusemap (i, 0);
    }
  }

  /// Set this thread's heap id to 0.
  void chooseZero (void) {
    HL::Guard<LockType> g (heapLock);
    setTidMap ((int) CPUInfo::getThreadId() % MaxThreads, 0);
  }

  int findUnusedHeap (void) {

    HL::Guard<LockType> g (heapLock);

    int tid = CPUInfo::getThreadId();

    tid = tid % MaxThreads;

    
    int i = 0;
    while ((i < MaxHeaps) && (getInusemap(i)))
      i++;
    if (i >= MaxHeaps) {
      // Every heap is in use: pick heap one.
      i = 0;
      // i = (int) ( MaxHeaps * ((double) rand() / (double) RAND_MAX));
    }

    setInusemap (i, 1);
    setTidMap (tid, i);

    return i;
  }

  void releaseHeap (void) {
    // Decrement the ref-count on the current heap.

    HL::Guard<LockType> g (heapLock);
    
    // Statically ensure that the number of threads is a power of two.
    enum { VerifyPowerOfTwo = 1 / ((MaxThreads & ~(MaxThreads-1))) };
    
    int tid = CPUInfo::getThreadId() & (MaxThreads - 1);
    int heapIndex = getTidMap (tid);
    
    setInusemap (heapIndex, 0);
    
    // Prevent underruns (defensive programming).
    
    if (getInusemap (heapIndex) < 0) {
      setInusemap (heapIndex, 0);
    }
  }


private:

  // Disable copying.

  HeapManager (const HeapManager&);
  HeapManager& operator= (const HeapManager&);

  /// The lock, to ensure mutual exclusion.
  LockType heapLock;
};

#endif
