/* -*- C++ -*- */

// Author: Emery Berger, http://www.cs.umass.edu/~emery
//
// Copyright (c) 1998-2003, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#if defined(_WIN32)
#error "This library is for UNIX only."
// This is because it requires dlsym and family.
#endif


#include <stdlib.h>
#include <dlfcn.h>

#if !defined(RTLD_NEXT)
#define RTLD_NEXT ((void *) -1)
#endif

volatile int anyThreadCreated = 1;

extern "C" {

  size_t malloc_usable_size (void *);

  typedef void * mallocFunction (size_t);
  typedef void freeFunction (void *);
  typedef size_t msizeFunction (void *);

  typedef void exitFunction (int);
  exitFunction * trueExitFunction;

}

#include <heaplayers.h>

#include "logheap.h"

#if 0
#include "sizeheap.h"
#include "ansiwrapper.h"
#include "staticheap.h"
#endif

using namespace HL;

class LocalMallocHeap {
public:

  LocalMallocHeap (void)
    : freefn (NULL),
      msizefn (NULL),
      mallocfn (NULL),
      firsttime (true)
  {}

  inline void * malloc (size_t sz) {
    if (firsttime) {

      // We haven't initialized anything yet.

      // Initialize all of the malloc shim functions.

      freefn = (freeFunction *) dlsym (RTLD_NEXT, "free");
      msizefn = (msizeFunction *) dlsym (RTLD_NEXT, "malloc_usable_size");
      trueExitFunction = (exitFunction *) dlsym (RTLD_NEXT, "exit");
      mallocfn = (mallocFunction *) dlsym (RTLD_NEXT, "malloc");

      if (!(freefn && msizefn && trueExitFunction && mallocfn)) {
	fprintf (stderr, "Serious problem!\n");
	abort();
      }

      assert (freefn);
      assert (msizefn);
      assert (trueExitFunction);
      assert (mallocfn);

      // Now we're done with the first time through.
      firsttime = false;

      // Go get some memory from malloc!
      return (*mallocfn)(sz);
    }

    // Now, once we have mallocfn resolved, we can use it.
    // Otherwise, we're still in dlsym-land, and have to use our local heap.

    if (mallocfn) {
      return (*mallocfn)(sz);
    } else {
      void * ptr = localHeap.malloc (sz);
      assert (ptr);
      return ptr;
    }
  }

  inline void free (void * ptr) {
    if (mallocfn) {
      if (localHeap.isValid (ptr)) {
	// We got a pointer to the temporary allocation buffer.
	localHeap.free (ptr);
      } else {
	(*freefn)(ptr);
      }
    }
  }

  inline size_t getSize (void * ptr) {
    if (localHeap.isValid (ptr)) {
      return localHeap.getSize (ptr);
    } else if (mallocfn) {
      return (*msizefn)(ptr);
    } else {
      // This should never happen.
      return 0;
    }
  }

private:

  bool firsttime;   /// True iff we haven't initialized the shim functions.

  // Shim functions below.

  freeFunction *   freefn;
  msizeFunction *  msizefn;
  mallocFunction * mallocfn;

  /// The local heap (for use while we are in dlsym, installing the
  /// shim functions). Hopefully 64K is enough...

  ANSIWrapper<SizeHeap<StaticHeap<65536> > > localHeap;

};

using namespace HL;

class TheCustomHeapType :
  public LockedHeap<SpinLockType, LogHeap<LocalMallocHeap> > {};

inline static TheCustomHeapType * getCustomHeap (void) {
  static char thBuf[sizeof(TheCustomHeapType)];
  static TheCustomHeapType * th = new (thBuf) TheCustomHeapType;
  return th;
}

#if 1
extern "C" {
  void __attribute__((noreturn)) exit (int status) {
    getCustomHeap()->write();
    exitFunction * fn = (exitFunction *) trueExitFunction;
    if (fn) {
      (*fn)(status);
    }
  }
}
#endif

#if defined(_WIN32)
#pragma warning(disable:4273)
#endif

#include "wrapper.cpp"

