/*
                   Vertex - Internal Plugin Management

	This defines the plug-in structure used by Vertex internally,
	all this is private and should not be #include'ed by plug-ins.

 */

#ifndef VPIINTERNAL_H
#define VPIINTERNAL_H

#include <sys/types.h>
#include <gtk/gtk.h>

#include "splash.h"
#include "vpi.h"

/*
 *	Loaded plugin structure:
 */
typedef struct {

	/* Pointer to OS specific plug-in id, can be NULL for
	 * failed load/not loaded.
	 */
	void *handle;

	/* Pointer back to core structure. */
	void *core_ptr;

	/* Full path to the plugin binary. */
	char *filename;

	/* If plugin is global, then this will be true. */
	int global;

	/* If plug-in is currently processing, then this will be
	 * marked true and no additional plug-in functions on this
	 * plug-in should be called untill processing is false.
	 *
	 * The only exception is when the shutdown function needs to
	 * be called, it will be called even if the plug-in is marked
	 * processing.
	 */
	int processing;

	/* Marks if the initialization function needs to be called
	 * during a regular loop cycle.
	 */
	int need_call_init;

	/* Marks if the shutdown function needs to be called just
	 * before unloading in VPIUnload().
	 */
	int need_call_shutdown;

	/* Version information of plug-in, to check against current
	 * version of Vertex.
	 */
	int	version_major,
		version_minor;

	/* Time stamps, in system seconds. Value of 0 means unset or
	 * not available.
	 */
	time_t	time_created,		/* When first loaded. */
		time_installed,		/* Binary's create date. */
		time_last_used;		/* When last enabled or disabled. */

	/* Appearance information. */
	char *title;
	char *description;
	u_int8_t **list_icon_data;
	int list_icon_lines;
	GdkPixmap *icon_pixmap;
	GdkBitmap *icon_mask;

	/* Client data set by VPISetClientData(). */
	void *client_data;

	/* Manage callback function, since the manage callback function
	 * is called very often we record the pointer to the function
	 * here.
	 */
	VPI_PROTO_RETURN_MANAGE (*manage)(VPI_PROTO_INPUT_MANAGE);

	/* Operation callback function pointers and associated client
	 * data. The function pointer values here are ONLY valid if
	 * and ONLY IF the member handle is not NULL.
	 */
	/* Null. */
        int (*null)(vpi_id *, vpi_op_null_struct *, void *);
        void *null_data;

        int (*import)(vpi_id *, vpi_op_import_struct *, void *);
        void *import_data;
        int (*import_fcheck)(
		vpi_id *,
		const char *,		/* Full path. */
		const char *,		/* Just the extension, ie ".txt". */
		void *			/* import_fcheck_data */
	);
        void *import_fcheck_data;
	char **import_ftype_ext;
	char **import_ftype_name;
	int total_import_ftypes;

        int (*export)(vpi_id *, vpi_op_export_struct *, void *);
        void *export_data;
        char **export_ftype_ext;
        char **export_ftype_name;
        int total_export_ftypes;

	int (*render)(vpi_id *, vpi_op_render_struct *, void *);
        void *render_data;
	char *render_label;	/* Displayed as the render menu item label. */

	int (*configure)(vpi_id *, vpi_op_configure_struct *, void *);
	void *configure_data;

	int (*configuration_changed)(
		vpi_id *, vpi_op_configuration_changed_struct *, void *
	);
	void *configuration_changed_data;

} vma_plugin_struct;


/* The given input should be a vpi_id *. */
#define VMA_PLUGIN(p)	((vma_plugin_struct *)(p))


extern const char *VPIGetOSLoadError(void);

extern int VPIEnable(vma_plugin_struct *pi);
extern int VPIDisable(vma_plugin_struct *pi);
extern void *VPIGetFunction(vma_plugin_struct *pi, const char *name);
extern vma_plugin_struct *VPILoad(
	const char *filename,
	int load_as_disabled,
	int *error_code,
	void *core_ptr
);
extern void VPIUnload(vma_plugin_struct *pi);

extern int VPILoadPluginsFromDirectory(
        vma_plugin_struct ***plugin, int *total_plugins,
        const char *path,
        int load_as_disabled,
	int is_global,
	void *core_ptr,
	splash_win_struct *splash_win
);
extern int VPICListUpdateRow(
        GtkCList *clist, gint row, vma_plugin_struct *plugin_ptr
);
extern int VPICListAppend(
        GtkCList *clist,
        vma_plugin_struct **plugin, int total_plugins
);
extern int VPIDoGetInfo(vma_plugin_struct *plugin_ptr);
extern int VPIDoOpConfigurationChanged(vma_plugin_struct *plugin_ptr);


#endif	/* VPIINTERNAL_H */
