/*
 * chan_oh323.c
 *
 * Generic OpenH323 Channel Driver for ASTERISK PBX.
 * 
 * Copyright (C) 2002-2005 Inaccess Networks.
 * Michael Manousos <manousos@inaccessnetworks.com>
 *
 * This file is part of "H.323 support for ASTERISK"
 *
 * "H.323 support for ASTERISK" is free software;
 * you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version. 
 *
 * "H.323 support for ASTERISK" is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details. 
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 * In-Band DTMF detection is based on code sent by Klaus-Peter Junghanns
 * in ASTERISK mailing list.
 *
 * $Id: chan_oh323.c,v 1.126.2.7 2005/01/21 15:14:08 manousos Exp $
 *
 */

/******************************************************************************/
/* System and application's include files *************************************/
#include <string.h>
#include <asterisk/lock.h>
#include <asterisk/channel.h>
#include <asterisk/channel_pvt.h>
#include <asterisk/frame.h>
#include <asterisk/config.h>
#include <asterisk/logger.h>
#include <asterisk/module.h>
#include <asterisk/pbx.h>
#include <asterisk/cli.h>
#include <asterisk/options.h>
#include <asterisk/dsp.h>
#include <asterisk/callerid.h>
#include <asterisk/utils.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#include <netinet/ip.h>
#include <sys/signal.h>

#include "wrapper.hxx"
#include "oh323.h"

/******************************************************************************/
/* String variables required by ASTERISK **************************************/
static char *type = "OH323";
static char *desc = "InAccess Networks OpenH323 Channel Driver";
static char *tdesc = "InAccess Networks OpenH323 Channel Driver";
static char *config = "oh323.conf";

/******************************************************************************/
/* G.723.1 frame handling *****************************************************/

#undef	DEBUG_CIRCBUFFER
static unsigned const G7231FrameSize[4] = { 24, 20, 4, 1};
#define	G7231SF_BUFSIZE	4096
struct G7231SF {
	unsigned char buffer[G7231SF_BUFSIZE];
	int head;			/* The position where free space begins */
	int tail;			/* The position of a whole G.723.1 frame */
	int free_space;		/* The available space of the buffer */
};

/* Function prototypes */
struct G7231SF * G7231SF_new(void);
void G7231SF_free(struct G7231SF *s);
int G7231SF_push(struct G7231SF *s, char *buf, int len);
int G7231SF_pop(struct G7231SF *s, char *data, int len, int *framenum);
int G7231SF_isempty(struct G7231SF *s);

/* Function implementation */
struct G7231SF * G7231SF_new(void)
{
	struct G7231SF *s;
	s = malloc(sizeof(struct G7231SF));
	if (s == NULL) {
		ast_log(LOG_WARNING, "Failed to create G.723.1 SF.\n");
		return(NULL);
	}
	memset(s, 0, sizeof(struct G7231SF));
	s->head = 0;
	s->tail = 0;
	s->free_space = G7231SF_BUFSIZE;
	return(s);
}
void G7231SF_free(struct G7231SF *s)
{
	if (s)
		free(s);
}
/**
 * buf - The buffer holding the data
 * len - the length of data in "buf"
 *
 * The return value is -1 on error, 0 on success.
 */
int G7231SF_push(struct G7231SF *s, char *buf, int len)
{
	int tmp1;
	if (len >= s->free_space) {
		ast_log(LOG_WARNING, "No more space in G.723.1 SF.\n");
		return(-1);
	}
	if (s->head + len > G7231SF_BUFSIZE) {	/* Wrap around */
		tmp1 = G7231SF_BUFSIZE - s->head;
		memcpy((char *)(s->buffer + s->head), buf, tmp1);
		memcpy((char *)(s->buffer), (char *)(buf + tmp1), len - tmp1);
		s->head = len - tmp1;
		s->free_space -= len;
#ifdef	DEBUG_CIRCBUFFER
		ast_log(LOG_DEBUG, "PUSH: Buffer wrap head=%d, tail=%d free=%d.\n", 
					s->head, s->tail, s->free_space);
#endif
	} else {	/* No wrap */
		memcpy((char *)(s->buffer + s->head), buf, len);
		s->head += len;
		s->free_space -= len;
	}
#ifdef	DEBUG_CIRCBUFFER
	ast_log(LOG_DEBUG, "PUSH: head=%d, tail=%d, free=%d.\n", 
					s->head, s->tail, s->free_space);
#endif
	return(0);
}
/**
 * data - The buffer where the data will be stored
 * len - The length of the "data" buffer
 * framenum - If this has a value <= 0, then
 *            the function will return all the whole frames available
 *            in the internal buffer. If this has a value > 0, it will
 *            return up to this number of frames. This argument is updated
 *            with the number of frames read.
 *
 * The return value is -1 on error or the number of bytes stored in the
 * "data" buffer.
 */
int G7231SF_pop(struct G7231SF *s, char *data, int len, int *framenum)
{
	int tmp0, tmp1, tmp2, cur_index, framelim;
	if (s->free_space == G7231SF_BUFSIZE) {
		ast_log(LOG_WARNING, "G.723.1 SF is empty.\n");
		*framenum = 0;
		return(-1);
	}
	framelim = *framenum;
	*framenum = 0;
	cur_index = s->tail;
	tmp1 = 0;
	tmp0 = 0;
	while (1) {
		cur_index += tmp1;
		if (cur_index >= G7231SF_BUFSIZE)
			cur_index = cur_index - G7231SF_BUFSIZE;
		if (tmp1 >= G7231SF_BUFSIZE - s->free_space) {
			if (tmp1 > G7231SF_BUFSIZE - s->free_space) {
				tmp1 -= tmp0;
				--(*framenum);
			}
			if (tmp1 == 0) {
				ast_log(LOG_WARNING, "G.723.1 SF contains no full frames.\n");
				*framenum = 0;
				return(-1);
			}
			if (tmp1 > len) {
				ast_log(LOG_WARNING, "Not enough space to store G.723.1 frame.\n");
				*framenum = 0;
				return(-1);
			}
			break;
		}
		if (*framenum == framelim)
			break;
		tmp0 = G7231FrameSize[((char *)(s->buffer))[cur_index] & 0x3];
#ifdef	DEBUG_CIRCBUFFER
		ast_log(LOG_DEBUG, "Frame length is %d\n", tmp0);
#endif
		tmp1 += tmp0;
		++(*framenum);
	}
	memset(data, 0, len); /* XXX */
	if (s->tail + tmp1 > G7231SF_BUFSIZE) { 	/* Wrap around */
		tmp2 = G7231SF_BUFSIZE - s->tail;
		memcpy(data, (char *)(s->buffer + s->tail), tmp2);
		memcpy((char *)(data + tmp2), s->buffer, tmp1 - tmp2);
		s->tail = tmp1 - tmp2;
		s->free_space += tmp1;
#ifdef	DEBUG_CIRCBUFFER
		ast_log(LOG_DEBUG, "POP: Buffer wrap head=%d, tail=%d free=%d.\n", 
					s->head, s->tail, s->free_space);
#endif
	} else {	/* No wrap */
		memcpy(data, (char *)(s->buffer + s->tail), tmp1);
		s->tail += tmp1;
		s->free_space += tmp1;
	}
#ifdef	DEBUG_CIRCBUFFER
	ast_log(LOG_DEBUG, "POP: head=%d, tail=%d free=%d.\n", 
					s->head, s->tail, s->free_space);
#endif
	return(tmp1);
}
int G7231SF_isempty(struct G7231SF *s)
{	
	if (s->free_space == G7231SF_BUFSIZE)
		return(1);
	return(0);
}
/******************************************************************************/

/* Configuration of the builtin call rate limiter (ingress direction) */

#define	MAX_INCALLS						20
#define	MAX_INCALLTIME					20000	/* In ms */
#define	MAX_INCALLTIME_CALLTHRESHOLD	30

typedef struct {
	struct timeval tv;
	struct timeval tdiff;
	int passed;
} time_struct;

/* Ingress call rate limiter function prototypes */
int in_call_rate_limiter_init(int, int);
void in_call_mark(int, int);
int in_call_rate_update(void);
int in_call_number_blocked(void);
int in_call_number_passed(void);
int in_call_time_get(void);
int in_call_blockratio_get(void);
int in_call_passratio_get(void);
int in_call_rate_get(void);

/* Ingress call rate limiter  parameters */
static int incall_times = 0;
static int incall_pos = 0;
static int incall_cur = 0;

static int incall_time_period = 0;
static int incall_max = 0;
static int incall_rate_limiter_status = 0;
static float in_call_max_rate = 999999.0;
static time_struct *intimes = NULL;

/**
 * Initialize the ingress call rate limiter.
 *
 * call_number : Max number of calls
 * call_time   : Time period in milliseconds
 * 
 *                         call_number
 * Maxrate (CPS) = --------------------------- * 1000
 *                   call_time (milliseconds)
 */
int in_call_rate_limiter_init(int call_number, int call_time)
{
	if (!call_number || !call_time) {
		/* Disable rate limiter */
		incall_rate_limiter_status = 0;
		in_call_max_rate = 999999.0;
		return(0);
	}
	incall_rate_limiter_status = 1;
	incall_max = call_number;
	incall_time_period = call_time;
	intimes = (time_struct *)malloc(sizeof(time_struct) * incall_max);
	if (intimes == NULL)
		return(-1);
	memset((char *)intimes, 0, sizeof(time_struct) * incall_max);
	if (incall_time_period > 0)
		in_call_max_rate = (float)incall_max * 1000 / (float)incall_time_period;
	else
		in_call_max_rate = 999999.0;
	return(0);
}

void in_call_mark(int index, int val)
{
	if ((!incall_rate_limiter_status) || (index < 0) || (index > incall_max-1))
		return;
	intimes[index].passed = val;
}

int in_call_rate_update(void)
{
	struct timeval tv;
	int prev;

	if (!incall_rate_limiter_status)
		return(0);

	if (gettimeofday(&tv, NULL) < 0)
		return(-1);
	if (incall_times < incall_max) {
		intimes[incall_times].tv = tv;
		if (incall_times == 0) {
			intimes[incall_times].tdiff.tv_sec = 0;
			intimes[incall_times].tdiff.tv_usec = 0;
		} else {
			if (tv.tv_usec < intimes[incall_times-1].tv.tv_usec) {
				tv.tv_usec += 1000000;
				tv.tv_sec -= 1;
			}
			intimes[incall_times].tdiff.tv_sec = tv.tv_sec - intimes[incall_times-1].tv.tv_sec;
			intimes[incall_times].tdiff.tv_usec = tv.tv_usec - intimes[incall_times-1].tv.tv_usec;
		}
		incall_cur = incall_times;
		incall_times++;
	} else {
		intimes[incall_pos].tv = tv;
		if (incall_pos == 0)
			prev = incall_max - 1;
		else
			prev = incall_pos - 1;
		if (tv.tv_usec < intimes[prev].tv.tv_usec) {
			tv.tv_usec += 1000000;
			tv.tv_sec -= 1;
		}
		intimes[incall_pos].tdiff.tv_sec = tv.tv_sec - intimes[prev].tv.tv_sec;
		intimes[incall_pos].tdiff.tv_usec = tv.tv_usec - intimes[prev].tv.tv_usec;
		incall_cur = incall_pos;
		incall_pos++;
		if (incall_pos == incall_max)
			incall_pos = 0;
	}
	/* Be pesimistic for the call */
	intimes[incall_cur].passed = 0;

	/*
	ast_log(LOG_DEBUG, "POS: %d, DIFF: %ld sec - %ld usec.\n",
					incall_cur,
					intimes[incall_cur].tdiff.tv_sec,
					intimes[incall_cur].tdiff.tv_usec);
	*/
	return(incall_cur);
}

int in_call_number_blocked(void)
{
	int i, blocked;

	if (!incall_rate_limiter_status)
		return(0);

	if (incall_times != incall_max)
		return(0);

	blocked = 0;
	for (i = 0; i < incall_max; i++)
		blocked += (intimes[i].passed ? 0 : 1);

	return(blocked);
}

int in_call_number_passed(void)
{
	int i, passed;

	if (!incall_rate_limiter_status)
		return(0);

	if (incall_times != incall_max)
		return(0);

	passed = 0;
	for (i = 0; i < incall_max; i++)
		passed += intimes[i].passed;

	return(passed);
}

int in_call_time_get(void)
{
	int i;
	struct timeval tv;
	long int total;

	if (!incall_rate_limiter_status)
		return(0);

	if (incall_times != incall_max)
		return(0);

	tv.tv_sec = 0;
	tv.tv_usec = 0;
	for (i = 0; i < incall_max; i++) {
		tv.tv_sec += intimes[i].tdiff.tv_sec;
		tv.tv_usec += intimes[i].tdiff.tv_usec;
	}
	total = tv.tv_sec * 1000 + tv.tv_usec / 1000; /* ms */
	/*ast_log(LOG_DEBUG, "%d in calls in %ld ms.\n", incall_max, total);*/

	return((int)total);
}

/* Return the % of ingress calls blocked */
int in_call_blockratio_get(void)
{
	int total_blocked, total_time;
	long int diff;
	struct timeval tv;

	if (!incall_rate_limiter_status)
		return(0);

	if (incall_times != incall_max)
		return(0);

	total_blocked = (in_call_number_blocked() * 100) / incall_max;
	total_time = in_call_time_get();

	if (total_time > 0) {
		if (gettimeofday(&tv, NULL) < 0)
			return(-1);
		diff = (tv.tv_sec - intimes[incall_cur].tv.tv_sec) * 1000 +
				(tv.tv_usec - intimes[incall_cur].tv.tv_usec) / 1000;
		return((total_blocked * total_time) / (total_time + diff));
	} else
		return(0);
}

/* Return the % of ingress calls passed */
int in_call_passratio_get(void)
{
	int total_passed, total_time;
	long int diff;
	struct timeval tv;

	if (!incall_rate_limiter_status)
		return(0);

	if (incall_times != incall_max)
		return(0);

	total_passed = (in_call_number_passed() * 100) / incall_max;
	total_time = in_call_time_get();

	if (total_time > 0) {
		if (gettimeofday(&tv, NULL) < 0)
			return(-1);
		diff = (tv.tv_sec - intimes[incall_cur].tv.tv_sec) * 1000 +
				(tv.tv_usec - intimes[incall_cur].tv.tv_usec) / 1000;
		return((total_passed * total_time) / (total_time + diff));
	} else
		return(0);
}

/* Return '100 x rate' in CPS */
int in_call_rate_get(void)
{
	int total_time;
	long int diff;
	struct timeval tv;

	if (!incall_rate_limiter_status)
		return(0);

	total_time = in_call_time_get();
	if (total_time > 0) {
		if (gettimeofday(&tv, NULL) < 0)
			return(-1);
		diff = (tv.tv_sec - intimes[incall_cur].tv.tv_sec) * 1000 +
				(tv.tv_usec - intimes[incall_cur].tv.tv_usec) / 1000;
		return((incall_max * 100000) / (total_time + diff));
	} else
		return(0);
}

/******************************************************************************/


/******************************************************************************/
/* Channel driver's internal stuff ********************************************/

/* Statistics of channel's usage */
typedef struct {
    int incall;
    int outcall;
	int setup_sent;
	int setup_recv;
	int block_incall;
	int block_outcall;
	int proto_err;
	int call_err;
	int answer_err;
	struct timeval boot_time;
} OH323STATS;

/* Enable/Disable verbose debug info */
static int	oh323_verbose_debug = 0;

/** The capabilities (codecs) supported by H.323 channel driver */
static int oh323_capability = OH323_CAPABILITY_FULLBANDWIDTH;
static int oh323_full_capability = OH323_CAPABILITY_FULLBANDWIDTH;

/** Static buffer for fifo name creation (out-dated) */
//static char name[MAX_FIFO_NAME];

/** Mutex for the "name" buffer (out-dated) */
//static ast_mutex_t name_lock;

/** Mutex for the usage counter */
static ast_mutex_t usecnt_lock;
static int usecnt = 0;

/** Mutex for the OpenH323 calls table */
static ast_mutex_t oh323_tab_lock;

/** Mutex for the OpenH323 statistics structure */
static ast_mutex_t oh323_stats_lock;
static OH323STATS oh323_stats;

/* Protect the monitoring thread, so only one process can kill or start it, 
   and not when it's doing something critical. */
static ast_mutex_t mon_lock;

/* This is the thread for the monitor which checks for input on the channels
   which are not currently in use.  */
static pthread_t monitor_thread = 0;
static int monitor_shutdown = 0;
static int monitor_running = 0;

/** Mutex for reloading the configuration of the driver */
static ast_mutex_t oh323_reload_lock;
static int oh323_reloading;

/* Scheduled actions (channel cleanups) on OH323 channels
 * and gatekeeper registration retries. */
static struct sched_context *oh323_sched;
static int oh323_sched_gkid;

/*
 * Channel variables used by OpenH323 channel driver
 */
static char *oh323_var_tab[] = {
	"OH323_TEST",
	"OH323_CTOKEN",		/* Call Token (W) */
	"OH323_SRCALIAS",	/* Source alias(es) (W) */
	"OH323_DSTALIAS",	/* Destination alias(es) (W) */
	"OH323_SRCE164",	/* Source E.164 number (W) */
	"OH323_DSTE164",	/* Destination E.164 number (W) */
	"OH323_REMOTEAPP",	/* Remote application name (W) */
	"OH323_CHANCODEC",	/* The codec used by the channel (W) */
	"OH323_OUTCODEC",	/* The codec to be used for an outgoing 
						   H.323 call (R) */
	"OH323_RADDR",		/* The IP address of the remote end (W) */
	"OH323_LADDR",		/* The IP address of the local end (W) */
	"OH323_RRTP",		/* The remote RTP end of the call (W) */
	"OH323_LRTP",		/* The local RTP end of the call (W) */
	"OH323_CALL_ENDREASON",	/* Call end reason (W) */
	"OH323_CALL_DURATION",	/* Call duration (W) */
	"OH323_STATSFILE",	/* The file where the session stats will be
						   saved (R) */
	NULL
};

/*
 * Structures for storing configuration 
 * options of H.323 channel driver.
 */
struct oh323_reginfo {
	char context[AST_MAX_EXTENSION];
	char **alias;
	int  alias_num;
	char **prefix;
	int  prefix_num;
	struct oh323_reginfo *next;
};

struct oh323_codecinfo {
	int codec;
	int format;
	int frames;
	struct oh323_codecinfo *next;
};

struct oh323_options {
	char listenAddress[128];
	int  listenPort;
	int  connectPort;
	int  tcpStart;		/* TCP (H.245) ports */
	int  tcpEnd;
	int  udpStart;		/* UDP (RAS) ports */
	int  udpEnd;
	int  rtpStart;		/* UDP (RTP) ports */
	int  rtpEnd;
	int  fastStart;
	int  h245Tunnelling;
	int  h245inSetup;
	int  inBandDTMF;
	int  silenceSuppression;
	int  jitterMin;
	int  jitterMax;
#ifdef HAS_OH323MODS
	int  jitterDec;
	int  jitterInc;
#endif
	int  ipTos;
	int  outboundMax;
	int  inboundMax;
	int  simultaneousMax;
	int  totalNum;
	int  bwlimit;
	int  crlCallNumber;			/* Call rate limiter params */
	int  crlCallTime;
	int  crlThreshold;
	int  wrapLibTraceLevel;		/* Wrapper library trace level */
	int  libTraceLevel;			/* OpenH323/Pwlib trace level */
	char libTraceFile[256];
	int  amaFlags;
	char accountCode[20];
	gkmode_t gatekeeperMode;
	char gatekeeperName[128];
	char gatekeeperPass[128];
	int gatekeeperTTL;			/* TTL for our registration with the GK */
	uimode_t userInputMode;
	char context[AST_MAX_EXTENSION];
	char **alias_tab;
	int alias_num;
	char **gateway_prefix_tab;
	int prefix_num;
	struct oh323_reginfo *regInfo;
	struct oh323_codecinfo *codecInfo;
	/* The data of the "extensions" section are held in a separate struct */
};
static struct oh323_options config_options;

struct rtp_session {
	/* Null terminated strings with the following format:
	 *   address:RTP_port
	 * The RTCP port is 'RTP_port + 1'
	 */
	char local_addr[128];
	char remote_addr[128];
};

/** Structure for transfering data occurred inside 
 * an exception of OpenH323 Library. */
struct exception_oh323 {

	/** Type of exception data. */
	int type;

	/** Buffer with exception data */
	char data[MAX_EXCEPTION_DATA];
};

/**
 * Private structure of a OpenH323 channel.
 */
struct chan_oh323_pvt {

	/** Channel's file descriptors. The two arrays hold 
	 * 2 socketpairs. The first member of each array (index 0)
	 * is used inside the channel driver, whereas the 
	 * second member (index 1) is used inside the OpenH323.
	 * player_fd[0] is used to read data from the network
	 * (OpenH323 library) and record_fd[0] is used to write
	 * data to the network. player_fd[0] is select()'ed 
	 * inside ASTERISK. */
	int	player_fd[2];
	int recorder_fd[2];
	int close_player_fd, close_recorder_fd;

	/** Player fifo name (data flow: OH323 --> ASTERISK) */
	char playername[MAX_FIFO_NAME];

	/** Recorder fifo name (data flow: ASTERISK --> OH323) */
	char recordername[MAX_FIFO_NAME];

	/** Event pipe to Asterisk. event_pipe[0] is select()'ed
	 * inside ASTERISK. */
	int event_pipe[2];

	/** Options to be used during call setup */
	call_options_t call_opt;

	/** Call details */
	call_details_t cd;

	/** Direction established */
	lchan_dir_t direction;

	/** State indication (OH323_STATE_*) */
	int i_state;

	/** Used for inband DTMF detection */
	struct ast_dsp *vad;

	/** The capability (codec) used by this channel */
	int capability;
	int rxcapability;
	int txcapability;
	
	/** Buffer sizes for RX/TX */
	int rx_buf_size, tx_buf_size;
	int rxcount, txcount;

	/** Endpoint's RTP info (address, port) */
	char ep_rtp_addr[4];
	int  ep_rtp_port;

	/** Owner if we have one */
	struct ast_channel *owner;

	/** Asterisk frame (used for reading frames) */
	struct ast_frame fr;

	/** Last known RX/TX formats */
	int last_tx_format, last_rx_format;

	/** Smoother struct for queuing frames for transmission */
	struct ast_smoother *tx_smooth;

	struct G7231SF *rx_g7231_smooth, *tx_g7231_smooth;

	/** Static buffer for reading frames */
	char buf[AST_FRIENDLY_OFFSET + OH323_MAX_BUF];
	char txtbuf[AST_FRIENDLY_OFFSET + OH323_MAX_BUF];

	/** Flag indicating whether the call was initiated from
	 * the remote endpoint or from ASTERISK. */
	int from_remote;

	/** Flag indicating that the call is being cleared */
	int clearing_call;

	/** Structure containing exception data */
	struct exception_oh323 except_struct;

	/** RTP session info */
	struct rtp_session rtp;

	/** RTP session statistics */
	rtp_stats_t rtp_stats[RTPSTATS_MAX_NUM];
	rtp_stats_t *head_stats, *tail_stats;
	ast_mutex_t stats_lock;

	/** Scheduler ID and current index */
	int sched_id, index;

	/** Already gone */
	int	alreadygone;

	/** The call must be destroyed */
	int needs_destroy;
	int normread;
};

/**
 * Array of OpenH323 calls (either established or inactive).
 */
static struct chan_oh323_pvt **oh323_tab;

/**
 * Data for H.323 terminals that are treated as PBX extensions.
 * These data are used basically for call processing/services 
 * (e.g. enable/disable call transfer,...).
 */
struct oh323_ep {

	/** The name of the entry */
	char name[OH323_MAX_EXTNAME];

	/** The IP address or hostname of the terminal (if known) */
	char host[OH323_MAX_EXTNAME];

	/** Caller ID for the terminal */
	char callerid[OH323_MAX_EXTNAME];

	/** This terminal can enable/disable call forward */
	int cancallforward;

	/** The extension to forward to */
	char forwardexten[AST_MAX_EXTENSION];

	struct oh323_ep *next;
};
static struct {
	struct oh323_ep *head;
	ast_mutex_t lock;
} oh323_ep_list;


/* Some function prototypes of the channel driver */
static int find_call(const char *);
static void oh323_close_call_fds(int index);
static void	oh323_destroy(int);
//static char	*build_fifo_name(lchan_dir_t, int);
static int	copy_call_details(call_details_t *, call_details_t *);
static void	clear_call_details(call_details_t *);
static int	oh323_codec2format(int);
static char *oh323_codecset2str(int, struct oh323_codecinfo *, char **);
static void oh323_format2codecset(int, int *, int);
static char	*oh323_state2str(int);
static char *oh323_direction2str(lchan_dir_t);
static int context_from_alias(char *, char **);
static int context_from_prefix(char *, char **);
static int oh323_release(void *);
void oh323_atexit(void);

/*
 * CLI extensions 
 */
static int oh323_show_conf(int fd, int argc, char *argv[])
{
	char	*tmp, *gk_res;
	char	*gk_ptr, tmp_gk[200];

	if (argc != 3)
		return RESULT_SHOWUSAGE;
	ast_mutex_lock(&oh323_stats_lock);
	switch (h323_get_gk(tmp_gk, sizeof(tmp_gk))) {
	case 0:
		gk_res = "(Registered)";
		gk_ptr = (char *)tmp_gk;
		break;
	case OH323GK_FAILED:
		gk_res = "Failed";
		gk_ptr = "";
		break;
	case OH323GK_NOTREGISTERED:
		gk_res = "(Not registered)";
		gk_ptr = (char *)tmp_gk;
		break;
	case OH323GK_NOTUSED:
		gk_res = "No gatekeeper";
		gk_ptr = "";
		break;
	default:
		gk_res = "";
		gk_ptr = "";
		break;
	}
	ast_cli(fd, "\n");
	ast_cli(fd, " Configuration of OpenH323 channel driver \n");
	ast_cli(fd, "------------------------------------------\n");
	ast_cli(fd, "Version: %d.%d.%d\n", 
				OH323_VERSION_MAJOR, OH323_VERSION_MINOR, OH323_VERSION_BUILD);
	ast_cli(fd, "Listening on address: %s:%d\n", 
				config_options.listenAddress, config_options.listenPort);
	ast_cli(fd, "Gatekeeper used: %s %s\n", gk_ptr, gk_res);
	ast_cli(fd, "FastStart/H245Tunnelling/H245inSetup: %s/%s/%s\n", 
				config_options.fastStart!=0?"ON":"OFF",
				config_options.h245Tunnelling!=0?"ON":"OFF",
				config_options.h245inSetup!=0?"ON":"OFF");
	ast_cli(fd, "Supported formats in pref. order: %s\n", 
				oh323_codecset2str(1, config_options.codecInfo, &tmp));
	ast_cli(fd, "Jitter buffer limits (min/max): %d-%d ms\n", 
				config_options.jitterMin,
				config_options.jitterMax);
#ifdef HAS_OH323MODS
	ast_cli(fd, "Jitter buffer modification amounts (dec/inc): %d/%d frames\n", 
				config_options.jitterDec,
				config_options.jitterInc);
#endif
	/*ast_cli(fd, "Bandwidth limit: %.2f Kbps\n", config_options.bwlimit*100.0/1024.0);*/
	ast_cli(fd, "TCP port range: %d - %d\n", config_options.tcpStart, config_options.tcpEnd);
	ast_cli(fd, "UDP (RAS) port range: %d - %d\n", config_options.udpStart, config_options.udpEnd);
	ast_cli(fd, "UDP (RTP) port range: %d - %d\n", config_options.rtpStart, config_options.rtpEnd);
	ast_cli(fd, "IP Type-of-Service value: %d\n", config_options.ipTos);
	ast_cli(fd, "User input mode: %d\n", config_options.userInputMode);
	ast_cli(fd, "Max number of inbound H.323 calls: %d\n", config_options.inboundMax);
	ast_cli(fd, "Max number of outbound H.323 calls: %d\n", config_options.outboundMax);
	ast_cli(fd, "Max number of simultaneous H.323 calls: %d\n", config_options.simultaneousMax);
	ast_cli(fd, "Max call rate (ingress direction): %.2f/%d\n", 
							in_call_max_rate, config_options.crlThreshold);
	ast_cli(fd, "\n");
	if (tmp != NULL)
		free(tmp);
	ast_mutex_unlock(&oh323_stats_lock);
	return RESULT_SUCCESS;
}
static int oh323_show_stats(int fd, int argc, char *argv[])
{
	if (argc != 3)
		return RESULT_SHOWUSAGE;
	ast_mutex_lock(&oh323_stats_lock);
	ast_cli(fd, "\n");
	ast_cli(fd, "           Statistics of OpenH323 channel driver               \n");
	ast_cli(fd, "---------------------------------------------------------------\n");
	ast_cli(fd, "                            Up since: %s", ctime((time_t *) &(oh323_stats.boot_time.tv_sec)));
	ast_cli(fd, "     Established inbound H.323 calls: %6d\n", oh323_stats.incall);
	ast_cli(fd, "    Established outbound H.323 calls: %6d\n", oh323_stats.outcall);
	ast_cli(fd, "         Dropped inbound H.323 calls: %6d\n", oh323_stats.block_incall);
	ast_cli(fd, "        Blocked outbound H.323 calls: %6d\n", oh323_stats.block_outcall);
	ast_cli(fd, "  Total inbound H.323 calls detected: %6d\n", oh323_stats.setup_recv);
	ast_cli(fd, "Total outbound H.323 calls attempted: %6d\n", oh323_stats.setup_sent);
	ast_cli(fd, "                     Protocol errors: %6d\n", oh323_stats.proto_err);
	ast_cli(fd, "                  Call answer errors: %6d\n", oh323_stats.answer_err);
	ast_cli(fd, "              Call initiation errors: %6d\n", oh323_stats.call_err);
	ast_cli(fd, "\n");
	ast_mutex_unlock(&oh323_stats_lock);
	return RESULT_SUCCESS;
}
static int oh323_show_channels(int fd, int argc, char *argv[])
{
#define FORMAT_STRING  "%5s %-30s %-7s %-7s %-7s %4s/%-4s %-12s %-21s %-21s\n"
#define FORMAT_STRING2 "%5d %-30.30s %-7.7s %-7s %-7s %4d/%-4d %-12s %-21s %-21s\n"

	int i, count, established, res;
	char remote[100], local[100];

	if (argc != 3)
		return RESULT_SHOWUSAGE;
	count = 0;
	established = 0;
	ast_mutex_lock(&oh323_tab_lock);
	ast_cli(fd, "\n");
	ast_cli(fd, " Information about active OpenH323 channel(s) \n");
	ast_cli(fd, "----------------------------------------------\n");
	for (i=0; i<config_options.totalNum; i++)
		if (oh323_tab[i] != NULL) {
			if (count == 0)
				ast_cli(fd, FORMAT_STRING, 
						"Num.", "Token", "State", "Dir", "Init", "RX", "TX", 
						"Format", "Remote RTP Addr.", "Local RTP Addr.");
			memset(remote, 0, sizeof(remote));
			memset(local, 0, sizeof(local));
			snprintf(remote, sizeof(remote)-1, "%s", 
							oh323_tab[i]->rtp.remote_addr);
			snprintf(local, sizeof(local)-1, "%s", 
							oh323_tab[i]->rtp.local_addr);
			ast_cli(fd, FORMAT_STRING2, count, 
				(oh323_tab[i]->cd.call_token != NULL ? oh323_tab[i]->cd.call_token:"NULL"),
				oh323_state2str(oh323_tab[i]->i_state),
				oh323_direction2str(oh323_tab[i]->direction),
				(oh323_tab[i]->from_remote ? "Remote":"Local"),
				oh323_tab[i]->rx_buf_size, oh323_tab[i]->tx_buf_size,
				ast_getformatname(oh323_tab[i]->capability),
				remote, local);
			++count;
			if (oh323_tab[i]->i_state == OH323_STATE_ESTABLISHED)
				++established;
		}
	if (count == 0)
		ast_cli(fd, "  [No active H.323 connections]\n");
	else {
		ast_cli(fd, "\n");
		ast_cli(fd, "  [%d established H.323 connection(s)]\n", established);
		if (incall_rate_limiter_status) {
			res = in_call_rate_get();
			ast_cli(fd, "  [In-call rate: %d.%02d CPS]\n", res / 100, res % 100);
			res = in_call_blockratio_get();
			ast_cli(fd, "  [In-call block ratio: %d %%]\n", res);
		}
	}
	ast_cli(fd, "\n");

	ast_mutex_unlock(&oh323_tab_lock);
	return RESULT_SUCCESS;
}
static int oh323_show_established(int fd, int argc, char *argv[])
{
	int i, established;

	if (argc != 3)
		return RESULT_SHOWUSAGE;
	established = 0;
	ast_mutex_lock(&oh323_tab_lock);
	for (i=0; i<config_options.totalNum; i++)
		if (oh323_tab[i] != NULL) {
			if (oh323_tab[i]->i_state == OH323_STATE_ESTABLISHED)
				++established;
		}
	ast_mutex_unlock(&oh323_tab_lock);
	ast_cli(fd, "\n");
	ast_cli(fd, "%d established H.323 connection(s)\n\n", established);
	return RESULT_SUCCESS;
}
#if 0
static int oh323_show_ep(int fd, int argc, char *argv[])
{
#define	FORMAT_STRING2a	"%-20s %-20s %-10s %-15s\n"

	struct oh323_ep *exttmp;

	if (argc != 3)
		return RESULT_SHOWUSAGE;
	ast_cli(fd, "\n");
	ast_cli(fd, FORMAT_STRING2a,
				"Host", "CallerID", "CallFwd", "FwdExten");
	ast_cli(fd, "--------------------------------------------------------------------\n");
	exttmp = oh323_ep_list.head;
	while (exttmp) {
		ast_cli(fd, FORMAT_STRING2a,
					exttmp->host, exttmp->callerid,
					exttmp->cancallforward?"Allowed":"Disallowed",
					!(exttmp->forwardexten[0])?"-":exttmp->forwardexten);
		exttmp = exttmp->next;
	}
	ast_cli(fd, "\n");
	return RESULT_SUCCESS;
}
#endif
static int oh323_debug_toggle(int fd, int argc, char *argv[])
{
	if (argc != 3)
		return RESULT_SHOWUSAGE;
	ast_mutex_lock(&oh323_tab_lock);
	if (oh323_verbose_debug != 0) {
		oh323_verbose_debug = 0;
		ast_cli(fd, "Verbose debug info for OpenH323 channel driver turned off.\n");
	} else {
		oh323_verbose_debug = 1;
		ast_cli(fd, "Verbose debug info for OpenH323 channel driver turned on.\n");
	}
	ast_mutex_unlock(&oh323_tab_lock);
	return RESULT_SUCCESS;
}
#if 0
static int oh323_mode(int fd, int argc, char *argv[])
{
	if (argc != 4)
		return RESULT_SHOWUSAGE;
	/*ast_mutex_lock(&oh323_tab_lock);*/
	h323_change_call(argv[2], argv[3]);
	/*ast_mutex_unlock(&oh323_tab_lock);*/
	return RESULT_SUCCESS;
}
#endif
static int oh323_vars(int fd, int argc, char *argv[])
{
	char *pvar, *pvar_val;
	int i, j, found = 0;

	if (argc == 3) {
		ast_mutex_lock(&oh323_tab_lock);
		for (i=0; i<config_options.totalNum; i++) {
			if (oh323_tab[i] != NULL) {
				found = 1;
				ast_cli(fd, "Variables for OpenH323 channel %s:\n",
								oh323_tab[i]->owner->name);
				j = 0;
				pvar = oh323_var_tab[j];
				while (pvar != NULL) {
					pvar_val = pbx_builtin_getvar_helper(oh323_tab[i]->owner, pvar);
					if (pvar_val)
						ast_cli(fd, "\t%s\t'%s'\n", pvar, pvar_val);
					pvar = oh323_var_tab[++j];
				}
			}
		}
		if (!found)
			ast_cli(fd, "No active H.323 connections\n");
		ast_mutex_unlock(&oh323_tab_lock);
		return RESULT_SUCCESS;
	} else
		return RESULT_SHOWUSAGE;
}
#if 0
static int oh323_reload(int fd, int argc, char *argv[])
{
	if (argc != 2)
		return RESULT_SHOWUSAGE;

	ast_mutex_lock(&oh323_reload_lock);
	if (oh323_reloading) {
		ast_cli(fd, "Previous reload not yet done.\n");
	} else
		oh323_reloading = 1;
	ast_mutex_unlock(&oh323_reload_lock);
	return RESULT_SUCCESS;
}
#endif

/* ----------- CLI extensions: Usage -------------- */
static char oh323_show_conf_usage[] =
"Usage: oh323 show conf\n"
"       Show configuration info and parameters of OpenH323 channel driver\n";
static char oh323_show_stats_usage[] =
"Usage: oh323 show stats\n"
"       Show various statistics of the operation of OpenH323 channel driver\n";
static char oh323_show_channels_usage[] =
"Usage: oh323 show channels\n"
"       Show verbose information about active OpenH323 channel(s)\n";
static char oh323_show_established_usage[] =
"Usage: oh323 show established\n"
"       Show the number of established H.323 channels\n";
/*static char oh323_show_ep_usage[] =
"Usage: oh323 show endpoints\n"
"       Show the configured H.323 endpoints\n"; */
static char oh323_debug_toggle_usage[] =
"Usage: oh323 debug toggle\n"
"       Toggle on/off (very) verbose debug info for OpenH323 channel driver\n";
/*static char oh323_mode_usage[] =
"Usage: oh323 mode\n"
"       Change the mode of a running OpenH323 channel\n"; */
static char oh323_vars_usage[] =
"Usage: oh323 show vars\n"
"       Show variables of running OpenH323 channels\n";
/* ----------- CLI extensions: Asterisk entries -------------- */
static struct ast_cli_entry	cli_show_conf = 
	{ { "oh323", "show", "conf", NULL }, oh323_show_conf, 
		"Show config info of OpenH323 channel driver", oh323_show_conf_usage };
static struct ast_cli_entry	cli_show_stats = 
	{ { "oh323", "show", "stats", NULL }, oh323_show_stats, 
		"Show statistics of OpenH323 channel driver", oh323_show_stats_usage };
static struct ast_cli_entry	cli_show_info = 
	{ { "oh323", "show", "channels", NULL }, oh323_show_channels, 
		"Show info about active OpenH323 channel(s)", oh323_show_channels_usage };
static struct ast_cli_entry	cli_show_established = 
	{ { "oh323", "show", "established", NULL }, oh323_show_established, 
		"Show the number of established H.323 channels", oh323_show_established_usage };
/*static struct ast_cli_entry	cli_show_ep = 
	{ { "oh323", "show", "endpoints", NULL }, oh323_show_ep, 
		"Show the configured H.323 endpoints", oh323_show_ep_usage };*/
static struct ast_cli_entry	cli_debug_toggle = 
	{ { "oh323", "debug", "toggle", NULL }, oh323_debug_toggle, 
		"Toggle on/off debug info for OpenH323 channel driver", oh323_debug_toggle_usage };
/*static struct ast_cli_entry	cli_mode = 
	{ { "oh323", "mode", NULL }, oh323_mode, 
		"Change the mode of a running OpenH323 channel", oh323_mode_usage };*/
static struct ast_cli_entry	cli_vars = 
	{ { "oh323", "show", "vars", NULL }, oh323_vars, 
		"Variables of running OpenH323 channels", oh323_vars_usage };


/**
 * Handle exceptions of the OpenH323 channel.
 * TODO: Serialize the exceptions (e.g. with a queue)...
 */
static struct ast_frame *oh323_exception(struct ast_channel *c)
{
	struct chan_oh323_pvt *p = c->pvt->pvt;
	char notify_buf[2] = { OH323_EVENT_NTFY };
	char *p1, *p2;

	MARK_IN();
	ast_mutex_lock(&oh323_tab_lock);

	/* Some nice norms */
	p->fr.datalen = 0;
	p->fr.data =  NULL;
	p->fr.src = type;
	p->fr.offset = 0;
	p->fr.mallocd = 0;
	p->fr.src = __FUNCTION__;

	/* -- User input */
	if (p->except_struct.type == OH323EXC_USER_INPUT_TONE) {
		if (p->except_struct.data[0] == '!') {
			p->fr.frametype = AST_FRAME_CONTROL;
			p->fr.subclass = AST_CONTROL_FLASH;
			if (option_debug)
				ast_log(LOG_DEBUG, "%s: Got flash hook.\n", c->name);
		} else {
			p->fr.frametype = AST_FRAME_DTMF;
			p->fr.subclass = p->except_struct.data[0];
			if (option_debug)
				ast_log(LOG_DEBUG, "%s: Got DTMF %c.\n", c->name, 
								p->fr.subclass);
		}
	/* -- User text message */
	} else if (p->except_struct.type == OH323EXC_USER_MESSAGE) {
		p->fr.frametype = AST_FRAME_TEXT;
		p->fr.subclass = 0;
		memset(p->txtbuf, 0, sizeof(p->txtbuf));
		strncpy((char*)(p->txtbuf + AST_FRIENDLY_OFFSET), p->except_struct.data,
							sizeof(p->txtbuf) - AST_FRIENDLY_OFFSET - 1);
		p->fr.data = p->txtbuf + AST_FRIENDLY_OFFSET;
		p->fr.offset = AST_FRIENDLY_OFFSET;
	/* -- Call alerted (remote endpoint is ringing) */
	} else if (p->except_struct.type == OH323EXC_CALL_ALERTED) {
		p->fr.frametype = AST_FRAME_CONTROL;
		p->fr.subclass = AST_CONTROL_RINGING;
	/* -- Call transfer */
	} else if (p->except_struct.type == OH323EXC_CALL_TRANSFER) {
		/* XXX Asterisk MUST read first the frame returned by this exception
		 *     and then the queued DTMFs (the extension to transfer to) */
		p->fr.frametype = AST_FRAME_DTMF;
		p->fr.subclass = '#';
		if (strlen((p->except_struct).data) > 0) {
			if (option_debug)
				ast_log(LOG_DEBUG, "%s: Native transfer to '%s'.\n", 
								c->name, (p->except_struct).data);
			memset(c->dtmfq, 0, sizeof(c->dtmfq));
			memcpy(c->dtmfq, p->except_struct.data, 
								strlen(p->except_struct.data));
			write(p->event_pipe[1], notify_buf, 1);
		}
	/* -- Call establishment notification */
	} else if (p->except_struct.type == OH323EXC_CALL_ESTABLISHED) {
		memset(p->rtp.local_addr, 0, sizeof(p->rtp.local_addr));
		memset(p->rtp.remote_addr, 0, sizeof(p->rtp.remote_addr));
		p1 = p->except_struct.data;
		p2 = strchr(p->except_struct.data, ':');
		if (p2) {
			p2 = strchr(p2, '-');
			if (p2) {
				p2[0] = '\0';
				++p2;
				strncpy(p->rtp.local_addr, p1, sizeof(p->rtp.local_addr)-1);
				strncpy(p->rtp.remote_addr, p2, sizeof(p->rtp.remote_addr)-1);
			} else
				ast_log(LOG_WARNING, "%s: Invalid format of RTP addresses. No common codecs???\n",
							c->name);
		} else
			ast_log(LOG_WARNING, "%s: Invalid format of RTP addresses. No common codecs???\n",
						c->name);
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Local RTP '%s', Remote RTP '%s'.\n",
						c->name, p->rtp.local_addr, p->rtp.remote_addr);
		if (c->_state != AST_STATE_UP) {
			if (option_debug)
				ast_log(LOG_DEBUG, "%s: Channel is going UP.\n", c->name);
			p->fr.frametype = AST_FRAME_CONTROL;
			p->fr.subclass = AST_CONTROL_ANSWER;
			ISTATE_LOG(p->i_state, OH323_STATE_ESTABLISHED);
			p->i_state = OH323_STATE_ESTABLISHED;
		} else {
			if (p->i_state != OH323_STATE_ESTABLISHED)
				ast_log(LOG_ERROR, "%s: Got ESTABLISHED while UP and %s!\n", 
								c->name, oh323_state2str(p->i_state));
			p->fr.frametype = AST_FRAME_NULL;
			p->fr.subclass = 0;
			if (option_debug)
				ast_log(LOG_DEBUG, "%s: Got ESTABLISHED while UP!\n", c->name);
		}
	/* -- Call clearing notification */
	} else if (p->except_struct.type == OH323EXC_CALL_CLEARED) {
		/* Handle the clearing of the call */
#if 0
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Generating HANGUP.\n", c->name);
		p->fr.frametype = AST_FRAME_CONTROL;
		p->fr.subclass = AST_CONTROL_HANGUP;
#endif
		switch (h323_get_reason_code(p->cd.call_end_reason)) {
			case OH323END_NO_ACCEPT:
			case OH323END_REFUSAL:
			case OH323END_NO_ANSWER:
			case OH323END_ANSWER_DENIED:
			case OH323END_LOCAL_BUSY:
			case OH323END_REMOTE_BUSY:
			case OH323END_NO_ENDPOINT:
			case OH323END_CONNECT_FAIL:
			case OH323END_HOST_OFFLINE:
			case OH323END_Q931CAUSE:
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Generating BUSY.\n", c->name);
				p->fr.frametype = AST_FRAME_CONTROL;
				p->fr.subclass = AST_CONTROL_BUSY;
				break;
			case OH323END_TRANSPORT_FAIL:
			case OH323END_GATEKEEPER:
			case OH323END_NO_USER:
			case OH323END_NO_BANDWIDTH:
			case OH323END_SECURITY:
			case OH323END_LOCAL_CONGESTION:
			case OH323END_UNREACHABLE:
			case OH323END_TEMP_FAILURE:
			case OH323END_DURATION_LIMIT:
			case OH323END_REMOTE_CONGESTION:
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Generating CONGESTION.\n", c->name);
				p->fr.frametype = AST_FRAME_CONTROL;
				p->fr.subclass = AST_CONTROL_CONGESTION;
				break;
			case OH323END_LOCAL_USER:
			case OH323END_REMOTE_USER:
			case OH323END_CALLER_ABORT:
			case OH323END_CALLFWD:
			case OH323END_CAPABILITY:
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Generating HANGUP.\n", c->name);
				p->fr.frametype = AST_FRAME_CONTROL;
				p->fr.subclass = AST_CONTROL_HANGUP;
				break;
			default:
				ast_log(LOG_WARNING, "%s: Call cleared with unknown reason (%d).\n",
								c->name,
								h323_get_reason_code(p->cd.call_end_reason));
				ast_mutex_unlock(&oh323_tab_lock);
				MARK_OUT();
				return(NULL);
		}
	/* -- Call clearing notification */
	} else if (p->except_struct.type == OH323EXC_CTRL_ERROR) {
		ast_queue_hangup(c);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return(NULL);
	/* -- Unknown exception */
	} else {
		/*p->fr.frametype = AST_FRAME_NULL;*/
		/*p->fr.subclass = 0;*/
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Got null/unknown exception.\n", c->name);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return(NULL);
	}

	ast_mutex_unlock(&oh323_tab_lock);
	MARK_OUT();
	return(&(p->fr));
}

/**
 * Indicate various conditions on a channel.
 * Return -1 on error, 0 on success.
 */
static int oh323_indicate(struct ast_channel *c, int condition)
{
	struct chan_oh323_pvt *pvt = c->pvt->pvt;
	int res;

	if (option_debug)
		ast_log(LOG_DEBUG, "%s: Indicating condition %d.\n", c->name, condition);

	//ast_mutex_lock(&oh323_tab_lock);
	switch (condition) {
		case AST_CONTROL_RINGING:
			if (c->_state == AST_STATE_RING) {
				if (h323_indicate_call(pvt->cd.call_token, IND_RINGING) == CALL_IND_OK)
					res = 0;
				else
					res = -1;
				break;
			}
			//ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		case AST_CONTROL_BUSY:
			if (c->_state != AST_STATE_UP) {
				if (h323_indicate_call(pvt->cd.call_token, IND_BUSY) == CALL_IND_OK)
					res = 0;
				else
					res = -1;
				break;
			}
			//ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		case AST_CONTROL_CONGESTION:
			if (c->_state != AST_STATE_UP) {
				if (h323_indicate_call(pvt->cd.call_token, IND_CONGESTION) == CALL_IND_OK)
					res = 0;
				else
					res = -1;
				break;
			}
			//ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		case AST_CONTROL_PROGRESS:
			/* FIXME ... */
			ast_log(LOG_NOTICE, "Ignoring PROGRESS indication.\n");
			res = 0;
			break;
		case -1:
			res = -1;
			break;
		default:
			ast_log(LOG_WARNING, "Don't know how to indicate condition %d\n", 
							condition);
			res = -1;
			break;
	}
	//ast_mutex_unlock(&oh323_tab_lock);
	return(res);
}

/**
 * Send (play) the specified digit to the channel.
 * Return -1 on error, 0 on success.
 */
static int oh323_digit(struct ast_channel *c, char digit)
{
	struct chan_oh323_pvt *pvt = c->pvt->pvt;

	if (option_debug)
		ast_log(LOG_DEBUG, "In oh323_digit (%s, dig=%c).\n", c->name, digit);

	if (c->_state != AST_STATE_UP) {
		ast_log(LOG_WARNING, "%s: Channel is not UP!\n", c->name);
		return(-1);
	}
	ast_mutex_lock(&oh323_tab_lock);
	h323_send_tone((pvt->cd).call_token, digit);
	ast_mutex_unlock(&oh323_tab_lock);

	return(0);
}

/**
 * Send text message to the channel.
 * Return -1 on error, 0 on success.
 */
static int oh323_text(struct ast_channel *c, char *text)
{
	struct chan_oh323_pvt *pvt = c->pvt->pvt;

	if (option_debug)
		ast_log(LOG_DEBUG, "In oh323_text (%s, text='%s').\n", c->name, text);

	if (c->_state != AST_STATE_UP) {
		ast_log(LOG_WARNING, "%s: Channel is not UP!\n", c->name);
		return(-1);
	}
	ast_mutex_lock(&oh323_tab_lock);
	h323_send_text((pvt->cd).call_token, text);
	ast_mutex_unlock(&oh323_tab_lock);

	return(0);
}

/**
 * Make a call over the specified channel to the specified 
 * destination. This function will parse the destination string
 * and determine the address-number to call.
 * Return -1 on error, 0 on success.
 */
static int oh323_call(struct ast_channel *c, char *dest, int timeout)
{
	struct chan_oh323_pvt *pvt = c->pvt->pvt;
	char called_addr[AST_OH323_MAX_CALLED_ADDR] = {0};
	char *cid, *cidname, oldcid[256], *tmpbuf;
	int retval;
	char *out_codec;
	int fmt;

	if (option_debug)
		ast_log(LOG_DEBUG, "In oh323_call (%s, dest=%s, timeout=%d).\n", 
						c->name, dest, timeout);

	if ((c->_state != AST_STATE_DOWN) && (c->_state != AST_STATE_RESERVED)) {
		ast_log(LOG_WARNING, "%s: Channel is already in use?\n", c->name);
		return(-1);
	}

	ast_mutex_lock(&oh323_tab_lock);

	/* Set CallerID on outgoing call */
	if (c->callerid) {
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Generating CallerID '%s'\n", c->name, c->callerid);
		memset(oldcid, 0, sizeof(oldcid));
		strncpy(oldcid, c->callerid, sizeof(oldcid) - 1);
		ast_callerid_parse(oldcid, &cidname, &cid);
		if (option_debug) {
			ast_log(LOG_DEBUG, "CID is '%s'.\n", cid == NULL ? "<NULL>":cid);
			ast_log(LOG_DEBUG, "CIDname is '%s'.\n", cidname == NULL ? "<NULL>":cidname);
		}
		if (pvt->call_opt.cid) {
			free(pvt->call_opt.cid);
			pvt->call_opt.cid = NULL;
		}
		if (pvt->call_opt.cidname) {
			free(pvt->call_opt.cidname);
			pvt->call_opt.cidname = NULL;
		}
		pvt->call_opt.cid = (char*)malloc(256);
		if (pvt->call_opt.cid == NULL) {
			ast_log(LOG_ERROR, "Not enough memory.\n");
			ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		}
		memset(pvt->call_opt.cid, 0, 256);
		if ((cid != NULL)&&(strlen(cid) > 0))
			strncpy(pvt->call_opt.cid, cid, 255);
		pvt->call_opt.cidname = (char*)malloc(256);
		if (pvt->call_opt.cidname == NULL) {
			ast_log(LOG_ERROR, "Not enough memory.\n");
			ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		}
		memset(pvt->call_opt.cidname, 0, 256);
		if ((cidname != NULL)&&(strlen(cidname) > 0))
			strncpy(pvt->call_opt.cidname, cidname, 255);
	} else {
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: No CallerID info.\n", c->name);
		if (pvt->call_opt.cid) {
			free(pvt->call_opt.cid);
			pvt->call_opt.cid = NULL;
		}
	}

	/* Set CalledID (DNID) on outgoing call */
	/* XXX ... */

	/* Set a temp name and call token for the channel */
	memset(c->name, 0, sizeof(c->name));
	snprintf(c->name, sizeof(c->name)-1, "OH323/%s", dest);

	/* Set the connect port at the remote H.323 endpoint */
	/* XXX Not used anymore */
	pvt->call_opt.connectPort = config_options.connectPort;

	/* Set the codec according to the value of the OH323_OUTCODEC
	 * variable. */
	out_codec = pbx_builtin_getvar_helper(c, "OH323_OUTCODEC");
	if (out_codec) {
		fmt = ast_getformatbyname(out_codec);
		oh323_format2codecset(fmt, pvt->call_opt.cap, WRAP_MAX_CAP_SET);
		if (pvt->call_opt.cap[0] == 0) {
			ast_log(LOG_ERROR, "%s: Unsupported ${OH323_OUTCODEC} value (%s)!\n", 
							c->name, out_codec);
			ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		}
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Set codec according to channel's "
						"${OH323_OUTCODEC} (%s)\n", c->name, out_codec);
		if (option_verbose > 2)
			ast_verbose(VERBOSE_PREFIX_3 "H.323 call to %s with codec %s\n",
							dest, ast_getformatname(fmt));
	} else {
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: No ${OH323_OUTCODEC}.\n", c->name);
		oh323_format2codecset(0xffffffff, pvt->call_opt.cap, 
							WRAP_MAX_CAP_SET);
		if (pvt->call_opt.cap[0] == 0) {
			ast_log(LOG_ERROR, "%s: Unsupported format (%d)!\n", 
							c->name, c->nativeformats);
			ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		}
		tmpbuf = NULL;
		if (option_verbose > 2)
			ast_verbose(VERBOSE_PREFIX_3 "H.323 call to %s with codec(s) %s\n",
					dest, oh323_codecset2str(0, config_options.codecInfo, &tmpbuf));
		if (tmpbuf != NULL)
			free(tmpbuf);
	}

	/* Clear the call token */
	memset((char *)(pvt->cd).call_token, 0, sizeof((pvt->cd).call_token));

	ISTATE_LOG(pvt->i_state, OH323_STATE_INIT);
	pvt->i_state = OH323_STATE_INIT;

	/* Build the address to call. */
	memset(called_addr, 0, AST_OH323_MAX_CALLED_ADDR);
	strncpy(called_addr, dest, sizeof(called_addr) - 1);

	/* Make the call */
	retval = h323_make_call(called_addr, &(pvt->cd), pvt->call_opt);

	if (pvt->call_opt.cid) {
		free(pvt->call_opt.cid);
		pvt->call_opt.cid = NULL;
	}
	if (pvt->call_opt.cidname) {
		free(pvt->call_opt.cidname);
		pvt->call_opt.cidname = NULL;
	}
	if (retval != CALL_START_OK) {
		ast_log(LOG_WARNING, "Failed to make call to '%s'.\n", called_addr);

		/* Update statistics */
		ast_mutex_lock(&oh323_stats_lock);
		++oh323_stats.call_err;
		++oh323_stats.setup_sent;
		ast_mutex_unlock(&oh323_stats_lock);

		ast_setstate(c, AST_STATE_DOWN);
		ast_mutex_unlock(&oh323_tab_lock);
		return(-1);
	}
	/* Update statistics */
	ast_mutex_lock(&oh323_stats_lock);
	++oh323_stats.setup_sent;
	ast_mutex_unlock(&oh323_stats_lock);

	if (option_debug)
		ast_log(LOG_DEBUG, "Created call '%s' (OUT).\n", pvt->cd.call_token);
	if (oh323_verbose_debug)
		ast_verbose("Outbound H.323 call '%s'.\n", pvt->cd.call_token);

	/* Set the state of the channel and fix the name */
	ast_setstate(c, AST_STATE_DIALING);
	memset(c->name, 0, sizeof(c->name));
	snprintf(c->name, sizeof(c->name)-1, "OH323/%c%d", 
						pvt->from_remote?'R':'L', pvt->cd.call_reference);

	/* The call initiated successfully */
	if (option_debug)
		ast_log(LOG_DEBUG, "%s: Call to %s initiated successfully.\n", 
						c->name, called_addr);
	ast_mutex_unlock(&oh323_tab_lock);

	return(0);
}

/**
 * Send a hangup to the specified channel. This function
 * will terminate the active call over this channel, will
 * decrease the usage counter and will free allocated resources.
 * It can not fail !!!
 */
static int oh323_hangup(struct ast_channel *c) 
{
	struct chan_oh323_pvt *pvt = c->pvt->pvt;
	int i, index;
	char buf[128];

	if (option_debug)
		ast_log(LOG_DEBUG, "In oh323_hangup (%s).\n", c->name);

	ast_mutex_lock(&oh323_tab_lock);
	if (pvt) {
		ast_setstate(c, AST_STATE_DOWN);
		ast_queue_hangup(c);

		if ((pvt->cd.call_reference == 0) || (pvt->cd.call_token == NULL)) {
			ast_log(LOG_NOTICE, "%s: Cannot hangup a call which doesn't exist.\n",
							c->name);
			for (i=0; i<config_options.totalNum; i++)
				if (oh323_tab[i] == pvt)
					break;
			if (i == config_options.totalNum) {
				ast_log(LOG_WARNING, "Channel's %s pvt is out of our space!\n",
								c->name);
			} else {
				index = i;
				pvt->needs_destroy = 1;
			}
		} else {

			ISTATE_LOG(pvt->i_state, OH323_STATE_CLEARED);
			pvt->i_state = OH323_STATE_CLEARED;

			/* Search the already allocated private structures for a */
			/* matching call reference number.*/
			for (i=0; i<config_options.totalNum; i++)
				if (oh323_tab[i] != NULL)
					if (oh323_tab[i]->cd.call_reference == pvt->cd.call_reference)
						if (strcmp(oh323_tab[i]->cd.call_token, pvt->cd.call_token) == 0)
							break;
			index = i;
			if (oh323_tab[index] != pvt) {
				for (i=0; i<config_options.totalNum; i++)
					if (oh323_tab[i] == pvt)
						break;
				ast_log(LOG_WARNING, "%s: Private struct of channel doesn't "
								"match with ours (is in %d but found in %d).\n", 
								c->name, index, i);
			}

			/* Send the hangup only on local hangups */
			if (!(pvt->clearing_call)) {
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Hanging up call '%s'.\n", 
									c->name, (pvt->cd).call_token);
				pvt->clearing_call = OH323_CLEAR_LOC;

				/* Make sure that resources of the call will be released */
				if ((index >= 0) && (index < config_options.totalNum)) {
					pvt->index = index;
					pvt->sched_id = ast_sched_add(oh323_sched, 25000, 
									oh323_release, pvt);
				}

				//ast_mutex_unlock(&oh323_tab_lock);
				if (h323_clear_call((pvt->cd).call_token) != CALL_END_OK) {
					ast_log(LOG_WARNING, "%s: Failed to hangup call '%s'.\n", 
									c->name, pvt->cd.call_token);
				}
				//oh323_close_call_fds(index);
				//ast_mutex_lock(&oh323_tab_lock);
			}
		}

		/* No match found ?!?!?! */
		if ((index<0) || (index>=config_options.totalNum)) {
			if (option_debug)
				ast_log(LOG_WARNING, "%s: Call '%s' not found (hangup)!\n", 
							c->name, pvt->cd.call_token);
		} else {
			if (option_debug)
				ast_log(LOG_DEBUG, "%s: Call '%s' found in %d (hangup).\n", 
								c->name, pvt->cd.call_token, index);
			ast_log(LOG_DEBUG, "%s: End reason %d, duration %d secs.\n",
								c->name, 
								pvt->cd.call_end_reason, 
								pvt->cd.call_duration);
			/* Set environment variables */
			snprintf(buf, sizeof(buf)-1, "%d", pvt->cd.call_duration);
			pbx_builtin_setvar_helper(pvt->owner, "OH323_CALL_DURATION", buf);
			snprintf(buf, sizeof(buf)-1, "%d", pvt->cd.call_end_reason);
			pbx_builtin_setvar_helper(pvt->owner, "OH323_CALL_ENDREASON", buf);
			if (pvt->owner->bridge) {
				ast_log(LOG_DEBUG, "Setting variables on bridged channel %s.\n",
							pvt->owner->bridge->name);
				/* Set environment variables */
				snprintf(buf, sizeof(buf)-1, "%d", pvt->cd.call_duration);
				pbx_builtin_setvar_helper(pvt->owner->bridge, "OH323_CALL_DURATION", buf);
				snprintf(buf, sizeof(buf)-1, "%d", pvt->cd.call_end_reason);
				pbx_builtin_setvar_helper(pvt->owner->bridge, "OH323_CALL_ENDREASON", buf);
			}
#if 0
#endif

#if 0
			if (pvt->clearing_call == OH323_CLEAR_REM) {
				if (option_debug)
					ast_log(LOG_DEBUG, "Releasing resources of call (%d).\n", 
									index);
				/* Deallocate the resources */
				oh323_close_call_fds(index);
				oh323_destroy(index);
				free(oh323_tab[index]);
				oh323_tab[index] = NULL;
			}
#endif
		}

		c->pvt->pvt = NULL;
		pvt->owner = NULL;

		/* Update usage counter */
		ast_mutex_lock(&usecnt_lock);
		usecnt--;
		if (usecnt < 0) 
	 		ast_log(LOG_WARNING, "Usecnt < 0???\n");
		ast_mutex_unlock(&usecnt_lock);
		ast_update_use_count();
		if (option_verbose > 2) 
			ast_verbose( VERBOSE_PREFIX_3 "Hungup '%s'\n", c->name);
	} else {
		ast_log(LOG_WARNING, "Channel '%s' has no private structure!\n",
						c->name);
	}
	ast_mutex_unlock(&oh323_tab_lock);

	return(0);
}

/**
 * Read data from the network, encapsulate them in a frame
 * and return it to Asterisk PBX.
 * Return NULL on error, a frame pointer on success.
 */
static struct ast_frame *oh323_read(struct ast_channel *c)
{
	struct chan_oh323_pvt *p = c->pvt->pvt;
	int res, read_size, framenum;
	struct ast_frame *f;
	char buf[100];
	int valid_g729_frame, i;
	char addr_buf[256];
	char *p1, *p2;

	/* If it's already gone, just return */
#if 0
	if (p->alreadygone) {
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Channel was shut down.\n", c->name);
		return(NULL);
	}
#endif

	/* Do this basic initialization here */
	p->fr.mallocd = 0;
	p->fr.datalen = 0;
	p->fr.data = p->buf + AST_FRIENDLY_OFFSET;
	p->fr.src = type;
	p->fr.offset = AST_FRIENDLY_OFFSET;
	p->fr.samples = 0;
	p->fr.src = __FUNCTION__;

	/* Check the event pipe */
	//CHECK_BLOCKING(c);
	if (read(p->event_pipe[0], buf, 1) == 1) {
		switch (buf[0]) {
			case OH323_EVENT_FDUPDATE:
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Channel needs to update its fds.\n", 
									c->name);
				c->fds[0] = p->player_fd[0];
				p->normread = 0;
				p->rxcount = 0;
				p->fr.frametype = AST_FRAME_NULL;
				p->fr.subclass = 0;
				return &(p->fr);
				//break;

			case OH323_EVENT_TERM:
			case OH323_EVENT_EXCE:
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Channel has generated an exception.\n", 
									c->name);
				//c->blocking = 0;
				return oh323_exception(c);

			case OH323_EVENT_NTFY:
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Channel has been notified.\n", 
									c->name);
				/* XXX ?????? */
				/* c->exception = 1; */
				p->fr.frametype = AST_FRAME_NULL;
				p->fr.subclass = 0;
				//c->blocking = 0;
				break;

			default:
				ast_log(LOG_DEBUG, "%s: Unknown event type (%d).\n", 
									c->name, buf[0]);
				//c->blocking = 0;
				return(NULL);
		}
	}
	//c->blocking = 0;

	/* Find out how much data we are going to read */
	if (p->rx_buf_size <= 0) {
		read_size = OH323_MAX_BUF;
	} else {
		if (p->rx_buf_size > OH323_MAX_BUF) {
			ast_log(LOG_WARNING, "%s: Requested read buffer size is too long (%d)!\n",
							c->name, p->rx_buf_size);
			ast_log(LOG_WARNING, "%s: Truncating to %d.\n", c->name, OH323_MAX_BUF);
			read_size = OH323_MAX_BUF;
		} else
			read_size = p->rx_buf_size;
	}

	/* Try to read some data... */
	//CHECK_BLOCKING(c);
	if (!p->normread) {	/* First-time actions */
		res = 0;
		while (read(p->player_fd[0], (char *)(p->buf + AST_FRIENDLY_OFFSET), 1) > 0) res++;
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Reading for the first time. Removed %d bytes.\n", 
							c->name, res);
		/* Do it again here */
		h323_get_conn_info((p->cd).call_token, addr_buf, sizeof(addr_buf));
		memset(p->rtp.local_addr, 0, sizeof(p->rtp.local_addr));
		memset(p->rtp.remote_addr, 0, sizeof(p->rtp.remote_addr));
		p1 = addr_buf;
		p2 = strchr(addr_buf, ':');
		if (p2) {
			p2 = strchr(p2, '-');
			if (p2) {
				p2[0] = '\0';
				++p2;
				strncpy(p->rtp.local_addr, p1, sizeof(p->rtp.local_addr)-1);
				strncpy(p->rtp.remote_addr, p2, sizeof(p->rtp.remote_addr)-1);
			} else
				ast_log(LOG_WARNING, "%s: Invalid format of RTP addresses. No common codecs???\n",
							c->name);
		} else
			ast_log(LOG_WARNING, "%s: Invalid format of RTP addresses. No common codecs???\n",
						c->name);
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Local RTP '%s', Remote RTP '%s'.\n",
						c->name, p->rtp.local_addr, p->rtp.remote_addr);
		p->normread = 1;
		//c->blocking = 0;
		p->fr.frametype = AST_FRAME_NULL;
		p->fr.subclass = 0;
		p->fr.offset = AST_FRIENDLY_OFFSET;
		return &(p->fr);
	}
	/* Normal read */
	res = read(p->player_fd[0], (char *)(p->buf + AST_FRIENDLY_OFFSET), read_size);
	//c->blocking = 0;
	if ((res < 0) && ((errno == EAGAIN)||(errno == EINTR))) {
		/* No error */
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: EAGAIN/EINTR on channel.\n", c->name);
		p->fr.frametype = AST_FRAME_NULL;
		p->fr.subclass = 0;
		p->fr.offset = AST_FRIENDLY_OFFSET;
		return &(p->fr);
	}
	if (res <= 0) {
		if (option_debug) {
			if (res < 0)
				ast_log(LOG_NOTICE, "%s: Error reading from channel (%s).\n", 
								c->name, strerror(errno));
			else
				ast_log(LOG_NOTICE, "%s: Read zero bytes.\n", c->name);
		}
		c->fds[0] = -1;	/* Remove the fd from the channel's fds. */
		p->fr.frametype = AST_FRAME_NULL;
		p->fr.subclass = 0;
		p->fr.offset = AST_FRIENDLY_OFFSET;
		return &(p->fr);
	}

	p->rxcount += res;
	if (oh323_verbose_debug && (p->rxcount % (100 * p->rx_buf_size) < p->rx_buf_size))
		ast_verbose("Channel %s (call '%s') RX byte count is %d.\n", 
					c->name, p->cd.call_token, p->rxcount);

	/* Check for format changes */
	if (p->last_rx_format != p->capability) {
		ast_log(LOG_NOTICE, "%s: Format changed from %s to %s.\n", 
						c->name, 
						ast_getformatname(p->last_rx_format), 
						ast_getformatname(p->capability));
		p->last_rx_format = p->capability;
	}

	/* Do specific stuff for each format type */
	switch (p->capability) {
		case 0:
			p->fr.frametype = AST_FRAME_NULL;
			p->fr.subclass = 0;
			return &(p->fr);

		case AST_FORMAT_ULAW:
		case AST_FORMAT_ALAW:
			p->fr.samples = res;
			break;

		case AST_FORMAT_GSM:
			if (res % 33 != 0) {
				ast_log(LOG_WARNING, "%s: Invalid size for GSM (%d bytes).\n", 
						c->name, res);
				p->fr.frametype = AST_FRAME_NULL;
				p->fr.subclass = 0;
				p->fr.offset = AST_FRIENDLY_OFFSET;
				return &(p->fr);
			}
			/* Check for silence */
			if (((char *)(p->buf + AST_FRIENDLY_OFFSET))[0] == 0) {
				p->fr.frametype = AST_FRAME_NULL;
				p->fr.subclass = 0;
				p->fr.offset = AST_FRIENDLY_OFFSET;
				return &(p->fr);
			}
			p->fr.samples = 160 * (res / 33);
			break;

		case AST_FORMAT_G729A:
			if (res % 10 != 0) {
				ast_log(LOG_WARNING, "%s: Invalid size for G.729 (%d bytes).\n", 
						c->name, res);
				p->fr.frametype = AST_FRAME_NULL;
				p->fr.subclass = 0;
				p->fr.offset = AST_FRIENDLY_OFFSET;
				return &(p->fr);
			}
			/* Drop zeroed G.729 frames */
			valid_g729_frame = 0;
			for (i = 0; i < res; i++) {
				if (((char*)(p->buf + AST_FRIENDLY_OFFSET))[i] != '\0') {
					valid_g729_frame = 1;
					break;
				}
			}
			if (!valid_g729_frame) {
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Dropped zeroed G.729 frame.\n",
									c->name);
				p->fr.frametype = AST_FRAME_NULL;
				p->fr.subclass = 0;
				p->fr.offset = AST_FRIENDLY_OFFSET;
				return &(p->fr);
			}

			p->fr.samples = 80 * (res / 10);
			break;

		case AST_FORMAT_G723_1:
			/* Create the smoother, if not already there */
			if (p->rx_g7231_smooth == NULL) {
				p->rx_g7231_smooth = G7231SF_new();
				if (p->rx_g7231_smooth == NULL) {
					ast_log(LOG_ERROR, "%s: Failed to create G.723.1 smoother.\n", c->name);
					p->fr.frametype = AST_FRAME_NULL;
					p->fr.subclass = 0;
					p->fr.offset = AST_FRIENDLY_OFFSET;
					return &(p->fr);
				}
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Created G.723.1 smoother.\n", c->name);
			}

			/* Push the data */
			p->fr.samples = 0;
			if (G7231SF_push(p->rx_g7231_smooth, (p->buf + AST_FRIENDLY_OFFSET), res) < 0) {
				p->fr.frametype = AST_FRAME_NULL;
				p->fr.subclass = 0;
				p->fr.offset = AST_FRIENDLY_OFFSET;
				return &(p->fr);
			}
#ifdef	DEBUG_CIRCBUFFER
			ast_log(LOG_DEBUG, "Checked-in %d bytes.\n", res);
#endif

			/* Checkout a whole number of frames */
			framenum = -1;
			res = G7231SF_pop(p->rx_g7231_smooth, (p->buf + AST_FRIENDLY_OFFSET), sizeof(p->buf) - AST_FRIENDLY_OFFSET, &framenum);
			if (res < 0) {
				p->fr.frametype = AST_FRAME_NULL;
				p->fr.subclass = 0;
				p->fr.offset = AST_FRIENDLY_OFFSET;
				return &(p->fr);
			}
#ifdef	DEBUG_CIRCBUFFER
			ast_log(LOG_DEBUG, "Checked-out %d G.723.1 frames (%d bytes).\n",
							framenum, res);
#endif
			p->fr.samples += 240 * framenum;
			break;

		default:
			/* Do nothing */
#if 0
			ast_log(LOG_NOTICE, "%s: Nothing specific for this format type (%d)?\n",
							c->name, p->capability);
#endif
			break;
	}

	/* H.323 channel read direction open? */
	if ((p->direction != PLAYER) && (p->direction != BOTH)) {
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Channel read direction not open (%s).\n", 
								c->name, oh323_direction2str(p->direction));
		p->fr.frametype = AST_FRAME_NULL;
		p->fr.subclass = 0;
		p->fr.offset = AST_FRIENDLY_OFFSET;
		return &(p->fr);
	}

	/* Save the data in a ASTERISK frame struct */
	p->fr.data = p->buf + AST_FRIENDLY_OFFSET;
	p->fr.datalen = res;
	p->fr.frametype = AST_FRAME_VOICE;
	p->fr.subclass = p->capability;
	p->fr.offset = AST_FRIENDLY_OFFSET;
	f = &(p->fr);

	/* Do in-band DTMF detection, if enabled */
	if ((p->capability==AST_FORMAT_ALAW)||(p->capability==AST_FORMAT_ULAW)) {
		if (config_options.inBandDTMF) {
			f = ast_dsp_process(p->owner, p->vad, f);
			if (f->frametype == AST_FRAME_DTMF)
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Got in-band digit %c.\n", 
									c->name, f->subclass);
		}
	}

	return(f);
}

/**
 * Write the data buffer of the frame to the network.
 * Return -1 on error, 0 on success.
 */
static int oh323_write(struct ast_channel *c, struct ast_frame *f)
{
	struct chan_oh323_pvt	*i = c->pvt->pvt;
	struct ast_frame		*ftmp;
	int						res, framenum, len;
	char					buf[100];

	ast_mutex_lock(&oh323_tab_lock);

	/* H.323 channel write direction open? */
	if ((i->direction != RECORDER) && (i->direction != BOTH)) {
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Channel write direction not open (%s).\n", 
								c->name, oh323_direction2str(i->direction));
		ast_mutex_unlock(&oh323_tab_lock);
		return(0);
	}

	/* If it's already gone, return failure */
	if ((i->alreadygone)||(c->_state == AST_STATE_DOWN)) {
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Channel was shut down.\n", c->name);
		ast_mutex_unlock(&oh323_tab_lock);
		return(-1);
	}

	/* Don't send null frames */
	if (f->frametype == AST_FRAME_NULL) {
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Ignoring NULL frame.\n", c->name);
		ast_mutex_unlock(&oh323_tab_lock);
		return(0);
	}

	/* Check for a proper frame type. */
	if (f->frametype != AST_FRAME_VOICE) {
		ast_log(LOG_WARNING, "%s: Don't know what to do with frame type '%d'.\n", 
						c->name, f->frametype);
		ast_mutex_unlock(&oh323_tab_lock);
		return(-1);
	}

	/* Check for format changes */
	if (f->subclass != i->owner->nativeformats) {
		ast_log(LOG_NOTICE, "%s: Format changed to %s (native %s).\n", 
						c->name, 
						ast_getformatname(f->subclass),
						ast_getformatname(c->nativeformats));
		if (ast_set_write_format(c, f->subclass) < 0) {
			ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		}
		if (ast_set_read_format(c, f->subclass) < 0) {
			ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		}
		if (i->tx_smooth != NULL) {
			ast_smoother_free(i->tx_smooth);
			i->tx_smooth = NULL;
		}
		ast_mutex_unlock(&oh323_tab_lock);
		return(0);
	}

	/* Create the smoother, if not already there */
	if (i->tx_smooth == NULL) {
		i->tx_smooth = ast_smoother_new(i->tx_buf_size);
		if (i->tx_smooth == NULL) {
			ast_log(LOG_ERROR, "%s: Failed to create smoother.\n", c->name);
			ast_mutex_unlock(&oh323_tab_lock);
			return(-1);
		}
		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Created smoother (size=%d, format=%d).\n", 
						c->name, i->tx_buf_size, f->subclass);
	}

	/* Do specific stuff for each codec type */
	switch (f->subclass) {
		case AST_FORMAT_ULAW:
		case AST_FORMAT_ALAW:
		case AST_FORMAT_G726:
			break;

		case AST_FORMAT_GSM:
			if (f->datalen % 33 != 0) {
				ast_log(LOG_WARNING, "%s: Invalid frame size for GSM (%d bytes).\n", 
						c->name, f->datalen);
				ast_mutex_unlock(&oh323_tab_lock);
				return(0);
			}
			break;

		case AST_FORMAT_G729A:
			if (f->datalen % 10 != 0) {
				ast_log(LOG_WARNING, "%s: Invalid frame size for G.729 (%d bytes).\n", 
						c->name, f->datalen);
				ast_mutex_unlock(&oh323_tab_lock);
				return(0);
			}
			break;

		case AST_FORMAT_G723_1:
			/* Create the smoother, if not already there */
			if (i->tx_g7231_smooth == NULL) {
				i->tx_g7231_smooth = G7231SF_new();
				if (i->tx_g7231_smooth == NULL) {
					ast_log(LOG_ERROR, "%s: Failed to create G.723.1 smoother.\n", c->name);
					ast_mutex_unlock(&oh323_tab_lock);
					return(-1);
				}
				if (option_debug)
					ast_log(LOG_DEBUG, "%s: Created G.723.1 smoother.\n", c->name);
			}
			/* Check-in all the data we received */
			if (G7231SF_push(i->tx_g7231_smooth, f->data, f->datalen) < 0) {
				ast_log(LOG_ERROR, "%s: Failed to fill G.723.1 smoother.\n", c->name);
				ast_mutex_unlock(&oh323_tab_lock);
				return(-1);
			}
#ifdef	DEBUG_CIRCBUFFER
			ast_log(LOG_DEBUG, "Checked-in %d bytes.\n", res);
#endif

			/* Checkout single frames and send them */
			while (!G7231SF_isempty(i->tx_g7231_smooth)) {
				framenum = 1;
				len = G7231SF_pop(i->tx_g7231_smooth, buf, sizeof(buf), &framenum);
				if (framenum != 1) {
					ast_mutex_unlock(&oh323_tab_lock);
					return(0);
				}
				if ((res = write(i->recorder_fd[0], buf, len)) < 0) {
					if (option_debug)
						ast_log(LOG_WARNING, "%s: Unable to write to network (errno=%d).\n", 
									c->name, errno);
					ast_mutex_unlock(&oh323_tab_lock);
					return(0);
				}
				i->txcount += res;
				if (oh323_verbose_debug && (i->txcount % (100 * i->tx_buf_size) < i->tx_buf_size))
					ast_verbose("Channel %s (call '%s') TX byte count is %d.\n", 
							c->name, i->cd.call_token, i->txcount);
			}
			/* Nothing else to do for G.723.1 */
			ast_mutex_unlock(&oh323_tab_lock);
			return(0);
			
		default:
			/* Do nothing */
			ast_log(LOG_NOTICE, "%s: Nothing specific for this format type (%s)?\n",
							c->name, ast_getformatname(f->subclass));
			break;
	}

	if (ast_smoother_feed(i->tx_smooth, f) != 0) {
		ast_log(LOG_ERROR, "%s: Failed to fill smoother.\n", c->name);
		ast_mutex_unlock(&oh323_tab_lock);
		return(-1);
	}

/**************
	if (oh323_verbose_debug)
		ast_log(LOG_DEBUG, "%s: Pushed %d bytes into smoother...\n", 
						c->name, f->datalen);
**************/
	ftmp = ast_smoother_read(i->tx_smooth);
	while (ftmp != NULL) {
		if ((res = write(i->recorder_fd[0], (char *)(ftmp->data), ftmp->datalen)) < 0) {
			if (option_debug)
				ast_log(LOG_WARNING, "%s: Unable to write to fd %d (%d, %s).\n", 
							c->name, i->recorder_fd[0], errno, strerror(errno));
			//ast_mutex_unlock(&oh323_tab_lock);
			//return(-1);
		}
		i->txcount += res;
		if (oh323_verbose_debug && (i->txcount % (100 * i->tx_buf_size) < i->tx_buf_size))
			ast_verbose("Channel %s (call '%s') TX byte count is %d.\n", 
					c->name, i->cd.call_token, i->txcount);
		ftmp = ast_smoother_read(i->tx_smooth);
	}

	ast_mutex_unlock(&oh323_tab_lock);
	return(0);
}

/**
 * Answer an H.323 channel.
 * Return 0 on success, -1 on error.
 *
 * XXX Needs locking XXX
 */
static int oh323_answer(struct ast_channel *c)
{
	struct chan_oh323_pvt *pvt = c->pvt->pvt;
	//int answer_codec[1];
	//int	i;

	if (option_debug)
		ast_log(LOG_DEBUG, "In oh323_answer (%s).\n", c->name);

	if (pvt->i_state == OH323_STATE_ESTABLISHED) {
		ast_log(LOG_DEBUG, "%s: Channel is ESTABLISHED???\n", c->name);
		return(0);
	}
	if (c->_state == AST_STATE_UP) {
		ast_log(LOG_DEBUG, "%s: Channel is UP???\n", c->name);
		return(0);
	}

	//oh323_format2codecset(c->nativeformats, answer_codec, 1);
	
	//ast_mutex_lock(&oh323_tab_lock);
	//if (h323_answer_call((pvt->cd).call_token, answer_codec[0]) != CALL_ANS_OK) {
	if (h323_answer_call((pvt->cd).call_token) != CALL_ANS_OK) {
		ast_log(LOG_WARNING, "%s: Failed to answer call with token %s.\n",
						c->name, (pvt->cd).call_token);
		/* Update statistics */
		ast_mutex_lock(&oh323_stats_lock);
		++oh323_stats.answer_err;
		ast_mutex_unlock(&oh323_stats_lock);
		//ast_mutex_unlock(&oh323_tab_lock);
		return(-1);
	}

	/* The call was answered successfully */
	if (option_debug)
		ast_log(LOG_DEBUG, "%s: Call answered.\n", c->name);
	if (oh323_verbose_debug)
		ast_verbose("Channel %s answered.\n", c->name);
	ast_setstate(c, AST_STATE_UP);
	//ast_mutex_unlock(&oh323_tab_lock);

	return(0);
}

static int oh323_fixup(struct ast_channel *oldchan, struct ast_channel *newchan)
{
	struct chan_oh323_pvt *p = newchan->pvt->pvt;

	ast_mutex_lock(&oh323_tab_lock);
	if (p->owner != oldchan) {
		ast_log(LOG_WARNING, "Old channel wasn't %p but was %p\n", oldchan, p->owner);
		ast_mutex_unlock(&oh323_tab_lock);
		return(-1);
	}
	p->owner = newchan;
	ast_mutex_unlock(&oh323_tab_lock);
	return(0);
}

/**
 * Allocate and initialize a "chan_oh323_pvt" private structure.
 * Return NULL on error, the pointer to the struct on success.
 */
static struct chan_oh323_pvt *new_oh323(int index)
{
	struct chan_oh323_pvt	*tmp;
	int						flags, i;

	tmp = malloc(sizeof(struct chan_oh323_pvt));
	if (tmp) {
		memset(tmp, 0, sizeof(struct chan_oh323_pvt));

		/* Create socketpairs */
#if 0
		snprintf(tmp->playername, sizeof(tmp->playername), "%s", 
						build_fifo_name(PLAYER, index));
		snprintf(tmp->recordername, sizeof(tmp->recordername), "%s", 
						build_fifo_name(RECORDER, index));
#endif
		if (socketpair(AF_LOCAL, SOCK_STREAM, 0, tmp->player_fd) != 0) {
			ast_log(LOG_ERROR, "Failed to create socketpair for player(%d, %s).\n",
							errno, strerror(errno));
			free(tmp);
			return NULL;
		}
		if (shutdown(tmp->player_fd[1], SHUT_RD) != 0) {
			ast_log(LOG_ERROR, "Failed to configure player socket.\n");
			free(tmp);
			return NULL;
		}
		if (socketpair(AF_LOCAL, SOCK_STREAM, 0, tmp->recorder_fd) != 0) {
			ast_log(LOG_ERROR, "Failed to create socketpair for recorder(%d, %s).\n",
							errno, strerror(errno));
			free(tmp);
			return NULL;
		}
		if (shutdown(tmp->recorder_fd[1], SHUT_WR) != 0) {
			ast_log(LOG_ERROR, "Failed to configure recorder socket.\n");
			free(tmp);
			return NULL;
		}
		/* Set all fds in non-blocking mode */
		for (i=0; i<2; i++) {
			flags = fcntl(tmp->player_fd[i], F_GETFL);
			if (fcntl(tmp->player_fd[i], F_SETFL, flags | O_NONBLOCK) < 0) {
				ast_log(LOG_ERROR, "Failed to configure player socket.\n");
				free(tmp);
				return NULL;
			}
		}
		for (i=0; i<2; i++) {
			flags = fcntl(tmp->recorder_fd[i], F_GETFL);
			if (fcntl(tmp->recorder_fd[i], F_SETFL, flags | O_NONBLOCK) < 0) {
				ast_log(LOG_ERROR, "Failed to configure recorder socket.\n");
				free(tmp);
				return NULL;
			}
		}
		/* Create the event pipe */
		if (pipe(tmp->event_pipe) < 0) {
			ast_log(LOG_ERROR, "Failed to create event pipe (%d, %s).\n",
							errno, strerror(errno));
			free(tmp);
			return NULL;
		} else {
			flags = fcntl(tmp->event_pipe[0], F_GETFL);
			fcntl(tmp->event_pipe[0], F_SETFL, flags | O_NONBLOCK);
			flags = fcntl(tmp->event_pipe[1], F_GETFL);
			fcntl(tmp->event_pipe[1], F_SETFL, flags | O_NONBLOCK);
		}

		if (option_debug)
			ast_log(LOG_DEBUG, "Player fds %d,%d - Recorder fds %d,%d - Event pipe %d,%d.\n",
						tmp->player_fd[0], tmp->player_fd[1], 
						tmp->recorder_fd[0], tmp->recorder_fd[1],
						tmp->event_pipe[0], tmp->event_pipe[1]);

		tmp->direction = NONE;
		tmp->alreadygone = 0;
		tmp->sched_id = -1;
		tmp->close_recorder_fd = 1;
		tmp->close_player_fd = 1;
		tmp->normread = 0;
		tmp->rxcount = 0;
		tmp->txcount = 0;
		tmp->from_remote = 0;
		tmp->clearing_call = 0;
		tmp->tx_smooth = NULL;
		tmp->i_state = OH323_STATE_NULL;
		tmp->needs_destroy = 0;
		/*ast_mutex_init(&(tmp->stats_lock));*/
		tmp->head_stats = tmp->rtp_stats;
		tmp->tail_stats = tmp->rtp_stats;
	}
	return tmp;
}


static void oh323_close_call_fds(int index)
{
	struct chan_oh323_pvt *pvt = oh323_tab[index];

	/* Close open sockets */
	if (option_debug)
		ast_log(LOG_DEBUG, "Player fds %d,%d - Recorder fds %d,%d.\n",
					pvt->player_fd[0], pvt->player_fd[1], 
					pvt->recorder_fd[0], pvt->recorder_fd[1]);
	/* Handle special cases where only the one logical channel of
	 * the connection has been opened. */
	if (pvt->player_fd[0] > -1)
		close(pvt->player_fd[0]);
	pvt->player_fd[0] = -1;
	if (pvt->close_player_fd) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Closing socket %d.\n", pvt->player_fd[1]);
		if (pvt->player_fd[1] > -1)
			close(pvt->player_fd[1]);
		pvt->player_fd[1] = -1;
		pvt->close_player_fd = 0;
	}
	if (pvt->recorder_fd[0] > -1)
		close(pvt->recorder_fd[0]);
	pvt->recorder_fd[0] = -1;
	if (pvt->close_recorder_fd) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Closing socket %d.\n", pvt->recorder_fd[1]);
		if (pvt->recorder_fd[1] > -1)
			close(pvt->recorder_fd[1]);
		pvt->recorder_fd[1] = -1;
		pvt->close_recorder_fd = 0;
	}
}

/**
 * Pick a codec.
 */
static int oh323_codec_choose(int formats)
{
	struct oh323_codecinfo *ci = config_options.codecInfo;

	return(ci->format);
}

/**
 * Destroy allocated resources of an existing "chan_oh323_pvt"
 * private structure. Do not free the allocated memory.
 * Returns nothing.
 */
static void oh323_destroy(int index)
{
	struct chan_oh323_pvt *pvt = oh323_tab[index];

	if (option_debug)
		ast_log(LOG_DEBUG, "Releasing allocated resources (%d).\n", index);

	clear_call_details(&(pvt->cd));
	if (option_debug)
		ast_log(LOG_DEBUG, "Event pipe %d,%d.\n",
					pvt->event_pipe[0], pvt->event_pipe[1]);
	close(pvt->event_pipe[0]);
	close(pvt->event_pipe[1]);
	if ((config_options.inBandDTMF)&&(pvt->vad != NULL))
		ast_dsp_free(pvt->vad);
	ast_smoother_free(pvt->tx_smooth);
	G7231SF_free(pvt->rx_g7231_smooth);
	pvt->rx_g7231_smooth = NULL;
	G7231SF_free(pvt->tx_g7231_smooth);
	pvt->tx_g7231_smooth = NULL;
	pvt->i_state = OH323_STATE_NULL;
	pvt->direction = NONE;
	pvt->alreadygone = 1;
	pvt->owner = NULL;

	/* XXX Don't free() the respective pvt, do it outside this function */
}

/**
 * Allocate and initialize a "ast_channel" (ASTERISK channel)
 * structure. Associate the provided H.323 private structure
 * with the new channel.
 * Return NULL on error, the pointer to the struct on success.
 */
static struct ast_channel *ast_oh323_new(struct chan_oh323_pvt *i, int state)
{
	struct ast_channel *tmp;
	int callerid_len, dnamelen;
	char *context;
	int fmt;

	if (i == NULL) {
		ast_log(LOG_ERROR, "Private structure is NULL!\n");
		return NULL;
	}

	tmp = ast_channel_alloc(1);
	if (tmp) {
		snprintf(tmp->name, sizeof(tmp->name)-1, "OH323/%c%d", 
						i->from_remote?'R':'L',	i->cd.call_reference);
		tmp->type = type;

		if (config_options.inBandDTMF) {
			i->vad = ast_dsp_new();
			ast_dsp_set_features(i->vad, DSP_FEATURE_DTMF_DETECT);
		}
		tmp->fds[0] = i->player_fd[0];
		tmp->fds[1] = i->event_pipe[0];

		if (tmp->pvt == NULL) {
			/* XXX Really bad situation, usually this happens with binaries
			 * compiled with different versions of Asterisk or compiled
			 * with different compile-time flags (!!!).
			 */
			ast_log(LOG_ERROR, "Internal channel initialization failed. Bad binary?\n");
			ast_hangup(tmp);
			return NULL;
		}

		/* Set our own data structure */
		tmp->pvt->pvt = i;

		/* Set our native format */
		/* XXX: Only one should be selected */
		tmp->nativeformats = oh323_capability;
		fmt = oh323_codec_choose(tmp->nativeformats);
		/*fmt = ast_best_codec(tmp->nativeformats);*/

		tmp->writeformat = fmt;
		tmp->pvt->rawwriteformat = fmt;
		tmp->readformat = fmt;
		tmp->pvt->rawreadformat = fmt;

		if (option_debug)
			ast_log(LOG_DEBUG, "%s: Raw format set to %s.\n", 
								tmp->name, ast_getformatname(fmt));

		/* Register the OpenH323 channel's functions. */
		tmp->pvt->send_digit = oh323_digit;
		tmp->pvt->call = oh323_call;
		tmp->pvt->hangup = oh323_hangup;
		tmp->pvt->answer = oh323_answer;
		tmp->pvt->read = oh323_read;
		tmp->pvt->write = oh323_write;
		tmp->pvt->send_text = oh323_text;
		tmp->pvt->send_image = NULL;
		tmp->pvt->send_html = NULL;
		tmp->pvt->exception = oh323_exception;
		tmp->pvt->bridge = NULL;
		tmp->pvt->indicate = oh323_indicate;
		tmp->pvt->fixup = oh323_fixup;
		tmp->pvt->setoption = NULL;
		tmp->pvt->queryoption = NULL;
		tmp->pvt->transfer = NULL;
		tmp->pvt->write_video = NULL;

		i->owner = tmp;
		//tmp->blocking = 0;

		/* Set initial context and extension */
		/* First, search for a matching alias, then for a matching prefix */
		/* or return the default context. */
		/* The extension is the same as the called E.164 address of the call, */
		/* or the default extension, if this cannot be determined */
		strncpy(tmp->context, config_options.context, sizeof(tmp->context)-1);
		strncpy(tmp->exten, "s", sizeof(tmp->exten)-1);
		tmp->priority = 1;
		if (i->from_remote) {
			if (i->cd.call_dest_e164 != NULL) {
				if (strlen(i->cd.call_dest_e164) > 0) {
					if (!context_from_alias((char*)i->cd.call_dest_e164, &context))
						strncpy(tmp->context, context, sizeof(tmp->context)-1);
					else if (!context_from_prefix((char*)i->cd.call_dest_e164, &context))
						strncpy(tmp->context, context, sizeof(tmp->context)-1);
					else
						strncpy(tmp->context, config_options.context, sizeof(tmp->context)-1);
					strncpy(tmp->exten, i->cd.call_dest_e164, sizeof(tmp->exten)-1);
				} 
			}
		}
		if (strlen(tmp->exten) && strcmp(tmp->exten, "s"))
			tmp->dnid = strdup(tmp->exten);

		/* Set caller ID */
		callerid_len = 10;
		if ((i->cd.call_source_alias != NULL) && 
						(strlen(i->cd.call_source_alias) > 0))
			callerid_len += strlen(i->cd.call_source_alias);            
		if ((i->cd.call_source_e164 != NULL) && 
						(strlen(i->cd.call_source_e164) > 0))
			callerid_len += strlen(i->cd.call_source_e164);
		tmp->callerid = (char *)malloc(callerid_len);
		if (tmp->callerid != NULL) {
			memset(tmp->callerid, 0, callerid_len);
			if ((i->cd.call_source_alias != NULL) && 
							(strlen(i->cd.call_source_alias) > 0)) { 
				dnamelen = strcspn(i->cd.call_source_alias, "([");
				if (dnamelen < strlen(i->cd.call_source_alias)) {
					strncat(tmp->callerid, "\"", 1);
					strncat(tmp->callerid, i->cd.call_source_alias, dnamelen);
					strncat(tmp->callerid, "\"", 1);
				} else {
					strncat(tmp->callerid, "\"\"", 2);
					dnamelen = 0;
				}
				if ((i->cd.call_source_e164 != NULL) && 
								(strlen(i->cd.call_source_e164) > 0)) {
					strncat(tmp->callerid, " <", 2);
					strncat(tmp->callerid, i->cd.call_source_e164,
									strlen(i->cd.call_source_e164));
					strncat(tmp->callerid, ">", 1);
				} else {
					strncat(tmp->callerid, "<>", 2);
				}
			} else if ((i->cd.call_source_e164 != NULL) && 
								(strlen(i->cd.call_source_e164) > 0))
				memcpy(tmp->callerid, i->cd.call_source_e164,
								strlen(i->cd.call_source_e164));
			/* Set ANI */
			tmp->ani = strdup(tmp->callerid);
		}

		if (option_debug) {
			ast_log(LOG_DEBUG, "Context is '%s', extension is '%s'.\n", 
							tmp->context, tmp->exten);
			if (tmp->callerid != NULL)
				ast_log(LOG_DEBUG, "CallerID/ANI is '%s'.\n", tmp->callerid);
		}

		/* Set environment variables */
		pbx_builtin_setvar_helper(tmp, "OH323_CTOKEN", 
					(char*)((i->cd).call_token));
		pbx_builtin_setvar_helper(tmp, "OH323_SRCALIAS", 
					(char*)((i->cd).call_source_alias));
		pbx_builtin_setvar_helper(tmp, "OH323_DSTALIAS", 
					(char*)((i->cd).call_dest_alias));
		pbx_builtin_setvar_helper(tmp, "OH323_SRCE164", 
					(char*)((i->cd).call_source_e164));
		pbx_builtin_setvar_helper(tmp, "OH323_DSTE164", 
					(char*)((i->cd).call_dest_e164));
		pbx_builtin_setvar_helper(tmp, "OH323_REMOTEAPP", 
					(char*)((i->cd).remote_app));
		pbx_builtin_setvar_helper(tmp, "OH323_RADDR", 
					(char*)((i->cd).remote_addr));
		pbx_builtin_setvar_helper(tmp, "OH323_LADDR", 
					(char*)((i->cd).local_addr));

		/* Set channel's AMA flags and account code */
		if (config_options.amaFlags >= 0)
			tmp->amaflags = config_options.amaFlags;
		if (strlen(config_options.accountCode))
			strncpy(tmp->accountcode, config_options.accountCode, sizeof(tmp->accountcode)-1);

		/* Update use count */
		ast_mutex_lock(&usecnt_lock);
		usecnt++;
		ast_mutex_unlock(&usecnt_lock);
		ast_update_use_count();

		ast_setstate(tmp, state);
		if (state != AST_STATE_DOWN) {
			if (ast_pbx_start(tmp)) {
				ast_log(LOG_WARNING, "Unable to start PBX on %s.\n", tmp->name);
				ast_hangup(tmp);
				tmp = NULL;
			}
		}
	}
	return tmp;
}

/**
 * Create an ASTERISK H.323 channel with the requested format.
 * This function limits the number of outbound H.323 calls.
 * Return NULL on error, the pointer to the channel on success.
 */
static struct ast_channel *oh323_request(char *type, int format, void *data)
{
	int					i=0, count, simcount, oldformat;
	struct ast_channel	*c;

	if (option_debug)
		ast_log(LOG_DEBUG, "In oh323_request: type=%s, format=%d, data=%s.\n", 
						type, format, (char *)data);

	/* Check the format requested */
	oldformat = format;
	/*format &= oh323_capability;*/
	format &= oh323_full_capability;
	if (!format) {
		ast_log(LOG_ERROR, "Asked to get a channel of unsupported format '%d'\n", format);
		return NULL;
	}

	ast_mutex_lock(&oh323_tab_lock);

	/* Check the number of active outbound H.323 calls */
	count = 0;
	simcount = 0;
	for (i=0; i<config_options.totalNum; i++)
		if (oh323_tab[i] != NULL) {
			if (oh323_tab[i]->from_remote == 0)
				++count;
			++simcount;
		}
	if ((count >= config_options.outboundMax) ||
		((config_options.simultaneousMax > 0) && 
		 (simcount >= config_options.simultaneousMax))) {
		ast_log(LOG_WARNING, "Blocking outbound H.323 call due to call-limit violation.\n");

		/* Update statistics */
		ast_mutex_lock(&oh323_stats_lock);
		++oh323_stats.block_outcall;
		ast_mutex_unlock(&oh323_stats_lock);

		ast_mutex_unlock(&oh323_tab_lock);
		return NULL;
	}

	/* Search for a free entry in the private structure table. */
	for (i=0; i<config_options.totalNum; i++)
		if (oh323_tab[i] == NULL)
			break;

	/* No free entry */
	if ((i<0) || (i>=config_options.totalNum)) {
		ast_log(LOG_ERROR, "Unable to accept more calls.\n");
		ast_mutex_unlock(&oh323_tab_lock);
		return NULL;
	}

	oh323_tab[i] = new_oh323(i);
	if (oh323_tab[i]) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Created new call structure %d (%d bytes).\n", 
						i, sizeof(*(oh323_tab[i])));
		c = ast_oh323_new(oh323_tab[i], AST_STATE_DOWN);
		if (c) {
			oh323_tab[i]->owner = c;

			/* Specify our native formats */
			c->nativeformats = format;
			//c->writeformat = format;
			c->pvt->rawwriteformat = format;
			//c->readformat = format;
			c->pvt->rawreadformat = format;
			if (option_debug)
				ast_log(LOG_DEBUG, "%s: Native format changed to %s.\n", 
									c->name, ast_getformatname(format));

			/* XXX Other stuff to do...??? */

			ast_mutex_unlock(&oh323_tab_lock);
			return c;
		}
	}

	ast_log(LOG_WARNING, "Failed to create new H.323 private structure %d.\n", i);
	ast_mutex_unlock(&oh323_tab_lock);
	return NULL;
}

/**
 * Copy the call_details structure to a local structure.
 * All the "const char *" variables are malloc'ed with this function.
 * The right cleanup should be done by calling the "clear_call_details".
 * Return -1 on error, 0 on success.
 */
static int copy_call_details(call_details_t *src_cd, call_details_t *dest_cd)
{
	*dest_cd = *src_cd;
	if (option_debug) {
		ast_log(LOG_DEBUG, "--- CALL DETAILS ---\n");
		ast_log(LOG_DEBUG, "call_token = '%s'\n", dest_cd->call_token);
		ast_log(LOG_DEBUG, "call_source_alias = '%s'\n", dest_cd->call_source_alias);
		ast_log(LOG_DEBUG, "call_dest_alias = '%s'\n", dest_cd->call_dest_alias);
		ast_log(LOG_DEBUG, "call_source_e164 = '%s'\n", dest_cd->call_source_e164);
		ast_log(LOG_DEBUG, "call_dest_e164 = '%s'\n", dest_cd->call_dest_e164);
		ast_log(LOG_DEBUG, "call_rdnis = '%s'\n", dest_cd->call_rdnis);
		ast_log(LOG_DEBUG, "remote_app = '%s'\n", dest_cd->remote_app);
		ast_log(LOG_DEBUG, "remote_addr = '%s'\n", dest_cd->remote_addr);
		ast_log(LOG_DEBUG, "local_addr = '%s'\n", dest_cd->local_addr);
	}
	
	return(0);
}

/**
 * Cleanup all the variables of the call_details structure.
 * Returns nothing.
 */
static void clear_call_details(call_details_t *cd)
{
	if (!cd) {
		ast_log(LOG_WARNING, "Call details struct is NULL!\n");
		CRASH;
	}
	memset((char*)cd, 0, sizeof(call_details_t));
}


/**
 * Determine the ASTERISK format equivalent to the specified codec.
 * Returns non-zero in success, 0 in case of an unsupported codec.
 */
static int oh323_codec2format(int codec)
{
	switch (codec) {
		case G711U:
			/*return(AST_FORMAT_SLINEAR);*/
			return(AST_FORMAT_ULAW);
		case G711A:
			/*return(AST_FORMAT_SLINEAR);*/
			return(AST_FORMAT_ALAW);
		case G7231: /* fall-through */
		case G72316K3: /* fall-through */
		case G72315K3: /* fall-through */
		case G7231A6K3: /* fall-through */
		case G7231A5K3:
			return(AST_FORMAT_G723_1);
		case GSM0610:
			return(AST_FORMAT_GSM);
		case LPC10:
			return(AST_FORMAT_SLINEAR);
		case G729:
		case G729A:
		case G729B:
		case G729AB:
			return(AST_FORMAT_G729A);
		case SPEEXN8K:
			return(AST_FORMAT_SPEEX);
		case G726:
		case G72616K:
		case G72624K:
		case G72632K:
		case G72640K: /* XXX This is very bad */
			return(AST_FORMAT_G726);
		case MSGSM: /* fall-through */
		default:
			return(0);
	}
}

static char *oh323_codecset2str(int show_pref, struct oh323_codecinfo *ci, char **res)
{
	char buf[20];
	int  i;

	*res = (char *)malloc(1024);
	if (*res == NULL)
		return(NULL);
	memset(*res, 0, 1024);
	i = 0;
	while (ci) {
		strcat(*res, ast_getformatname(ci->format));
		memset(buf, 0, sizeof(buf));
		if (show_pref)
			sprintf(buf, "<%d> ", i);
		else
			sprintf(buf, " ");
		strcat(*res, buf);
		ci = ci->next;
		i++;
	}
	return(*res);
}

static void oh323_format2codecset(int formats, int *set, int set_size)
{
	int	i, count;
	struct oh323_codecinfo *p, *ci = config_options.codecInfo;

	if (set_size < 1)
		return;

	count = 0;
	p = ci;
	while (p) {
		if (p->format & formats) {
			if (count < set_size) {
				set[count] = p->codec;
				count++;
			}
		}
		p = p->next;
	}
	/* Rest entries are unused */
	for (i = count; i < set_size; i++)
		set[i] = 0;

	if (option_debug)
		for (i = 0; i < set_size; i++)
			ast_log(LOG_DEBUG, "capability_set[%d] - %d\n", i, set[i]);
}

/**
 * Convert H.323 codec names into numerical values.
 */
static int oh323_str2codec(char *name)
{
	if (!strcasecmp(name, "G711U")) 
		return G711U;
	else if (!strcasecmp(name, "G711A"))
		return G711A;
	else if (!strcasecmp(name, "G7231"))
		return G7231;
	else if (!strcasecmp(name, "G72316K3"))
		return G72316K3;
	else if (!strcasecmp(name, "G72315K3"))
		return G72315K3;
	else if (!strcasecmp(name, "G7231A6K3"))
		return G7231A6K3;
	else if (!strcasecmp(name, "G7231A5K3"))
		return G7231A5K3;
	else if (!strcasecmp(name, "G726"))
		return G726;
	else if (!strcasecmp(name, "G72616K"))
		return G72616K;
	else if (!strcasecmp(name, "G72624K"))
		return G72624K;
	else if (!strcasecmp(name, "G72632K"))
		return G72632K;
	else if (!strcasecmp(name, "G72640K"))
		return G72640K;
	else if (!strcasecmp(name, "G728"))
		return G728;
	else if (!strcasecmp(name, "G729"))
		return G729;
	else if (!strcasecmp(name, "G729A"))
		return G729A;
	else if (!strcasecmp(name, "G729B"))
		return G729B;
	else if (!strcasecmp(name, "G729AB"))
		return G729AB;
	else if (!strcasecmp(name, "GSM0610"))
		return GSM0610;
	else if (!strcasecmp(name, "MSGSM"))
		return MSGSM;
	else if (!strcasecmp(name, "LPC10"))
		return LPC10;
	else if (!strcasecmp(name, "SPEEXN8K"))
		return SPEEXN8K;
	return 0;
}

/**
 * Convert H.323 channel numerical states into string.
 */
static char *oh323_state2str(int state)
{
	switch (state) {
		case OH323_STATE_NULL:
			return("NULL");
		case OH323_STATE_INIT:
			return("INIT");
		case OH323_STATE_RING:
			return("RING");
		case OH323_STATE_RINGING:
			return("RINGING");
		case OH323_STATE_ESTABLISHED:
			return("ESTABLISHED");
		case OH323_STATE_CLEARED:
			return("CLEARED");
		default:
			return("UNKNOWN");
	}
}

/**
 * Convert H.323 channel open direction into string.
 */
static char *oh323_direction2str(lchan_dir_t dir)
{
	switch (dir) {
		case NONE:
			return("NONE");
		case PLAYER:
			return("PLAYER");
		case RECORDER:
			return("RECORDER");
		case BOTH:
			return("BOTH");
		default:
			return("UNKNOWN");
	}
}

/**
 * Find under which context the specified "alias" resides.
 * Return the pointer to the context in "context" and 0 on success,
 * -1 on error or no match.
 */
static int context_from_alias(char *alias, char **context)
{
	struct oh323_reginfo *regtmp;
	int i;

	if ((alias == NULL)||(context == NULL)||(strlen(alias) <= 0))
		return(-1);

	regtmp = config_options.regInfo;
	*context = NULL;
	while (regtmp) {
		for (i=0; i<regtmp->alias_num; i++)
			if (!strcasecmp(regtmp->alias[i], alias)) {
				*context = regtmp->context;
				if (option_debug)
					ast_log(LOG_DEBUG, "Routing alias '%s' in context '%s'.\n",
									alias, *context);
				return(0);
			}
		regtmp = regtmp->next;
	}
	return(-1);
}

/**
 * Find under which context the specified "prefix" resides.
 * Return the pointer to the context in "context" and 0 on success,
 * -1 on error or no match.
 */
static int context_from_prefix(char *prefix, char **context)
{
	struct oh323_reginfo *regtmp;
	int i;

	if ((prefix == NULL)||(context == NULL)||(strlen(prefix) <= 0))
		return(-1);

	regtmp = config_options.regInfo;
	*context = NULL;
	while (regtmp) {
		for (i=0; i<regtmp->prefix_num; i++)
			if (!strncasecmp(regtmp->prefix[i], prefix, strlen(regtmp->prefix[i]))) {
				*context = regtmp->context;
				if (option_debug)
					ast_log(LOG_DEBUG, "Routing alias '%s' in context '%s'.\n",
									prefix, *context);
				return(0);
			}
		regtmp = regtmp->next;
	}
	return(-1);
}

/**
 * Search the already allocated private structures for a matching call 
 * token. It returns the index of the call with that token or -1 if not
 * found.
 *
 * NOTE: The caller must lock the list of H.323 calls before calling
 * this function.
 */
static int find_call(const char *token)
{
	int i;

	if (token && !ast_strlen_zero(token)) {
		for (i = 0; i < config_options.totalNum; i++)
			if (oh323_tab[i] != NULL) {
				if (oh323_tab[i]->cd.call_token) {
					if (!strcmp(oh323_tab[i]->cd.call_token, token))
						return(i);
				} else {
					ast_log(LOG_WARNING, "The call token in %d is NULL!!!\n", i);
				}
			}
	}
	return(-1);
}

/**
 * Try to lock a mutex but wait no more than 'tv' duration.
 * Returns 0 on success, -1 on error, 1 on timeout waiting to
 * grab the lock.
 */
static int mutex_trylock_timeout(ast_mutex_t *lock, struct timeval tv)
{
	int	i, res, count, lim;

	if ((tv.tv_sec == 0) && (tv.tv_usec == 0)) {
		res = ast_mutex_trylock(lock);
		if (!res)
			return(0);
		if (res == EBUSY)
			return(1);
		else
			return(0);
	}
	for (i = 0; i < tv.tv_sec; i++) {
		count = 0;
		res = 0;
		while ((res = ast_mutex_trylock(lock)) != 0) {
			if (res != EBUSY)
				break;
			usleep(10);
			count++;
			if (count == 100000)
				break;
		}
		if ((res) && (res != EBUSY))
			return(-1);
		/* OK, lock acquired */
		if (count < 100000)
			return(0);
	}
	i = 0;
	lim = tv.tv_usec / 10;
	for (i = 0; i < lim; i++) {
		res = ast_mutex_trylock(lock);
		if (!res)
			return(0);
		if (res == EBUSY)
			usleep(10);
		else
			return(-1);
	}
	if (i < lim)
		return(0);
	return(1);
}

/******************************************************************************/
/* Wrapper library's callbacks ************************************************/

/* Call-back function protos */
char *setup_h323_connection(call_details_t cd, lchan_dir_t dir, int bs, 
				int codec, int *fd);
char *cleanup_h323_connection(call_details_t cd);
int alerted_h323_connection(call_details_t cd);
int init_h323_connection(call_details_t cd);
int exception_h323_connection(call_details_t cd, int type, char *data);
int stats_h323_connection(call_details_t cd, rtp_stats_t rs);

/**
 * Call-back function for establishing communication (media channels)
 * between Asterisk PBX and OpenH323 library.
 * Returns NULL on error, the fifo name on success.
 */
/**************
char *setup_h323_connection(call_details_t cd, lchan_dir_t dir, int bs, 
				int codec, int *fd, char *rtp_addr, int rtp_port)
*************/
char *setup_h323_connection(call_details_t cd, lchan_dir_t dir, int bs, 
				int codec, int *fd)
{
	int	i;
	//int	res, lock_fail;
	struct timeval tv;
	int j, flags;
	char fdupdate_buf[2] = { OH323_EVENT_FDUPDATE };
	int needsfdupdate = 0;

	MARK_IN();

	if (option_debug)
		ast_log(LOG_DEBUG, "Setting up call '%s'.\n", cd.call_token);

	tv.tv_sec = 4;
	tv.tv_usec = 0;
	switch (mutex_trylock_timeout(&oh323_tab_lock, tv)) {
	case -1:
		ast_log(LOG_WARNING, "Call '%s' failed to acquire lock.\n",
						cd.call_token);
		return(NULL);
	case 1:
		ast_log(LOG_WARNING, "Call '%s' failed to be serviced.\n",
						cd.call_token);
		return(NULL);
	default:
		/* Lock is held */
		break;
	}
	//ast_mutex_lock(&oh323_tab_lock);

	switch (dir) {
	case RECORDER:
		if (option_debug)
			ast_log(LOG_DEBUG, "Call '%s', direction RECORDER.\n",
								cd.call_token);
		break;
	case PLAYER:
		if (option_debug)
			ast_log(LOG_DEBUG, "Call '%s', direction PLAYER.\n",
								cd.call_token);
		break;
	default:
		ast_log(LOG_NOTICE, "Call '%s', unknown direction %d.\n", 
								cd.call_token, dir);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return NULL;
	}

#if 0
	res = 0;
	lock_fail = 0;
	while (ast_mutex_trylock(&oh323_tab_lock)) {
		usleep(100);
		res++;
		if (res == 10000) {
			lock_fail = 1;
			break;
		}
	}
	if (lock_fail) {
		ast_log(LOG_WARNING, "%s: Cannot acquire lock!\n", cd.call_token);
		return(NULL);
	}
	//ast_mutex_lock(&oh323_tab_lock);
#endif

	*fd = -1;

	i = find_call(cd.call_token);
	if (i < 0) {
		if (option_debug)
			ast_log(LOG_WARNING, "Call '%s' not found (setup).\n", cd.call_token);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return NULL;
	}

	/* Call reference match, the struct is already there. */
	/* Check the direction requested. */
	if (oh323_tab[i]->direction == BOTH || oh323_tab[i]->direction == dir) {
		ast_log(LOG_WARNING, "Call '%s' requests direction re-setup (%d).\n", 
						cd.call_token, dir);
		/* Re-setup the direction ... */
		oh323_tab[i]->direction = dir;
		switch (dir) {
		case PLAYER:
			if (oh323_tab[i]->player_fd[0] > -1)
				close(oh323_tab[i]->player_fd[0]);
			oh323_tab[i]->player_fd[0] = -1;
			/* Open a new player_fd pair */
			if (socketpair(AF_LOCAL, SOCK_STREAM, 0, oh323_tab[i]->player_fd) != 0) {
				ast_log(LOG_ERROR, "Failed to create socketpair for player(%d, %s).\n",
								errno, strerror(errno));
				ast_mutex_unlock(&oh323_tab_lock);
				MARK_OUT();
				return NULL;
			}
			if (shutdown(oh323_tab[i]->player_fd[1], SHUT_RD) != 0) {
				ast_log(LOG_ERROR, "Failed to configure player socket.\n");
				ast_mutex_unlock(&oh323_tab_lock);
				MARK_OUT();
				return NULL;
			}
			/* Set fds in non-blocking mode */
			for (j=0; j<2; j++) {
				flags = fcntl(oh323_tab[i]->player_fd[j], F_GETFL);
				if (fcntl(oh323_tab[i]->player_fd[j], F_SETFL, flags | O_NONBLOCK) < 0) {
					ast_log(LOG_ERROR, "Failed to configure player socket.\n");
					ast_mutex_unlock(&oh323_tab_lock);
					MARK_OUT();
					return NULL;
				}
			}
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' re-opened player fds %d, %d.\n", 
								cd.call_token, 
								oh323_tab[i]->player_fd[0],
								oh323_tab[i]->player_fd[1]);
			break;
		case RECORDER:
			if (oh323_tab[i]->recorder_fd[0] > -1)
				close(oh323_tab[i]->recorder_fd[0]);
			oh323_tab[i]->recorder_fd[0] = -1;
			/* Open a new player_fd pair */
			if (socketpair(AF_LOCAL, SOCK_STREAM, 0, oh323_tab[i]->recorder_fd) != 0) {
				ast_log(LOG_ERROR, "Failed to create socketpair for recorder(%d, %s).\n",
								errno, strerror(errno));
				ast_mutex_unlock(&oh323_tab_lock);
				MARK_OUT();
				return NULL;
			}
			if (shutdown(oh323_tab[i]->recorder_fd[1], SHUT_WR) != 0) {
				ast_log(LOG_ERROR, "Failed to configure recorder socket.\n");
				ast_mutex_unlock(&oh323_tab_lock);
				MARK_OUT();
				return NULL;
			}
			/* Set fds in non-blocking mode */
			for (j=0; j<2; j++) {
				flags = fcntl(oh323_tab[i]->recorder_fd[j], F_GETFL);
				if (fcntl(oh323_tab[i]->recorder_fd[j], F_SETFL, flags | O_NONBLOCK) < 0) {
					ast_log(LOG_ERROR, "Failed to configure recorder socket.\n");
					ast_mutex_unlock(&oh323_tab_lock);
					MARK_OUT();
					return NULL;
				}
			}
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' re-opened recorder fds %d, %d.\n", 
								cd.call_token, 
								oh323_tab[i]->recorder_fd[0],
								oh323_tab[i]->recorder_fd[1]);
			break;
		default:
			CRASH;
		}
		needsfdupdate = 1;
	}
#if 0
	if (oh323_tab[i]->direction == dir) {
		ast_log(LOG_WARNING, "Call '%s' invalid direction request (%d).\n", 
						cd.call_token, dir);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return NULL;
	}
#endif
	if (oh323_tab[i]->direction == NONE) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Call '%s' requests direction %d.\n", 
							cd.call_token, dir);
		if (copy_call_details(&cd, &(oh323_tab[i]->cd)) != 0){
			ast_log(LOG_ERROR, "Failed to copy call details.\n");
			ast_mutex_unlock(&oh323_tab_lock);
			MARK_OUT();
			return NULL;
		}
		oh323_tab[i]->direction = dir;
		if (dir == PLAYER) {
			oh323_tab[i]->rx_buf_size = bs;
			*fd = oh323_tab[i]->player_fd[1];
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' player fd %d sent.\n", 
								cd.call_token, oh323_tab[i]->player_fd[1]);
			oh323_tab[i]->close_player_fd = 0; /* Don't close this fd */
		} else if (dir == RECORDER) {
			oh323_tab[i]->tx_buf_size = bs;
			*fd = oh323_tab[i]->recorder_fd[1];
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' recorder fd %d sent.\n", 
								cd.call_token, oh323_tab[i]->recorder_fd[1]);
			oh323_tab[i]->close_recorder_fd = 0; /* Don't close this fd */
		}

		/* Set the capability of the channel */
		oh323_tab[i]->capability = oh323_codec2format(codec);
		oh323_tab[i]->last_tx_format = oh323_tab[i]->capability;
		oh323_tab[i]->last_rx_format = oh323_tab[i]->capability;
		if (oh323_tab[i]->capability == 0) {
			ast_log(LOG_NOTICE, "Call '%s' requested unsupported codec %d!\n", 
							cd.call_token, codec);
			ast_mutex_unlock(&oh323_tab_lock);
			MARK_OUT();
			return NULL;
		}
		if (oh323_tab[i]->owner != NULL) {
			if (option_debug)
				ast_log(LOG_DEBUG, "Setting channel '%s' native format to %s!\n", 
								oh323_tab[i]->owner->name,
								ast_getformatname(oh323_tab[i]->capability));
			if (oh323_verbose_debug)
				ast_verbose("Setting channel '%s' (%s) native format to %s!\n", 
								oh323_tab[i]->owner->name,
								cd.call_token,
								ast_getformatname(oh323_tab[i]->capability));
			oh323_tab[i]->owner->nativeformats = oh323_tab[i]->capability;
		}

		/* Set environment variables */
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_CHANCODEC", 
					ast_getformatname(oh323_codec2format(codec)));
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_CTOKEN", 
					(char*)((oh323_tab[i]->cd).call_token));
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_SRCALIAS", 
					(char*)((oh323_tab[i]->cd).call_source_alias));
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_DSTALIAS", 
					(char*)((oh323_tab[i]->cd).call_dest_alias));
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_SRCE164", 
					(char*)((oh323_tab[i]->cd).call_source_e164));
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_DSTE164", 
					(char*)((oh323_tab[i]->cd).call_dest_e164));
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_REMOTEAPP", 
					(char*)((oh323_tab[i]->cd).remote_app));
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_RADDR", 
					(char*)((oh323_tab[i]->cd).remote_addr));
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_LADDR", 
					(char*)((oh323_tab[i]->cd).local_addr));

		/******************
		if (rtp_addr != NULL)
			ast_log(LOG_DEBUG, "Endpoint's RTP address is %s.\n", rtp_addr);
		else
			ast_log(LOG_DEBUG, "Endpoint's RTP address is unspecified.\n");
		if (rtp_port >= 0)
			ast_log(LOG_DEBUG, "Endpoint's RTP port is %d.\n", rtp_port);
		else
			ast_log(LOG_DEBUG, "Endpoint's RTP port is unspecified.\n");
		*******************/
		if (needsfdupdate) {
			if (write(oh323_tab[i]->event_pipe[1], fdupdate_buf, 1) != 1) {
				ast_log(LOG_WARNING, "Failed to write to event pipe of call '%s'.\n",
								cd.call_token);
			}
		}

		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return "asterisk-oh323";
	}

	if (option_debug)
		ast_log(LOG_DEBUG, "Call '%s' found in entry %d (setup).\n", 
						cd.call_token, i);
	oh323_tab[i]->direction = BOTH;
	if (dir == PLAYER) {
		oh323_tab[i]->rx_buf_size = bs;
		*fd = oh323_tab[i]->player_fd[1];
		if (option_debug)
			ast_log(LOG_DEBUG, "Call '%s' player fd %d sent.\n", 
							cd.call_token, oh323_tab[i]->player_fd[1]);
		oh323_tab[i]->close_player_fd = 0; /* Don't close this fd */
	} else if (dir == RECORDER) {
		oh323_tab[i]->tx_buf_size = bs;
		*fd = oh323_tab[i]->recorder_fd[1];
		if (option_debug)
			ast_log(LOG_DEBUG, "Call '%s' recorder fd %d sent.\n", 
							cd.call_token, oh323_tab[i]->recorder_fd[1]);
		oh323_tab[i]->close_recorder_fd = 0; /* Don't close this fd */
	}
	/******************
	if (rtp_addr != NULL)
		ast_log(LOG_DEBUG, "Endpoint's RTP address is %s.\n", rtp_addr);
	else
		ast_log(LOG_DEBUG, "Endpoint's RTP address is unspecified.\n");
	if (rtp_port >= 0)
		ast_log(LOG_DEBUG, "Endpoint's RTP port is %d.\n", rtp_port);
	else
		ast_log(LOG_DEBUG, "Endpoint's RTP port is unspecified.\n");
	************/
	/* Set (again) the capability of the channel */
	if (oh323_tab[i]->capability != oh323_codec2format(codec)) {
		ast_log(LOG_WARNING, "Call '%s' format changed from %s to %s!\n", 
						cd.call_token, 
						ast_getformatname(oh323_tab[i]->capability), 
						ast_getformatname(oh323_codec2format(codec)));
		oh323_tab[i]->capability = oh323_codec2format(codec);
		oh323_tab[i]->last_tx_format = oh323_tab[i]->capability;
		oh323_tab[i]->last_rx_format = oh323_tab[i]->capability;
		if (oh323_tab[i]->capability == 0) {
			ast_log(LOG_DEBUG, "Call '%s' requested unsupported codec %d!\n", 
							cd.call_token, codec);
			ast_mutex_unlock(&oh323_tab_lock);
			MARK_OUT();
			return NULL;
		}
		if (oh323_tab[i]->owner != NULL) {
			if (option_debug)
				ast_log(LOG_DEBUG, "Setting channel '%s' native format to %s!\n", 
								oh323_tab[i]->owner->name,
								ast_getformatname(oh323_tab[i]->capability));
			if (oh323_verbose_debug)
				ast_verbose("Setting channel '%s' (%s) native format to %s!\n", 
								oh323_tab[i]->owner->name,
								cd.call_token,
								ast_getformatname(oh323_tab[i]->capability));
			oh323_tab[i]->owner->nativeformats = oh323_tab[i]->capability;
			if (oh323_tab[i]->tx_smooth) {
				ast_smoother_free(oh323_tab[i]->tx_smooth);
				oh323_tab[i]->tx_smooth = NULL;
			}
		}
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_CHANCODEC", 
					ast_getformatname(oh323_codec2format(codec)));
	}
	if (needsfdupdate) {
		if (write(oh323_tab[i]->event_pipe[1], fdupdate_buf, 1) != 1) {
			ast_log(LOG_WARNING, "Failed to write to event pipe of call '%s'.\n",
							cd.call_token);
		}
	}

	ast_mutex_unlock(&oh323_tab_lock);
	MARK_OUT();
	return "asterisk-oh323";
}


/**
 * Call-back function to cleanup communication
 * previously established by "setup_h323_connection(...)"
 * Always returns NULL.
 */
char *cleanup_h323_connection(call_details_t cd)
{	
	int i;
	char shutdown_buf[2] = { OH323_EVENT_TERM };
	char buf[256];

	MARK_IN();
	ast_mutex_lock(&oh323_tab_lock);

	i = find_call(cd.call_token);
	if (i < 0) {
		if (option_debug)
			ast_log(LOG_WARNING, "Call '%s' not found (clear).\n", cd.call_token);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return NULL;
	}

	/* Call token match. */
	if (option_debug) {
		ast_log(LOG_DEBUG, "Call '%s' found in %d (clear).\n", 
						cd.call_token, i);
	}
	if (option_debug)
		ast_log(LOG_DEBUG, "Call '%s' cleared.\n", cd.call_token);
	if (oh323_verbose_debug)
		ast_verbose("Call '%s' cleared.\n", cd.call_token);

	oh323_tab[i]->cd.call_duration = cd.call_duration;
	oh323_tab[i]->cd.call_end_reason = cd.call_end_reason;
	oh323_tab[i]->alreadygone = 1;
	if (option_verbose > 2)
		ast_verbose(VERBOSE_PREFIX_3 "H.323 call '%s' cleared, reason %d (%s)\n", 
						cd.call_token,
						h323_get_reason_code(cd.call_end_reason),
						h323_get_reason_desc(cd.call_end_reason));

	/* Set useful environment variables */
	if (oh323_tab[i]->owner) {
		ast_log(LOG_DEBUG, "Setting variables on channel %s.\n",
						oh323_tab[i]->owner->name);
		snprintf(buf, sizeof(buf)-1, "%d", oh323_tab[i]->cd.call_duration);
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_CALL_DURATION", buf);
		snprintf(buf, sizeof(buf)-1, "%d", oh323_tab[i]->cd.call_end_reason);
		pbx_builtin_setvar_helper(oh323_tab[i]->owner, "OH323_CALL_ENDREASON", buf);
		if (oh323_tab[i]->owner->bridge) {
			ast_log(LOG_DEBUG, "Setting variables on bridged channel %s.\n",
						oh323_tab[i]->owner->bridge->name);
			/* Set environment variables */
			snprintf(buf, sizeof(buf)-1, "%d", oh323_tab[i]->cd.call_duration);
			pbx_builtin_setvar_helper(oh323_tab[i]->owner->bridge, "OH323_CALL_DURATION", buf);
			snprintf(buf, sizeof(buf)-1, "%d", oh323_tab[i]->cd.call_end_reason);
			pbx_builtin_setvar_helper(oh323_tab[i]->owner->bridge, "OH323_CALL_ENDREASON", buf);
		}
	}

	/* Keep track of cleared calls */
	oh323_tab[i]->needs_destroy = 1;

	if (oh323_tab[i]->i_state == OH323_STATE_INIT) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Call '%s' cleared in INIT state.\n",
						cd.call_token);
		if (oh323_verbose_debug)
			ast_verbose("Call '%s' cleared in INIT state.\n",
						cd.call_token);

		ISTATE_LOG(oh323_tab[i]->i_state, OH323_STATE_CLEARED);
		oh323_tab[i]->i_state = OH323_STATE_CLEARED;
		//oh323_tab[i]->alreadygone = 1;
		//ast_mutex_unlock(&oh323_tab_lock);
		//MARK_OUT();
		//return NULL;
	}
	if (!(oh323_tab[i]->clearing_call)) {
		/* The call cleared from the remote endpoint */
		if (option_debug)
			ast_log(LOG_DEBUG, "Forcing hangup of call '%s'.\n", cd.call_token);
		oh323_tab[i]->except_struct.type = OH323EXC_CALL_CLEARED;
		oh323_tab[i]->clearing_call = OH323_CLEAR_REM;
#if 0
		while ((oh323_tab[i]->owner) && 
				(ast_mutex_trylock(&(oh323_tab[i]->owner->lock)))) {
			ast_mutex_unlock(&oh323_tab_lock);
			usleep(10);
			ast_mutex_lock(&oh323_tab_lock);
		}
		if (oh323_tab[i]->owner != NULL) {
			ast_setstate(oh323_tab[i]->owner, AST_STATE_DOWN);
			ast_queue_control(oh323_tab[i]->owner, AST_CONTROL_HANGUP);
			ast_mutex_unlock(&(oh323_tab[i]->owner->lock));
		}
#else
		if (write(oh323_tab[i]->event_pipe[1], shutdown_buf, 1) != 1) {
			ast_log(LOG_WARNING, "Failed to write to event pipe of call '%s'.\n",
							cd.call_token);
		}
#endif
	}

#if 0
	if (oh323_tab[i]->clearing_call == OH323_CLEAR_LOC) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Releasing resources of entry %d, call '%s'.\n", 
							i, cd.call_token);
		if (oh323_tab[i]->sched_id != -1) {
			ast_sched_del(oh323_sched, oh323_tab[i]->sched_id);
			oh323_tab[i]->sched_id = -1;
			if (option_debug)
				ast_log(LOG_DEBUG, "Cancelled scheduled release of entry %d, call '%s'.\n", 
								i, cd.call_token);
		}
#if 0
		/* Deallocate the resources */
		oh323_close_call_fds(i);
		oh323_destroy(i);
		free(oh323_tab[i]);
		oh323_tab[i] = NULL;
#endif

	}
#endif

	ast_mutex_unlock(&oh323_tab_lock);
	MARK_OUT();
	return NULL;
}

/**
 * Call-back function to inform "oh323_call()" that the
 * remote endpoint was found and is ringing.
 * Always returns 0.
 */
int alerted_h323_connection(call_details_t cd)
{	
	int i;

	MARK_IN();
	ast_mutex_lock(&oh323_tab_lock);

	i = find_call(cd.call_token);
	if (i < 0) {
		if (option_debug)
			ast_log(LOG_WARNING, "Call '%s' not found (alert).\n", cd.call_token);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return(0);
	}

	/* Call token match. */
	if (option_debug) {
		ast_log(LOG_DEBUG, "Call '%s' found in %d (alert).\n", 
						cd.call_token, i);
	}
	if (oh323_tab[i]->i_state == OH323_STATE_INIT) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Call '%s' alerted in INIT state.\n",
						cd.call_token);
		if (oh323_verbose_debug)
			ast_verbose("Call '%s' alerted in INIT state.\n",
						cd.call_token);
		ISTATE_LOG(oh323_tab[i]->i_state, OH323_STATE_RINGING);
		oh323_tab[i]->i_state = OH323_STATE_RINGING;
		while ((oh323_tab[i]->owner) && 
				(ast_mutex_trylock(&(oh323_tab[i]->owner->lock)))) {
			ast_mutex_unlock(&oh323_tab_lock);
			usleep(10);
			ast_mutex_lock(&oh323_tab_lock);
		}
		if (oh323_tab[i]->owner != NULL) {
			ast_setstate(oh323_tab[i]->owner, AST_STATE_RINGING);
			ast_queue_control(oh323_tab[i]->owner, AST_CONTROL_RINGING);
			ast_mutex_unlock(&(oh323_tab[i]->owner->lock));
		}
	} else if (oh323_tab[i]->i_state == OH323_STATE_RINGING) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Call '%s' alerted in RINGING state.\n",
						cd.call_token);
		if (oh323_verbose_debug)
			ast_verbose("Call '%s' alerted in RINGING state.\n",
						cd.call_token);
	/*****************
	} else if (oh323_tab[i]->i_state == OH323_STATE_OPEN_BOTH) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Call %s alerted while media open.\n",
						cd.call_token);
		ISTATE_LOG(oh323_tab[i]->i_state, OH323_STATE_ESTABLISHED);
		oh323_tab[i]->i_state = OH323_STATE_ESTABLISHED;
	************/
	} else {
		ast_log(LOG_WARNING, "Call '%s' alerted in unexpected state (%s).\n",
						cd.call_token, 
						oh323_state2str(oh323_tab[i]->i_state));
	}

	ast_mutex_unlock(&oh323_tab_lock);
	MARK_OUT();
	return(0);
}

/**
 * Call-back function for initializing a incoming H.323 call
 * inside ASTERISK. This function limits the number of inbound
 * H.323 calls and the rate of the incoming calls.
 *
 * Returns 0 on success, -1 on error.
 */
int init_h323_connection(call_details_t cd)
{
	int	i, count, simcount, intime, index, calls_passed;
	float cur_rate;
	struct ast_channel	*c;
	struct timeval tv;

	MARK_IN();

	if (option_debug)
		ast_log(LOG_DEBUG, "Inbound H.323 call '%s' detected.\n", cd.call_token);
	if (oh323_verbose_debug)
		ast_verbose("Inbound H.323 call '%s' detected.\n", cd.call_token);

	/* Update statistics */
	ast_mutex_lock(&oh323_stats_lock);
	++oh323_stats.setup_recv;
	ast_mutex_unlock(&oh323_stats_lock);

	tv.tv_sec = 3;
	tv.tv_usec = 0;
	switch (mutex_trylock_timeout(&oh323_tab_lock, tv)) {
	case -1:
		ast_log(LOG_WARNING, "Inbound call '%s' failed to acquire lock.\n",
						cd.call_token);
		return(-1);
	case 1:
		ast_log(LOG_WARNING, "Inbound call '%s' failed to be serviced.\n",
						cd.call_token);
		return(-1);
	default:
		/* Lock is held */
		break;
	}
	//ast_mutex_lock(&oh323_tab_lock);

	index = in_call_rate_update();

	/* Check the number of active inbound and simultaneous H.323 calls */
	count = 0;
	simcount = 0;
	for (i=0; i<config_options.totalNum; i++)
		if (oh323_tab[i] != NULL) {
			if (oh323_tab[i]->from_remote == 1)
				++count;
			++simcount;
		}

	/* Apply call number limitations */
	if ((count >= config_options.inboundMax) ||
		((config_options.simultaneousMax > 0) && 
		(simcount >= config_options.simultaneousMax))) {

		ast_log(LOG_WARNING, "Inbound call '%s' dropped due to call-limit violation.\n",
							cd.call_token);

		/* Update statistics */
		in_call_mark(index, 0);
		ast_mutex_lock(&oh323_stats_lock);
		++oh323_stats.block_incall;
		ast_mutex_unlock(&oh323_stats_lock);

		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return(-1);
	}

	intime = in_call_time_get();
	calls_passed = in_call_number_passed() + 1; /* Assume that this call will
												   go through */
	if (intime > 0)
		cur_rate = (float)calls_passed * 1000 / (float)intime;
	else
		cur_rate = 0.0;
#if 0
	ast_log(LOG_DEBUG, "Cur_rate = %f, Max_rate = %f, Sim_calls = %d.\n", 
					cur_rate, in_call_max_rate, simcount);
#endif

	/* Apply call rate limitations */
	if ((simcount > config_options.crlThreshold) &&
		(cur_rate > in_call_max_rate)) {

		ast_log(LOG_WARNING, "Inbound call '%s' dropped due to in-call-rate violation (%.2f).\n",
							cd.call_token, cur_rate);

		/* Update statistics */
		in_call_mark(index, 0);
		ast_mutex_lock(&oh323_stats_lock);
		++oh323_stats.block_incall;
		ast_mutex_unlock(&oh323_stats_lock);

		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return(-1);
	}

	/* Search for a free private structure entry */
	for (i=0; i<config_options.totalNum; i++)
		if (oh323_tab[i] == NULL)
			break;

	/* No free entry */
	if ((i<0) || (i>=config_options.totalNum)) {
		ast_log(LOG_ERROR, "Inbound call '%s' dropped because there is no space.\n",
							cd.call_token);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return(-1);
	}

	/* We found a free entry */
	oh323_tab[i] = new_oh323(i);
	if (oh323_tab[i]) {
		if (option_debug)
			ast_log(LOG_DEBUG, "Inbound call '%s' stored in entry %d.\n", 
							cd.call_token, i);
		/* The call was initiated from the remote endpoint */
		oh323_tab[i]->from_remote = 1;
		in_call_mark(index, 1);

		/* Copy the call details */
		if (copy_call_details(&cd, &(oh323_tab[i]->cd)) != 0){
			ast_log(LOG_ERROR, "Failed to copy call details.\n");
			ast_mutex_unlock(&oh323_tab_lock);
			MARK_OUT();
			return(-1);
		}

		/* Attach the channel to ASTERISK PBX */
		if ((c = ast_oh323_new(oh323_tab[i], AST_STATE_RING))) {
			if (option_debug)
				ast_log(LOG_DEBUG, "Channel %s created and attached for inbound H.323 call '%s'.\n",
								c->name, cd.call_token);
			if (oh323_verbose_debug)
				ast_verbose("Channel %s created and attached for inbound H.323 call '%s'.\n",
								c->name, cd.call_token);
			ISTATE_LOG(oh323_tab[i]->i_state, OH323_STATE_RING);
			oh323_tab[i]->i_state = OH323_STATE_RING;
			ast_mutex_unlock(&oh323_tab_lock);
			MARK_OUT();
			return(0);
		} else {
			ast_log(LOG_ERROR, "Failed to create channel for inbound call '%s'.\n",
							cd.call_token);
			oh323_close_call_fds(i);
			oh323_destroy(i);
			free(oh323_tab[i]);
			oh323_tab[i] = NULL;
			ast_mutex_unlock(&oh323_tab_lock);
			MARK_OUT();
			return(-1);
		}
	}
	ast_log(LOG_WARNING, "Failed to create private structure in entry %d for inbound call '%s'.\n", 
					i, cd.call_token);
	ast_mutex_unlock(&oh323_tab_lock);
	MARK_OUT();
	return(-1);
}

/**
 * Call-back function to activate the exception handling mechanism
 * of ASTERISK. Copy 'type' and 'data' into private channel's "except_struct"
 * data structure and trigger the processing of the pending exception.
 */
int exception_h323_connection(call_details_t cd, int type, char *data)
{
	int i;
	char exception_buf[2] = { OH323_EVENT_EXCE };

	MARK_IN();
	ast_mutex_lock(&oh323_tab_lock);

	i = find_call(cd.call_token);
	if (i < 0) {
		if (option_debug)
			ast_log(LOG_WARNING, "Call '%s' not found.\n", cd.call_token);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return(-1);
	}

	if (type == OH323EXC_CTRL_ERROR) {
		/* Update statistics */
		ast_mutex_lock(&oh323_stats_lock);
		++oh323_stats.proto_err;
		ast_mutex_unlock(&oh323_stats_lock);
	}

	/* Do we have an owner? */
	if (oh323_tab[i]->owner == NULL) {
		ast_log(LOG_WARNING, "Call '%s' has no owner. Autodestroying it.\n", 
						cd.call_token);
		oh323_tab[i]->needs_destroy = 1;
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return(0);
	}

	/* Fill the exception structure */
	switch (type) {
		case OH323EXC_USER_INPUT_TONE:
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' has exception USER_INPUT_TONE.\n",
								oh323_tab[i]->cd.call_token);
			if (oh323_verbose_debug)
				ast_verbose("Call '%s' has exception USER_INPUT_TONE (%c).\n",
								oh323_tab[i]->cd.call_token,
								data[0]);
			(oh323_tab[i]->except_struct).type = type;
			(oh323_tab[i]->except_struct).data[0] = data[0];
			break;

		case OH323EXC_USER_MESSAGE:
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' has exception USER_MESSAGE.\n",
								oh323_tab[i]->cd.call_token);
			if (oh323_verbose_debug)
				ast_verbose("Call '%s' has exception USER_MESSAGE.\n",
								oh323_tab[i]->cd.call_token);
			(oh323_tab[i]->except_struct).type = type;
			memset((oh323_tab[i]->except_struct).data, 0, MAX_EXCEPTION_DATA);
			strncpy((oh323_tab[i]->except_struct).data, data, 
							MAX_EXCEPTION_DATA - 1);
			break;

		case OH323EXC_CALL_ALERTED:
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' has exception CALL_ALERTED.\n",
								oh323_tab[i]->cd.call_token);
			if (oh323_verbose_debug)
				ast_verbose("Call '%s' has exception CALL_ALERTED.\n",
								oh323_tab[i]->cd.call_token);
			(oh323_tab[i]->except_struct).type = type;
			break;

		case OH323EXC_CALL_TRANSFER:
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' has exception CALL_TRANSFER.\n",
								oh323_tab[i]->cd.call_token);
			if (oh323_verbose_debug)
				ast_verbose("Call '%s' has exception CALL_TRANSFER.\n",
								oh323_tab[i]->cd.call_token);
			(oh323_tab[i]->except_struct).type = type;
			memset((oh323_tab[i]->except_struct).data, 0, MAX_EXCEPTION_DATA);
			strncpy((oh323_tab[i]->except_struct).data, data, 
							MAX_EXCEPTION_DATA - 1);
			break;

		case OH323EXC_CALL_ESTABLISHED:
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' has exception CALL_ESTABLISHED.\n",
								oh323_tab[i]->cd.call_token);
			if (oh323_verbose_debug)
				ast_verbose("Call '%s' has exception CALL_ESTABLISHED.\n",
								oh323_tab[i]->cd.call_token);
			(oh323_tab[i]->except_struct).type = type;
			memset((oh323_tab[i]->except_struct).data, 0, MAX_EXCEPTION_DATA);
			strncpy((oh323_tab[i]->except_struct).data, data, 
							MAX_EXCEPTION_DATA - 1);
			ISTATE_LOG(oh323_tab[i]->i_state, OH323_STATE_ESTABLISHED);
			oh323_tab[i]->i_state = OH323_STATE_ESTABLISHED;

			if (oh323_tab[i]->from_remote == 1) {
				if (option_debug)
					ast_log(LOG_DEBUG, "Call '%s' established (remote).\n",
									oh323_tab[i]->cd.call_token);
				if (oh323_verbose_debug)
					ast_verbose("Call '%s' established (remote).\n",
									oh323_tab[i]->cd.call_token);

				/* Update statistics */
				ast_mutex_lock(&oh323_stats_lock);
				++oh323_stats.incall;
				ast_mutex_unlock(&oh323_stats_lock);
			} else {
				if (option_debug)
					ast_log(LOG_DEBUG, "Call '%s' established (local).\n",
									oh323_tab[i]->cd.call_token);
				if (oh323_verbose_debug)
					ast_verbose("Call '%s' established (local).\n",
									oh323_tab[i]->cd.call_token);

				/* Update statistics */
				ast_mutex_lock(&oh323_stats_lock);
				++oh323_stats.outcall;
				ast_mutex_unlock(&oh323_stats_lock);
			}
			break;

		case OH323EXC_CTRL_ERROR:
			if (option_debug)
				ast_log(LOG_DEBUG, "Call '%s' has exception CTRL_ERROR.\n",
								oh323_tab[i]->cd.call_token);
			if (oh323_verbose_debug)
				ast_verbose("Call '%s' has exception CTRL_ERROR.\n",
								oh323_tab[i]->cd.call_token);
			(oh323_tab[i]->except_struct).type = type;
			/* Destroy the call */
			oh323_tab[i]->needs_destroy = 1;
			break;

		default:
			ast_log(LOG_ERROR, "Call '%s' has an unknown exception %d!\n", 
								oh323_tab[i]->cd.call_token, type);
			oh323_tab[i]->needs_destroy = 1;
			ast_mutex_unlock(&oh323_tab_lock);
			MARK_OUT();
			return(-1);
	}

	/* XXX Wake-up Asterisk */
	if (write(oh323_tab[i]->event_pipe[1], exception_buf, 1) != 1) {
		ast_log(LOG_WARNING, "Failed to write to event pipe (%d) for call '%s'.\n",
						type, cd.call_token);
	}

	ast_mutex_unlock(&oh323_tab_lock);

	MARK_OUT();
	return(0);
}

int stats_h323_connection(call_details_t cd, rtp_stats_t rs)
{
/* token - pack_sent - oct_sent - pack_recv - oct_recv - pack_lost - pack_late - pack_ooo - tx_min - tx_avg - tx_max - rx_min - rx_avg - rx_max - jit_max - jit_cur - jit_cur_ms */
#define FORMAT_STRING3  "%30s %5ld %7ld %5ld %7ld %5ld %5ld %5ld (%4ld/%4ld/%4ld)ms (%4ld/%4ld/%4ld)ms (%3ld/%3ld)ms %5ld (%5ldms)\n"
#define FORMAT_STRING4  "%30s (RR) %5ld:%5ld %5ld %5ld %5ld\n"

	int i;
	rtp_stats_t *r;

	MARK_IN();
	ast_mutex_lock(&oh323_tab_lock);

	i = find_call(cd.call_token);
	if (i < 0) {
		if (option_debug)
			ast_log(LOG_WARNING, "Call '%s' not found.\n", cd.call_token);
		ast_mutex_unlock(&oh323_tab_lock);
		MARK_OUT();
		return(-1);
	}

	/* Call reference match. */
	(oh323_tab[i]->rtp_stats)[0] = rs;
	r = &((oh323_tab[i]->rtp_stats)[0]);
	printf(FORMAT_STRING3, 
						(oh323_tab[i]->cd.call_token != NULL ? oh323_tab[i]->cd.call_token:"NULL"),
						r->packets_sent,
						r->octets_sent,
						r->packets_recv,
						r->octets_recv,
						r->packets_lost,
						r->packets_late,
						r->packets_ooo,
						r->min_send_time,
						r->average_send_time,
						r->max_send_time,
						r->min_recv_time,
						r->average_recv_time,
						r->max_recv_time,
						r->average_jitter,
						r->max_jitter,
						r->current_jitter_size,
						(r->current_jitter_size/8));
	if (r->with_rr)
		printf(FORMAT_STRING4, 
						(oh323_tab[i]->cd.call_token != NULL ? oh323_tab[i]->cd.call_token:"NULL"),
						(r->reported_sequence & 0xffff0000) >> 16,
						(r->reported_sequence & 0xffff),
						r->reported_fraction_lost,
						r->reported_total_lost,
						r->reported_jitter);
	ast_mutex_unlock(&oh323_tab_lock);
	MARK_OUT();
	return(0);
}

/******************************************************************************/
/* Monitoring thread and queue call-back functions ****************************/

static int oh323_release(void *data)
{
	struct chan_oh323_pvt *pvt = (struct chan_oh323_pvt *)data;
	int index;

	if (!pvt)
		return(0);

	ast_mutex_lock(&oh323_tab_lock);
	if (pvt->owner) {
		ast_log(LOG_WARNING, "Call '%s' is owned?\n", pvt->cd.call_token);
		ast_mutex_unlock(&oh323_tab_lock);
		return(0);
	}
	index = pvt->index;
	ast_log(LOG_WARNING, "Forcing the release of entry %d (call '%s').\n", 
					index, pvt->cd.call_token);
#if 0
	if (h323_clear_call((pvt->cd).call_token) != CALL_END_OK) 
		ast_log(LOG_WARNING, "Failed to hangup call %s.\n", 
							(pvt->cd).call_token);
#endif
	pvt->needs_destroy = 1;
	pvt->sched_id = -1;
#if 0
	oh323_destroy(index);
	free(oh323_tab[index]);
	oh323_tab[index] = NULL;
#endif
	ast_mutex_unlock(&oh323_tab_lock);
	return(0);
}

static int oh323_gk_check(void *data)
{
	char gkname[256];

	if (config_options.gatekeeperMode != GKMODE_DISABLE) {
		switch (h323_get_gk(gkname, sizeof(gkname))) {
		case 0:
			ast_verbose(VERBOSE_PREFIX_3 "Registered with gatekeeper '%s'.\n",
							(char*)gkname);
			break;
		case OH323GK_FAILED:
			ast_verbose(VERBOSE_PREFIX_3 "Failed to get gatekeeper status.\n");
			break;
		case OH323GK_NOTREGISTERED:
			ast_verbose(VERBOSE_PREFIX_3 "Gatekeeper '%s' found but failed to register.\n",
							(char*)gkname);
			break;
		case OH323GK_NOTUSED:
			if (config_options.gatekeeperMode == GKMODE_DISCOVER) {
				ast_log(LOG_WARNING, "Gatekeeper discovery failed.\n");
			} else {
				ast_log(LOG_WARNING, "Failed to register with gatekeeper '%s'.\n",
							(char*)config_options.gatekeeperName);
			}
			/* Retry the gatekeeper discovery/registration */
			ast_verbose(VERBOSE_PREFIX_3 "Retrying gatekeeper registration.\n");
			if (h323_reset_gk(config_options.gatekeeperMode, 
						config_options.gatekeeperName)) {
				ast_log(LOG_ERROR, "H.323 gatekeeper setup failed.\n");
				return(0);
			}
			/* Reschedule */
			return(1);
		}
	}
	/* Do not reschedule the gatekeeper discovery/registration */
	return(0);
}

/**
 * The monitor thread.
 * This thread executes all the scheduled actions and supervises
 * the active channels.
 */
static void *do_monitor(void *ignore)
{
	struct timeval tv;
	int res, i;

	if (option_debug)
		ast_log(LOG_DEBUG, "Monitor thread started.\n");

	while (1) {
		/* <ENDLESS LOOP> */
		if (ast_mutex_lock(&mon_lock)) {
			ast_log(LOG_ERROR, "Unable to grab monitor lock.\n");
			monitor_running = 0;
			break;
		}
		monitor_running = 1;
		/* Check for a request to die */
		if (monitor_shutdown) {
			if (option_debug)
				ast_log(LOG_DEBUG, "Monitor thread is going down.\n");
			monitor_running = 0;
			ast_mutex_unlock(&mon_lock);
			break;
		}
		ast_mutex_unlock(&mon_lock);

		/* Wait some time */
		tv.tv_sec = 0;
		tv.tv_usec = 765432;
		res = select(0, NULL, NULL, NULL, &tv);
		if (res < 0) {
			if ((errno != EAGAIN) && (errno != EINTR))
				ast_log(LOG_WARNING, "Select failed: %s.\n", strerror(errno));
			continue;
		} else {
			ast_mutex_lock(&mon_lock);

			/* Check and process the run queue */
			ast_sched_runq(oh323_sched);

			/* Check the existing channels */
			ast_mutex_lock(&oh323_tab_lock);
			for (i=0; i<config_options.totalNum; i++)
				if ((oh323_tab[i] != NULL) && (oh323_tab[i]->needs_destroy)) {
					if (oh323_tab[i]->owner == NULL) {
						if (option_debug)
							ast_log(LOG_DEBUG, "Call '%s' without owner needs to be destroyed.\n",
										oh323_tab[i]->cd.call_token);
						if (option_debug) {
							if (oh323_tab[i]->clearing_call) {
								ast_log(LOG_DEBUG, "Call '%s' without owner has already been cleared (%d).\n",
											oh323_tab[i]->cd.call_token, oh323_tab[i]->clearing_call);
							} else {
								ast_log(LOG_DEBUG, "Call '%s' without owner not already cleared???\n",
											oh323_tab[i]->cd.call_token);
							}
						}
						if (oh323_verbose_debug) {
							if (oh323_tab[i]->clearing_call) {
								ast_verbose("Call '%s' without owner has already been cleared (%d).\n",
											oh323_tab[i]->cd.call_token, oh323_tab[i]->clearing_call);
							} else {
								ast_verbose("Call '%s' without owner not already cleared???\n",
											oh323_tab[i]->cd.call_token);
							}
						}
						if (oh323_tab[i]->sched_id != -1) {
							ast_sched_del(oh323_sched, oh323_tab[i]->sched_id);
							if (option_debug)
								ast_log(LOG_DEBUG, "Cancelled scheduled release of call '%s'.\n", 
											oh323_tab[i]->cd.call_token);
							if (oh323_verbose_debug)
								ast_verbose("Cancelled scheduled release of call '%s'.\n", 
											oh323_tab[i]->cd.call_token);
						}
						/* Deallocate the resources */
						oh323_close_call_fds(i);
						oh323_destroy(i);
						free(oh323_tab[i]);
						oh323_tab[i] = NULL;
					} else {
						if (option_debug)
							ast_log(LOG_DEBUG, "Call '%s' with owner needs to be destroyed.\n",
										oh323_tab[i]->cd.call_token);
						if (option_debug) {
							if (oh323_tab[i]->clearing_call) {
								/* Weird situation */
								ast_log(LOG_DEBUG, "Call '%s' with owner has already been cleared (%d).\n",
											oh323_tab[i]->cd.call_token, oh323_tab[i]->clearing_call);
							} else {
								ast_log(LOG_DEBUG, "Call '%s' with owner not already cleared???\n",
											oh323_tab[i]->cd.call_token);
							}
						}
						if (oh323_verbose_debug) {
							if (oh323_tab[i]->clearing_call) {
								/* Weird situation */
								ast_verbose("Call '%s' with owner has already been cleared (%d).\n",
											oh323_tab[i]->cd.call_token, oh323_tab[i]->clearing_call);
							} else {
								ast_verbose("Call '%s' with owner not already cleared???\n",
											oh323_tab[i]->cd.call_token);
							}
						}
						while ((oh323_tab[i]) && (oh323_tab[i]->owner) && 
								(ast_mutex_trylock(&(oh323_tab[i]->owner->lock)))) {
							ast_mutex_unlock(&oh323_tab_lock);
							usleep(10);
							ast_mutex_lock(&oh323_tab_lock);
						}
						if ((oh323_tab[i]) && (oh323_tab[i]->owner)) {
							ast_queue_control(oh323_tab[i]->owner, AST_CONTROL_HANGUP);
							ast_mutex_unlock(&(oh323_tab[i]->owner->lock));
							if (option_debug)
								ast_log(LOG_DEBUG, "Call '%s' has been hungup.\n",
										oh323_tab[i]->cd.call_token);
							if (oh323_verbose_debug)
								ast_verbose("Call '%s' has been hungup.\n",
										oh323_tab[i]->cd.call_token);
						}
					}
				}
			ast_mutex_unlock(&oh323_tab_lock);

			ast_mutex_unlock(&mon_lock);
		}
		/* </ENDLESS LOOP> */
	}
	sched_context_destroy(oh323_sched);
	return NULL;
}


/**
 * Restart the monitor thread.
 */
static int restart_monitor(void)
{
	/* If we're supposed to be stopped -- stay stopped */
	if (monitor_thread == AST_PTHREADT_STOP)
		return 0;
	if (ast_mutex_lock(&mon_lock)) {
		ast_log(LOG_WARNING, "Unable to lock monitor.\n");
		return -1;
	}
	if (monitor_thread == pthread_self()) {
		ast_mutex_unlock(&mon_lock);
		ast_log(LOG_WARNING, "Cannot kill myself.\n");
		return -1;
	}
	if (monitor_thread) {
		/* Wake up the thread */
		pthread_kill(monitor_thread, SIGURG);
		if (option_debug)
			ast_log(LOG_DEBUG, "Monitor thread restarted.\n");
	} else {
		/* Start a new monitor */
		if (ast_pthread_create(&monitor_thread, NULL, do_monitor, NULL) < 0) {
			ast_mutex_unlock(&mon_lock);
			ast_log(LOG_ERROR, "Unable to start monitor thread.\n");
			return -1;
		}
		if (option_debug)
			ast_log(LOG_DEBUG, "New monitor thread started.\n");
	}
	ast_mutex_unlock(&mon_lock);
	return 0;
}

/**
 * Kill the monitor thread.
 */
static int kill_monitor(void)
{
	struct timeval tv;
	int maxtries = 5;
	int res;

	if (!ast_mutex_lock(&mon_lock)) {
		if (!monitor_running)
			return(0);
		if (monitor_thread) {
			monitor_shutdown = 1;
			pthread_kill(monitor_thread, SIGURG);
		}
		ast_mutex_unlock(&mon_lock);
		usleep(100);
		if (option_debug)
			ast_log(LOG_DEBUG, "Waiting monitor thread to come down...\n");
		while (1) {
			ast_mutex_lock(&mon_lock);
			if (!monitor_running) {
				if (option_debug)
					ast_log(LOG_DEBUG, "Monitor thread terminated.\n");
				ast_mutex_unlock(&mon_lock);
				break;
			}
			ast_mutex_unlock(&mon_lock);
			tv.tv_sec = 0;
			tv.tv_usec = 100000;
			res = select(0, NULL, NULL, NULL, &tv);
			if (res < 0) {
				if ((errno != EAGAIN) && (errno != EINTR))
					ast_log(LOG_WARNING, "Select failed: %s.\n", strerror(errno));
			}
			--maxtries;
			if (maxtries < 0) {
				ast_log(LOG_WARNING, "Failed to kill monitor thread.\n");
				return(-1);
			}
		}
		monitor_thread = -2;
	} else {
		ast_log(LOG_WARNING, "Unable to lock the monitor.\n");
		return(-1);
	}
	return(0);
}


/******************************************************************************/
/* Configuration file parsing *************************************************/

static int init_registry_info(struct ast_config *cfg, struct oh323_reginfo **reg,
							  char *default_context)
{
	struct ast_variable	*v = NULL;
	struct oh323_reginfo *regtmp, *regprev;
	int nocontext;

	if ((cfg == NULL)||(reg == NULL)||(default_context == NULL))
		return(-1);
	nocontext = 1;
	*reg = NULL;
	regtmp = NULL;
	regprev = NULL;
	v = ast_variable_browse(cfg, "register");
	while (v) {
		if (!strcasecmp(v->name, "context")) {
			if (regtmp != NULL) {
				if (regtmp->alias_num > 0) {
					regtmp->alias = (char **)malloc(regtmp->alias_num * sizeof(char*));
					if (regtmp->alias == NULL) {
						ast_log(LOG_ERROR, "Memory allocation failed.\n");
						return(-1);
					}
					memset(regtmp->alias, 0, regtmp->alias_num * sizeof(char*));
				}
				if (regtmp->prefix_num > 0) {
					regtmp->prefix = (char **)malloc(regtmp->prefix_num * sizeof(char*));
					if (regtmp->prefix == NULL) {
						ast_log(LOG_ERROR, "Memory allocation failed.\n");
						return(-1);
					}
					memset(regtmp->prefix, 0, regtmp->prefix_num * sizeof(char*));
				}
				regprev = regtmp;
			}
			/* Allocate a new struct */
			regtmp = (struct oh323_reginfo *)malloc(sizeof(struct oh323_reginfo));
			if (regtmp == NULL) {
				ast_log(LOG_ERROR, "Memory allocation failed.\n");
				return(-1);
			}
			memset(regtmp, 0, sizeof(struct oh323_reginfo));
			if (*reg == NULL)	/* Save the head of the list */
				*reg = regtmp;
			else				/* Connect the new struct to the end of the list */
				regprev->next = regtmp;
			strncpy(regtmp->context, v->value, AST_MAX_EXTENSION-1);
			nocontext = 0;
		} else if (!strcasecmp(v->name, "alias")) {
			if ((nocontext)&&(regtmp == NULL)) {
				/* Allocate a new struct */
				regtmp = (struct oh323_reginfo *)malloc(sizeof(struct oh323_reginfo));
				if (regtmp == NULL) {
					ast_log(LOG_ERROR, "Memory allocation failed.\n");
					return(-1);
				}
				memset(regtmp, 0, sizeof(struct oh323_reginfo));
				*reg = regtmp; /* Save the head of the list */
				strncpy(regtmp->context, default_context, AST_MAX_EXTENSION-1);
			}
			++regtmp->alias_num;
		} else if (!strcasecmp(v->name, "gwprefix")) {
			if ((nocontext)&&(regtmp == NULL)) {
				/* Allocate a new struct */
				regtmp = (struct oh323_reginfo *)malloc(sizeof(struct oh323_reginfo));
				if (regtmp == NULL) {
					ast_log(LOG_ERROR, "Memory allocation failed.\n");
					return(-1);
				}
				memset(regtmp, 0, sizeof(struct oh323_reginfo));
				*reg = regtmp; /* Save the head of the list */
				strncpy(regtmp->context, default_context, AST_MAX_EXTENSION-1);
			}
			++regtmp->prefix_num;
		} else
			ast_log(LOG_NOTICE, "Ignoring unknown H.323 [register] keyword '%s', line %d.\n", 
								v->name, v->lineno);
		v = v->next;
	}
	/* Update the last struct of the list */
	if (regtmp != NULL) {
		if (regtmp->alias_num > 0) {
			regtmp->alias = (char **)malloc(regtmp->alias_num * sizeof(char*));
			if (regtmp->alias == NULL) {
				ast_log(LOG_ERROR, "Memory allocation failed.\n");
				return(-1);
			}
			memset(regtmp->alias, 0, regtmp->alias_num * sizeof(char*));
		}
		if (regtmp->prefix_num > 0) {
			regtmp->prefix = (char **)malloc(regtmp->prefix_num * sizeof(char*));
			if (regtmp->prefix == NULL) {
				ast_log(LOG_ERROR, "Memory allocation failed.\n");
				return(-1);
			}
			memset(regtmp->prefix, 0, regtmp->prefix_num * sizeof(char*));
		}
		regtmp->next = NULL;
	}

	return(0);
}

static int reload_config(void)
{
	struct ast_config *cfg = NULL;
	struct ast_variable	*v = NULL;
	struct oh323_reginfo *regtmp;
	struct oh323_codecinfo *codectmp, *codecprev;
	struct oh323_ep *extprev, *exttmp;
	char *s;
	int format, alias_index, prefix_index;
	int codec_found, frames_found, id_count;
	int vcodec, flags;
	int i;


	/* Initialize stuff */
	memset((char *)&config_options, 0, sizeof(config_options));
	config_options.listenPort = 1720;
	config_options.connectPort = 1720;
	config_options.tcpStart = 5000;
	config_options.tcpEnd = 31000;
	config_options.udpStart = 5000;
	config_options.udpEnd = 31000;
	config_options.rtpStart = 5000;
	config_options.rtpEnd = 31000;
	config_options.fastStart = 0;
	config_options.h245Tunnelling = 0;
	config_options.h245inSetup = 0;
	config_options.inBandDTMF = 0;
	config_options.silenceSuppression = 0;
	config_options.jitterMin = 20;		/* ms */
	config_options.jitterMax = 100;		/* ms */
#ifdef HAS_OH323MODS
	config_options.jitterDec = 2;		/* frames */
	config_options.jitterInc = 1;		/* frames */
#endif
	config_options.ipTos = 0;
	config_options.outboundMax = 0;
	config_options.inboundMax = 0;
	config_options.simultaneousMax = -1;
	config_options.totalNum = 0;
	config_options.bwlimit = 1024*1024;		/* 1 Mbit/sec */
	config_options.crlCallNumber = MAX_INCALLS;
	config_options.crlCallTime = MAX_INCALLTIME;
	config_options.crlThreshold = MAX_INCALLTIME_CALLTHRESHOLD;
	config_options.wrapLibTraceLevel = 0;
	config_options.libTraceLevel = 0;
	strncpy(config_options.libTraceFile, "stdout", 
					sizeof(config_options.libTraceFile)-1);
	config_options.amaFlags = 0;
	config_options.gatekeeperMode = GKMODE_DISABLE;
	config_options.gatekeeperTTL = 0;		/* for ever */
	config_options.userInputMode = UIMODE_TONE;
	strncpy(config_options.context, "default", AST_MAX_EXTENSION-1);
	oh323_capability = 0;

	/* Get the RTP port range from "rtp.conf" */
	cfg = ast_load("rtp.conf");
	if (cfg) {
		if ((s = ast_variable_retrieve(cfg, "general", "rtpstart"))) {
			config_options.rtpStart = atoi(s);
			if (config_options.rtpStart < 1024)
				config_options.rtpStart = 1024;
			if (config_options.rtpStart > 65535)
				config_options.rtpStart = 65535;
		}
		if ((s = ast_variable_retrieve(cfg, "general", "rtpend"))) {
			config_options.rtpEnd = atoi(s);
			if (config_options.rtpEnd < 1024)
				config_options.rtpEnd = 1024;
			if (config_options.rtpEnd > 65535)
				config_options.rtpEnd = 65535;
		}
		ast_destroy(cfg);
	}

	/* We must have a config file, otherwise stop */
	cfg = ast_load(config);
	if (!cfg) {
		ast_log(LOG_ERROR, "Unable to load configuration file (%s).\n", config);
		return(-1);
	}

	/* There must be a [general] category, otherwise stop */
	v = ast_variable_browse(cfg, "general");
	if (v == NULL) {
		ast_log(LOG_ERROR, "Category [general] not present in configuration file (%s).\n", config);
		return(-1);
	}
	while (v) {
		if (!strcasecmp(v->name, "listenAddress")) {
			strncpy(config_options.listenAddress, v->value, 
							sizeof(config_options.listenAddress)-1);
		} else if (!strcasecmp(v->name, "listenPort")) {
			config_options.listenPort = (int) strtol(v->value, NULL, 10);
		} else if (!strcasecmp(v->name, "connectPort")) {
			config_options.connectPort = (int) strtol(v->value, NULL, 10);
		} else if (!strcasecmp(v->name, "tcpStart")) {
			config_options.tcpStart = (int) strtol(v->value, NULL, 10);
		} else if (!strcasecmp(v->name, "tcpEnd")) {
			config_options.tcpEnd = (int) strtol(v->value, NULL, 10);
		} else if (!strcasecmp(v->name, "udpStart")) {
			config_options.udpStart = (int) strtol(v->value, NULL, 10);
		} else if (!strcasecmp(v->name, "udpEnd")) {
			config_options.udpEnd = (int) strtol(v->value, NULL, 10);
		} else if (!strcasecmp(v->name, "fastStart")) {
			config_options.fastStart = ast_true(v->value);
		} else if (!strcasecmp(v->name, "h245Tunnelling")) {
			config_options.h245Tunnelling = ast_true(v->value);
		} else if (!strcasecmp(v->name, "h245inSetup")) {
			config_options.h245inSetup = ast_true(v->value);
		} else if (!strcasecmp(v->name, "inBandDTMF")) {
			config_options.inBandDTMF = ast_true(v->value);
		} else if (!strcasecmp(v->name, "silenceSuppression")) {
			config_options.silenceSuppression = ast_true(v->value);
		} else if (!strcasecmp(v->name, "jitterMin")) {
			config_options.jitterMin = (int) strtol(v->value, NULL, 10);
			if (config_options.jitterMin < 20) {
				ast_log(LOG_WARNING, "Invalid min jitter value (<20ms)! Setting it to 20ms.\n");
				config_options.jitterMin = 20;
			} else if (config_options.jitterMin > 10000) {
				ast_log(LOG_WARNING, "Invalid min jitter value (>10s)! Setting it to 10s.\n");
				config_options.jitterMin = 10000;
			}
		} else if (!strcasecmp(v->name, "jitterMax")) {
			config_options.jitterMax = (int) strtol(v->value, NULL, 10);
			if (config_options.jitterMax < 20) {
				ast_log(LOG_WARNING, "Invalid max jitter value (<20ms)! Setting it to 20ms.\n");
				config_options.jitterMax = 20;
			} else if (config_options.jitterMax > 10000) {
				ast_log(LOG_WARNING, "Invalid max jitter value (>10s)! Setting it to 10s.\n");
				config_options.jitterMax = 10000;
			}
#ifdef	HAS_OH323MODS
		} else if (!strcasecmp(v->name, "jitterDec")) {
			config_options.jitterDec = (int) strtol(v->value, NULL, 10);
		} else if (!strcasecmp(v->name, "jitterInc")) {
			config_options.jitterInc = (int) strtol(v->value, NULL, 10);
#endif
		} else if (!strcasecmp(v->name, "ipTos")) {
			if (sscanf(v->value, "%i", &format) == 1)
				config_options.ipTos = format & 0xff;
			else if (!strcasecmp(v->value, "lowdelay"))
				config_options.ipTos = IPTOS_LOWDELAY;
			else if (!strcasecmp(v->value, "throughput"))
				config_options.ipTos = IPTOS_THROUGHPUT;
			else if (!strcasecmp(v->value, "reliability"))
				config_options.ipTos = IPTOS_RELIABILITY;
			else if (!strcasecmp(v->value, "mincost"))
				config_options.ipTos = IPTOS_MINCOST;
			else if (!strcasecmp(v->value, "none"))
				config_options.ipTos = 0;
			else
				ast_log(LOG_WARNING, "Invalid ToS value (%s) at line %d, "
								"should be 'lowdelay', 'throughput', "
								"'reliability', 'mincost', or 'none'.\n",
								v->value, v->lineno);
		} else if (!strcasecmp(v->name, "outboundMax")) {
			config_options.outboundMax = (int) strtol(v->value, NULL, 10);
			if (config_options.outboundMax < 0) {
				ast_log(LOG_WARNING, "Invalid max number of outbound H.323 connections (<0)! Setting it to 0.\n");
				config_options.outboundMax = 0;
			}
		} else if (!strcasecmp(v->name, "inboundMax")) {
			config_options.inboundMax = (int) strtol(v->value, NULL, 10);
			if (config_options.inboundMax < 0) {
				ast_log(LOG_WARNING, "Invalid max number of inbound H.323 connections (<0)! Setting it to 0.\n");
				config_options.inboundMax = 0;
			}
		} else if (!strcasecmp(v->name, "simultaneousMax")) {
			config_options.simultaneousMax = (int) strtol(v->value, NULL, 10);
			if (config_options.simultaneousMax < 0) {
				ast_log(LOG_WARNING, "Invalid max number of simultaneous H.323 connections (<0)! Setting it to 0.\n");
				config_options.simultaneousMax = 0;
			}
		} else if (!strcasecmp(v->name, "bandwidthLimit")) {
			config_options.bwlimit = (int) strtol(v->value, NULL, 10);
			config_options.bwlimit = (int) ((float)config_options.bwlimit*(1024.0/100.0));
			if (config_options.bwlimit < 0) {
				ast_log(LOG_WARNING, "Invalid bandwidth limit value for H.323 connections (<0)! Setting it to 0.\n");
				config_options.bwlimit = 0;
			}
		} else if (!strcasecmp(v->name, "crlCallNumber")) {
			config_options.crlCallNumber = (int) strtol(v->value, NULL, 10);
			if (config_options.crlCallNumber < 0) {
				ast_log(LOG_WARNING, "Invalid call rate limiter parameter (crlCallNumber)! Setting it to 0.\n");
				config_options.crlCallNumber = 0;
			}
		} else if (!strcasecmp(v->name, "crlCallTime")) {
			config_options.crlCallTime = (int) strtol(v->value, NULL, 10);
			if (config_options.crlCallTime < 0) {
				ast_log(LOG_WARNING, "Invalid call rate limiter parameter (crlCallTime)! Setting it to 0.\n");
				config_options.crlCallTime = 0;
			}
		} else if (!strcasecmp(v->name, "crlThreshold")) {
			config_options.crlThreshold = (int) strtol(v->value, NULL, 10);
			if (config_options.crlThreshold < 0) {
				ast_log(LOG_WARNING, "Invalid call rate limiter parameter (crlThreshold)! Setting it to 0.\n");
				config_options.crlThreshold = 0;
			}
		} else if (!strcasecmp(v->name, "wrapLibTraceLevel")) {
			config_options.wrapLibTraceLevel = (int) strtol(v->value, NULL, 10);
			if (config_options.wrapLibTraceLevel < 0) {
				ast_log(LOG_WARNING, "Invalid wrapper library trace level! Disabling it.\n");
				config_options.wrapLibTraceLevel = 0;
			}
		} else if (!strcasecmp(v->name, "libTraceLevel")) {
			config_options.libTraceLevel = (int) strtol(v->value, NULL, 10);
			if (config_options.libTraceLevel < 0) {
				ast_log(LOG_WARNING, "Invalid library trace level! Disabling it.\n");
				config_options.libTraceLevel = 0;
			}
		} else if (!strcasecmp(v->name, "libTraceFile")) {
			if ((v->value[0] == '\0') || (!strcasecmp(v->value, "stdout")))
				memset(config_options.libTraceFile, 0, sizeof(config_options.libTraceFile));
			else
				strncpy(config_options.libTraceFile, v->value, 
								sizeof(config_options.libTraceFile)-1);
		} else if (!strcasecmp(v->name, "accountCode")) {
			strncpy(config_options.accountCode, v->value, 
								sizeof(config_options.accountCode)-1);
		} else if (!strcasecmp(v->name, "amaFlags")) {
			flags = ast_cdr_amaflags2int(v->value);
			if (flags < 0) {
				ast_log(LOG_WARNING, "Invalid AMA Flags: %s at line %d\n", v->value, v->lineno);
			} else {
				config_options.amaFlags = flags;
			}
		} else if (!strcasecmp(v->name, "gatekeeper")) {
			if (!strcasecmp(v->value, "DISABLE")) {
				config_options.gatekeeperMode = GKMODE_DISABLE;
			} else if (!strcasecmp(v->value, "DISCOVER")) {
				config_options.gatekeeperMode = GKMODE_DISCOVER;
			} else if (!strncasecmp(v->value, "GKID:", 5)){
				config_options.gatekeeperMode = GKMODE_ID;
				strncpy(config_options.gatekeeperName, (char*)(v->value+5), 
								sizeof(config_options.gatekeeperName)-1);
			} else {
				config_options.gatekeeperMode = GKMODE_NAME;
				strncpy(config_options.gatekeeperName, v->value, 
								sizeof(config_options.gatekeeperName)-1);
			}
		} else if (!strcasecmp(v->name, "gatekeeperPassword")) {
			strncpy(config_options.gatekeeperPass, v->value, 
							sizeof(config_options.gatekeeperPass)-1);
		} else if (!strcasecmp(v->name, "gatekeeperTTL")) {
			config_options.gatekeeperTTL = (int) strtol(v->value, NULL, 10);
			if (config_options.gatekeeperTTL < 0) {
				ast_log(LOG_WARNING, "Invalid gatekeeper TTL value %s at line %d\n", 
								v->value, v->lineno);
				config_options.gatekeeperTTL = 0;
			}
		} else if (!strcasecmp(v->name, "userInputMode")) {
			if (!strcasecmp(v->value, "q931")) {
				config_options.userInputMode = UIMODE_Q931;
			} else if (!strcasecmp(v->value, "string")) {
				config_options.userInputMode = UIMODE_STRING;
			} else if (!strcasecmp(v->value, "tone")) {
				config_options.userInputMode = UIMODE_TONE;
			} else if (!strcasecmp(v->value, "rfc2833")) {
				config_options.userInputMode = UIMODE_RFC2833;
			} else
				ast_log(LOG_WARNING, "%s: Unknown user-input mode.\n", v->value);
		} else if (!strcasecmp(v->name, "context")) {
			strncpy(config_options.context, v->value, sizeof(config_options.context)-1);
		} else
			ast_log(LOG_NOTICE, "Ignoring unknown H.323 [general] keyword '%s', line %d.\n", 
								v->name, v->lineno);
		v = v->next;
	}

	if (config_options.tcpStart >= config_options.tcpEnd) {
		ast_log(LOG_WARNING, "Invalid TCP port values (start=%d, end=%d). Resetting to defaults.\n",
						config_options.tcpStart, config_options.tcpEnd);
		config_options.tcpStart = 5000;
		config_options.tcpEnd = 31000;
	}
	if (config_options.udpStart >= config_options.udpEnd) {
		ast_log(LOG_WARNING, "Invalid UDP port values (start=%d, end=%d). Resetting to defaults.\n",
						config_options.udpStart, config_options.udpEnd);
		config_options.udpStart = 5000;
		config_options.udpEnd = 31000;
	}
	if (config_options.jitterMin > config_options.jitterMax) {
		ast_log(LOG_WARNING, "Invalid jitter buffer values (min=%d, max=%d). Resetting to defaults.\n",
						config_options.jitterMin, config_options.jitterMax);
		config_options.jitterMin = 20;
		config_options.jitterMax = 100;
	}
	/* XXX We should restrict this number to an upper limit */
	config_options.totalNum = config_options.inboundMax+config_options.outboundMax;
	if (config_options.simultaneousMax > -1) {
		if (config_options.simultaneousMax > config_options.totalNum) {
			ast_log(LOG_NOTICE, "Value %d of max simultaneous H.323 calls "
						"doesn't make sense (max total number of H.323 calls "
						"is %d). Disabled.\n", 
						config_options.simultaneousMax,
						config_options.totalNum);
			config_options.simultaneousMax = -1;
		}
		i = config_options.inboundMax < config_options.outboundMax ? 
					config_options.outboundMax : config_options.inboundMax;
		if (config_options.simultaneousMax < i) {
			ast_log(LOG_NOTICE, "Value %d of max simultaneous H.323 calls "
						"doesn't make sense (max number of H.323 calls on one "
						"direction is %d). Disabled.\n", 
						config_options.simultaneousMax, i);
			config_options.simultaneousMax = -1;
		}
	}

	/* Check for a [register] category */
	if (init_registry_info(cfg, &(config_options.regInfo), 
							config_options.context) != 0) {
		return(-1);
	}
	if (config_options.regInfo) {
		regtmp = config_options.regInfo;
		alias_index = 0;
		prefix_index = 0;
		v = ast_variable_browse(cfg, "register");
		while (v) {
			if ((alias_index == regtmp->alias_num)&&(prefix_index == regtmp->prefix_num)) {
				regtmp = regtmp->next;
				alias_index = 0;
				prefix_index = 0;
			}
			if (!strcasecmp(v->name, "alias")) {
				regtmp->alias[alias_index] = (char *)malloc(strlen(v->value)+1);
				if (regtmp->alias[alias_index] == NULL) {
					ast_log(LOG_ERROR, "Memory allocation failed.\n");
					return(-1);
				}
				memset(regtmp->alias[alias_index], 0, strlen(v->value)+1);
				strcpy(regtmp->alias[alias_index], v->value);
				++alias_index;
			} else if (!strcasecmp(v->name, "gwprefix")) {
				regtmp->prefix[prefix_index] = (char *)malloc(strlen(v->value)+1);
				if (regtmp->prefix[prefix_index] == NULL) {
					ast_log(LOG_ERROR, "Memory allocation failed.\n");
					return(-1);
				}
				memset(regtmp->prefix[prefix_index], 0, strlen(v->value)+1);
				strcpy(regtmp->prefix[prefix_index], v->value);
				++prefix_index;
			}
			v = v->next;
		}
	} else {
		ast_log(LOG_NOTICE, "H.323 [register] category contains no entries.\n");
	}

	/* Check for a [endpoints] category */
	id_count = 0;
	exttmp = NULL;
	extprev = NULL;
	v = ast_variable_browse(cfg, "endpoints");
	while (v) {
		if (!strcasecmp(v->name, "name")) {
			if (exttmp != NULL)
				extprev = exttmp;
			exttmp = (struct oh323_ep *)malloc(sizeof(struct oh323_ep));
			if (exttmp == NULL) {
				ast_log(LOG_ERROR, "Memory allocation failed.\n");
				return(-1);
			}
			memset((char*)exttmp, 0, sizeof(struct oh323_ep));
			strncpy(exttmp->name, v->value, sizeof(exttmp->name)-1);
			exttmp->next = NULL;
			if (oh323_ep_list.head == NULL) /* Save the head */
				oh323_ep_list.head = exttmp;
			else /* Connect the next element */
				extprev->next = exttmp;
			id_count++;
		} else if (!strcasecmp(v->name, "callerid")) {
			if (exttmp)
				strncpy(exttmp->callerid, v->value, sizeof(exttmp->callerid)-1);
		} else if (!strcasecmp(v->name, "host")) {
			if (exttmp)
				strncpy(exttmp->host, v->value, sizeof(exttmp->host)-1);
		} else if (!strcasecmp(v->name, "cancallforward")) {
			if (exttmp)
				exttmp->cancallforward = ast_true(v->value);
		} else if (!strcasecmp(v->name, "forwardexten")) {
			if (exttmp)
				strncpy(exttmp->forwardexten, v->value, sizeof(exttmp->forwardexten)-1);
		} else
			ast_log(LOG_NOTICE, "Ignoring unknown H.323 [endpoints] keyword '%s'.\n", 
							v->name);
		v = v->next;
	}
	if (option_debug) {
		if (id_count)
			ast_log(LOG_DEBUG, "H.323 [endpoints] category contains %d IDs.\n",
							id_count);
		else
			ast_log(LOG_DEBUG, "H.323 [endpoints] category contains no entries.\n");
	}

	/* There must be a [codecs] category, otherwise stop */
	codec_found = 0;
	frames_found = 0;
	codectmp = NULL;
	config_options.codecInfo = NULL;
	v = ast_variable_browse(cfg, "codecs");
	if (v == NULL) {
		ast_log(LOG_WARNING, "Category [codecs] not present in configuration file (%s).\n", config);
		ast_destroy(cfg);
		return(0);
	}
	while (v) {
		if (!strcasecmp(v->name, "codec")) {
			vcodec = oh323_str2codec(v->value);
			if (vcodec == 0) {
				ast_log(LOG_WARNING, "Unknown codec '%s'.\n", v->value);
				codec_found = 0;
			} else {
				format = oh323_codec2format(vcodec);
				if (format == 0) {
					ast_log(LOG_WARNING, "Unsupported codec '%s'.\n", v->value);
					codec_found = 0;
				} else {
					oh323_capability |= format;

					/* Allocate a new struct */
					if (codectmp != NULL)
						codecprev = codectmp;
					codectmp = (struct oh323_codecinfo*)malloc(sizeof(struct oh323_codecinfo));
					if (codectmp == NULL) {
						ast_log(LOG_ERROR, "Memory allocation failed.\n");
						return(-1);
					}
					memset(codectmp, 0, sizeof(struct oh323_codecinfo));
					codectmp->codec = vcodec;
					codectmp->format = format;
					codectmp->frames = 1;
					if (config_options.codecInfo == NULL)	/* Save the head of the list */
						config_options.codecInfo = codectmp;
					else	 /* Connect the new struct to the end of the list */
						codecprev->next = codectmp;
					codec_found = 1;
					frames_found = 0;
				}
			}
		} else if (!strcasecmp(v->name, "frames")) {
			if ((!codec_found)||(frames_found))
				ast_log(LOG_NOTICE, "Ignoring unexpected 'frames' options at line %d.\n", v->lineno);
			else {
				codectmp->frames = (int) strtol(v->value, NULL, 10);
				frames_found = 1;
				codec_found = 0;
			}
		} else
			ast_log(LOG_NOTICE, "Ignoring unknown H.323 [codecs] keyword '%s'.\n", v->name);
		v = v->next;
	}
	oh323_full_capability = oh323_capability;

	ast_destroy(cfg);

	/******** DEBUG START *****
	regtmp = config_options.regInfo;
	while (regtmp) {
		printf("Context: %s\n", regtmp->context);
		for (i=0; i<regtmp->alias_num; i++)
			printf("   Alias: %s\n", regtmp->alias[i]);
		for (i=0; i<regtmp->prefix_num; i++)
			printf("   Prefix: %s\n", regtmp->prefix[i]);
		regtmp = regtmp->next;
	}
	codectmp = config_options.codecInfo;
	while (codectmp) {
		printf("Codec: %d\n", codectmp->codec);
		printf("   Format: %d\n", codectmp->format);
		printf("   Frames: %d\n", codectmp->frames);
		codectmp = codectmp->next;
	}
	******** DEBUG END *****/

	return(0);
}

/**
 * Deallocate the dynamically allocated parts of the configuration
 * structure.
 */
static void destroy_config(void)
{
	struct oh323_reginfo *regtmp, *regtmp1;
	struct oh323_codecinfo *codectmp, *codectmp1;
	struct oh323_ep *exttmp, *exttmp1;
	int i;

	/* [register] section */
	regtmp = config_options.regInfo;
	while (regtmp) {
		for (i=0; i<regtmp->alias_num; i++)
			if (regtmp->alias[i])
				free(regtmp->alias[i]);
		for (i=0; i<regtmp->prefix_num; i++)
			if (regtmp->prefix[i])
				free(regtmp->prefix[i]);
		regtmp1 = regtmp->next;
		free(regtmp);
		regtmp = regtmp1;
	}

	/* [endpoints] section */
	exttmp = oh323_ep_list.head;
	while (exttmp) {
		exttmp1 = exttmp->next;
		free(exttmp);
		exttmp = exttmp1;
	}
	oh323_ep_list.head = NULL;

	/* [codecs] section */
	codectmp = config_options.codecInfo;
	while (codectmp) {
		codectmp1 = codectmp->next;
		free(codectmp);
		codectmp = codectmp1;
	}
	memset((char *)&config_options, 0, sizeof(config_options));
}

/**
 * Perform cleanup of the OpenH323 channel driver.
 * Return nothing.
 */
void oh323_atexit(void)
{
	int i, res;

	ast_mutex_lock(&usecnt_lock);
	res = usecnt;
	ast_mutex_unlock(&usecnt_lock);
	if (res > 0) {
		ast_log(LOG_WARNING, "OpenH323 channel driver is busy!\n");
		return;
	}

	if (option_verbose > 1)
		ast_verbose(VERBOSE_PREFIX_2 "Cleaning up OpenH323 channel driver.\n");

	if (option_debug)
		ast_log(LOG_DEBUG, "Unregistering CLI extensions.\n");
	ast_cli_unregister(&cli_show_conf);
	ast_cli_unregister(&cli_show_stats);
	ast_cli_unregister(&cli_show_info);
	ast_cli_unregister(&cli_show_established);
	//ast_cli_unregister(&cli_show_ep);
	ast_cli_unregister(&cli_debug_toggle);
	//ast_cli_unregister(&cli_mode);
	ast_cli_unregister(&cli_vars);

	/* Unregister channel type */
	if (option_debug)
		ast_log(LOG_DEBUG, "Unregistering channel type.\n");
	ast_channel_unregister(type);

	/* Kill the monitor thread */
	if (option_debug)
		ast_log(LOG_DEBUG, "Killing monitor thread.\n");
	kill_monitor();

	/* Cleanup stuff */
	if (option_debug)
		ast_log(LOG_DEBUG, "Freeing up resources.\n");
	for (i=0; i<config_options.totalNum; i++)
		if (oh323_tab[i] != NULL) {
			oh323_close_call_fds(i);
			oh323_destroy(i);
			free(oh323_tab[i]);
			oh323_tab[i] = NULL;
		}

	if (option_debug)
		ast_log(LOG_DEBUG, "Removing capabilities.\n");
	if (h323_removeall_capabilities() != CAP_REMOVEALL_OK)
		ast_log(LOG_ERROR, "Unable to remove H323 capabilities.\n");
	if (option_debug)
		ast_log(LOG_DEBUG, "Removing listener.\n");
	if (h323_removeall_listeners() != LIS_REMOVEALL_OK)
		ast_log(LOG_ERROR, "Unable to remove H323 listeners.\n");

	if (option_debug)
		ast_log(LOG_DEBUG, "Destroying endpoint.\n");
	h323_end_point_destroy();
	destroy_config();

	if (option_debug)
		ast_log(LOG_DEBUG, "Done...\n");

	return;
}

/******************************************************************************/
/* Required ASTERISK's stuff **************************************************/

int load_module()
{
	//char **gateway_prefix_tab, **alias_tab;
	//int alias_num;
	//int prefix_num;
	struct oh323_reginfo *regtmp;
	struct oh323_codecinfo *codectmp;
	int i, j, k;
	int res;

	/* Init our locks */
	ast_mutex_init(&usecnt_lock);
	ast_mutex_init(&mon_lock);
	ast_mutex_init(&oh323_tab_lock);
	ast_mutex_init(&oh323_stats_lock);
	ast_mutex_init(&oh323_reload_lock);
	oh323_reloading = 0;
	monitor_running = 0;
	monitor_shutdown = 0;

	oh323_sched = sched_context_create();
	if (!oh323_sched) {
		ast_log(LOG_WARNING, "Unable to create schedule context.\n");
		return(-1);
	}

	/* Register our CLI extensions */
	if (option_debug)
		ast_log(LOG_DEBUG, "Registering CLI extensions.\n");
	ast_cli_register(&cli_show_conf);
	ast_cli_register(&cli_show_stats);
	ast_cli_register(&cli_show_info);
	ast_cli_register(&cli_show_established);
	//ast_cli_register(&cli_show_ep);
	ast_cli_register(&cli_debug_toggle);
	//ast_cli_register(&cli_mode);
	ast_cli_register(&cli_vars);

	/* Init the endpoints list */
	oh323_ep_list.head = NULL;
	ast_mutex_init(&oh323_ep_list.lock);

	/* Load our config file */
	if (option_debug)
		ast_log(LOG_DEBUG, "Loading config file.\n");
	if (reload_config() < 0)
		return(-1);

	if (config_options.totalNum == 0) {
		ast_log(LOG_NOTICE, "Total number of allowed H.323 calls is 0! Disabling H.323 channel driver.\n");
		return(0);
	}
	if (config_options.codecInfo == NULL) {
		ast_log(LOG_NOTICE, "No codecs configured! Disabling H.323 channel driver.\n");
		return(0);
	}

	/* Init table with private structures of H.323 calls. */
	if (option_debug)
		ast_log(LOG_DEBUG, "Allocating H.323 channel space.\n");
	oh323_tab = (struct chan_oh323_pvt **)malloc(config_options.totalNum * 
												sizeof(struct chan_oh323_pvt *));
	if (oh323_tab == NULL) {
		ast_log(LOG_ERROR, "Memory allocation failed.\n");
		return(-1);
	}
	memset(oh323_tab, 0, config_options.totalNum * sizeof(struct chan_oh323_pvt *));

	/* Init the ingress call rate limiter */
	if (in_call_rate_limiter_init(config_options.crlCallNumber,
							config_options.crlCallTime) < 0) {
		return(-1);
	}
	if (option_debug)
		ast_log(LOG_DEBUG, "Ingress call rate limit set at %.2f CPS.\n", 
						in_call_max_rate);

	/* Init statistics */
	if (option_debug)
		ast_log(LOG_DEBUG, "Initializing statistics.\n");
	ast_mutex_lock(&oh323_stats_lock);
	oh323_stats.incall = 0;
	oh323_stats.outcall = 0;
	oh323_stats.setup_sent = 0;
	oh323_stats.setup_recv = 0;
	oh323_stats.block_incall = 0;
	oh323_stats.block_outcall = 0;
	oh323_stats.proto_err = 0;
	oh323_stats.call_err = 0;
	oh323_stats.answer_err = 0;
	if (gettimeofday(&(oh323_stats.boot_time), NULL)) {
		ast_log(LOG_ERROR, "Failed to get current system time!\n");
		ast_mutex_unlock(&oh323_stats_lock);
		return(-1);
	}
	ast_mutex_unlock(&oh323_stats_lock);

	/* Get aliases and prefixes */
	if (option_debug)
		ast_log(LOG_DEBUG, "Initializing aliases/prefixes.\n");
	config_options.gateway_prefix_tab = NULL;
	config_options.alias_tab = NULL;
	regtmp = config_options.regInfo;
	config_options.prefix_num = 0;
	config_options.alias_num = 0;
	while (regtmp) {
		config_options.prefix_num += regtmp->prefix_num;
		config_options.alias_num += regtmp->alias_num;
		regtmp = regtmp->next;
	}
	if (config_options.prefix_num > 0) {
		config_options.gateway_prefix_tab = (char **)malloc(config_options.prefix_num * sizeof(char *));
		if (config_options.gateway_prefix_tab == NULL) {
			ast_log(LOG_ERROR, "Memory allocation failed.\n");
			return(-1);
		}
		memset(config_options.gateway_prefix_tab, 0, 
						config_options.prefix_num * sizeof(char *));
	} else {
		ast_log(LOG_DEBUG, "Zero prefix(es) for ASTERISK.\n");
	}
	if (config_options.alias_num > 0) {
		config_options.alias_tab = (char **)malloc(config_options.alias_num * sizeof(char *));
		if (config_options.alias_tab == NULL) {
			ast_log(LOG_ERROR, "Memory allocation failed.\n");
			return(-1);
		}
		memset(config_options.alias_tab, 0, config_options.alias_num * sizeof(char *));
	} else {
		ast_log(LOG_DEBUG, "Zero alias(es) for ASTERISK.\n");
	}

	regtmp = config_options.regInfo;
	j = 0;
	k = 0;
	while (regtmp) {
		for (i=0; i<regtmp->prefix_num; i++) {
			(config_options.gateway_prefix_tab)[j] = regtmp->prefix[i];
			++j;
		}
		for (i=0; i<regtmp->alias_num; i++) {
			(config_options.alias_tab)[k] = regtmp->alias[i];
			++k;
		}
		regtmp = regtmp->next;
	}

	h323_callback_register(NULL, NULL, NULL, NULL, NULL, NULL); 
	h323_appinfo_set("asterisk-oh323", OH323_VERSION_MAJOR, 
					OH323_VERSION_MINOR, OH323_VERSION_BUILD);

	/* Create our H.323 endpoint */
	if (option_debug)
		ast_log(LOG_DEBUG, "Initializing endpoint.\n");
	h323_end_point_create(config_options.gateway_prefix_tab, 
					config_options.prefix_num, 
					config_options.wrapLibTraceLevel, 
					config_options.libTraceLevel, config_options.libTraceFile);

	/* Configure the endpoint */
	if (option_debug)
		ast_log(LOG_DEBUG, "Configuring endpoint.\n");
	if ((h323_set_options(!config_options.fastStart,
						!config_options.h245Tunnelling,
						!config_options.h245inSetup,
						!config_options.silenceSuppression,
						config_options.bwlimit,
						config_options.jitterMin,
						config_options.jitterMax,
#ifdef HAS_OH323MODS
						config_options.jitterDec,
						config_options.jitterInc,
#endif
						config_options.ipTos) < 0) || 
		(h323_set_ports(config_options.tcpStart, 
						config_options.tcpEnd, 
						config_options.udpStart, 
						config_options.udpEnd,
						config_options.rtpStart, 
						config_options.rtpEnd) < 0)) {
		ast_log(LOG_ERROR, "H.323 endpoint setup failed.\n");
		unload_module();
		return(-1);
	}

	/* Create the H.323 listener */
	if (option_debug)
		ast_log(LOG_DEBUG, "Starting listener.\n");
	if (h323_start_listener(TCP, config_options.listenAddress,
							config_options.listenPort) != LIS_START_OK) {
		ast_log(LOG_ERROR, "H.323 listener creation failed.\n");
		unload_module();
		return(-1);
	}

	/* Add the codecs */
	/* Note: 
	 * 		AddCapability puts the codec into the list of codecs we can send
	 * 		SetCapability puts the codec into the list of codecs we can send and receive
	 */
	if (option_debug)
		ast_log(LOG_DEBUG, "Capability setup.\n");
	h323_removeall_capabilities();
	codectmp = config_options.codecInfo;
	while (codectmp) {
		res = h323_set_capability(codectmp->codec, codectmp->frames);
		if (res != CAP_INSERT_OK) {
			ast_log(LOG_ERROR, "Failed to insert capability %d.\n", codectmp->codec);
			unload_module();
			return(-1);
		}
		codectmp = codectmp->next;
	}

	/* Set the user-input mode */
	if (option_debug)
		ast_log(LOG_DEBUG, "User-input mode setup.\n");
	if (h323_set_senduimode(config_options.userInputMode) != CAP_INSERT_OK) {
		ast_log(LOG_ERROR, "Failed to set user-input mode to %d.\n", config_options.userInputMode);
		unload_module();
		return(-1);
	}

	/* Gatekeeper setup */
	if (option_debug)
		ast_log(LOG_DEBUG, "Gatekeeper setup.\n");
	if (h323_set_gk(config_options.gatekeeperMode, 
					config_options.gatekeeperName,
					config_options.gatekeeperPass, 
					config_options.gatekeeperTTL, 
					config_options.alias_tab, 
					config_options.alias_num)) {
		ast_log(LOG_ERROR, "H.323 gatekeeper setup failed.\n");
		unload_module();
		return(-1);
	}
	if (config_options.gatekeeperMode != GKMODE_DISABLE) {
		oh323_sched_gkid = ast_sched_add(oh323_sched, 30000, 
									oh323_gk_check, NULL);
	}

	/* Register our callback functions */
	if (option_debug)
		ast_log(LOG_DEBUG, "Callback functions setup.\n");
	if (h323_callback_register(setup_h323_connection, cleanup_h323_connection, 
					alerted_h323_connection, exception_h323_connection, 
					init_h323_connection, NULL) < 0) {
/*					init_h323_connection, stats_h323_connection) < 0) {*/
		ast_log(LOG_ERROR, "Failed to register callback function(s).\n");
		unload_module();
		return(-1);
	}

	/* Register this channel type with Asterisk PBX. */
	if (option_debug)
		ast_log(LOG_DEBUG, "Channel registration, with capabilities '%x'.\n", oh323_full_capability);
	if (ast_channel_register(type, tdesc, oh323_full_capability, oh323_request)) {
		ast_log(LOG_ERROR, "Unable to register channel class %s\n", type);
		unload_module();
		return(-1);
	}

	/* Start the monitor for the first time */
	restart_monitor();

	/* Register our "atexit" function */
	ast_register_atexit(oh323_atexit);

	if (option_verbose > 1)
		ast_verbose(VERBOSE_PREFIX_2 "OpenH323 Channel Ready (v%d.%d.%d)\n", 
						OH323_VERSION_MAJOR, OH323_VERSION_MINOR, OH323_VERSION_BUILD);

	return(0);
}

int unload_module()
{
	oh323_atexit();

	/* Unregister our "atexit" function */
	ast_unregister_atexit(oh323_atexit);

	return(0);
}

int usecount()
{
	int res;
	ast_mutex_lock(&usecnt_lock);
	res = usecnt;
	ast_mutex_unlock(&usecnt_lock);
	return res;
}

char *description()
{
	return desc;
}

char *key()
{
	return ASTERISK_GPL_KEY;
}

/* End of file ****************************************************************/
