/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: Tools.pmod,v 1.98.2.5 2004/03/22 13:23:32 vida Exp $
 */

#include <camas/globals.h>
#include <module.h>

inherit "caudiumlib";

// is_7bitsafe() uses it for sscanf template, handle with care
#define IS_7BITSAFE_TEMPLATE "%*[\040-\0177]%*c"

//! file: Tools.pmod
//!  General tools for CAMAS
//! cvs_version: $Id: Tools.pmod,v 1.98.2.5 2004/03/22 13:23:32 vida Exp $

/*
 * Tools and others functions used by CAMAS internaly without the 
 * need to import/export the session id system
 */

//! method: string formdrawbutton(object id, string img, string action, string value, void|mapping args)
//!  Calls appropriate method from the form button module, with the button method the user has in his session.
//!  If the user don't have any form button method in his session, the default from the form button module is used.
//! Returns:
//!  RXML code used in form actions.
//! arg: object id
//!  Caudium ID object
//! arg: string img
//!  The image needed to be loaded. Used also as a key on
//!  CAMAS Basic and extended mode.
//! arg: string action
//!  The form action to use with this button
//! arg: string value
//!  The text to add to the button coming from language file (wide-
//!  string compliant).
//! arg: void|array args
//!  Optional args that should be appended in the buttons.
string formdrawbutton(object id, string img, string action, string value, void|mapping args)
{
  string _buttonmode = stringp(CSESSION->buttonmode) ? CSESSION->buttonmode : "";

  object cfbuttn = id->conf->get_provider("camas_formbuttons");

  if (objectp(cfbuttn))
    if (sizeof((string)_buttonmode))
      return cfbuttn->button(id, img, action, value, args, _buttonmode);
		// if no _buttonmode, use the default from the formbutton module
    else
      return cfbuttn->button(id, img, action, value, args);

  else
    return "<!-- Camas formbutton not installed ??? -->";
}

//! method: void setup_state(object id, mapping arg)
//!  Create CSESSION-&gt;nextpage, CSESSION-&gt;target and parse arguments
void setup_state (object id, mapping arg)
{

  if(CSESSION) {
    string target = "",sidetarget="",uptarget="",login="", passwd="",
      frame = id->misc->camas->frame;
    if (arg["target"])
      target = sidetarget = uptarget = arg["target"];
    if (arg["sidetarget"]) // Target to a frame in the same level
      sidetarget = arg["sidetarget"];
    if (arg["uptarget"]) // Target in a frame level up
      uptarget = arg["uptarget"];
    if (sizeof(target) == 0)
      target = uptarget;
    if (sizeof(target) == 0)
      target = sidetarget;
    if (sizeof(target) > 0)
      id->misc->camas->nexttarget = CSESSION->nexturl + target;
    else
      id->misc->camas->nexttarget = CSESSION->nexturl + frame;
    if (sizeof(sidetarget) > 0)
      id->misc->camas->nextsidetarget = CSESSION->nexturl + sidetarget;
    else
      id->misc->camas->nextsidetarget = CSESSION->nexturl + frame;
    if (sizeof(uptarget) > 0)
      id->misc->camas->nextuptarget = CSESSION->nexturl + uptarget;
    else
      id->misc->camas->nextuptarget = CSESSION->nexturl + frame;
    id->misc->camas->nextpage = CSESSION->nexturl + frame;
    id->misc->camas->target = target;
    id->misc->camas->sidetarget = sidetarget;
    id->misc->camas->uptarget = uptarget;
    if (arg->face)
    {
      CSESSION->face = arg->face;

      if (arg->size)
        CSESSION->size= arg->size;
      else
        CSESSION->size=0;
    }
    else
      CSESSION->face = 0;
    CSESSION->notopbuttons = 0;
    if (arg->notopbuttons)
      CSESSION->notopbuttons = 1;
    CSESSION->nobottombuttons = 0;
    if (arg->nobottombuttons)
      CSESSION->nobottombuttons = 1;
    if(arg->mixedtopbottombuttons)
      CSESSION->mixedtopbottombuttons = 1;
    CSESSION->mixedtopbuttombuttons = 0;
    if(arg->nobackbutton)
      CSESSION->nobackbutton=1;
    if(arg->columns)
      CSESSION->columns=arg->columns;
  }
}

//! method: string make_get_url(object id, void|mapping args, void|mapping vars, void|string filename)
//!  Builds a full GET URL for passing arguments to CAMAS, and sets up the target.
//!    - server url
//!    - CAMAS mountpoint
//!    - frame target
//!    - fake screen to use
//!    - random string for buggy browsers with SSL (if needed)
//!    - fake file name (if needed)
//!    - anchor to go (if needed)
//!    - http encoded arguments
//! arg: object id
//!  The Caudium request id object
//! arg: void|mapping args
//!  The args to build the url:
//!    - screen
//!    - anchor
//! arg: void|mapping vars
//!  The args to pass with the GET method
//! arg: void|string filename
//!  The fake filename to append to the URL
//! returns:
//!  The full GET request as a string eg: http://your.server.tld/mail/frame-screen-123456/filename#anchor?foo=bar&gazonk=baz
string make_get_url(object id, void|mapping args, void|mapping vars, void|string filename)
{
  // set arg->target and arg->action to correct values
  args = set_target(id, args, filename);

  // build the string from args->action
  string out = args->action;

  // args->action is not usefull for <a></>
  m_delete(args, "action");
 
	// The "&" character have to be encoded with "&amp;" in a web page.
	// If http_encode_query handled that himself, it would be nicer and faster
  if(mappingp(vars) && sizeof(vars))
    out += "?"+replace(Protocols.HTTP.http_encode_query(vars), "&", "&amp;");
 
 return out;
}

//! method: mapping set_target(object id, mapping args, void|string filename)
//!  Builds a full POST URL and sets up the target.
//!  The target the web browser wants to refresh is not passed to the web server during the HTTP
//!  request. That's why the url have to be changed to be sync with the form's "target" argument
//!  More, if no target is given, it should be set to the current target, because the browser will
//!  refresh the data in the current target if none is provided.
//!  This method is used by make_get_url that append GET variables behind it
//! arg: object id
//!  The Caudium request id object
//! arg: mapping args
//!  The args passed to the <camas_form></> container
//! arg: void|string filename
//!  The fake filename to append to the URL
//! returns:
//!  The full POST request as a mapping with:
//!   - The full POST request as a string in args->action eg:  http://your.server.tld/mounpointframe-screen-random123456/filename_for_download#anchor
//!   - The target set into args->target
mapping set_target(object id, mapping args, void|string filename)
{
  // If target not specified in the args, use the current frame
  // This avoids the recursive frames phenomenon
  args->target = HTTP_ENCODE_URL(stringp(args->target) ? args->target : id->misc->camas->frame || "");

  // put the target in the URL
  args->action = CSESSION->nexturl+args->target+"-";
  
  // put the overriding screen if provided
	// else put the current screen as a target screen
  if(args->screen)
    args->action += HTTP_ENCODE_URL(args->screen);
  
  // put some random string for browsers that cache too much (read: damned MSIE)
  args->action += "-"+rand_string(20, id->supports->ssl_faulty);
  
  // put a filename if needed (eg giving the good filename when downloading a mailpart)
  if(stringp(filename) && sizeof(filename))
    args->action += "/"+HTTP_ENCODE_URL(filename);

  // add an anchor if needed
  if(args->anchor)
    args->action += "#"+HTTP_ENCODE_URL(args->anchor);

	// removing non-html args from the RXML tags for not leaking them in the generated HTML source
	m_delete(args, "screen");
	m_delete(args, "anchor");
 
  return args;
}

//! method: mapping extract_html_attributes(mapping args, string begin)
//!  Fake namespace implementation of in HTML attributes: namespace are separated with "_"
//!  Extract the keys from args which begins with the arg begin.
//!  These keys are put into a new mapping with begin value removed from their indices
//!  The original entry in mapping args is removed destructively
//! example :
//!  mapping args = ([ "bar_baz":"foo", "bar_foo":"gazonk", "stuff":"otherstuff" ]);
//!  mapping newargs = extract_html_attributes(args, "bar");
//!       args : ([ "stuff":"otherstuff" ]);
//!    newargs : ([ "baz":"foo", "foo":"gazonk" ]);
//! returns:
//!  A mapping with begin-values inside
mapping extract_html_attributes(mapping args, string begin)
{ 
 	mapping out = ([ ]);

	begin += "_";

  foreach(indices(args), string arg)
  {
    if(search(arg, begin)!=-1 && arg!=begin)
    {
      out += ([ arg[sizeof(begin)..] : (string)args[arg] ]);
      m_delete(args, arg);
    }
  }

  return out;
} 


//! method: void low_destroy_session(object sess)
//!  Destroy the Camas session object
//! arg: mapping|array sess
//!  The Camas session object
void low_destroy_session(object sess)
{ 
  if(sess->imapcachefeeder && objectp(sess->imapcachefeeder))
  {
    if(sess->imapcachefeeder->cache && objectp(sess->imapcachefeeder->cache))
      destruct(sess->imapcachefeeder->cache);
    destruct(sess->imapcachefeeder);
  }
  if(sess->imapclient && objectp(sess->imapclient))
  {
    if(sess->imapclient->cache && objectp(sess->imapclient->cache))
      destruct(sess->imapclient->cache);
    destruct(sess->imapclient);
  }
  if(sess)
    destruct(sess);
  else
    report_warning("Camas Tools.pmod: Try to destroy a non existing session!");
}

//! method: void destroy_csession (object id)
//!  Destroy the current session
//! arg: object id
//!  The caudium id object
void destroy_csession(object id)
{
  object temp_session = CSESSION;
  // try to remove the reference from the id object
  m_delete(id->misc->session_variables, "CAMAS");
  low_destroy_session(temp_session);
#ifdef CAMAS_DEBUG
#if constant(Debug.pp_memory_usage)
  write("mem usage=%s\n", Debug.pp_memory_usage());
#endif
#if constant(Debug.gc_status)
  write("gc status=%O\n", Debug.gc_status());
#endif
#endif
}

//! method: string current_layout (object id)
//!  Returns the name of the layout being currently used
//! arg: object id
//!  The caudium id object
//! note:
//!  This function is stupid. Maybe we could create a CSESSION->currentlayout
//!  or set CSESSION->layout with the correct value.
string current_layout (object id)
{
  string _layout = "";

  if(CSESSION->overridelayout)
    _layout = CSESSION->overridelayout;
  else
    _layout = CSESSION->layout;

  return _layout;
}

//! method: string encode_pref (string s)
//!  Encode the user preferences
//! arg: string s
//!  String used to encode preferences
string encode_pref (string s) {
  if (!s)
    return "";
  return replace(s, ({ "<br", "\n", "\r" }), ({ "<br0", "<br>", "" }));
}

//! method: string decode_pref (string s)
//!  Decode the user preferences
//! arg: string s
//!  String used to decode preferences
string decode_pref (string s) {
  if (!s)
    return "";
  return replace(s, ({ "<br0", "<br>", "\r" }), ({ "<br", "\n", "" }));
}

//! method: string rand_string(int len, int faulty)
//!  Random a string (used to add it in url to avoid caching problems with
//!  some browser like MSIE)
//! returns:
//!  A random string
//! arg: int len
//!  Size of the string. Note that the size is depending of hex encoding
//!  of the internal Crypto thing.
//! arg: int faulty
//!  Wherether the client browser is ssl faulty. SSL faulty clients
//!  are the MSIE that cache too much and thus need this.
string rand_string(int len, int faulty) {
  if(faulty)
#if constant(Caudium.Crypto.rand_string)
    return Caudium.Crypto.rand_string(len);
#else
    return Crypto.string_to_hex(Crypto.randomness.reasonably_random()->read(5+len));
#endif
  return "";
}

string get_hash( string data ) {
#if constant(Caudium.Crypto)
  return sprintf("%@02x", (array(int))Caudium.Crypto.hash_md5(data));
#else
  if(String.width(data) > 8)
    data = string_to_utf8(data);
  return sprintf("%@02x", (array(int))Crypto.md5()->update(data)->digest());
#endif
}

//! method: string strip(string s)
//!  Strip '\r' in a string
//! returns:
//!  The stripped string
//! arg: string s
//!  String to strip
string strip (string s) {
  if (!s)
    return "";
  return (s - "\r");
}

//! method: string strip_path(string s)
//!  Remove '..' in a string
//! returns:
//!  A string with all '..' removed in
//! arg: string s
//!  String to strip
string strip_path (string s) {
  if (!s)
    return "";
  return (s - "..");
}

//! method: string scut(string s, int len)
//!  Cut the string s into 'len' size max and add '...'
//! returns:
//!  A string cuted to the right size
//! arg: string s
//!  String to cut
string scut(string s, int len)
{
  if (len > 1) len = len -1;
  if (sizeof(s) > len) {
    if (len > 3) len = len - 3;
    return s[..len] + "...";
  }
  else
    return s;		// just in case
}

//! method: int is_7bitsafe(string s)
//!  determinate either a string is 7bit safe or not.
//! returns:
//!  1 if it is safe, 0 otherwise
//! arg: string s
//!  String to check either 7bit safe
int is_7bitsafe( string s ) {
  return ( sscanf( s, IS_7BITSAFE_TEMPLATE ) == 1 );
}

//! method: string encode_header (string s,string cs)
//!  encode a header for the mail example:
//!  encode_header("", "iso-8859-1");
//!  result = "=?iso-8859-1?q?=E9=E0?="
//! returns:
//!  the encoded string
//! arg: string s
//!  the string to be encoded
//! arg: string cs
//!  the charset
string encode_header(string s,string|void cs) {
  //FIXME: doesn't honour the length limits, 75,76 chars.
  //       it would be very difficult due to the integral constraints
  //       for multibyte character sets
  if(!cs)
    cs = "UTF-8";
  array a1=(Locale.Charset.encoder(cs,"")->feed(s)->drain())/" ";
  array a2=Array.map(a1, lambda(string s,string cs)
			 { return MIME.encode_word(({s,cs}),"quoted-printable"); }, cs);
  array a3=Array.map(a2, lambda(string s) { return (s / "?"); } );

  for(int i=0;i<sizeof(a1);i++){
    if(a1[i]!=a3[i][3] && replace(a1[i],"_","=5F")!=a3[i][3]) { 
    //MIME word encode has changed this word
      if(i && a1[i-1][0..1]=="=?") {
	// if previous word was encoded, we encode those words as one
	a3[i][3] = a3[i-1][3]+"_"+a3[i][3];
	a1[i] = a3[i]*"?";
	a1[i-1] = "";
      }
      else
	a1[i]=a2[i];
    }
  }
  return a1*" ";
}

//! method: string encode_addresses(array a,string cs)
//!  encode a set of given addresses.
//! returns:
//!  a string containing the addresses separated by ,
//!  eg, "Foo bar  <foobar@ece.fr>,Foo\351 Bar foobar2@ece.fr>"
//! arg:
//!  array a
//!  the array containing the addresses
//!  eg, ({ "Foo bar <foobar@ece.fr>", "Foo Bar
//!           foobar2@ece.fr>" })
//!      string cs
//!  the encoding ex: iso-8859-1
string encode_addresses(array a,string cs) {
  array a2=({ });
  string s1,s2;
  foreach(a,string s) {
    if(sscanf(s, "%s<%s>",s1,s2)==2)
      a2 += ({ encode_header(s1,cs)+" <"+s2+">"  });
    else
      a2 += ({ s });
  }

  return a2*",";
}

//! method: string wrap_lines(string s, int len)
//! returns:
//!   Wrap lines in a text to max len chars (i.e insert CRLFs at
//!   appropriate positions) if the line contains spaces (actually it tries 
//!   to split the best it can: splitting before len if there are spaces before
//!   and splitting just after if there are space just after). 
//!   Also make sure LF is really CRLF.
//!   Finally remove spaces in the end.
//!   
//! arg:
//!   string s: the string to wrap
//!   the len at which the string is wrapped
string wrap_lines(string s, int len)
{
  string out = "";
  foreach(s / "\n", string line)
  {
    line = replace(line, "\r", "");
    //write("line=%O\n", line);
    if(len > 0 && sizeof(line) > len)
    {
      //write("is %d long => too much\n", sizeof(line));
      // if there is a white space before max len
      // wrap at this position
      // Also make sure to wrap the resulting line if it is too long
      int wrap_pos = len - search(reverse(line[..len]), " ");
      //write("wrap_pos=%d\n", wrap_pos);
      if(wrap_pos > 0 && wrap_pos <= len)
      {
        //write("wrapping before at pos %d\n", wrap_pos);
        line = line[..wrap_pos-1] + 
           "\r\n" + wrap_lines(line[wrap_pos..], len);
      }
      else
      {
        // if it is not possible try to break after
        wrap_pos = search(line, " ");
        if(wrap_pos > 0)
        {
          //write("wrapping at pos %d\n", wrap_pos); 
          line = line[..wrap_pos-1] + 
            "\r\n" + String.trim_whites(wrap_lines(line[wrap_pos..], len));
        }
        else if(sizeof(line) > 0)
        {
           line = line + "\r\n";
           //write("no need to beak\n");
        }
      }
    }
    // add the \n that we removed in the foreach loop
    else
    {
      line = line + "\r\n";
      //write("no need to beak\n");
    }
    out += line;
  }
  return out;
}

//! method: string decode_fname(string fname)
//! To be documented
string decode_fname(string fname) {
  return replace(fname,({ "@0","@1" }),({ "@","_" }) );
}

//! method: string encode_fname(string fname)
//! To be documented
string encode_fname(string fname) {
  return replace(fname,({ "@","_" }),({ "@0","@1" }) );
}

//! method: string smtp_format(string s, int keepbcc)
//!   Make sure each line is max 1000 bytes and that no lines begins with "."
//!   Also remove BCC address
//! returns:
//!   string containing the formated mail
//! arg:
//!   string s: the string to format
//!   int keepbcc: if 1, keep bcc in the headers
string smtp_format(string s, int keepbcc)
{ // Make sure each line is max 1000 bytes and that no lines begins with "."
  // Also remove BCC address
  string ret = "";
  int headers = 1;
  array lines = s/"\r\n";
  foreach(lines, string l) {
    string bcc = "";
    if (!(headers && !keepbcc &&(sscanf(lower_case(l), "bcc%*[ ]:%s",bcc) > 0) && sizeof(bcc) > 0)) {
      if (sizeof(l) == 0)
        headers = 0;
      while (sizeof(l) > 1000) {
        string first = l[0..999];
        if (headers)
          l = " "+l[1000..];
        else
          l = l[1000..];
        if (first[0] == '.')
          ret+= ".";
        ret+=first+"\r\n";
      }
      if (sizeof(l) > 0 && l[0] == '.')
        ret+= ".";
      ret+=l+"\r\n";
    }
  }
  return(ret);
}

//! method: string decode_mboxname (string s)
//! To be documented
//
// Modified UTF-7 encoding/decoding
//
// Note: this function is not used
string decode_mboxname (string s) {
  string out="";
  int coded=0;
  string codeds="";
  string decodeds="";
  array ss = s/"";
  foreach(ss,string s1) {

    switch(s1) {
    case "&":
      coded=1;
      break;

    case "-":
      if(coded) {
        if(sizeof(codeds)) {
          codeds+=({ "","===","==","=" })[sizeof(codeds)%4];
          decodeds=MIME.decode_base64(replace(codeds,",","/"));
          if(sizeof(decodeds)&1)
            decodeds=decodeds[..sizeof(decodeds)-2];
          out+=unicode_to_string(decodeds);
        }
        else
          out+="&";
        coded=0;
        codeds="";
      }
      else
        out+="-";
      break;

    default:
      if(coded)
        codeds+=s1;
      else
        out+=s1;
      break;

    }
  }

  if(coded) {
    if(sizeof(codeds)) {
      codeds+({ "","===","==","=" })[sizeof(codeds)%4];
      decodeds=MIME.decode_base64(replace(codeds,",","/"));
      if(sizeof(decodeds)&1)
        decodeds=decodeds[..sizeof(decodeds)-2];
      out+=unicode_to_string(decodeds);
    }
    else
      out+="&";
  }


  return out;
}

//! method: int find_mail (array mails, int value)
//!   Find a mail with a certain key and return the array index.
//! returns:
//!   an int containing the array index
int find_mail (array mails, int value)
{
  int nr = Array.search_array (mails,
                               lambda (mapping mail, int value)
                               {
                                 if (mail->imap->UID == value)
                                   return 1;
                                 return 0;
                               }, value);
  return (nr);
}

//! method: int allowed_mbox_char (string s)
//!   Can be used to know if the name of the mbox is ok
//! returns:
//!   1 if the first char of a mbox is allowed and 0 otherwise
//! arg:
//!   string s : the name of the mbox
int allowed_mbox_char (string s) {
  int i=s[0];
  return (i!='&' && i>=0x20 && i<=0x7e);
}

//! method: string encode_mboxname (string s)
//! To be documented
string encode_mboxname (string s) {
  string out="";
  int coded=0;
  string codeds="";
  array ss = s/"";
  foreach(ss,string s1) {

    if(s1=="&") {
      if(coded) {
        out+="&"+replace(MIME.encode_base64(string_to_unicode(codeds)),({"/","="}),({",",""}) )+"-";
        coded=0;
        codeds="";
      }
      out+="&-";
    }
    else {
      if(allowed_mbox_char(s1)) {
        if(coded) {
          out+="&"+replace(MIME.encode_base64(string_to_unicode(codeds)),({"/","="}),({",",""}) )+"-";
          coded=0;
          codeds="";
        }
        out+=s1;
      }
      else {
        coded=1;
        codeds+=s1;
      }
    }
  }

  if(coded)
    out+="&"+replace(MIME.encode_base64(string_to_unicode(codeds)),({"/","="}),({",",""}) )+"-";

  return out;
}

//! method: string filetype (string fname, object conf)
//!  Give the type of the file named fname
//! returns:
//!  The type of the file
//! arg:
//!  string fname: the name of the file
//!  object conf: the Caudium my_configuration object
string filetype (string fname, object conf) {
  return conf->type_from_filename (fname, 0);
}

private constant safe_characters = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"/"";
private constant empty_strings = ({
  "","","","","","","","","","","","","","","","","","","","","","","","","",
  "","","","","","","","","","","","","","","","","","","","","","","","","",
  "","","","","","","","","","","","",
});

//! method: int is_safe_string(string in)
//!  Check if a string contains only safe characters, which are defined as
//!  a-z, A-Z and 0-9. Mainly used internally by make_tag_attributes.
//!  Taken from caudiumlib for make_tag_attributes
//! arg: string in
//!  The string to check.
//! returns: 1 if the test contains only the safe characters, 0 otherwise.
//! name: IS_SAFE_STRING - check if a string contains unsafe characters
int is_safe_string(string in)
{
  return in && strlen(in) && !strlen(replace(in, safe_characters, empty_strings));
}

//! method: string make_tag_attributes(mapping in)
//!  Convert a mapping with key-value pairs to tag attribute format.
//!  Taken from caudiumlib because the one from Caudium doesn't work
//!  good and it will take years to have a fix.
//! arg: mapping in
//!  The mapping with the attributes
//! returns:
//!  The string of attributes.
//! name: make_tag_attributes - convert a mapping to tag attributes
string make_tag_attributes(mapping in)
{
  array a=indices(in), b=values(in);
  for(int i=0; i<sizeof(a); i++)
  {
    b[i] = (string) b[i];
    if(IS_SAFE_STRING(b[i]))
      a[i]+="=\""+b[i]+"\"";
    else
      // Bug inserted again. Grmbl.
      a[i]+="=\""+replace(b[i], ({ "\"", "<", ">" //, "&"
      }) ,
                       ({ "&quot;", "&lt;", "&gt;" //, "&amp;"
                       }))+"\"";
  }
  return a*" ";
} 

//! method: string make_tag(string tag, mapping in)
//!  Build a tag with the specified name and attributes.
//!  Taken from caudiumlib because the one from Caudium doesn't work
//!  good and it will take years to have a fix.
//! arg: string tag
//!  The name of the tag.
//! arg: mapping in
//!  The mapping with the attributes
//! returns:
//!  A string containing the tag with attributes.
//! name: make_tag - build a tag 
string make_tag(string tag, mapping in)
{
  string q;
#if constant(Caudium._make_tag_attributes)
  q = Caudium._make_tag_attributes(in);
#else
  q = make_tag_attributes(in);
#endif
  return "<" + tag + (strlen(q)?" " + q:"") + "/>";
}

//! method: string make_container(string tag, mapping in, string contents)
//!  Build a container with the specified name, attributes and content.
//!  Taken from caudiumlib because the one from Caudium doesn't work
//!  good and it will take years to have a fix.
//! arg: string tag
//!  The name of the container.
//! arg: mapping in
//!  The mapping with the attributes
//! arg: string contents
//!  The contents of the container.
//! returns:
//!  A string containing the finished container
//! name: make_container - build a container
string make_container(string tag,mapping in, string contents)
{
  string q;
#if constant(Caudium._make_tag_attributes)
  q = Caudium._make_tag_attributes(in);
#else
  q = make_tag_attributes(in);
#endif
  return "<"+ tag + (strlen(q)?" "+q:"") + ">" + contents + "</"+tag+">";
}

//! method: string fix_coding(string str)
//!  Invoques _fix_coding(string str) if no lookup in the cache
//! arg: string str
//!  The string to safe and to check the safed result in the cache
//! returns:
//!  The safed result, from the cache, or from _fix_coding()
string fix_coding(string str)
{
  string s = cache_lookup("camas_fix_coding", str);

  if(!s)
  {
    s = _fix_coding(str);
    cache_set("camas_fix_coding", str, s, CACHE_TIMEOUT);
  }

  return s;
}

//! method: private string _fix_coding(string str)
//!  Fix the coding of a string (general string from a mail)
//! arg: string arg
//!  String to fix
//! returns:
//!  String fixed.
private string _fix_coding (string str) {
  if (!str)
    return "?";

  str = replace(str, ({ "(", ")" }), ({ " ( ", " ) " }) );

  array a = Array.map ((str / " "), MIME.decode_word);
  string spaces = " ";
  string ret = "";
  if (sizeof (a)) {
    mixed err = catch { ret += Locale.Charset.decoder (a[0][1])->feed (a[0][0])->drain (); };
  if (err) {
      //DW ("fix_coding: " + err[0] );
      ret += a[0][0];
    }
  }

  int last_coded = sizeof (a) && a[0][1];
  for (int i = 1; i < sizeof (a); i++) {
    if (a[i][0] != "") {
      if (!(last_coded && a[i][1]))
        ret += spaces;
      mixed err = catch { ret += Locale.Charset.decoder (a[i][1])->feed (a[i][0])->drain (); };
    if (err) {
        //DW ("fix_coding: " + err[0] );
        ret += a[i][0];
      }
      spaces = " ";
      last_coded = !!a[i][1];
    }
    else {
      if (last_coded)
        spaces += " ";
      else
        ret += " ";
    }
  }
  //DW ("fix_coding (" + str + ") = [" + ret + "]");

  return replace(ret, ({ " ( ", " ) " }), ({ "(", ")" }) );
}

//! method: array address_split(string s)
//!  Split addresses from a string to an array (for address book and maybe other things)
array address_split(string s) {
  array i=({ 0 });
  return (Array.map(s/"\"",
                    lambda(string s,array i){return (i[0]=i[0]?0:1)?replace(s,({",",";"}),({"\0","\0"})):s;},
                    i)*"\"")/"\0";
}

#if 0
//! method: array fixaddresses(string addr)
//!  Make sure the all email addresses are on correct form
//!  See RFC 1036 section 2.1.1
array fixaddresses(string addr) {
  array ret = ({ "", "" });
  array address_split_addr = address_split(addr);
  foreach(address_split_addr, string s) {
    string fixedaddr = "";
    string name = "", address = "";
    int endlimit = ')';
    int pos = 0,pos2=0, quote = 0;
    int broken = 0;
    while (pos < sizeof(s) && ( quote ||
                                (s[pos] != '(' && s[pos] != '<'))) {
      if (s[pos] == '"')
        quote = quote?0:1;
      pos++;
    }
    if (pos < sizeof(s)) {
      if (s[pos] == '<') {
        endlimit = '>';
        if (pos > 1)
          name = s[0..pos-2];
      } else {
        if (pos > 1)
          address = s[0..pos-2];
      }
      pos++;
      pos2 = pos;
      while (pos < sizeof(s) && ( quote || s[pos] != endlimit)) {
        if (s[pos] == '"')
          quote = quote?0:1;
        pos++;
      }
      if (pos == sizeof(s))
        broken = 1;
      if (endlimit == '>')
        address = s[pos2..pos-1];
      else
        name = s[pos2..pos-1];
    } else {
      address = s;
    }
    while (sizeof(name) > 0 && name[0] == ' ')
      name = name[1..];
    while (sizeof(name) > 0 && name[-1] == ' ')
      name = name[0..sizeof(name)-2];
    while (sizeof(address) > 0 && address[0] == ' ')
      address = address[1..];
    while (sizeof(address) > 0 && address[-1] == ' ')
      address = address[0..sizeof(address)-2];
    if (search(address, " ") >= 0)
      broken = 1;
    if (name == "")
      fixedaddr = address;
    else {
      name = replace(name, "\"", "");
      fixedaddr = "\""+name+"\" <"+address+">";
    }
    if (sizeof(ret[0]) == 0)
      ret[0] = fixedaddr;
    else
      ret[0] += ", "+fixedaddr;
    if (broken)
      ret[1] = s;
  }
  return(ret);
}

#endif
//! method: array fixaddresses(string addr)
//!  Make sure the all email addresses are on correct form
//!  See RFC 1036 section 2.1.1
array fixaddresses(string addr) {
  array(string) ret = ({ "", "" });
  array(string) address_split_addr = address_split(addr);
  foreach(address_split_addr, string s) 
  {
    int broken;
    string name = "", address = "", fixedaddr = "";
    string comment;
    s = String.trim_all_whites(s);
    sscanf(s, "%*s(%s)%*s", comment);
    // remove comment, they only had problems
    // some poeple use them in their name but we 
    // remove then anyway
    if(comment)
      s -= "(" + comment + ")";
    if(sscanf(s, "%s<%s>", name, address) != 2)
    {
      if(sscanf(s, "%*s\"%s\" %s", name, address) != 3)
        address = s;
    }
    name = String.trim_all_whites(name);
    address = String.trim_all_whites(address);
    // some ppl like to put "" around their email address
    sscanf(address, "\"%s\"", address);
    if(search(address, "@") == -1)
      broken = 1;
    if(search(address, " ") > 0)
    {
      // then we propably don't have a valid email address
      int pos = search(address, "@");
      if(pos == -1)
        broken = 1;
      else
      {
        array(string) possible_addrs = address / " ";
        address = "";
        foreach(possible_addrs, string possible_addr)
          if(search(possible_addr, "@") != -1)
            address = possible_addr;
      }
    }
    if(!sizeof(address))
      broken = 1;
    else
    {
      if(sizeof(name))
        fixedaddr = name + " <" + address + ">";
      else
        fixedaddr = "<" + address + ">";
    }
    if (sizeof(ret[0]) == 0)
      ret[0] = fixedaddr;
    else
      ret[0] += ", "+fixedaddr;
    if (broken)
      ret[1] = s;
  }
  return ret;
}

//! method: string fixstring(string s)
//!  Fix a string to be HTML compliant.
string fixstring (string s) {
  //if(!s) s="";
  //return replace(s,({"<",">","&","\n", "\r"}),({"&lt;","&gt;","&amp;","<br />", "" }));
  return (s ?
          replace (s, ({"<",">","&","\n", "\r", "\""}), ({"&lt;","&gt;","&amp;","<br />", "", "&quot;" }))
          : "");
}

//! method: string fix_header(string str)
//!  Invoques _fix_header(string str) if no lookup in the cache
//! arg: string str
//!  The string to safe and to check the safed result in the cache
//! returns:
//!  The safed result, from the cache, or from _fix_header()
string fix_header(string str)
{
  string s = cache_lookup("camas_fix_header", str);

  if(!s)
  {
    s = _fix_header(str);
    cache_set("camas_fix_header", str, s, CACHE_TIMEOUT);
  }

  return s;
}

//! method: private string _fix_header(string str)
//!  Replace "dangerous strings" into more HTML safe ones
//! returns:
//!  String with dangerous strings replaced by html ones
//! arg: string str
//!  The string to safe
private string _fix_header (string str)
{
  //DW("fix_header("+str+")");
  if (!str || !sizeof (str))
    return "?";
  else {
    string tmp;
    mixed err = catch {
                  tmp = fix_coding (str);
                };
    if (err) {
      tmp = str;
    }
    // Experimental RFC1522 decoding of headers
    array decode;
    err = catch {
      decode = MIME.decode_word(tmp);
    };
    if(err)
    {
      // Outlook Express might put a " before the MIME encoded string and that fools pike
      int headquote = 0;
      if(tmp[0..0]=="\"")
      { 
        tmp = tmp[1..];
        headquote = 1;
      }
      decode = MIME.decode_word(tmp);
      if(headquote)
        decode[0] = "\""+decode[0];
    }
    string tmp2 = tmp;
    err = catch {
      tmp = Locale.Charset->decoder(decode[1])->feed(decode[0])->drain();
    };
    if(!err)
      tmp2 = tmp;
    return fixstring(tmp2);
  }
}

//! method: mapping(int:array(object)) get_providers_bypriority(string provider, object id)
//!  Like standard id->conf->get_providers but returns the object in the order
//!  of their internal priority.
//! returns:
//!  mapping of array of modules indexed by priority
//! arg: string provider
//!  The provider string like "camas_auth"
//! arg: object id
//!  Caudium object id
mapping(int:array(object)) get_providers_bypriority(string provider, object id)
{
  array(object) camas_mods = id->conf->get_providers(provider);
  mapping(int:array(object)) modules = ([ ]);
  foreach(camas_mods, object module)
    if(!modules[module->QUERY(_priority)])
   	  modules += ([ module->QUERY(_priority): ({ module }) ]);
    else
  	  modules[module->QUERY(_priority)] += ({ module });
  return modules;
}

//! method: void save_preferences(object id)
//!  Save current state of preferences to every preferences module
//! returns:
//!  An array containing the imap command if imap commands have to be executed
//!  or void if no imap commands have to be executed
array|void save_preferences(object id)
{
  array imap_commands = ({ });
  foreach(id->conf->get_providers("camas_preferences"), object pref_module)
  {
    mixed res = pref_module->set_preferences(id);
    if(arrayp(res))
      return res;
  }
}
 
//! method: void init_temp_vars(object id)
//!  Initialize some temporary variables used in Camas actions modules
//!  before going or leaving the compose screen
//! arg: object id
//!  The Caudium object id
void init_temp_vars(object id)
{
  CSESSION->replytouid = -1;
  CSESSION->replytoidx = -1;
  CSESSION->forwarduid = -1;
  CSESSION->forwardidx = -1;
  CSESSION->draftidx   = -1;
  CSESSION->draftuid   = -1;
  CSESSION->usermove   =  0;
}

//! method: string html2text(string html)
//!  Convert the HTML formatted string html into plain text
string html2text(string html)
{
  int add = 1;
  int added = 0;
  string message = "";
  for(int i=0; i<sizeof(html); i++)
  {
    switch(add)
    {
      case 1:
      if(html[i]!=60)
      {
        message += sprintf("%c",html[i]);
        added ++;
      }
      else
        add = 0;
      case 0:
      if(html[i]==62)
        add = 1;
    }
  }
  return HTML_DECODE_STRING(message);
}

//! method: string display_size(int bytes)
//!  Format the given bytes in human readable format
//!  For example 390483204 becomes 382MB
//!  and 39048 becomes 38KB
string display_size(int bytes)
{
  int mbytes = bytes >> 20;
  if(mbytes)
    return (string) mbytes + "MB";
  int kbytes = bytes >> 10;
  if(kbytes)
    return (string) kbytes + "KB";
  if(bytes)
    return (string) bytes + "B";
  return "0";
}

//! method: string serialize_prefs (mapping prefs)
//!  Serialize Camas preferences into a string for writing to an IMAP 
//!  mail or to a file
//! arg: mapping prefs
//!  The preferences to serialize. This is a mapping that maps preferences
//!  names to their values
string serialize_prefs(mapping(string:string) prefs)
{
  string foo = "2\r\n"; // Version 2
  foreach (indices(prefs), string prop) 
  {
    foo += "#"+ replace(prop +" = "+ encode_pref(prefs[prop]),
              ({ "\\", "#"}), ({ "\\1", "\\2" }))+"\r\n";
  }
  return string_to_utf8(foo);
}

//! method: mapping(string:string) deserialize_prefs(string prefs) 
//!  De-serialize Camas preferences into a mapping that maps preferences
//!  names to their values
mapping(string:string) deserialize_prefs(string prefs)
{ 
  mapping result = ([ ]);
  string data=replace(prefs ||"","\r","");
  // UTF-8 encoded prefs starts with a '@' == prefs version 1
  // Prefs version 2 and up start with the version number
  // Prefs 2 lines start with #, and # is encoded \2 ,  \ is encoded \1
  ///array (string) lines = (data[0..0]=="@"?FROM_UTF8(data):data) / "\n";
  int utf8=data[0..0]=="@";
  int version = 0;
  if (utf8)
    version = 1;
  else
    version = (int) data[0..0];
  array (string) lines;
  if (version >= 2)
    lines= data / "\n#";
  else
    lines= data / "\n";
  foreach(indices(lines), int i) {
    string prop, value;
    string line = ((version >= 2)?
                   replace(lines[i],({ "\n","\\2", "\\1" }),
                           ({ "","#", "\\" })):lines[i]);
    if (sscanf(line, "%s = %s", prop, value ) > 0) {
      result += ([ prop: decode_pref((version>=1)?utf8_to_string(value):value) ]);
    }
  }
  return result;
}

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */
