/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_sql_logger.pike,v 1.18.2.1 2004/02/23 11:26:00 vida Exp $
 */

#include <module.h>
#include <camas/globals.h>
inherit "module";

inherit "caudiumlib";

//
//! module: CAMAS: SQL Logger Module
//!  SQL Logger Module for CAMAS.<br />
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_sql_logger.pike,v 1.18.2.1 2004/02/23 11:26:00 vida Exp $
//

constant cvs_version = "$Id: camas_sql_logger.pike,v 1.18.2.1 2004/02/23 11:26:00 vida Exp $";
constant module_type = MODULE_PROVIDER|MODULE_EXPERIMENTAL;
constant module_name = "CAMAS: SQL Logger Module";
constant module_doc  = "SQL Logger Module for CAMAS.<br />";

constant module_unique = 1;
// vida: is it always right after the SQL stuff ?
constant thread_safe = 1;

#if constant(thread_create)
object log_lock = Thread.Mutex();
#endif


// SQL class db_handler take from sqllogger
object db;		// The db stack
int num;
mapping stats= ([ ]);

#if constant(thread_create)
static inherit Thread.Mutex;
#define THREAD_SAFE
#define LOCK() do { object key; catch(key=lock())
#define UNLOCK() key=0; } while(0)
#else
#undef THREAD_SAFE
#define LOCK() do {
#define UNLOCK() } while(0)
#endif

class db_handler {
#ifdef THREAD_SAFE
  static inherit Thread.Mutex;
#endif

  array (object) dbs = ({});
  string db_url;
  int num_dbs;

  void create(string _dburl) {
    db_url = _dburl;
    num_dbs=num;
    mixed err;
    for(int i = 0; i < num; i++) {
      err=catch( dbs += ({ Sql.Sql(db_url) }));
      if(err) perror("Error creating db object:\n" + describe_backtrace(err)+"\n");
    }
  }

  void|object handle(void|object d) {
    mixed err;
    LOCK();
    err = catch {
      int count;
      dbs -= ({0});
      if(objectp(d)) {
	if(search(dbs, d) == -1) {
	  if(sizeof(dbs)>(2*num_dbs)) {
	    CDEBUG("Dropping db because of inventory...\n");
	    destruct(d);
	  } else {
	    dbs += ({d});
	  }
	}
      } else {
	if(!sizeof(dbs)) {
	  d = Sql.Sql(db_url);
	} else {
	  d = dbs[0];
	  dbs -= ({d});
	}
      }
    };
    UNLOCK();
    if(err)
      report_error("error in camas_sql_logger.pike: %s\n", describe_backtrace(err));
    return d;
  }
}

constant events = ({ "login", "send_mail", "smtp_errors", "imap_errors" });

string describe_what (array var, mixed path) {
  return "";
  // return (sizeof (var[VAR_VALUE]) ? replace (sprintf ("<pre>%O</pre>", var[VAR_VALUE]), "\n", "<br />") : "");
}

string describe_form_what (array var, mixed path) {
  array misc = var[VAR_MISC][3];
  string res = "<select name=\"" + path + "\" multiple size=\"" + sizeof (misc) + "\">";
  for (int i = 0; i < sizeof (misc); i++) {
    if (var[VAR_VALUE] && has_value (var[VAR_VALUE], misc[i]))
      res += "<option selected>" + misc[i] + "</option>";
    else
      res += "<option>" + misc[i] + "</option>";
  }
  res += "</select>&nbsp;<input type=\"submit\" value=\"Ok\">";
  return res;
}

array set_from_form_what (string val, int type, object o, mixed ... rest) {
  return (val / "\0");
}

void create () {
  defvar ("on", 0, "Logging", TYPE_FLAG,
          "Set to yes to turn logging on.");
  defvar ("what", ({ }), "What to log", TYPE_CUSTOM,
          "The events to log. <br />"
          "Select from :<ul>"
          "<li>login: logins (in/out)</li>"
          "<li>send_mail: sent mails info</li>"
          "<li>smtp_errors: SMTP errors</li>"
          "<li>imap_errors: IMAP errors</li>"
          "</ul>",
          // function callbacks for the configuration interface
          ({ describe_what, describe_form_what, set_from_form_what,
             // extra argument
             events }),
          lambda () { return !QUERY (on); });
 #ifdef CAMAS_DEBUG
  // Debug
  defvar("debug",1,"Debug",TYPE_FLAG,"Debug the call / errors into Caudium "
         "error log ?");
 #endif
  defvar("dburl", "mysql://user:pass@host/database", "SQL: Database URL",
         TYPE_STRING,
         "This is the conncetion to the database in a SQL-URL.\n");
  defvar("logtable_send_mail", "camas_send_mail", "SQL: Table: send mail",
         TYPE_STRING,
         "This is the table into which all send mail actions will be put.<br>\n"
         "It have to be of the form (date date, time time, text smtpfrom, text smtpto, text smtpcc, int size)");
  defvar("logtable_login", "camas_login", "SQL: Table: login table",
         TYPE_STRING,
         "This is the table into which all users login actions (login, failed login, logout, autologout) will be put.<br>\n"
         "It have to be of the form (date date, time time, char login, int action)."
         "action = 0 for login, 1 for failed login, 2 for logout and 3 for autologout");
  defvar("logtable_imap_errors", "camas_imap_errors", "SQL: Table: IMAP error",
         TYPE_STRING,
         "This is the table into which all IMAP errors will be put.<br>\n"
         "It have to be of the form (date date, time time, text command, text line, text history).");
  defvar("logtable_smtpfail", "camas_smtpfail", "SQL: Table: SMTP error",
         TYPE_STRING,
         "This is the table into which all SMTP errors will be put.<br>\n"
         "It have to be of the form (date date, time time, text command, text line)");
  defvar("logtable_default", "camas_default", "SQL: Table: Default log",
         TYPE_STRING,
         "This the table into which all unknown actions will be put (should have none).<br>\n"
         "It have to be of the form (date date, time time, text data)");
  defvar("dbcount", 3, "SQL: Number of Connections",
         TYPE_INT,
         "Number of connections to make.\n");
  defvar("failtime", 5,"SQL: Warning Timeout",
         TYPE_INT,
         "Time between reconnect attempts if SQL server is down, in minutes.\n");
}

string query_provides () {
  return ("camas_sql_logger");
}

void start() {
  num=QUERY(dbcount);
  db=db_handler( QUERY(dburl) );
}

string status()
{
  if (catch {
        object o;
        o = Sql.Sql(QUERY(dburl));
        return(sprintf("Connected to %s-server on %s<br>\n",
                       o->server_info(), o->host_info()));
      }) {
    return("<font color=red>Not connected.</font><br>\n");
  }
}

void stop() {
  destruct(db);			// We're done, so close the connections.
}

void do_log (string ssql) {
  if (!ssql)
  {
    perror("CAMAS SQL LOGGER: query empty!\n");
    return;
  }
  object sql_conn=db->handle();
  if(catch(sql_conn->query(ssql)))
    perror(sprintf("CAMAS SQL LOGGER: Error running query '%s'.\n", ssql));
  db->handle(sql_conn);
}

void log (string event, mapping data) {

  switch (event) {
  case "send_mail":
    stats->nomailsent++;
    stats->nobytessent += data->size;
    if (has_value (QUERY (what), "send_mail"))
      do_log (sprintf ("INSERT INTO %s VALUES (%s, %s, '%s', '%s', '%s', '%d')", QUERY(logtable_send_mail), "NOW()", "NOW()", data->from, data->to, data->cc, data->size)); // note we don't log bccs for now
    break;

  case "login":
    stats->nologins++;
    if (has_value (QUERY (what), "login"))
      do_log (sprintf("INSERT INTO %s VALUES (%s, %s, '%s', 0)", QUERY(logtable_login), "NOW()", "NOW()", data->login));
    break;

  case "loginfailed":
    stats->nofailedlogins++;
    if (has_value (QUERY (what), "login"))
      do_log (sprintf("INSERT INTO %s VALUES (%s, %s, '%s', 1)", QUERY(logtable_login), "NOW()", "NOW()", data->login));
    break;

  case "logout":
    stats->nologouts++;
    if (has_value (QUERY (what), "login"))
      do_log (sprintf("INSERT INTO %s VALUES (%s, %s, '%s', 2)", QUERY(logtable_login), "NOW()", "NOW()", data->login));
    break;

  case "autologout":
    stats->noautologouts++;
    if (has_value (QUERY (what), "login"))
      do_log (sprintf("INSERT INTO %s VALUES (%s, %s, '%s', 3)", QUERY(logtable_login), "NOW()", "NOW()", data->login));

    break;

  case "imap_errors":
    stats->noimapfailures++;
    if (has_value (QUERY (what), "imap_errors"))
      do_log (sprintf("INSERT INTO %s VALUES (%s, %s, '%s', '%s', '%s')", QUERY(logtable_imap_errors), "NOW()", "NOW()", data->command, data->line, data->history));
    break;

  case "smtpfail":
    stats->nosmtpfailures++;
    if (has_value (QUERY (what), "smtp_errors"))
      do_log (sprintf("INSERT INTO %s VALUES (%s, %s, '%s', '%s')", QUERY(logtable_smtpfail), "NOW()", "NOW()", data->command, data->line ));
    break;

  default:
    do_log (sprintf("INSERT INTO %s VALUES (%s, %s, '%O')", QUERY(logtable_default), "NOW()", "NOW()", data));
    break;
  }
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: on
//! Set to yes to turn logging on.
//!  type: TYPE_FLAG
//!  name: Logging
//
//! defvar: what
//! The events to log. <br />Select from :<ul><li>login: logins (in/out)</li><li>send_mail: sent mails info</li><li>smtp_errors: SMTP errors</li><li>imap_errors: IMAP errors</li></ul>
//!  type: TYPE_CUSTOM
//!  name: What to log
//
//! defvar: debug
//! Debug the call / errors into Caudium error log ?
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: dburl
//! This is the conncetion to the database in a SQL-URL.
//!
//!  type: TYPE_STRING
//!  name: SQL: Database URL
//
//! defvar: logtable_send_mail
//! This is the table into which all send mail actions will be put.<br />
//!It have to be of the form (date date, time time, text smtpfrom, text smtpto, text smtpcc, int size)
//!  type: TYPE_STRING
//!  name: SQL: Table: send mail
//
//! defvar: logtable_login
//! This is the table into which all users login actions (login, failed login, logout, autologout) will be put.<br />
//!It have to be of the form (date date, time time, char login, int action).action = 0 for login, 1 for failed login, 2 for logout and 3 for autologout
//!  type: TYPE_STRING
//!  name: SQL: Table: login table
//
//! defvar: logtable_imap_errors
//! This is the table into which all IMAP errors will be put.<br />
//!It have to be of the form (date date, time time, text command, text line, text history).
//!  type: TYPE_STRING
//!  name: SQL: Table: IMAP error
//
//! defvar: logtable_smtpfail
//! This is the table into which all SMTP errors will be put.<br />
//!It have to be of the form (date date, time time, text command, text line)
//!  type: TYPE_STRING
//!  name: SQL: Table: SMTP error
//
//! defvar: logtable_default
//! This the table into which all unknown actions will be put (should have none).<br />
//!It have to be of the form (date date, time time, text data)
//!  type: TYPE_STRING
//!  name: SQL: Table: Default log
//
//! defvar: dbcount
//! Number of connections to make.
//!
//!  type: TYPE_INT
//!  name: SQL: Number of Connections
//
//! defvar: failtime
//! Time between reconnect attempts if SQL server is down, in minutes.
//!
//!  type: TYPE_INT
//!  name: SQL: Warning Timeout
//
