/* g_save_image.c
 * 
 * Copyright (C) 2001 Claudio Girardi
 *
 * This file is derived from the 
 * GdkPixbuf library - convert X drawable information to RGB
 *
 * Copyright (C) 1999 Michael Zucchi
 *
 * Authors: Michael Zucchi <zucchi@zedzone.mmc.com.au>
 *          Cody Russell <bratsche@gnome.org>
 *          Federico Mena-Quintero <federico@gimp.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>

#ifdef HAVE_LIBJPEG
#include <jpeglib.h>
#endif

#include "g_save_image.h"
#include "util.h"

#ifdef HAVE_DMALLOC_H
#include <dmalloc.h>		/* dmalloc.h should be included last */
#endif

//#include "gdk-pixbuf-private.h"

#if (G_BYTE_ORDER == G_LITTLE_ENDIAN)
#define LITTLE
#endif

/* function prototypes */


/* This should work correctly with any display/any endianness, but will probably run quite slow */
static void convert_real_slow(GdkImage * image, guchar * pixels, int rowstride, GdkColormap * cmap, int alpha)
{
  int xx, yy;
  int width, height;
  int bpl;
  guint8 *srow = image->mem, *orow = pixels;
  guint8 *s;
  guint8 *o;
  guint32 pixel;
  GdkVisual *v;
  guint8 component;
  int i;

  width = image->width;
  height = image->height;
  bpl = image->bpl;
  v = gdk_colormap_get_visual(cmap);

  D(printf("rgb  mask/shift/prec = %x:%x:%x %d:%d:%d  %d:%d:%d\n", v->red_mask, v->green_mask, v->blue_mask, v->red_shift, v->green_shift, v->blue_shift, v->red_prec, v->green_prec, v->blue_prec));

  for (yy = 0; yy < height; yy++) {
    s = srow;
    o = orow;
    for (xx = 0; xx < width; xx++) {
      pixel = gdk_image_get_pixel(image, xx, yy);
      switch (v->type) {
	/* I assume this is right for static & greyscale's too? */
      case GDK_VISUAL_STATIC_GRAY:
      case GDK_VISUAL_GRAYSCALE:
      case GDK_VISUAL_STATIC_COLOR:
      case GDK_VISUAL_PSEUDO_COLOR:
	*o++ = cmap->colors[pixel].red;
	*o++ = cmap->colors[pixel].green;
	*o++ = cmap->colors[pixel].blue;
	break;
      case GDK_VISUAL_TRUE_COLOR:
	/* This is odd because it must sometimes shift left (otherwise
	   I'd just shift >> (*_shift - 8 + *_prec + <0-7>). This logic
	   should work for all bit sizes/shifts/etc. */
	component = 0;
	for (i = 24; i < 32; i += v->red_prec)
	  component |= ((pixel & v->red_mask) << (32 - v->red_shift - v->red_prec)) >> i;
	*o++ = component;
	component = 0;
	for (i = 24; i < 32; i += v->green_prec)
	  component |= ((pixel & v->green_mask) << (32 - v->green_shift - v->green_prec)) >> i;
	*o++ = component;
	component = 0;
	for (i = 24; i < 32; i += v->blue_prec)
	  component |= ((pixel & v->blue_mask) << (32 - v->blue_shift - v->blue_prec)) >> i;
	*o++ = component;
	break;
      case GDK_VISUAL_DIRECT_COLOR:
	*o++ = cmap->colors[((pixel & v->red_mask) << (32 - v->red_shift - v->red_prec)) >> 24].red;
	*o++ = cmap->colors[((pixel & v->green_mask) << (32 - v->green_shift - v->green_prec)) >> 24].green;
	*o++ = cmap->colors[((pixel & v->blue_mask) << (32 - v->blue_shift - v->blue_prec)) >> 24].blue;
	break;
      }
      if (alpha)
	*o++ = 0xff;
    }
    srow += bpl;
    orow += rowstride;
  }
}


/*
   perform actual conversion

   If we can, try and use the optimized code versions, but as a default
   fallback, and always for direct colour, use the generic/slow but complete
   conversion function.
 */
static void rgbconvert(GdkImage * image, guchar * pixels, int rowstride, int alpha, GdkColormap * cmap)
{
  int index = (image->byte_order == GDK_MSB_FIRST) | (alpha != 0) << 1;
  int bank = 5;			/* default fallback converter */
  GdkVisual *v = gdk_colormap_get_visual(cmap);

  D(printf("masks = %x:%x:%x\n", v->red_mask, v->green_mask, v->blue_mask));
  D(printf("image depth = %d, bpp = %d\n", image->depth, image->bpp));

  switch (v->type) {
    /* I assume this is right for static & greyscale's too? */
  case GDK_VISUAL_STATIC_GRAY:
  case GDK_VISUAL_GRAYSCALE:
  case GDK_VISUAL_STATIC_COLOR:
  case GDK_VISUAL_PSEUDO_COLOR:
    switch (image->bpp) {
    case 1:
      bank = 0;
      break;
    case 8:
      bank = 1;
      break;
    }
    break;
  case GDK_VISUAL_TRUE_COLOR:
    switch (image->depth) {
    case 15:
      if (v->red_mask == 0x7c00 && v->green_mask == 0x3e0 && v->blue_mask == 0x1f && image->bpp == 16)
	bank = 2;
      break;
    case 16:
      if (v->red_mask == 0xf800 && v->green_mask == 0x7e0 && v->blue_mask == 0x1f && image->bpp == 16)
	bank = 3;
      break;
    case 24:
    case 32:
      if (v->red_mask == 0xff0000 && v->green_mask == 0xff00 && v->blue_mask == 0xff && image->bpp == 32)
	bank = 4;
      break;
    }
    break;
  case GDK_VISUAL_DIRECT_COLOR:
    /* always use the slow version */
    break;
  }

  D(printf("converting using conversion function in bank %d\n", bank));

  convert_real_slow(image, pixels, rowstride, cmap, alpha);
}


/* Exported functions */

/**
 * gdk_pixbuf_get_from_drawable:
 * @dest: Destination pixbuf, or NULL if a new pixbuf should be created.
 * @src: Source drawable.
 * @cmap: A colormap if @src is a pixmap.  If it is a window, this argument will
 * be ignored.
 * @src_x: Source X coordinate within drawable.
 * @src_y: Source Y coordinate within drawable.
 * @dest_x: Destination X coordinate in pixbuf, or 0 if @dest is NULL.
 * @dest_y: Destination Y coordinate in pixbuf, or 0 if @dest is NULL.
 * @width: Width in pixels of region to get.
 * @height: Height in pixels of region to get.
 *
 * Transfers image data from a Gdk drawable and converts it to an RGB(A)
 * representation inside a GdkPixbuf.
 *
 * If the drawable @src is a pixmap, then a suitable colormap must be specified,
 * since pixmaps are just blocks of pixel data without an associated colormap.
 * If the drawable is a window, the @cmap argument will be ignored and the
 * window's own colormap will be used instead.
 *
 * If the specified destination pixbuf @dest is #NULL, then this function will
 * create an RGB pixbuf with 8 bits per channel and no alpha, with the same size
 * specified by the @width and @height arguments.  In this case, the @dest_x and
 * @dest_y arguments must be specified as 0, otherwise the function will return
 * #NULL.  If the specified destination pixbuf is not NULL and it contains alpha
 * information, then the filled pixels will be set to full opacity.
 *
 * If the specified drawable is a pixmap, then the requested source rectangle
 * must be completely contained within the pixmap, otherwise the function will
 * return #NULL.
 *
 * If the specified drawable is a window, then it must be viewable, i.e. all of
 * its ancestors up to the root window must be mapped.  Also, the specified
 * source rectangle must be completely contained within the window and within
 * the screen.  If regions of the window are obscured by noninferior windows, the
 * contents of those regions are undefined.  The contents of regions obscured by
 * inferior windows of a different depth than that of the source window will also
 * be undefined.
 *
 * Return value: The same pixbuf as @dest if it was non-NULL, or a newly-created
 * pixbuf with a reference count of 1 if no destination pixbuf was specified; in
 * the latter case, NULL will be returned if not enough memory could be
 * allocated for the pixbuf to be created.
 **/
guchar *gdk_pixbuf_get_from_drawable(guchar * dest, GdkDrawable * src, GdkColormap * cmap, int src_x, int src_y, int width, int height)
{
  GdkWindowType window_type;
  int src_width, src_height;
  GdkImage *image;
  int rowstride, alpha;

  /* General sanity checks */
  g_return_val_if_fail(src != NULL, NULL);
  window_type = gdk_window_get_type(src);
  g_return_val_if_fail(cmap != NULL, NULL);

  /* Coordinate sanity checks */
  gdk_window_get_size(src, &src_width, &src_height);
  g_return_val_if_fail(src_x >= 0 && src_y >= 0, NULL);
  g_return_val_if_fail(src_x + width <= src_width && src_y + height <= src_height, NULL);

  /* Get Image in ZPixmap format (packed bits). */
  image = gdk_image_get(src, src_x, src_y, width, height);
  g_return_val_if_fail(image != NULL, NULL);

  /* Create the destination buffer */
  dest = (guchar *) g_malloc((width * height * 3) * sizeof(guchar));
  if (!dest) {
    gdk_image_destroy(image);
    return NULL;
  }
  alpha = 0;			/* no alpha */
  rowstride = 3 * width;	/* packed RGB ? */

  /* we offset into the image data based on the position we are retrieving from */
  rgbconvert(image, dest, rowstride, alpha, cmap);

  gdk_image_destroy(image);

  return dest;
}


/* this is based on example.c in libjpeg, by the Independent JPEG Group */
int write_jpeg_file(char *filename, int quality, char *image_buffer, int image_width, int image_height)
{
#ifdef HAVE_LIBJPEG
  struct jpeg_compress_struct cinfo;
  FILE *outfile;
  int row_stride;
  JSAMPROW row_pointer[1];
  struct jpeg_error_mgr jerr;

  if (filename == NULL) {
    return -1;
  };
  if (image_buffer == NULL) {
    return -1;
  };
  if ((outfile = fopen(filename, "wb")) == NULL) {
    fprintf(stderr, "write_jpeg_file: can't open %s\n", filename);
    return -1;
  };

  /* set up the error handler */
  cinfo.err = jpeg_std_error(&jerr);
  /* initialise the JPEG compression object */
  jpeg_create_compress(&cinfo);
  /* specify data destination */
  jpeg_stdio_dest(&cinfo, outfile);
  /* image width and height, in pixels */
  cinfo.image_width = image_width;
  cinfo.image_height = image_height;
  /* number of color components per pixel */
  cinfo.input_components = 3;
  /* colorspace of input image */
  cinfo.in_color_space = JCS_RGB;
  /* set default compression parameters */
  jpeg_set_defaults(&cinfo);
  /* limit to baseline-JPEG values */
  jpeg_set_quality(&cinfo, quality, TRUE);
  /* start compressor */
  jpeg_start_compress(&cinfo, TRUE);

  /* from gd-1.8.4
     sprintf(comment, "CREATOR: gd-jpeg v%s (using IJG JPEG v%d),",
     GD_JPEG_VERSION, JPEG_LIB_VERSION);
     if (quality >= 0)
     sprintf(comment + strlen(comment), " quality = %d\n",
     quality);
     else
     strcat(comment + strlen(comment), " default quality\n");
     jpeg_write_marker(&cinfo, JPEG_COM, (unsigned char *)comment,
     (unsigned int)strlen(comment));
   */

  row_stride = image_width * 3;
  while (cinfo.next_scanline < cinfo.image_height) {
    row_pointer[0] = &image_buffer[cinfo.next_scanline * row_stride];
    (void) jpeg_write_scanlines(&cinfo, row_pointer, 1);
  };
  /* finish compression */
  jpeg_finish_compress(&cinfo);
  /* close the output file */
  fclose(outfile);
  /* release JPEG compression object */
  jpeg_destroy_compress(&cinfo);
  /* everyting went right */
  return 0;
#else
  return -1;
#endif /* #ifdef HAVE_LIBJPEG */
}
