/* mg-canvas-tip.c
 *
 * Copyright (C) 2002 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "mg-defs.h"
#include "mg-canvas-tip.h"
#include "marshal.h"

static void mg_canvas_tip_class_init (MgCanvasTipClass * class);
static void mg_canvas_tip_init       (MgCanvasTip * tip);
static void mg_canvas_tip_finalize    (GObject   * object);

static void mg_canvas_tip_set_property    (GObject              *object,
					guint                 param_id,
					const GValue         *value,
					GParamSpec           *pspec);
static void mg_canvas_tip_get_property    (GObject              *object,
					guint                 param_id,
					GValue               *value,
					GParamSpec           *pspec);

static void mg_canvas_tip_post_init  (MgCanvasTip * tip);

struct _MgCanvasTipPrivate
{
	gchar  *text;

	/* presentation parameters */
	gdouble x_text_space;
	gdouble y_text_space;
};

enum
{
	PROP_0,
	PROP_TEXT
};


/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *tip_parent_class = NULL;


guint
mg_canvas_tip_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (MgCanvasTipClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_canvas_tip_class_init,
			NULL,
			NULL,
			sizeof (MgCanvasTip),
			0,
			(GInstanceInitFunc) mg_canvas_tip_init
		};		

		type = g_type_register_static (GNOME_TYPE_CANVAS_GROUP, "MgCanvasTip", &info, 0);
	}

	return type;
}

static void
mg_canvas_tip_class_init (MgCanvasTipClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	tip_parent_class = g_type_class_peek_parent (class);

	object_class->finalize = mg_canvas_tip_finalize;

	/* Properties */
	object_class->set_property = mg_canvas_tip_set_property;
	object_class->get_property = mg_canvas_tip_get_property;

	g_object_class_install_property (object_class, PROP_TEXT,
					 g_param_spec_string ("tip_text", NULL, NULL, NULL,
							      (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
mg_canvas_tip_init (MgCanvasTip * tip)
{
	tip->priv = g_new0 (MgCanvasTipPrivate, 1);
	tip->priv->text = NULL;
	tip->priv->x_text_space = 3.;
	tip->priv->y_text_space = 3.;
}


static void
mg_canvas_tip_finalize (GObject   * object)
{
	MgCanvasTip *tip;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CANVAS_TIP (object));

	tip = MG_CANVAS_TIP (object);
	if (tip->priv) {
		if (tip->priv->text)
			g_free (tip->priv->text);
		g_free (tip->priv);
		tip->priv = NULL;
	}

	/* for the parent class */
	tip_parent_class->finalize (object);
}

static void 
mg_canvas_tip_set_property    (GObject              *object,
			    guint                 param_id,
			    const GValue         *value,
			    GParamSpec           *pspec)
{
	MgCanvasTip *tip;
	const gchar *str;

	tip = MG_CANVAS_TIP (object);

	switch (param_id) {
	case PROP_TEXT:
		if (tip->priv->text) {
			g_free (tip->priv->text);
			tip->priv->text = NULL;
		}
		str = g_value_get_string (value);
		if (str)
			tip->priv->text = g_strdup (str);
		break;
	}

	if (tip->priv->text)
		mg_canvas_tip_post_init (tip);
}

static void 
mg_canvas_tip_get_property    (GObject              *object,
			    guint                 param_id,
			     GValue              *value,
			    GParamSpec           *pspec)
{
	TO_IMPLEMENT;
}


static void 
mg_canvas_tip_post_init  (MgCanvasTip * tip)
{
	GnomeCanvasItem *text, *bg;
	double x1, y1, x2, y2;

	text = gnome_canvas_item_new (GNOME_CANVAS_GROUP (tip),
				      GNOME_TYPE_CANVAS_TEXT,
				      "x", tip->priv->x_text_space,
				      "y", tip->priv->y_text_space,
				      "font-desc", GTK_WIDGET (GNOME_CANVAS_ITEM (tip)->canvas)->style->font_desc,
				      "fill_color", "black",
				      "text", tip->priv->text,
				      "justification", GTK_JUSTIFY_LEFT, 
				      "anchor", GTK_ANCHOR_NORTH_WEST,
				      NULL);
	gnome_canvas_item_get_bounds (text, &x1, &y1, &x2, &y2);

	bg = gnome_canvas_item_new (GNOME_CANVAS_GROUP (tip),
				    GNOME_TYPE_CANVAS_RECT,
				    "x1", (double) 0,
				    "y1", (double) 0,
				    "x2", x2 - x1 + 2*tip->priv->x_text_space,
				    "y2", y2 - y1 + 2*tip->priv->y_text_space,
				    "outline_color", "black",
				    "fill_color", "lightyellow",
				    "width_pixels", 1,
				    NULL);
	
	gnome_canvas_item_lower_to_bottom (bg);
}

