
/*
 * $Header: /cvsroot/mpdist/mpdist/mp/header.c,v 1.1.1.1 2002/04/12 16:47:25 richbastard Exp $
 * 
 * Copyright (c) 1987-2002 Rich Burridge, Sun Microsystems Inc.
 * All rights reserved.
 * 
 * This software is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "option.h"
#include "header.h"
#include "input.h"
#include "print.h"

static char *from;             /* From: */
static char *from_;            /* From_ (UNIX from) */
static char *app_from;         /* Apparently_from: */
static char *to[MAXCONT + 1];  /* To: (can have multiple lines) */
static char *app_to;           /* Apparently_to: */
static char *cc[MAXCONT + 1];  /* Cc: (can have multiple lines) */
static char *subject;          /* Subject: (can be set from command line) */
static char *date;             /* Date: */
static char *newsgroups;       /* Newsgroups: (news articles only) */
static char *reply_to;         /* Reply-to: */
static char *content_len;      /* Content-Length: */

static char *hdrs[MAXHDRS];    /* Contents of additional headers. */

static int  emptyline       (char *);
static int  toprint         (char *, char *);
static void get_extra_hdrs  (void);
static void get_header      (char *hdr, char **dest);
static void get_mult_hdr    (char *, char **);
static void show_all_hdrs   (void);
static void show_mult_hdr   (char *, char **);


/* Emptyline returns true if its argument is empty or whitespace only. */

static int
emptyline(char *str)
{
    while (*str) {
        if (!isspace((int) *str)) {
            return(0);
        }
        str++;
    }
    return(1);
}


#ifdef MIME_SUPPORT
char *
get_appfrom_hdr(void)
{
    return(app_from);
}


char *
get_appto_hdr(void)
{
    return(app_to);
}


char **
get_cc_hdr(void)
{
    return(cc);
}


char *
get_replyto_hdr(void)
{
    return(reply_to);
}


char **
get_to_hdr(void)
{
    return(to);
}


int
printable(char *str)
{
    if (EQUAL(str, FROMHDR)) {
        return(toprint(from, FROMHDR));
    }
    if (EQUAL(str, FROM_HDR)) {
        return(toprint(from_, FROM_HDR));
    }
    if (EQUAL(str, TOHDR)) {
        return(toprint(to[0], TOHDR));
    }
    if (EQUAL(str, CCHDR)) {
        return(toprint(cc[0], CCHDR));
    }
    if (EQUAL(str, SUBJECTHDR)) {
        return(toprint(subject, SUBJECTHDR));
    }
    if (EQUAL(str, APP_FROMHDR)) {
        return(toprint(app_from, APP_FROMHDR));
    }
    if (EQUAL(str, APP_TOHDR)) {
        return(toprint(app_to, APP_TOHDR));
    }
    if (EQUAL(str, DATEHDR)) {
        return(toprint(date, DATEHDR));
    }
    if (EQUAL(str, REPLYHDR)) {
        return(toprint(reply_to, REPLYHDR));
    }
    if (EQUAL(str, NEWSGROUPHDR)) {
        return(toprint(newsgroups, NEWSGROUPHDR));
    }
    if (EQUAL(str, NEWSGROUPSHDR)) {
        return(toprint(newsgroups, NEWSGROUPSHDR));
    }

    return(0);
}
#endif /* MIME_SUPPORT */


char *
get_date_hdr(void)
{
    return(date);
}


static void
get_extra_hdrs(void)       /* Get user header preferences. */
{
    int i;

    for (i = 0; i < int_opt_val(O_ADDHDRN); i++) {
        get_header(get_hdr(H_ADD, i), &hdrs[i]);
    }
}


char *
get_from_hdr(void)
{
    return(from);
}


char *
get_header_n(int n)
{
    return(hdrs[n]);
}


/*
 * If input line is header of type 'hdr', get the contents of the header into
 * 'dest' (dynamically allocated).
 */

static void
get_header(char *hdr, char **dest)
{
    char c, *line, *ptr;
    int  len;    /* Length of the header text to the left of the colon. */

    if (hdr_equal(hdr)) {
        c = (EQUAL(hdr, FROM_HDR)) ? ' ' : ':';
        line = current_line();
        ptr = strchr(line, c);
        len = ptr - line + 1;
        *dest = (char *) malloc((unsigned) (strlen(line) - len + 1));
        STRCPY(*dest, line + len);
    }
}


/*
 * If input line is header of type 'hdr', get header into dest. The header
 * may have multiple lines. This skips input to next line.
 */

static void
get_mult_hdr(char *hdr, char **dest)
{
    char *line;
    int  i = 0;

    if (hdr_equal(hdr)) {
        get_header(hdr, dest);
        i++;
        readline();
        line = current_line();
        while (i < MAXCONT && !emptyline(line) && isspace((int) line[0])) {
            dest[i] = (char *) malloc((unsigned) (strlen(line) + 1));
            STRCPY(dest[i], line);
            i++;
            readline();
        }
        dest[i] = NULL;
    }
}


char *
get_newsgroups_hdr(void)
{
    return(newsgroups);
}


char *
get_subject_hdr(void)
{
    return(subject);
}


void
Header(void)
{
    int i;

    for (i = 0; i < MAXHDRS; i++) {
        hdrs[i] = NULL;
    }

    from = NULL;           /* From: */
    from_ = NULL;          /* From_ (UNIX from)  */
    app_from = NULL;       /* Apparently_from:  */
    app_to = NULL;         /* Apparently_to:  */
    for (i = 0; i < MAXCONT + 1; i++) {
        to[i] = NULL;      /* To: (can have multiple lines) */
        cc[i] = NULL;      /* Cc: (can have multiple lines) */
    }
    subject = NULL;        /* Subject: (can be set from command line)  */
    date = NULL;           /* Date:  */
    newsgroups = NULL;     /* Newsgroups: (news articles only)  */
    reply_to = NULL;       /* Reply-to:  */
    content_len = NULL;    /* Content-Length: */
}


/*
 * Compare the first word of the current line (converted to lower-case, with
 * the given header definition. Determine if they are equal. Allows for extra
 * white space before the final colon.
 */

int
hdr_equal(char *val)
{
    char *nptr = current_line();
    char *wptr = val;
    int  n, w;

    do {
        n = *nptr++;
        w = *wptr++;
        if (isupper(n)) {
            n = tolower(n);
        }
        if (isupper(w)) {
            w = tolower(w);
        }

        if (w == ':') {
            while ((n == ' ' || n == '\t') && n != '\0') {
                n = *nptr++;
            }
        }

        if (n != w && w != '\0') {
            return(0);
        }
    } while (n != '\0' && w != '\0');

    return(1);
}


/*
 * Parse_headers is a function which reads and parses the message headers,
 * extracting the bits which are of interest.
 * 
 * The document is on standard input; the document is read up to the end of the
 * header; the next line is read ahead into 'nextline'.
 * 
 * Parameter: digest  indicates if parsing is of digest headers instead of
 * message headers
 * 
 * Implicit Input: nextline contains the next line from standard input
 * 
 * Side-effects: The function fills in the global header variables with headers
 * found. The variable dtype is set to the document type The variable
 * nextline is set The document is read up to the line following the headers
 */

void
parse_headers(int digest)
{
    char *colon;                  /* Pointer to colon in line. */
    char *c;                      /* General character pointer. */
    char *line, tmpstr[MAXLINE];
    int  mlen;

/*
 * If not processing digest headers, determine if this article is an
 * ordinary text file.
 */

    line = current_line();
    if (!digest) {                       /* Parsing digest headers? */
        if (!hdr_equal(FROM_HDR)) {      /* UNIX From_ header? */
            colon = strchr(line, ':');
            if (colon == NULL) {         /* No colon => not a header line. */
                set_doc_type(DO_TEXT);
                return;
            }
            c = line;
            while (c < colon && (!isspace((int) *c))) {
                c++;
            }
            if (c != colon) {    /* Whitespace in header name => not header. */
                set_doc_type(DO_TEXT);
                return;
            }
        }
    }
    set_doc_type(DO_MAIL);          /* Default to mail document. */
    if (int_opt_val(O_ALLHDRS)) {
        return;                     /* Parse/show headers in show_headers(). */
    }

/* Parse headers. */

    for (;;) {
        if (emptyline(line)) {
            break;    /* End of headers. */
        }

        if (!digest) {
            get_header(FROM_HDR, &from_);
            get_header(APP_FROMHDR, &app_from);
            get_header(APP_TOHDR, &app_to);
            get_header(NEWSGROUPSHDR, &newsgroups);
            get_header(NEWSGROUPHDR, &newsgroups);
            get_header(REPLYHDR, &reply_to);

            get_mult_hdr(TOHDR, to);
            if (emptyline(line)) {
                break;
            }

            get_mult_hdr(CCHDR, cc);
            if (emptyline(line)) {
                break;
            }

            if (doc_type() != DO_NEWS && hdr_equal(NEWSGROUPSHDR)) {
                set_doc_type(DO_NEWS);
            }
            if (doc_type() != DO_NEWS && hdr_equal(NEWSGROUPHDR)) {
                set_doc_type(DO_NEWS);
            }
        }
        get_header(FROMHDR, &from);
        get_header(SUBJECTHDR, &subject);
        get_header(DATEHDR, &date);

        get_extra_hdrs();    /* Get user's header preferences. */

        if (int_opt_val(O_CONTENT)) {
            get_header(CONTENT_LEN, &content_len);
            if (hdr_equal(CONTENT_LEN)) {
                SSCANF(line, "%s %d", tmpstr, &mlen);

/*
 * The Content-Length: doesn't seem to include the initial blank line 
 * between the mail header and the message body, or the blank line after 
 * the message body and before the start of the next "From " header, so 
 * add in two for those.
 */

                set_message_len(mlen + 2);
            }
        }
        if (digest || (!hdr_equal(TOHDR) && !hdr_equal(CCHDR))) {
            while (!end_of_file() && !end_of_line()) {
                readline();    /* Skip rest of long lines. */
            }
            readline();
        }
    }
}


void
reset_headers(void)          /* Reset header values for next message. */
{
    int i;

    if (int_opt_val(O_ALLHDRS)) {
        return;
    }

    if (from != NULL) {
        FREE(from);
    }
    if (from_ != NULL) {
        FREE(from_);
    }
    if (app_from != NULL) {
        FREE(app_from);
    }
    if (app_to != NULL) {
        FREE(app_to);
    }
    if (content_len != NULL) {
        FREE(content_len);
    }
    if (date != NULL) {
        FREE(date);
    }
    if (newsgroups != NULL) {
        FREE(newsgroups);
    }
    if (reply_to != NULL) {
        FREE(reply_to);
    }

    from       = NULL;
    from_      = NULL;
    app_from   = NULL;
    app_to     = NULL;
    date       = NULL;
    newsgroups = NULL;
    reply_to   = NULL;
    subject    = NULL;

    for (i = 0; i < MAXCONT + 1; i++) {
        if (to[i] != NULL) {
            FREE(to[i]);
        }
        to[i] = NULL;
        if (cc[i] != NULL) {
            FREE(cc[i]);
        }
        cc[i] = NULL;
    }
}


void
set_date_hdr(char *value)
{
    date = value;
    date[24] = '\0';
}


void
set_subject_hdr(char *value)
{
    subject = value;
}


/*
 * Parse and show all mail/news headers. Note that the parse and printing is
 * done here, in this one single routine, which is different from the normal
 * way of handling mail/news headers. The parse_headers() and reset_headers()
 * routines immediately return if the user wants to print all headers.
 */

static void
show_all_hdrs(void)           /* Read and print all mail/news headers. */
{
    char hdrtype[MAXLINE];    /* Header type (upto the colon). */
    char *shdr;               /* Single line header. */
    char c, *line, *ptr;
    int  i;

    for (;;) {
        shdr = NULL;
        line = current_line();
        if (emptyline(line)) {
            break;                              /* End of headers. */
        }

        if (line[0] == ' ' || line[0] == '\t') {
            show_text(T_ROMAN, NULL, line);
        } else {
            STRCPY(hdrtype, line);              /* Extract header type. */
            c = hdr_equal(FROM_HDR) ? ' ' : ':';
            i = 0;
            if ((ptr = strchr(hdrtype, c)) != NULL) {
                i = ptr - hdrtype + 1;
            }
            hdrtype[i] = '\0';

            get_header(hdrtype, &shdr);
            if (hdr_equal(FROM_HDR)) {
                show_text(T_BOLD, hdrtype, shdr);
            } else {
                show_text(T_MIXED, hdrtype, shdr);
            }
        }

        while (!end_of_file() && !end_of_line()) {
            readline();    /* Skip rest of long lines. */
        }
        readline();

        if (shdr != NULL) {
            FREE(shdr);
        }
    }
}


/*
 * Show_headers outputs the headers in PostScript. Different headers are
 * output depending 'digest'.
 */

void
show_headers(int digest)
{
    if (int_opt_val(O_ALLHDRS)) {
        show_all_hdrs();
        return;
    }
    if (digest) {
        if (toprint(from, FROMHDR)) {
            show_text(T_MIXED, FROMHDR, from);
        }
        if (toprint(subject, SUBJECTHDR)) {
            show_text(T_MIXED, SUBJECTHDR, subject);
        }
        if (toprint(date, DATEHDR)) {
            show_text(T_MIXED, DATEHDR, date);
        }
    } else {
        if (toprint(from_, FROM_HDR)) {
            show_text(T_BOLD, FROM_HDR, from_);
        }
        if (toprint(from, FROMHDR)) {
            show_text(T_MIXED, FROMHDR, from);
        }
        if (toprint(app_from, APP_FROMHDR)) {
            show_text(T_MIXED, APP_FROMHDR, app_from);
        }
        if (toprint(to[0], TOHDR)) {
            show_mult_hdr(TOHDR, to);
        }
        if (toprint(app_to, APP_TOHDR)) {
            show_text(T_MIXED, APP_TOHDR, app_to);
        }
        if (toprint(cc[0], CCHDR)) {
            show_mult_hdr(CCHDR, cc);
        }
        if (toprint(reply_to, REPLYHDR)) {
            show_text(T_MIXED, REPLYHDR, reply_to);
        }
        if (toprint(newsgroups, NEWSGROUPSHDR)) {
            show_text(T_MIXED, NEWSGROUPSHDR, newsgroups);
        }
        if (toprint(subject, SUBJECTHDR)) {
            show_text(T_MIXED, SUBJECTHDR, subject);
        }
        if (toprint(date, DATEHDR)) {
            show_text(T_MIXED, DATEHDR, date);
        }
    }
    print_extra();        /* Print out user header preferences. */
    write_output("sf ");
}


static void
show_mult_hdr(char *hdr, char **val)
{
    show_text(T_MIXED, hdr, *val);
    val++;
    while (*val) {
        show_text(T_ROMAN, NULL, *val++);
    }
}


static int
toprint(char *hdr, char *str)
{
    int i;

    if (!hdr) {
        return(0);
    }
    for (i = 0; i < int_opt_val(O_REMHDRN); i++) {
        if (CASE_EQUAL(get_hdr(H_REMOVE, i), str)) {
            return(0);
        }
    }
    return(1);
}
