#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <termios.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>

#include <config.h>
#include <support.h>
#include <xcio.h>

#include <sysmsg.h>

#include <option.h>
#include <timer.h>
#include <log.h>
#include <env.h>

#include "device.h"

typedef enum {
    FLOW_NONE,
    FLOW_XONXOFF,
    FLOW_CRTSCTS,
    FLOW_MAX
} flow_t;

typedef enum {
    PARITY_NONE,
    PARITY_ODD,
    PARITY_EVEN,
    PARITY_MAX
} parity_t;

struct sioopt_s {
    speed_t speed;
    parity_t parity;
    flow_t flow;
    char *modem;
    char *lockpath;
    char *loginpath;
    struct numlist_s {
	char *number;
	struct numlist_s *next;
    } *numhp;
    uid_t uid;
};

#define	DEFAULT_SPEED	B38400

static struct sioopt_s sioOpt;
static struct termios oldTio;
static char *lockPath;

static struct {
    speed_t b;
    u_int n;
} speedTab[]={
    {B1200, 1200},
    {B2400, 2400},
    {B4800, 4800},
    {B9600, 9600},
#ifdef	B19200
    {B19200, 19200},
#endif
#ifdef	B38400
    {B38400, 38400},
#endif
#ifdef	B57600
    {B57600, 57600},
#endif
#ifdef	B115200
    {B115200, 115200},
#endif
#ifdef	B230400
    {B230400, 230400},
#endif
#ifdef	B460800
    {B460800, 460800},
#endif
    {0, 0}
};

u_int
GetSpeed()
{
    int i;

    for (i = 0; speedTab[i].n; i ++) {
	if (speedTab[i].b == sioOpt.speed)
	    return(speedTab[i].n);
    }
    return(0);
}

static bool_t
EnvSpeed(int argc, char **argv, char *outs)
{
    int i;
    u_int nspeed;

    if (!argc) {
	if ((nspeed = GetSpeed()) != 0) sprintf(outs, "%d", nspeed);
	return FALSE;
    }
    if ((nspeed = atoi(argv[1])) == 0) return FALSE;
    for (i = 0; speedTab[i].n; i ++) {
	if (speedTab[i].n == nspeed) {
	    sioOpt.speed = speedTab[i].b;
	    break;
	}
    }
    if (!sioOpt.speed) sioOpt.speed = DEFAULT_SPEED;
    return TRUE;
}

static bool_t
EnvUser(int argc, char **argv, char *outs)
{
    struct passwd *pwd;

    if (!argc) {
	if (sioOpt.uid) {
	    pwd = getpwuid(sioOpt.uid);
	    if (pwd) strcpy(outs, pwd->pw_name);
	}
	return FALSE;
    }
    if ((pwd = getpwnam(argv[1])) != NULL) sioOpt.uid = pwd->pw_uid;
    return TRUE;
}

static bool_t
EnvFlow(int argc, char **argv, char *outs)
{
    flow_t f;
    const char *flows[]={
	"none",
	"xonxoff",
	"crtscts"
    };

    if (!argc) {
	strcpy(outs, flows[sioOpt.flow]);
	return FALSE;
    }
    for (f = FLOW_NONE; f < FLOW_MAX; f ++) {
	if (!strcasecmp(argv[1], flows[f])) {
	    sioOpt.flow = f;
	    return TRUE;
	}
    }
    sioOpt.flow = FLOW_NONE;
    return TRUE;
}

static bool_t
EnvParity(int argc, char **argv, char *outs)
{
    parity_t p;
    const char *paritys[]={
	"none",
	"odd",
	"even"
    };

    if (!argc) {
	strcpy(outs, paritys[sioOpt.parity]);
	return FALSE;
    }
    for (p = PARITY_NONE; p < PARITY_MAX; p ++) {
	if (!strcasecmp(argv[1], paritys[p])) {
	    sioOpt.parity = p;
	    return TRUE;
	}
    }
    sioOpt.parity = PARITY_NONE;
    return TRUE;
}

static bool_t
EnvModem(int argc, char **argv, char *outs)
{
    unsigned int n;
    int i;
    char *dev;
    extern char *ModemGetStr();
    const char *nullmodems[]={
	"direct",
	"null",
	"no",
	"none",
    };

    if (!argc) {
	if (sioOpt.modem) {
	    extern char *ModemGetName();
	    char *name=ModemGetName();

	    if (name) sprintf(outs, "%s (%s)", sioOpt.modem, name);
	    else strcpy(outs, sioOpt.modem);
	} else strcpy(outs, nullmodems[0]);
	return FALSE;
    }
    if (sioOpt.modem) Free(sioOpt.modem);
    sioOpt.modem = NULL;
    for (n = 0; n < sizeof(nullmodems)/sizeof(nullmodems[0]); n ++)
	if (!strcasecmp(argv[1], nullmodems[n])) return FALSE;
    if (LoadModemInfo(argv[1])) {
	char *new="default";

	ConsoleOutf("%s: unknown modem...using '%s'\n",
		    argv[1], new);
	sioOpt.modem = strdup(new);
	LoadModemInfo(new);
    } else sioOpt.modem = strdup(argv[1]);
    n = ModemGetNum("MaxDTESpeed");
    for (i = 0; speedTab[i].n; i ++) {
	if (speedTab[i].n == n)
	    sioOpt.speed = speedTab[i].b;
    }

    if ((dev = ModemGetStr("Device")) != NULL) {
	char *av[]={argv[0], dev, NULL};

	EnvLine(2, av, NULL);
    }

#if 0
    DumpModemInfo();
#endif

    return TRUE;
}

static int
MakeSioLock(pid_t pid)
{
    FILE *fp;

    if (lockPath && (fp = fopen(lockPath, "w")) != NULL) {
	fprintf(fp, "%10d\n", pid);
	fclose(fp);
	return(0);
    }
    if (errno == EACCES || errno == EPERM)
	ConsoleMsg(MS_E_PERM_DENIED, lockPath);
    LogError(lockPath);
    return(-1);
}

void
SioIgnoreCD(int fd, bool_t sw)
{
    struct termios tio;

    tcgetattr(fd, &tio);
    if (sw) tio.c_cflag |= CLOCAL;
    else tio.c_cflag &= ~CLOCAL;
    tcsetattr(fd, TCSADRAIN, &tio);
}

static int
SioOpen(char **devname, int *status, int argc, char *argv[])
{
    struct termios newtio;
    int fdf, fd, ret;
    char *dev, *p;
    char *lkpath;

    if ((dev = strchr(devname[0], ':')) != NULL) dev ++;
    else dev = devname[0];
    if ((p = strrchr(dev, '/')) != NULL && sioOpt.lockpath) {
	int pid;
	struct stat st;
	FILE *fp;

	p ++;
	lkpath = Malloc(strlen(sioOpt.lockpath) + strlen(p) + 1);
	sprintf(lkpath, sioOpt.lockpath, p);
	if (!stat(lkpath, &st)) {
	    if ((fp = fopen(lkpath, "r")) != NULL) {
		fscanf(fp, "%d", &pid);
		fclose(fp);
		if (pid < 0 || pid == getpid()
		    || (kill(pid, 0) < 0 && errno == ESRCH)) {
		    unlink(lkpath);
		    goto makelock;
		}
	    }
	    fprintf(stderr, "%s: lock file exists\n", lkpath);
	    Free(lkpath);
	    *status = OPEN_AGAIN;
	    return(-1);
	}
    makelock:
	lockPath = lkpath;
	if (sioOpt.uid) {
	    SuperPrivilege(TRUE);
	    SwitchUser(sioOpt.uid);
	}
	ret = MakeSioLock(getpid());
	if (sioOpt.uid) SuperPrivilege(FALSE);
	if (ret) {
	    Free(lockPath);
	    lockPath = NULL;
	    *status = OPEN_FAILED;
	    return(-1);
	}
    }
    if (sioOpt.uid) {
	SuperPrivilege(TRUE);
	SwitchUser(sioOpt.uid);
    }
    fd = open(dev, O_RDWR | O_NONBLOCK);
    if (sioOpt.uid) SuperPrivilege(FALSE);
    if (fd < 0) {
	if (errno == EACCES || errno == EPERM)
	    ConsoleMsg(MS_E_PERM_DENIED, dev);
	LogError(dev);
	*status = OPEN_FAILED;
	return(-1);
    }
    if ((fdf = fcntl(fd, F_GETFL)) == -1 ||
	fcntl(fd, F_SETFL, fdf & ~O_NONBLOCK) < 0) {
	close(fd);
	*status = OPEN_FAILED;
	return(-1);
    }
    if (ISLOG(LOG_OS)) Logf(LOG_OS, "open(%s)=%d\n", dev, fd);
    tcgetattr(fd, &oldTio);

    newtio = oldTio;
    newtio.c_cflag &=
	~(CS8|CSIZE|CSTOPB|PARENB|PARODD|CLOCAL|CRTSCTS|HUPCL);
    newtio.c_cflag |= CLOCAL|CREAD|(sioOpt.modem ? HUPCL: 0);
    newtio.c_iflag = IGNBRK|IGNPAR;
    newtio.c_oflag = 0;
    newtio.c_lflag = 0;
    newtio.c_cc[VMIN] = 1;
    newtio.c_cc[VTIME] = 0;
    switch (sioOpt.parity) {
    case PARITY_NONE:
	newtio.c_cflag |= CS8;
	break;
    case PARITY_ODD:
	newtio.c_cflag |= (CS7|PARENB|PARODD);
	break;
    case PARITY_EVEN:
	newtio.c_cflag |= (CS7|PARENB);
	break;
    default:
    }
    switch (sioOpt.flow) {
    case FLOW_CRTSCTS:
	newtio.c_cflag |= CRTSCTS;
	break;
    case FLOW_XONXOFF:
	newtio.c_iflag |= (IXON|IXOFF);
	newtio.c_cc[VSTART] = 0x11;
	newtio.c_cc[VSTOP] = 0x13;
	break;
    default:
    }
    pppInfo.minfo = 0;
#if 0
    newtio.c_cflag |= CCTS_OFLOW | CRTS_IFLOW;
    newtio.c_iflag |= IXON;
    newtio.c_iflag &= ~IGNPAR;
    Logf(LOG_OS, "modem (put): iflag = %x, oflag = %x, cflag = %x\n",
	 newtio.c_iflag, newtio.c_oflag, newtio.c_cflag);
#endif
    cfsetispeed(&newtio, sioOpt.speed);
    cfsetospeed(&newtio, sioOpt.speed);
    tcsetattr(fd, TCSAFLUSH, &newtio);
    tcflush(fd, TCIOFLUSH);
    if (!(pppInfo.l_stat & LSTAT_TTY) && sioOpt.modem) {
	*status = DialerStart(fd, argc, argv);
    }
    return(fd);
}

static bool_t
SioProbe(char *devname)
{
    char *dev;
    struct stat st;

    if ((dev = strchr(devname, ':')) != NULL) dev ++;
    else dev = devname;
    return((stat(dev, &st) || !S_ISCHR(st.st_mode)) ? FALSE: TRUE);
}

static void
SioClose(int fd)
{
    DialerClose(fd);
    if (fd >= 0) {
	if (sioOpt.modem) {
	    struct termios tio;

	    tcgetattr(fd, &tio);
	    cfsetispeed(&tio, B0);
	    cfsetospeed(&tio, B0);
	    tcsetattr(fd, TCSAFLUSH, &tio);	/* hang up!! */
	}
	tcsetattr(fd, TCSAFLUSH, &oldTio);
	Close(fd);
    }
    if (lockPath) {
	if (sioOpt.uid) {
	    SuperPrivilege(TRUE);
	    SwitchUser(sioOpt.uid);
	}
	unlink(lockPath);
	if (sioOpt.uid) SuperPrivilege(FALSE);
	Free(lockPath);
	lockPath = NULL;
    }
    pppInfo.minfo = (unsigned int)-1;
}

static int
SioCheck()
{
    static unsigned int minfo;
    extern int devFd;

    if (devFd >= 0) {
	if (ioctl(devFd, TIOCMGET, &minfo) < 0) return(0);
	if (sioOpt.modem && ((minfo ^ pppInfo.minfo) & TIOCM_CD)
	    && !(minfo & TIOCM_CD)) {
	    if (ISLOG(LOG_OS))
		Logf(LOG_OS, "No Carrier(MODEM=%x)\n", minfo);
	    pppInfo.minfo = minfo;
	    minfo = 0;
	    return(-1);
	}
	pppInfo.minfo = minfo;
    }
    return(0);
}

void
SioSetup()
{
    static struct env_s sioenv[]={
	{"SPEED", {EnvSpeed}, ENV_SET, 0, 0, 0666},
	{"FLOW", {EnvFlow}, ENV_SET, 0, 0, 0666},
	{"PARITY", {EnvParity}, ENV_SET, 0, 0, 0666},
	{"MODEM", {EnvModem}, ENV_SET, 0, 0, 0666},
	{"LOCK", {&sioOpt.lockpath}, ENV_STRING|ENV_SECURE|ENV_SECRET,
		0, 0, 0600},
	{"LOGIN", {&sioOpt.loginpath}, ENV_STRING|ENV_SECURE|ENV_SECRET,
		0, 0, 0600},
	{"USER", {EnvUser}, ENV_SET|ENV_SECURE|ENV_SECRET, 0, 0, 0600},
	{NULL}
    };
    static struct device_s dev={
	NULL, SioProbe, SioOpen, AsyncWrite, AsyncRead, SioClose, SioCheck
    };

    RegisterEnvs(sioenv, "SERIAL", NULL);
    RegisterDevice(&dev);
    DialerSetup();
}
