/* Symbol lists. */

#include <stdio.h>
#include <stdlib.h>
#include <obstack.h>
#include "xmalloc.h"
#include "symbol.h"
#include "pprint.h"
#include "formula.h"

#define CLASS_VAR "c"
#define PERMISSION_VAR "p"
#define TYPE_VAR "t"
#define ROLE_VAR "r"
#define USER_VAR "u"
#define NEXT_SUFFIX "'"

/* Allocation using obstack */

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

static struct obstack *stack = 0;

struct obstack *
formula_stack(struct obstack *new)
{
  struct obstack *old = stack;
  if (new)
    stack = new;
  return old;
}

struct symbol_list {
  symbol_t head;
  symbol_list_t tail;
};

struct tran {
  tran_type_t type;
  union {
    symbol_list_t list;
    struct {
      tran_t arg1;
      tran_t arg2;
    } args;
  } u;
};

struct tran_list {
  tran_t head;
  tran_list_t tail;
};

struct arrow {
  tran_t state;
  tran_t action;
  int more;
};

struct arrow_list {
  arrow_t head;
  arrow_list_t tail;
};

struct diagram {
  arrow_list_t arrows;
  tran_t state;
  tran_t except_state;
  tran_t except_action;
};

struct diagram_list {
  diagram_t head;
  diagram_list_t tail;
};

struct lts {
  symbol_list_t types;
  symbol_list_t roles;
  symbol_list_t users;
  symbol_list_t classes;
  symbol_list_t permissions;
  tran_t initial;
  tran_t transition;
  tran_list_t specifications;
};

static struct tran tran_true[] = { { TRAN_TRUE } };
static struct tran tran_false[] = { { TRAN_FALSE } };
static struct tran same_types[] = { { SAME_TYPES } };
static struct tran same_roles[] = { { SAME_ROLES } };
static struct tran same_users[] = { { SAME_USERS } };
static struct tran tran_okay[] = { { TRAN_OKAY } };

static tran_t
mk_tran()
{
  return (tran_t)obstack_alloc(stack, sizeof(struct tran));
}

static tran_t
mk_list(tran_type_t type, symbol_list_t list)
{
  tran_t tran = mk_tran();
  tran->type = type;
  tran->u.list = list;
  return tran;
}

static tran_t
mk_binary(tran_type_t type, tran_t arg1, tran_t arg2)
{
  tran_t tran = mk_tran();
  tran->type = type;
  tran->u.args.arg1 = arg1;
  tran->u.args.arg2 = arg2;
  return tran;
}

/* Constructors */

symbol_list_t
mk_symbol_list(symbol_t symbol, symbol_list_t tail)
{
  symbol_list_t list
    = (symbol_list_t)obstack_alloc(stack, sizeof(struct symbol_list));
  list->head = symbol;
  list->tail = tail;
  return list;
}

tran_t
mk_tran_true()
{
  return tran_true;
}

tran_t
mk_tran_false()
{
  return tran_false;
}

tran_t
mk_tran_classes(symbol_list_t list)
{
  return mk_list(TRAN_CLASSES, list);
}

tran_t
mk_tran_permissions(symbol_list_t list)
{
  return mk_list(TRAN_PERMISSIONS, list);
}

tran_t
mk_tran_types(symbol_list_t list)
{
  return mk_list(TRAN_TYPES, list);
}

tran_t
mk_next_types(symbol_list_t list)
{
  return mk_list(NEXT_TYPES, list);
}

tran_t
mk_same_types()
{
  return same_types;
}

tran_t
mk_tran_roles(symbol_list_t list)
{
  return mk_list(TRAN_ROLES, list);
}

tran_t
mk_next_roles(symbol_list_t list)
{
  return mk_list(NEXT_ROLES, list);
}

tran_t
mk_same_roles()
{
  return same_roles;
}

tran_t
mk_tran_users(symbol_list_t list)
{
  return mk_list(TRAN_USERS, list);
}

tran_t
mk_next_users(symbol_list_t list)
{
  return mk_list(NEXT_USERS, list);
}

tran_t
mk_same_users()
{
  return same_users;
}

tran_t
mk_tran_not(tran_t arg)
{
  tran_t tran = mk_tran();
  tran->type = TRAN_NOT;
  tran->u.args.arg1 = arg;
  return tran;
}

tran_t
mk_tran_and(tran_t arg1, tran_t arg2)
{
  return mk_binary(TRAN_AND, arg1, arg2);
}

tran_t
mk_tran_or(tran_t arg1, tran_t arg2)
{
  return mk_binary(TRAN_OR, arg1, arg2);
}

tran_t
mk_tran_imply(tran_t arg1, tran_t arg2)
{
  return mk_binary(TRAN_IMPLY, arg1, arg2);
}

tran_t
mk_tran_iff(tran_t arg1, tran_t arg2)
{
  return mk_binary(TRAN_IFF, arg1, arg2);
}

tran_t
mk_tran_okay()
{
  return tran_okay;
}

tran_t
mk_tran_ax(tran_t arg)
{
  tran_t tran = mk_tran();
  tran->type = TRAN_AX;
  tran->u.args.arg1 = arg;
  return tran;
}

tran_t
mk_tran_next(tran_t arg)
{
  tran_t tran = mk_tran();
  tran->type = TRAN_NEXT;
  tran->u.args.arg1 = arg;
  return tran;
}

tran_t
mk_tran_until(tran_t arg1, tran_t arg2)
{
  return mk_binary(TRAN_UNTIL, arg1, arg2);
}

tran_list_t
mk_tran_list(tran_t tran, tran_list_t tail)
{
  tran_list_t list
    = (tran_list_t)obstack_alloc(stack, sizeof(struct tran_list));
  list->head = tran;
  list->tail = tail;
  return list;
}

arrow_t
mk_arrow(tran_t state, tran_t action, int more)
{
  arrow_t arrow
    = (arrow_t)obstack_alloc(stack, sizeof(struct arrow));
  arrow->state = state;
  arrow->action = action;
  arrow->more = more;
  return arrow;
}

arrow_list_t
mk_arrow_list(arrow_t arrow, arrow_list_t tail)
{
  arrow_list_t list
    = (arrow_list_t)obstack_alloc(stack, sizeof(struct arrow_list));
  list->head = arrow;
  list->tail = tail;
  return list;
}

diagram_t
mk_diagram(arrow_list_t arrows, tran_t state,
	   tran_t except_state, tran_t except_action)
{
  diagram_t diagram
    = (diagram_t)obstack_alloc(stack, sizeof(struct diagram));
  diagram->arrows = arrows;
  diagram->state = state;
  diagram->except_state = except_state;
  diagram->except_action = except_action;
  return diagram;
}

diagram_list_t
mk_diagram_list(diagram_t diagram, diagram_list_t tail)
{
  diagram_list_t list
    = (diagram_list_t)obstack_alloc(stack, sizeof(struct diagram_list));
  list->head = diagram;
  list->tail = tail;
  return list;
}

lts_t
mk_lts(symbol_list_t types,
       symbol_list_t roles,
       symbol_list_t users,
       symbol_list_t classes,
       symbol_list_t permissions,
       tran_t initial,
       tran_t transition,
       tran_list_t specifications)
{
  lts_t lts = (lts_t)obstack_alloc(stack, sizeof(struct lts));
  lts->types = types;
  lts->roles = roles;
  lts->users = users;
  lts->classes = classes;
  lts->permissions = permissions;
  lts->initial = initial;
  lts->transition = transition;
  lts->specifications = specifications;
  return lts;
}

/* Observers */

symbol_t
symbol_list_head(symbol_list_t list)
{
  return list->head;
}

symbol_list_t
symbol_list_tail(symbol_list_t list)
{
  return list->tail;
}

tran_type_t
tran_type(tran_t tran)
{
  return tran->type;
}

symbol_list_t
tran_list(tran_t tran)
{
  return tran->u.list;
}

tran_t
tran_arg1(tran_t tran)
{
  return tran->u.args.arg1;
}

tran_t
tran_arg2(tran_t tran)
{
  return tran->u.args.arg2;
}

tran_t
tran_list_head(tran_list_t list)
{
  return list->head;
}

tran_list_t
tran_list_tail(tran_list_t list)
{
  return list->tail;
}

tran_t
arrow_state(arrow_t arrow)
{
  return arrow->state;
}

tran_t
arrow_action(arrow_t arrow)
{
  return arrow->action;
}

int
arrow_more(arrow_t arrow)
{
  return arrow->more;
}

arrow_t
arrow_list_head(arrow_list_t list)
{
  return list->head;
}

arrow_list_t
arrow_list_tail(arrow_list_t list)
{
  return list->tail;
}

arrow_list_t
diagram_arrows(diagram_t diagram)
{
  return diagram->arrows;
}

tran_t
diagram_state(diagram_t diagram)
{
  return diagram->state;
}

tran_t
diagram_except_state(diagram_t diagram)
{
  return diagram->except_state;
}

tran_t
diagram_except_action(diagram_t diagram)
{
  return diagram->except_action;
}

diagram_t
diagram_list_head(diagram_list_t list)
{
  return list->head;
}

diagram_list_t
diagram_list_tail(diagram_list_t list)
{
  return list->tail;
}

symbol_list_t
lts_types(lts_t lts)
{
  return lts->types;
}

symbol_list_t
lts_roles(lts_t lts)
{
  return lts->roles;
}

symbol_list_t
lts_users(lts_t lts)
{
  return lts->users;
}

symbol_list_t
lts_classes(lts_t lts)
{
  return lts->classes;
}

symbol_list_t
lts_permissions(lts_t lts)
{
  return lts->permissions;
}

tran_t
lts_initial(lts_t lts)
{
  return lts->initial;
}

tran_t
lts_transition(lts_t lts)
{
  return lts->transition;
}

tran_list_t
lts_specifications(lts_t lts)
{
  return lts->specifications;
}

/* Predicates */

int is_state(tran_t tran)
{
  switch (tran_type(tran)) {
  case TRAN_TRUE:
  case TRAN_FALSE:
  case TRAN_TYPES:
  case TRAN_ROLES:
  case TRAN_USERS:
    return 1;
  case TRAN_NOT:
    return is_state(tran_arg1(tran));
  case TRAN_AND:
  case TRAN_OR:
  case TRAN_IMPLY:
  case TRAN_IFF:
    return is_state(tran_arg1(tran))
      && is_state(tran_arg2(tran));
  default:
    return 0;
  }
}

/* Utilities */

symbol_list_t symbol_list_reverse(symbol_list_t list)
{
  symbol_list_t rest;
  if (!list)
    return list;
  rest = list->tail;
  list->tail = 0;
  while (rest) {
    symbol_list_t temp = rest;
    rest = rest->tail;
    temp->tail = list;
    list = temp;
  }
  return list;
}

tran_list_t tran_list_reverse(tran_list_t list)
{
  tran_list_t rest;
  if (!list)
    return list;
  rest = list->tail;
  list->tail = 0;
  while (rest) {
    tran_list_t temp = rest;
    rest = rest->tail;
    temp->tail = list;
    list = temp;
  }
  return list;
}

arrow_list_t arrow_list_reverse(arrow_list_t list)
{
  arrow_list_t rest;
  if (!list)
    return list;
  rest = list->tail;
  list->tail = 0;
  while (rest) {
    arrow_list_t temp = rest;
    rest = rest->tail;
    temp->tail = list;
    list = temp;
  }
  return list;
}

diagram_list_t diagram_list_reverse(diagram_list_t list)
{
  diagram_list_t rest;
  if (!list)
    return list;
  rest = list->tail;
  list->tail = 0;
  while (rest) {
    diagram_list_t temp = rest;
    rest = rest->tail;
    temp->tail = list;
    list = temp;
  }
  return list;
}

/* Pretty printers */

#define INDENT 2
#define MARGIN 72

static pretty_t
pset_aux(symbol_t sym, symbol_list_t list)
{
  pretty_t next;
  if (list) {
    next = pset_aux(symbol_list_head(list),
		    symbol_list_tail(list));
    next = pbreak(1, next);
    next = pstring(",", next);
  }
  else
    next = pstring("}", PPRINT_NULL);
  return pstring(symbol_name(sym), next);
}

static pretty_t
pset(symbol_list_t list, pretty_t next)
{
  if (!list)
    return pstring("{}", next);
  else {
    pretty_t p = pset_aux(symbol_list_head(list),
			  symbol_list_tail(list));
    return pblock(1, pstring("{", p), next);
  }
}

static pretty_t
pin(const char *var, symbol_list_t list, int neg, pretty_t next)
{
  if (!list)
    return neg ? pstring("TRUE", next) : pstring("FALSE", next);
  else if (!symbol_list_tail(list)) {
    next = pstring(symbol_name(symbol_list_head(list)), next);
    if (neg)
      next = pstring(" != ", next);
    else
      next = pstring(" = ", next);
  }
  else {
    next = pset(list, next);
    if (neg)
      next = pstring(" !: ", next);
    else
      next = pstring(" : ", next);
  }
  return pstring(var, next);
}

static pretty_t
psame(const char *var, int neg, pretty_t next)
{
  next = pstring(NEXT_SUFFIX, next);
  next = pstring(var, next);
  if (neg)
    next = pstring(" != ", next);
  else
    next = pstring(" = ", next);
  next = pstring(var, next);
  return next;
}

typedef enum {MIN_PREC, IMPLY, IFF, OR, AND, NOT, MAX_PREC} prec_t;

static prec_t
prec(tran_t tran)
{
  switch (tran_type(tran)) {
  case TRAN_NOT:
    return NOT;
  case TRAN_AND:
    return AND;
  case TRAN_OR:
    return OR;
  case TRAN_IFF:
    return IFF;
  case TRAN_IMPLY:
    return IMPLY;
  default:
    return MAX_PREC;
  }
}

typedef enum {ASSOC, LEFT, RIGHT, NON_ASSOC} assoc_t;

static assoc_t
assoc(tran_t tran)
{
  switch (tran_type(tran)) {
  case TRAN_AND:
  case TRAN_OR:
    return ASSOC;
  case TRAN_IMPLY:
    return RIGHT;
  case TRAN_IFF:
    return LEFT;
  default:
    return NON_ASSOC;
  }
}

static prec_t
arg_prec(assoc_t side, assoc_t assoc, prec_t prec)
{
  if (side == assoc || assoc == ASSOC)
    return prec;
  else
    return prec + 1;
}

static pretty_t
ptran(tran_t tran, int neg, int level, pretty_t next);

static pretty_t
pbinary(tran_t tran, prec_t prec, const char *op, pretty_t next)
{
  assoc_t tran_assoc = assoc(tran);
  next = ptran(tran_arg2(tran), 0, arg_prec(RIGHT, tran_assoc, prec), next);
  next = pstring(op, next);
  next = pbreak(1, next);
  next = ptran(tran_arg1(tran), 0, arg_prec(LEFT, tran_assoc, prec), next);
  return next;
}

static pretty_t
ptran(tran_t tran, int neg, int level, pretty_t next)
{
  int tran_prec = prec(tran);
  if (level > tran_prec) {
    pretty_t p = pstring(" )", PPRINT_NULL);
    p = ptran(tran, 0, MIN_PREC, p);
    if (neg)
      p = pstring("!(", p);
    else
      p = pstring("( ", p);
    return pblock(INDENT, p, next);
  }
  else {
    switch (tran_type(tran)) {
    case TRAN_TRUE:
      return neg ? pstring("FALSE", next) : pstring("TRUE", next);
    case TRAN_FALSE:
      return neg ? pstring("TRUE", next) : pstring("FALSE", next);
    case TRAN_CLASSES:
      return pin(CLASS_VAR, tran_list(tran), neg, next);
    case TRAN_PERMISSIONS:
      return pin(PERMISSION_VAR, tran_list(tran), neg, next);
    case TRAN_TYPES:
      return pin(TYPE_VAR, tran_list(tran), neg, next);
    case NEXT_TYPES:
      return pin(TYPE_VAR NEXT_SUFFIX, tran_list(tran), neg, next);
    case SAME_TYPES:
      return psame(TYPE_VAR, neg, next);
    case TRAN_ROLES:
      return pin(ROLE_VAR, tran_list(tran), neg, next);
    case NEXT_ROLES:
      return pin(ROLE_VAR NEXT_SUFFIX, tran_list(tran), neg, next);
    case SAME_ROLES:
      return psame(ROLE_VAR, neg, next);
    case TRAN_USERS:
      return pin(USER_VAR, tran_list(tran), neg, next);
    case NEXT_USERS:
      return pin(USER_VAR NEXT_SUFFIX, tran_list(tran), neg, next);
    case SAME_USERS:
      return psame(USER_VAR, neg, next);
    case TRAN_NOT:
      return ptran(tran_arg1(tran), !neg, tran_prec, next);
    case TRAN_AND:
      /* assert neg is zero */
      return pbinary(tran, tran_prec, "& ", next);
    case TRAN_OR:
      /* assert neg is zero */
      return pbinary(tran, tran_prec, "| ", next);
    case TRAN_IMPLY:
      /* assert neg is zero */
      return pbinary(tran, tran_prec, "-> ", next);
    case TRAN_IFF:
      /* assert neg is zero */
      return pbinary(tran, tran_prec, "<-> ", next);
    default:
      abort();
    }
  }
}

int
print_transition(FILE *out, tran_t tran)
{
  pretty_t p = ptran(tran, 0, MIN_PREC, PPRINT_NULL);
  return pprint(out, p, MARGIN);
}

static void
emit_comment(FILE *out, const char *title)
{
  fprintf(out, "\n# %s\n", title);
}

static void
emit_section_start(FILE *out, const char *name)
{
  fprintf(out, "\n %s\n\n", name);
}

static void
emit_state_var(FILE *out, const char *name, symbol_list_t list)
{
  pretty_t p = pset(list, PPRINT_NULL);
  p = pbreak(1, p);
  p = pstring(":", p);
  p = pstring(name, p);
  p = pstring(" STATE ", p);
  p = pblock(INDENT, p, PPRINT_NULL);
  pprint(out, p, MARGIN);
  fputc('\n', out);
}

static void
emit_action_var(FILE *out, const char *name, symbol_list_t list)
{
  pretty_t p = pset(list, PPRINT_NULL);
  p = pbreak(1, p);
  p = pstring(":", p);
  p = pstring(name, p);
  p = pstring(" ACTION ", p);
  p = pblock(INDENT, p, PPRINT_NULL);
  pprint(out, p, MARGIN);
  fputc('\n', out);
}

static void
emit_decls(FILE *out, lts_t lts)
{
  emit_state_var(out, TYPE_VAR, lts_types(lts));
  emit_state_var(out, ROLE_VAR, lts_roles(lts));
  emit_state_var(out, USER_VAR, lts_users(lts));
  emit_action_var(out, CLASS_VAR, lts_classes(lts));
  emit_action_var(out, PERMISSION_VAR, lts_permissions(lts));
}

void
print_lts(FILE *out, lts_t lts)
{
  tran_list_t list = lts_specifications(lts);
  emit_decls(out, lts);

  emit_comment(out, "Initial States");
  emit_section_start(out, "INIT");
  print_transition(out, lts_initial(lts));
  fputc('\n', out);

  emit_comment(out, "Transition Relation");
  emit_section_start(out, "TRAN");
  print_transition(out, lts_transition(lts));
  fputc('\n', out);

  if (list)
    emit_comment(out, "Specifications");
  for(; list; list = tran_list_tail(list)) {
    emit_section_start(out, "SPEC");
    print_transition(out, tran_list_head(list));
    fputc('\n', out);
  }
}
