/**************************************************************/
/* ********************************************************** */
/* *                                                        * */
/* *             PROOF GENERATOR MAIN                       * */
/* *                                                        * */
/* *  Copyright (C) 1998, 1999, 2000, 2001                  * */
/* *  MPI fuer Informatik                                   * */
/* *                                                        * */
/* *  This program is free software; you can redistribute   * */
/* *  it and/or modify it under the terms of the GNU        * */
/* *  General Public License as published by the Free       * */
/* *  Software Foundation; either version 2 of the License, * */
/* *  or (at your option) any later version.                * */
/* *                                                        * */
/* *  This program is distributed in the hope that it will  * */
/* *  be useful, but WITHOUT ANY WARRANTY; without even     * */
/* *  the implied warranty of MERCHANTABILITY or FITNESS    * */
/* *  FOR A PARTICULAR PURPOSE.  See the GNU General Public * */
/* *  License for more details.                             * */
/* *                                                        * */
/* *  You should have received a copy of the GNU General    * */
/* *  Public License along with this program; if not, write * */
/* *  to the Free Software Foundation, Inc., 59 Temple      * */
/* *  Place, Suite 330, Boston, MA  02111-1307  USA         * */
/* *                                                        * */
/* $Revision: 1.10 $                                         * */
/* $State: Exp $                                            * */
/* $Date: 2002/02/17 16:44:01 $                             * */
/* $Author: weidenb $                                       * */
/* *                                                        * */
/* *                                                        * */
/* *             Contact:                                   * */
/* *             Christoph Weidenbach                       * */
/* *             MPI fuer Informatik                        * */
/* *             Stuhlsatzenhausweg 85                      * */
/* *             66123 Saarbruecken                         * */
/* *             Email: weidenb@mpi-sb.mpg.de               * */
/* *             Germany                                    * */
/* *                                                        * */
/* ********************************************************** */

/* $RCSfile: pgen.c,v $ */

/**************************************************************

   Given a dfg proof file, this program generates
   a set of proof files corresponding to each proof step.

	Options:

	-d    Prefix of generated files
	-t    time limit for proof attempt
	      (to set the TimeLimit option in the
	       file passed to SPASS)
	-s    file suffix of generated proof files
	-q    quiet mode

**************************************************************/

#include "proofcheck.h"

extern int   pcheck_Timelimit;
extern const char* pcheck_ProofFileSuffix;
extern BOOL  pcheck_Quiet;

extern BOOL pcheck_ClauseCg;
extern BOOL pcheck_GenNamedCg, pcheck_GenRedCg;
extern const char *pcheck_CgName, *pcheck_RedCgName;
extern GRAPHFORMAT pcheck_GraphFormat;

#define PGEN__VERSION "1.9"

int main(int argc, const char* argv[])
{
  /* option defaults */
  /* time allowed for each spass proof check */
  const int  DefaultTimeLimit = 3;
  /* file suffix of generated proof files */
  const char *const DefaultProofFileSuffix = ".dfg";

  LIST       Proof;            /* proof with DFG terms */
  LIST       ConvertedProof;   /* proof with clauses   */
  char       *Filename, *Path;
  const char *FullFilename, *Prefix;
  int        i;
  BOOL       PrefixDefined;
  FILE*      ProofFileHandle;
  TABLEAU    Tableau;
  FLAGSTORE  Flags;
  PRECEDENCE Precedence;

  OPTID   Timelimit_OptId; /* set time limit in SPASS flags in generated files */
  OPTID   Suffix_OptId;    /* suffix of generated files */
  OPTID   Quiet_OptId;     /* do not print any comments on progress */
  OPTID   Prefix_OptId;    /* prefix of generated files */
  OPTID   JustCg_OptId;    /* do not generate proof files */
  BOOL    pcheck_JustCg;   /* do not generate any proof files */

  OPTID GenRedCg_OptId;    /* write the reduced tableau to a named file */
  OPTID GenNamedCg_OptId;  /* write the (non-reduced) tableau to a named file */
  OPTID ClauseCg_OptId;    /* do not show clauses in tableau output */
  OPTID XvcgFormat_OptId;  /* print tableau in xvcg format */
  OPTID DaVinciFormat_OptId; /* print tableau in daVinci format */

  /* Initalization */
  memory_Init(memory__UNLIMITED);
  symbol_Init(TRUE);
  stack_Init();
  term_Init();
  flag_Init();

  Flags = flag_CreateStore();
  flag_InitStoreByDefaults(Flags);
  Precedence = symbol_CreatePrecedence();

  fol_Init(TRUE, Precedence);
  clause_Init();
  cont_Init();
  unify_Init();
  vec_Init();
  opts_Init();


  /* Declare options, set to defaults and read command line */
  /* See *_OptId variable declarations for option semantics */
  Timelimit_OptId = opts_Declare("t", opts_REQARGTYPE);
  Suffix_OptId    = opts_Declare("s", opts_REQARGTYPE);
  Prefix_OptId    = opts_Declare("d", opts_REQARGTYPE);
  Quiet_OptId     = opts_Declare("q", opts_NOARGTYPE);
  JustCg_OptId    = opts_Declare("j", opts_NOARGTYPE);

  GenNamedCg_OptId    = opts_Declare("n", opts_REQARGTYPE);
  GenRedCg_OptId      = opts_Declare("r", opts_REQARGTYPE);
  ClauseCg_OptId      = opts_Declare("c", opts_NOARGTYPE);
  XvcgFormat_OptId    = opts_Declare("xvcg", opts_NOARGTYPE);
  DaVinciFormat_OptId = opts_Declare("vinci", opts_NOARGTYPE);
  
  pcheck_ProofFileSuffix = DefaultProofFileSuffix;
  pcheck_Timelimit       = DefaultTimeLimit;
  pcheck_Quiet           = FALSE;
  Prefix                 = NULL;
  PrefixDefined          = FALSE;
  pcheck_JustCg          = FALSE;

  pcheck_GenNamedCg      = FALSE;
  pcheck_GenRedCg        = FALSE;
  pcheck_ClauseCg        = TRUE;
  pcheck_GraphFormat     = XVCG;

  /* translate command line options into flags */
  if (!opts_Read(argc, argv))
    return 0;
  if (opts_GetValue(Prefix_OptId, &Prefix))
    PrefixDefined = TRUE;
  if (opts_IsSet(Timelimit_OptId)) {
    if (!opts_GetIntValue(Timelimit_OptId, &pcheck_Timelimit)) {
      misc_StartUserErrorReport();
      misc_UserErrorReport("\n Time limit given to pgen must be an integer.\n");
      misc_FinishUserErrorReport();
    }
  }
  if (opts_IsSet(Suffix_OptId))
    opts_GetValue(Suffix_OptId, &pcheck_ProofFileSuffix);
  pcheck_Quiet  = opts_IsSet(Quiet_OptId);
  pcheck_JustCg = opts_IsSet(JustCg_OptId);
  
  pcheck_GenNamedCg = opts_GetValue(GenNamedCg_OptId, &pcheck_CgName);
  pcheck_GenRedCg   = opts_GetValue(GenRedCg_OptId, &pcheck_RedCgName);
  if (opts_IsSet(ClauseCg_OptId))
    pcheck_ClauseCg = FALSE;
  if (opts_IsSet(DaVinciFormat_OptId))
    pcheck_GraphFormat = DAVINCI;
  if (opts_IsSet(XvcgFormat_OptId))
    pcheck_GraphFormat = XVCG;

  /* get file name argument */
  if (opts_Indicator() > argc-1) {
    /* print options */
    fputs("\n\t         pgen Version ", stdout);
    fputs(PGEN__VERSION, stdout);
    puts("\nUsage: pgen [ -dstqjnr] [-vinci -xvcg] filename");
    puts("\nSee the man page or the postscript documentation for more details.");
    return EXIT_FAILURE;
  }
  FullFilename = argv[opts_Indicator()];
  
  /* derive prefix of proof file from filename */
  /* extract pathname, append prefix parameter, if given. */
  for (i = strlen(FullFilename)-1; i >= 0 && FullFilename[i] != '/' ; i--)
    /* empty */;
  if (i == 0)
    i = -1;
  Path     = string_Prefix(FullFilename,i+1);
  Filename = string_Suffix(FullFilename, i+1);   /* extract file name */
  if (!PrefixDefined)
    Prefix = Path;

  /* read and check proof */
  ProofFileHandle = misc_OpenFile(FullFilename, "r");
  Proof           = dfg_ProofParser(ProofFileHandle, Flags, Precedence);
  misc_CloseFile(ProofFileHandle, FullFilename);

  if (list_Empty(Proof)) {
    misc_StartUserErrorReport();
    misc_UserErrorReport("\n Proof given to pgen is empty.\n");
    misc_FinishUserErrorReport();
  }

  /* Create clauses from the terms */
  ConvertedProof = pcheck_ConvertTermListToClauseList(Proof, Flags, Precedence);

  if (!pcheck_BuildTableauFromProof(ConvertedProof, &Tableau)) {
    misc_StartUserErrorReport();
    misc_UserErrorReport("\n Could not built a closed tableau from the proof.\n");
    misc_FinishUserErrorReport();
  }
  if (pcheck_TableauProof(&Tableau, ConvertedProof)) {
    if (!pcheck_JustCg) {
      pcheck_TableauToProofTask(Tableau, Filename, Prefix);
      puts("\nAll files generated!");
    }
  }
  else
    puts("\nProof not correct.");
  
  tab_Delete(Tableau);
  clause_DeleteClauseList(ConvertedProof);

  /* clean up  */

  string_StringFree(Filename);
  string_StringFree(Path);
  pcheck_DeleteProof(Proof);

  flag_DeleteStore(Flags);
  symbol_DeletePrecedence(Precedence);

  opts_Free();
  cont_Free();
  unify_Free();
  fol_Free();
  symbol_FreeAllSymbols();
  dfg_Free();

#ifdef CHECK
  memory_Print();
  putchar('\n');
#endif

 return 0;
}


