// ti994.c - TI-99/4 Emulator
//
// Copyright (c) 2001-2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "ti99/ti994.h"

// TI-99/4(A) Memory Map
//
//       +-------------------------------------+
// >0000 |            Console ROM              |
//       +-------------------------------------+
// >2000 |        Low Memory Expansion         |
//       +-------------------------------------+
// >4000 |       Device Service Routine        |
//       +-------------------------------------+
// >6000 |          Module Expansion           |
//       +-------------------------------------+
// >8000 |          PAD RAM and I/O            |
//       +-------------------------------------+
// >A000 |       High Memory Expansion         |
//       +-------------------------------------+

uint8 ti994_ReadGROM(TI994_CPU *ti99cpu, uint16 addr)
{
	uint8 data;

	if (addr & 1)
		return 0;

	if (addr & 2) {
		ti99cpu->grmAddr = ((ti99cpu->grmAddr + 1) & 0x1FFF) |
			(ti99cpu->grmAddr & 0xE000);
		data             = ti99cpu->grmAddr >> 8;
		ti99cpu->grmAddr = ti99cpu->grmAddr << 8;
#ifdef DEBUG
//		dbg_Printf("%s: (R) GROM Address %04X => %02X\n",
//			ti99cpu->cpu.devName, ti99cpu->grmAddr, data);
#endif /* DEBUG */
	} else {
		data = ti99cpu->gplMemory[ti99cpu->grmAddr];
#ifdef DEBUG
//		dbg_Printf("%s: (R) GROM Data (>%04X) => %02X\n",
//			ti99cpu->cpu.devName, ti99cpu->grmAddr, data);
#endif /* DEBUG */
		ti99cpu->grmAddr = ((ti99cpu->grmAddr + 1) & 0x1FFF) |
			(ti99cpu->grmAddr & 0xE000);
	}
	return data;
}

void ti994_WriteGROM(TI994_CPU *ti99cpu, uint16 addr, uint8 data)
{
	if (addr & 1)
		return;

	if (addr & 2) {
		ti99cpu->grmAddr = (ti99cpu->grmAddr << 8) | data;
#ifdef DEBUG
//		dbg_Printf("%s: (W) GROM Address %04X <= %02X\n",
//			ti99cpu->cpu.devName, ti99cpu->grmAddr, data);
#endif /* DEBUG */
	}
}

uint16 ti994_ReadW(TI994_CPU *ti99cpu, uint16 addr)
{
	switch (addr & 0xE000) {
		case 0x0000: // Console ROM
			return ti99cpu->ROM[addr >> 1];

		case 0x8000:
			switch (addr & 0xFC00) {
				case 0x8000: // PAD RAM (>8300 - >83FF)
					return ti99cpu->padRAM[(addr & 0xFF) >> 1];
				case 0x9800: // GROM - Read Access
					return ti994_ReadGROM(ti99cpu, addr) << 8;
			}
			return 0;

		default:
			return ti99cpu->cpuMemory[addr >> 1];
	}
}

void ti994_WriteW(TI994_CPU *ti99cpu, uint16 addr, uint16 data)
{
	switch (addr & 0xE000) {
		case 0x0000: // Console ROM
			return;

		case 0x8000: //
			switch (addr & 0xFC00) {
				case 0x8000: // PAD RAM (>8300 to >83FF)
					ti99cpu->padRAM[(addr & 0xFF) >> 1] = data;
					return;
				case 0x9C00: // GROM - Read Access
					return ti994_WriteGROM(ti99cpu, addr, data >> 8);
			}
			return;

		default:
			ti99cpu->cpuMemory[addr >> 1] = data;
			return;
	}
}

#define GETBYTE(addr, mem) \
	((mem) >> ((addr & 1) ? 0 : 8))
#define PUTBYTE(addr, mem, data) \
	if (addr & 1)  (mem) = (data)      | (mem & 0xFF00); \
	else           (mem) = (data << 8) | (mem & 0x00FF);

uint8 ti994_ReadB(TI994_CPU *ti99cpu, uint16 addr)
{
	switch (addr & 0xE000) {
		case 0x0000: // Console ROM
			return GETBYTE(addr, ti99cpu->ROM[addr >> 1]);

		case 0x8000:
			switch (addr & 0xFC00) {
				case 0x8000: // PAD RAM (>8300 - >83FF)
					addr &= 0xFF;
					return GETBYTE(addr, ti99cpu->padRAM[addr >> 1]);
				case 0x9800: // GROM - Read Access
					return ti994_ReadGROM(ti99cpu, addr);
			}
			return 0;

		default:
			return GETBYTE(addr, ti99cpu->cpuMemory[addr >> 1]);
	}
}

void ti994_WriteB(TI994_CPU *ti99cpu, uint16 addr, uint8 data)
{
	switch (addr & 0xE000) {
		case 0x0000: // Console ROM
			return;

		case 0x8000:
			switch (addr & 0xFC00) {
				case 0x8000:
					addr &= 0xFF;
					PUTBYTE(addr, ti99cpu->padRAM[addr >> 1], data);
					return;
				case 0x9C00: // GROM - Read Access
					return ti994_WriteGROM(ti99cpu, addr, data);
			}
			return;

		default:
			PUTBYTE(addr, ti99cpu->cpuMemory[addr >> 1], data);
			return;
	}
}

uint16 ti994_ReadCRU(TI994_CPU *ti99cpu, uint16 addr)
{
}

void ti994_WriteCRU(TI994_CPU *ti99cpu, uint16 addr, uint16 data)
{
}

// ********************************************************

void ti994_SystemTimer(void *dptr)
{
	register TI994_CPU *ti99cpu = (TI994_CPU *)dptr;

	if (ti99cpu->TickClock++ >= TI99_SECOND) {
		ti99cpu->TickClock = 0;

#ifdef DEBUG
		if (dbg_Check(DBG_IPS))
			dbg_Printf("%s: %d ips\n", ti99cpu->cpu.Unit.devName, IPS);
#else
		printf("%s: %d ips\n", ti99cpu->cpu.Unit.devName, IPS);
#endif /* DEBUG */

		IPS = 0;
	}
}

void ti994_EnableTimer(register TI994_CPU *ti99cpu)
{
	ti99cpu->TickClock          = 0;
	ti99cpu->SystemTimer.Flags |= CLK_ENABLE;
}

void ti994_DisableTimer(register TI994_CPU *ti99cpu)
{
	ti99cpu->SystemTimer.Flags &= ~CLK_ENABLE;
}

// ********************************************************

void *ti994_Create(MAP_DEVICE *newMap, int argc, char **argv)
{
	TI99_SYSTEM *ti99sys;
	TI994_CPU   *ti99cpu;
	CLK_QUEUE   *timer;

	if (ti99cpu = (TI994_CPU *)calloc(1, sizeof(TI994_CPU))) {
		ti99cpu->cpu.Unit.devName    = newMap->devName;
		ti99cpu->cpu.Unit.keyName    = newMap->keyName;
		ti99cpu->cpu.Unit.emuName    = newMap->emuName;
		ti99cpu->cpu.Unit.emuVersion = newMap->emuVersion;

		ti99cpu->cpuMemory = (uint16 *)calloc(1, 65536);
		ti99cpu->gplMemory = (uint8 *)calloc(1, 65536);

		// Set up function calls
		ReadW    = ti994_ReadW;
		WriteW   = ti994_WriteW;
		ReadB    = ti994_ReadB;
		WriteB   = ti994_WriteB;
		ReadCRU  = ti994_ReadCRU;
		WriteCRU = ti994_WriteCRU;

		EnableTimer  = ti994_EnableTimer;
		DisableTimer = ti994_DisableTimer;

		// Set up a link between CPU and System device.
		ti99sys = (TI99_SYSTEM *)newMap->devParent->Device;
		ti99cpu->System    = ti99sys;
		ti99sys->Processor = (TI99_CPU *)ti99cpu;

		// Set up a system timer
		timer           = &ti99cpu->SystemTimer;
		timer->Next     = NULL;
		timer->Flags    = CLK_REACTIVE;
		timer->outTimer = TI99_TICK;
		timer->nxtTimer = TI99_TICK;
		timer->Device   = ti99cpu;
		timer->Execute  = ti994_SystemTimer;

		// Now enable system timer.
		ts10_SetRealTimer(timer);

		// Build opcode table for execution.
		ti99_BuildCPU((TI99_CPU *)ti99cpu, 0);

		// Finally, set up device identification and return.
		newMap->Device = ti99cpu;
	}
	return ti99cpu;
}

int ti994_Boot(MAP_DEVICE *map, int argc, char **argv)
{
	register TI994_CPU *ti99cpu = (TI994_CPU *)map->Device;

	// Power-On State - Load WP and PC register.
	WP = ReadW(ti99cpu, 0);
	PC = ReadW(ti99cpu, 2);

	printf("%s: Load WP = >%04X, PC = >%04X\n",
		ti99cpu->cpu.Unit.devName, WP, PC);

	// Enable TMS9900 to run.
	ti99cpu->cpu.State = TI99_RUN;
	emu_State          = TI99_RUN;

	return TI99_OK;
}

extern COMMAND ti994_Commands[];
//extern COMMAND ti994_SetCommands[];
//extern COMMAND ti994_ShowCommands[];

DEVICE ti99_System_994 =
{
	TI994_KEY,
	TI994_NAME,
	TI994X_VERSION,
	NULL,
	DF_USE,
	DT_PROCESSOR,

	ti994_Commands, NULL, NULL,

	// Function Calls
	ti994_Create,    // Create Routine
	NULL,            // Configure Routine
	NULL,            // Delete Routine
	NULL,            // Reset Routine
	NULL,            // Attach Routine
	NULL,            // Detach Routine
	NULL,            // Info Routine
	ti994_Boot,      // Boot Routine
	ti99_Execute,    // Execute Routine
#ifdef DEBUG
	NULL,            // Debug Routine
#endif /* DEBUG */
};

DEVICE ti99_System_994A =
{
	TI994A_KEY,      // Key (Device Type) Name
	TI994A_NAME,     // Emulator Name
	TI994X_VERSION,  // Emulator Version
	NULL,            // Listing of Devices
	DF_USE,          // Device Flags
	DT_PROCESSOR,    // Device Type

	// Commands
	ti994_Commands, NULL, NULL,

	// Function Calls
	ti994_Create,    // Create Routine
	NULL,            // Configure Routine
	NULL,            // Delete Routine
	NULL,            // Reset Routine
	NULL,            // Attach Routine
	NULL,            // Detach Routine
	NULL,            // Info Routine
	ti994_Boot,      // Boot Routine
	ti99_Execute,    // Execute Routine
#ifdef DEBUG
	NULL,            // Debug Routine
#endif /* DEBUG */
};
