// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "configuration.h"
#include "icons.h"
#include "decoder.h"
#include "colorbutton.h"
#include "mq3.h"

#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qapplication.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qspinbox.h>
#include <qslider.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qfiledialog.h>
#include <qsettings.h>
#include <qradiobutton.h>
#include <qwidgetstack.h>

#include <stdio.h>


Configuration::Configuration(QObject *parent, const char *name)
    : QObject(parent, name)
{
}

Configuration::~Configuration()
{
}

void Configuration::init()
{
    {   // read in the icon theme paths first
	QSettings settings;
	QStringList paths = settings.readListEntry("/MQ3/iconThemePath", ':');
	if (! paths.isEmpty())
	    IconLoader::addPaths(paths);

	// and the global icon theme
	QString theme = settings.readEntry("/MQ3/iconTheme");
	if (theme.isEmpty())
	    theme = "default";
	IconLoader::setTheme(theme);
    }

    emit configChanged();
}


void Configuration::show()
{
    RealConfigDialog cd( qApp->activeWindow() );
    cd.init();

    if (cd.exec() == QDialog::Accepted) {
	cd.write();
	emit configChanged();
    }
}


RealConfigDialog::RealConfigDialog(QWidget *parent)
    : ConfigDialog(parent, "real config dialog", TRUE), visualPrefs( 0 )
{
    connect(pathAdd, SIGNAL(clicked()), SLOT(addPath()));
    connect(pathRem, SIGNAL(clicked()), SLOT(removePath()));
    connect(visualMode, SIGNAL(activated(const QString &)),
	    SLOT(changeVisualPrefs(const QString &)));

    visualGroupBox->setColumns( 1 );
    visualGroupBox->setOrientation( Horizontal );
}

void RealConfigDialog::init()
{
    // read current config into the dialog
    QSettings settings;

    playOnStartup->
	setChecked(settings.readBoolEntry("/MQ3/playOnStartup", false));
    useLargeIcons->
	setChecked(settings.readBoolEntry("/MQ3/useLargeIcons", false));

    QString val;
    val = settings.readEntry( "/MQ3/Visual/frameShape", "Boxed Panel" );
    {
	// find the current shape
	bool found = FALSE;
	for ( int i = 0; i < frameShape->count(); i++ ) {
	    if ( frameShape->text( i ) == val ) {
		frameShape->setCurrentItem( i );
		found = TRUE;
		break;
	    }
	}
	if ( ! found )
	    frameShape->setCurrentItem( 2 ); // Boxed Panel
    }
    val = settings.readEntry( "/MQ3/Visual/frameStyle", "Sunken" );
    {
	// find the current style
	bool found = FALSE;
	for ( int i = 0; i < frameStyle->count(); i++ ) {
	    if ( frameStyle->text( i ) == val ) {
		frameStyle->setCurrentItem( i );
		found = TRUE;
		break;
	    }
	}
	if ( ! found )
	    frameStyle->setCurrentItem( 1 ); // Sunken

    }
    val = settings.readEntry("/MQ3/Visual/mode");
    visualMode->insertStringList( MainVisual::visuals() );
    {
	// find the currently configured visual
	bool found = FALSE;
	for ( int i = 0; i < visualMode->count(); i++ ) {
	    if ( visualMode->text( i ) == val ) {
		visualMode->setCurrentItem( i );
		found = TRUE;
		break;
	    }
	}
	if ( ! found )
	    visualMode->setCurrentItem( 0 ); // None
    }
    framerate->setValue(settings.readNumEntry("/MQ3/Visual/frameRate", 20));
    val = settings.readEntry( "/MQ3/Visual/backColor" );
    if ( ! val.isEmpty() )
	bgColor->setColor( QColor( val ) );
    else
	bgColor->setColor( MQ3::instance()->backgroundColor() );

    val = settings.readEntry("/MQ3/Organizer/playMode").lower();
    int pm = (Organizer::Normal);
    if (val == "normal")
	pm = (Organizer::Normal);
    else if (val == "random")
	pm = (Organizer::Random);
    else if (val == "onesong")
	pm = (Organizer::OneSong);
    else if (val == "repeatall")
	pm = (Organizer::RepeatAll);
    else if (val == "repeatone")
	pm = (Organizer::RepeatOne);
    playmode->setCurrentItem(pm);

    labelSpeed->setValue(settings.readNumEntry("/MQ3/scrollSpeed", 50));
    stopDelay->setValue(settings.readNumEntry("/MQ3/scrollStopDelay", 50));
    if (settings.readBoolEntry("/MQ3/showRemainingTime", false))
	remainingtime->setChecked(true);
    else
	elapsedtime->setChecked(true);
    oShowOnStartup->
	setChecked(settings.readBoolEntry("/MQ3/Organizer/showOnStartup", false));
    eShowOnStartup->
	setChecked(settings.readBoolEntry("/MQ3/Equalizer/showOnStartup", false));
    oSaveChanges->
	setChecked(settings.readBoolEntry("/MQ3/Organizer/saveChangesOnExit",
					  false));
    oUseLargeIcons->
	setChecked(settings.readBoolEntry("/MQ3/Organizer/useLargeIcons", false));
    oTextInToolbars->
	setChecked(settings.readBoolEntry("/MQ3/Organizer/useTextLabels", false));
    oSearchRegexp->
	setChecked(settings.readBoolEntry("/MQ3/Organizer/regexpSearching", false));
    eSaveOnExit->
	setChecked(settings.readBoolEntry("/MQ3/Equalizer/saveChangesOnExit",
					  false));
    eEnabled->
	setChecked(settings.readBoolEntry("/MQ3/Equalizer/enabled", false));
    iconTheme->insertStringList(IconLoader::themes());
    iconTheme->setCurrentItem(iconTheme->findItem(IconLoader::theme()));
    themeSearch->insertStringList(IconLoader::paths());
    decoderlist->insertStringList(Decoder::all());
    obufsize->setValue(settings.readNumEntry("/MQ3/Output/bufferSize", 128));

    // now, create the visual prefs for the current visual
    visualPrefs = MainVisual::createPrefs( visualMode->currentText(),
				     visualGroupBox, "visual options" );
    if ( visualPrefs )
	visualPrefs->init( settings );
}

void RealConfigDialog::write()
{
    QSettings settings;
    settings.writeEntry("/MQ3/playOnStartup",
			playOnStartup->isChecked());
    settings.writeEntry("/MQ3/useLargeIcons",
			useLargeIcons->isChecked());
    settings.writeEntry("/MQ3/showRemainingTime",
			remainingtime->isChecked());
    settings.writeEntry( "/MQ3/Visual/frameShape",
			 frameShape->currentText() );
    settings.writeEntry( "/MQ3/Visual/frameStyle",
			 frameStyle->currentText() );
    settings.writeEntry("/MQ3/Visual/mode",
			visualMode->currentText());
    settings.writeEntry("/MQ3/Visual/frameRate",
			framerate->value());
    settings.writeEntry( "/MQ3/Visual/backColor",
			 bgColor->color().name() );
    settings.writeEntry("/MQ3/Organizer/playMode",
			playmode->currentText().lower());
    settings.writeEntry("/MQ3/scrollSpeed",
			labelSpeed->value());
    settings.writeEntry("/MQ3/scrollStopDelay",
			stopDelay->value());
    settings.writeEntry("/MQ3/Organizer/showOnStartup",
			oShowOnStartup->isChecked());
    settings.writeEntry("/MQ3/Equalizer/showOnStartup",
			eShowOnStartup->isChecked());
    settings.writeEntry("/MQ3/Organizer/saveChangesOnExit",
			oSaveChanges->isChecked());
    settings.writeEntry("/MQ3/Organizer/useLargeIcons",
			oUseLargeIcons->isChecked());
    settings.writeEntry("/MQ3/Organizer/useTextLabels",
			oTextInToolbars->isChecked());
    settings.writeEntry("/MQ3/Organizer/regexpSearching",
			oSearchRegexp->isChecked());
    settings.writeEntry("/MQ3/Equalizer/saveChangesOnExit",
			eSaveOnExit->isChecked());
    settings.writeEntry("/MQ3/Equalizer/enabled",
			eEnabled->isChecked());
    settings.writeEntry("/MQ3/iconTheme",
			iconTheme->currentText());
    settings.writeEntry("/MQ3/Output/bufferSize",
			obufsize->value());

    if ( visualPrefs )
	visualPrefs->write( settings );
}

void RealConfigDialog::addPath()
{
    QString path = QFileDialog::getExistingDirectory(QString::null, this, 0,
						     tr("Add Icon Theme Path"));

    if (path.isNull())
	return;

    QFileInfo fi(path);
    if (fi.isDir())
	IconLoader::addPath(path);

    themeSearch->clear();
    themeSearch->insertStringList(IconLoader::paths());
}

void RealConfigDialog::removePath()
{
    IconLoader::removePath(themeSearch->currentText());
    themeSearch->clear();
    themeSearch->insertStringList(IconLoader::paths());
}

void RealConfigDialog::changeVisualPrefs( const QString &visualname )
{
    if ( visualPrefs ) {
	// write any changes
	QSettings settings;
	visualPrefs->write( settings );
    }
    delete visualPrefs;
    visualPrefs = MainVisual::createPrefs( visualname,
					   visualGroupBox, "visual options" );
    if ( visualPrefs ) {
	QSettings settings;
	visualPrefs->init( settings );
    }
}
