;;; Major mode for editing archive files (.tar, .tgz, .zip, .deb, .rpm, etc.)
;;;
;;; Copyright (C) 1990-1998, Massimo Dal Zotto <dz@cs.unitn.it>
;;;
;;; This file is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; This file is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to the
;;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;;; Boston, MA 02111-1307, USA.
;;;
;;; DESCRIPTION:
;;;
;;; This is archive-mode, a major mode to "edit" archive files (.tar, .zip,
;;; .lha, .deb, .rpm, etc.). I wrote this code nearly 10 years ago (it was
;;; my first emacs code) and I still use it every day to work with tar files
;;; without having to extract the single members by hand.
;;; The code uses external programs to do all the work on the archives, and
;;; obviously it handles also compressed, gzippped, bzipped tar files and
;;; archives stored recursively inside other archives.
;;; You can even delete or store files back into the archive, with some
;;; limitations. It is however not possible to rename files or add new ones.
;;; The code uses a temporary directory created inside the directory defined
;;; by the environment variable TMP (or /tmp if undefined).
;;; New archive types can be added by defining handlers with the function
;;; archive-define-handler, see the code at the end of this file.
;;; Bug reports or suggestions to: Massimo Dal Zotto <dz@cs.unitn.it>
;;;
;;; KNOWN BUGS OR LIMITATIONS:
;;;
;;; If the archive contains files stored with thier full pathname, i.e.
;;; "/usr/include/stdio.h", the file is not be extracted to avoid accidental
;;; overwriting of existing files by dumb archiver (i.e. Sun's tar).
;;;
;;; You should use GNU tar instead of standard tar because it can extract
;;; files even if you dont' have the permission stored in the file header
;;; and it can also handle gzip compression.
;;;
;;; You must use unzip version 4.00 or later, because previous versions can't
;;; reconstruct sub-directories.
;;;
;;; When updating a file in a .tar archive the new file is always added at
;;; the end and old copies are still left in the archive.
;;;
;;; When updating a .lzh archive the old copy of archive is renamed .bak and
;;; is not deleted.
;;;
;;; When extracting from a MacIntosh .sit archive only the data fork is
;;; extracted.
;;;
;;; It is not possible to update a file stored in a compressed or gzipped tar
;;; archive, unless you use some helper script.
;;;
;;; It is not possible to update a file stored in a .arj, .sit, .rpm or .deb
;;; archive.
;;;
;;; It is not possible to update an archive stored recursively in an another
;;; archive file.
;;;
;;; It is not possible to put a new file into an archive or to rename a file
;;; inside the archive.
;;;
;;; Stored filename can't contain the strings "'" or "../", for security
;;; reasons.
;;;
;;; Save-buffer after a write-buffer to a normal file doesn't work.

(require 'cl)
(provide 'archive-mode)

(defvar archive-version "Archive 3.4.2 - Tue Jun  9 11:20:42 MET DST 1998")

(defvar archive-handlers nil
  "Handlers for known archive types. To add new handlers use 
the function archive-define-handler.")

(defvar archive-mode-map nil "Local keymap for archive-mode buffers.")
(when (not archive-mode-map)
  (setf archive-mode-map (make-sparse-keymap))
  (suppress-keymap archive-mode-map)
  (define-key archive-mode-map "g"    'revert-buffer)
  (define-key archive-mode-map "e"    'archive-find-file)
  (define-key archive-mode-map "\r"   'archive-find-file)
  (define-key archive-mode-map "\n"   'archive-find-file)
  (define-key archive-mode-map "f"    'archive-find-file)
  (define-key archive-mode-map "c"    'archive-copy-file)
  (define-key archive-mode-map "C"    'archive-copy-file)
  (define-key archive-mode-map "d"    'archive-mark-for-deletion)
  (define-key archive-mode-map "u"    'archive-unmark)
  (define-key archive-mode-map "x"    'archive-do-deletions)
  (define-key archive-mode-map " "    'archive-next-line)
  (define-key archive-mode-map "n"    'archive-next-line)
  (define-key archive-mode-map "p"    'archive-previous-line)
  (define-key archive-mode-map "\177" 'archive-previous-line)
  (define-key archive-mode-map "\M-d" 'archive-delete-file)
  (define-key archive-mode-map "\C-n" 'archive-next-line)
  (define-key archive-mode-map "\C-p" 'archive-previous-line)
  (define-key archive-mode-map [mouse-2] 'archive-find-file)
  (define-key archive-mode-map [double-down-mouse-1] 'archive-find-file))

(defvar archive-buffer-local-map nil "Local keymap for archive files buffers.")
(when (not archive-buffer-local-map)
  (setf archive-buffer-local-map (make-sparse-keymap))
  (define-key archive-buffer-local-map "\C-x\C-s" 'archive-save-buffer))

(defvar archive-tempdir-prefix ".arctmp"
  "Prefix of the temporary subdirectory created in the TMP directory.
Username is always prepended to avoid conflicts with other users.")

(defvar archive-confirm-changes t
  "If set asks confirmation for every change to the archive.")

(defvar archive-delete-tempdirs t
  "Delete archive subdirs created under temporary-directory.
You may want to enable this ... at your own risk.")

(put 'archive-mode 'mode-class 'special) 	; (?) from dired.el

(defun archive-mode (&optional extension)
  "Mode for \"editing\" .tar, .zip, .lzh, .arc, .zoo, etc... archives.
In archive mode, you are \"browsing\" a list of the files in an archive.
You can move using the usual cursor motion commands.
Letters are no longer self-inserting.
Archive commands:
 e, f, return	to Find the current line's file.
 c, C      	to Copy the current line's file to an ordinary file.
 d		to mark the current line's file for deletion.
 u		to unmark the current line's file.
 x		to do deletions of marked files.
 g      	to read the archive listing again.
 M-d      	to delete the current line's file.
 [2],[double-1]	to Find the current line's file.
 n, p, Space Backspace and cursors can be used to move down and up by lines.
Files can be deleted or stored back in the archive (with C-x C-s), if the
archiver has this capability, and can be copied to ordinary files."
  (interactive)
  (kill-all-local-variables)
  (let (short-filename item)
    (make-local-variable 'revert-buffer-function)
    (make-local-variable 'archive-filename)
    (make-local-variable 'archive-type)
    (make-local-variable 'temporary-directory)
    (make-local-variable 'list-cmd)
    (make-local-variable 'extract-cmd)
    (make-local-variable 'store-cmd)
    (make-local-variable 'delete-cmd)
    (make-local-variable 'filename-finder)
    (make-local-variable 'kill-buffer-hook)
    (add-hook 'kill-buffer-hook 'archive-mode-kill-buffer-hook)
    (setf major-mode 'archive-mode)
    (setf mode-name "Archive")
    (setf mode-line-buffer-identification '("Archive: %17b"))
    (setf revert-buffer-function 'archive-revert)
    (setf archive-filename (buffer-file-name))
    (setf short-filename (file-name-nondirectory (buffer-file-name)))
    (setf case-fold-search nil)
    (setf temporary-directory
	  (concat (or (getenv "TMP") "/tmp") "/"
		  archive-tempdir-prefix "."
		  (user-login-name)))
    (unless (file-exists-p temporary-directory)
      (call-process "mkdir" nil nil nil temporary-directory)
      (call-process "chmod" nil nil nil "700" temporary-directory))
    (unless (file-exists-p temporary-directory)
      (error "Unable to create temporary directory: %s" temporary-directory))
    (unless (string-equal extension "tar.Z")	; hack for tar.Z files
      (setf extension
	    (downcase (or extension
			  (archive-filename-extension short-filename)))))
    (setf item (assoc extension archive-handlers))
    (setf archive-type    (first item))
    (setf list-cmd        (second item))
    (setf extract-cmd     (third item))
    (setf store-cmd       (fourth item))
    (setf delete-cmd      (fifth item))
    (setf filename-finder (sixth item))
    (unless list-cmd
      (error "Unknown archive type: %s" extension))
    (unless (file-exists-p short-filename) 	; probably a compressed file
      						; or an archived archive
      (when (memq 'crypt features)	
	(funcall 'compress-mode -1))		; turn off compress mode
      (setf require-final-newline nil)		; this is a binary file
      (make-local-variable 'temporary-file)
      (setf temporary-file (concat temporary-directory "/" short-filename))
      (write-file temporary-file)		; create temp uncompressed file
      (setf default-directory (file-name-directory archive-filename))
      (setf archive-filename temporary-file))
    (use-local-map archive-mode-map)
    (archive-readin archive-filename (current-buffer))))

(defun archive-local-variable (buffer name &optional error)
  "Function to get a local variable of a buffer."
  (save-excursion
    (set-buffer buffer)
    (or
     (and (boundp name) (symbol-value name))
     (and error (symbol-value name)))))

(defun archive-filename-extension (filename)
  "Return the extension of a filename or an empty string if none."
  (let ((start (length filename))
	(i     (length filename)))
    (while (plusp i)
      (if (= (aref filename (1- i)) ?.)
	  (setf start i i 0))
      (decf i))
    (substring filename start (length filename))))

;; Hack to allow selection of a modified archive from dired without revert
(unless (fboundp 'verify-visited-file-modtime-orig)
  (setf (symbol-function 'verify-visited-file-modtime-orig)
	(symbol-function 'verify-visited-file-modtime)))

(defun archive-verify-visited-file-modtime (buf &rest args)
  "Called by find-file-noselect in <files.el>"
  (if (eq (archive-local-variable buf 'major-mode) 'archive-mode)
      t
    (apply 'verify-visited-file-modtime-orig buf args)))

;; Hook to delete temporary files when killing a buffer
(defun archive-mode-kill-buffer-hook ()
  (when (and (boundp 'temporary-file) (symbol-value 'temporary-file))
    (call-process "rm" nil nil nil temporary-file)))

(defun tar-mode ()
  "Visit a .tar archive."
  (interactive)
  (archive-mode "tar"))

(defun tgz-mode ()
  "Visit a .tar.gz archive."
  (interactive)
  (archive-mode "tgz"))

(defun tbz-mode ()
  "Visit a .tar.bz2 archive."
  (interactive)
  (archive-mode "tar.bz2"))

(defun zip-mode ()
  "Visit a .zip archive."
  (interactive)
  (archive-mode "zip"))

(defun lha-mode ()
  "Visit a .lha archive."
  (interactive)
  (archive-mode "lha"))

(defun lzh-mode ()
  "Visit a .lzh archive."
  (interactive)
  (archive-mode "lzh"))

(defun zoo-mode ()
  "Visit a .zoo archive."
  (interactive)
  (archive-mode "zoo"))

(defun arc-mode ()
  "Visit a .arc archive."
  (interactive)
  (archive-mode "arc"))

(defun arj-mode ()
  "Visit a .arc archive."
  (interactive)
  (archive-mode "arj"))

(defun sit-mode ()
  "Visit a .sit archive."
  (interactive)
  (archive-mode "sit"))

(defun deb-mode ()
  "Visit a .deb archive."
  (interactive)
  (archive-mode "deb"))

(defun rpm-mode ()
  "Visit a .rpm archive."
  (interactive)
  (archive-mode "rpm"))

(defun tarzan-mode ()
  "I like the name. Visit a .tar.Z archive."
  (interactive)
  (if (string-match ".z$" buffer-file-name)
      (tgz-mode)		; hack for gzipped tar files
    (archive-mode "tar.Z")))

(defun archive-next-line (arg)
  "Move down ARG lines then position at filename."
  (interactive "p")
  (next-line arg)
  (archive-move-to-filename))

(defun archive-previous-line (arg)
  "Move up ARG lines then position at filename."
  (interactive "p")
  (previous-line arg)
  (archive-move-to-filename))

(defun archive-revert (&rest args)
  "Revert the archive directory."
  (let ((opoint           (point))
	(ofile            (archive-get-filename t t)))
    (archive-readin archive-filename (current-buffer))
    (setf buffer-file-name archive-filename)
    (or (and ofile (re-search-forward (concat " " (regexp-quote ofile) "$")
				      nil t))
	(goto-char opoint))
    (beginning-of-line)))

(defun archive-readin (archive buffer)
  "Read the archive listing."
  (let ((oldfile buffer-file-name)
	(archive (expand-file-name archive))
	(command (format "%s '%s' 2>/dev/null" list-cmd archive)))
    (save-excursion
      (set-buffer buffer)
      (clear-visited-file-modtime)
      (setf buffer-read-only nil)
      (setf buffer-file-name nil)
      (widen)
      (erase-buffer)
      (if (not (and (file-exists-p archive)
		    (zerop (call-process "sh" nil buffer nil "-c" command))))
	  (error "Unable to read archive %s" archive))
      (goto-char (point-min))
      (while (not (eobp))
	(insert "  ")
	(forward-line 1))
      (goto-char (point-min))
      (setf buffer-file-name oldfile)
      (setf buffer-read-only t)
      (setf buffer-backed-up t)
      (setf buffer-read-only t)
      (set-buffer-modified-p nil)
      (auto-save-mode 0))))

(defun archive-find-buffer (archive)
  (let ((blist (buffer-list))
	found)
    (while blist
      (save-excursion
        (set-buffer (car blist))
	(if (and (eq major-mode 'archive-mode)
		 (equal archive-filename archive))
	    (setf found (car blist)
		  blist nil)
	  (setf blist (cdr blist)))))
    (or found
	(create-file-buffer (file-name-nondirectory archive)))))

(defun archive-get-filename (&optional ignore no-error-if-not-filep)
  "In archive-mode, return name of the file on the current line.
1st argument is ignored.  A non-nil 2nd argument says return nil
if no filename on this line, otherwise an error occurs."
  (save-excursion
    (let ((end      (archive-move-to-filename))
	  (filename nil))
      (cond (end
	     (setf filename (buffer-substring (point) end))
	     (if (string= (substring filename 0 1) "/")
		 (error "File begins with \"/\", can't extract it")))
	    (no-error-if-not-filep
	     nil)
	    (t
	     (error "No file on this line")))
      filename)))

(defun archive-move-to-filename (&optional ignore no-error-if-not-filep)
  "In archive-mode, move to first char of filename on the current line.
Returns the name end position or nil if no filename on this line."
  (funcall (if (stringp filename-finder)
	       'archive-find-filename-default
	     filename-finder)
	   ignore 
	   no-error-if-not-filep))

(defun archive-find-filename-default (&optional ignore no-error-if-not-filep)
  "Default function to find the name of file on the current line.
Position point at start of filename and return the end point of filename."
  (let (start end)
    (save-excursion
      (end-of-line)
      (setf end (point))
      (beginning-of-line)
      (if (re-search-forward filename-finder end t)
	  (setf start (point))
	(setf start nil)
	(setf end nil)))
    (if start
	(goto-char start))
    end))

(defun archive-find-lzh-filename (&optional ignore no-error-if-not-filep)
  "For lzh archives, function to find the name of file
on the current line."
  (let (start end)
    (save-excursion
      (end-of-line)
      (setf end (point))
      (beginning-of-line)
      (if (or 
	   (re-search-forward 
	    "[0-9]\.[0-9]\% ... .. ..... [^ ]" end t)	          ; lharc 1.00
	   (re-search-forward 
	    "[0-9]\.[0-9]\% .... ... .. ..... .... [^ ]" end t))  ; lharc 1.02
	  (progn 
	    (backward-char 1)
	    (setf start (point)))
	(setf start nil)
	(setf end nil)))
    (if start
	(goto-char start))
    end))

(defun archive-find-arc-filename (&optional ignore no-error-if-not-filep)
  "For arc/arj archives, function to find the name of file
on the current line."
  (let (start end)
    (save-excursion
      (end-of-line)
      (setf end (point))
      (beginning-of-line)
      (if (re-search-forward ":[0-9][0-9][ap ]" end t)
	  (progn (beginning-of-line)
		 (forward-char 2)
		 (re-search-forward " " nil t)
		 (setf end (1- (point)))
		 (beginning-of-line)
		 (forward-char 2)
		 (setf start (point)))
	(setf start nil)
	(setf end nil)))
    (if start
	(goto-char start))
    end))

(defun archive-find-sit-filename (&optional ignore no-error-if-not-filep)
  "For sit archives, function to find the name of file
on the current line."
  (let (start end)
    (save-excursion
      (end-of-line)
      (setf end (point))
      (beginning-of-line)
      (if (re-search-forward "name=\"" end t)
	  (progn (setf start (point))
		 (re-search-forward "\", type=" end t)
		 (setf end (- (point) 8)))
	(setf start nil)
	(setf end nil)))
    (if start
	(goto-char start))
    end))

(defun archive-find-file ()
  "In archive-mode, visit the file named on the current line."
  (interactive)
  (let* ((filename (archive-get-filename t nil))
	 (buffname (format "%s: %s" archive-filename filename))
	 (buffer   (find-buffer-visiting buffname)))
    (unless (and buffer 
		 (eq (archive-local-variable buffer 'archive-buffer) 
		     (current-buffer)))
      (setf buffer (create-file-buffer buffname))
      (archive-extract-file buffer filename))
    (switch-to-buffer buffer)))

(defun archive-check-filename (filename &optional ok-if-directory)
  (if (string-match "^/" filename)
      (error "File is an absolute pathname: %s" filename))
  (if (and (not ok-if-directory) (string-match "/$" filename))
      (error "File is a directory: %s" filename))
  (if (string-match "/\\.\\./" (concat "/" filename "/"))
      (error "Filename contains ../: %s" filename))
  (if (string-match "['\r\n]" filename)
      (error "Filename contains invalid characters: %s" filename)))

(defun archive-check-tempdir (tempdir)
  (if (or (not tempdir) (string= tempdir ""))
      (error "Temporary directory is nil")))

(defun archive-extract-file (buffer filename)
  "Call the extract-cmd to extract FILENAME to a temporary file
and insert its content in BUFFER."
  (archive-check-filename filename)
  (archive-check-tempdir temporary-directory)
  (let ((arcbuff     (current-buffer))
	(arcname     archive-filename)
	(tempdir     temporary-directory)
	(extractcmd  extract-cmd)
	(storecmd    store-cmd)
	(command     nil)
	(topdir      nil)
	(pathname    nil)
	(tempfile    nil)
	(rc          nil))
    (save-excursion
      (set-buffer buffer)
      (setf buffer-read-only nil)
      (widen)
      (goto-char (point-min))
      (erase-buffer)
      (insert (concat tempdir "/" filename))
      (goto-char (point-min))
      (replace-string "//" "/")		; zap double slashes, like "aaa//bbb"
      (setf tempfile (buffer-substring (point-min) (point-max)))
      (erase-buffer)
      (insert filename)
      (goto-char (point-min))
      (replace-regexp "/.*" "")		; get filename top directory
      (setf topdir (buffer-substring (point-min) (point-max)))
      (erase-buffer)
      (message "Extracting %s..." filename)
      (setf command
	    (concat (format "cd '%s';" tempdir)
		    (format "rm -rf '%s';" tempfile)
		    (format "%s '%s' '%s' 2>/dev/null;" 
			    extractcmd arcname filename)
		    (format "chmod u+r '%s'" filename)))
      (if (not (zerop (call-process "sh" nil t nil "-c" command)))
	  (error "Unable to extract file %s" filename))
      (insert-file tempfile)		; insert the file text
      (goto-char (point-min))
      (set-buffer-modified-p nil)
      (cond ((or (not archive-delete-tempdirs) (string= topdir ""))
	     (call-process "rm" nil nil nil "-rf" tempfile))
	    (t
	     (setf pathname (concat tempdir "/" topdir))
	     (call-process "rm" nil nil nil "-rf" pathname) ))
      (setf buffer-file-name  (format "%s: %s" arcname filename))
      (setf default-directory (file-name-directory arcname))
      (after-find-file)			; set major mode of extracted file
      (auto-save-mode 0)		; no autosave for archived files
      (make-local-variable 'revert-buffer-function)
      (make-local-variable 'require-final-newline)
      (make-local-variable 'temporary-filename)
      (make-local-variable 'temporary-topdir)
      (make-local-variable 'short-name)
      (make-local-variable 'archive-buffer)
      (setf revert-buffer-function 'archive-revert-buffer)
      (setf temporary-filename tempfile)
      (setf temporary-topdir topdir)
      (setf archive-buffer arcbuff)
      (setf short-name filename)
      (setf require-final-newline nil)
      (cond ((eq major-mode 'archive-mode)
	     (setf store-cmd nil)	; can't store or delete in stored
	     (setf delete-cmd nil))	; archives
	    (t
	     (use-local-map archive-buffer-local-map)
	     (setf buffer-read-only
		   (not (and storecmd (file-writable-p arcname))))))
      (if (not (input-pending-p))
	  (message "Extracting %s...done" (buffer-name))))))

(defun archive-revert-buffer (&rest args)
  "In archive-mode, revert the current file buffer."
  (interactive)
  (let* ((buffer   (current-buffer))
	 (filename short-name))
    (save-excursion
      (set-buffer archive-buffer)
      (archive-extract-file buffer filename))))

(defun archive-save-buffer ()
  "Store the current buffer back in its archive file."
  (interactive)
  (unless (buffer-name archive-buffer)
    (error "Archive of file %s id not being visited" (buffer-name)))
  (let ((filename (buffer-name))
	(oldfile  buffer-file-name)
	(olddir   default-directory)
	(tempfile temporary-filename)
	(topdir   temporary-topdir)
	(tempdir  (archive-local-variable archive-buffer 'temporary-directory))
	(arcname  (archive-local-variable archive-buffer 'archive-filename))
	(arctype  (archive-local-variable archive-buffer 'archive-type))
	(storecmd (archive-local-variable archive-buffer 'store-cmd))
	(pathname nil)
	(command  nil)
	(rc       nil))
    (unless storecmd
      (error "Store into this archive is not possible"))
    (unless (file-writable-p arcname)
      (error "No write permission for archive %s"))
    (archive-check-filename (file-relative-name tempfile tempdir))
    (archive-check-tempdir tempdir)
    (if (not (buffer-modified-p))
	(message "(No changes need to be saved)")
      (when (or 
	     (not archive-confirm-changes)
	     (y-or-n-p (format "Save %s in archive %s ? " filename arcname)))
	(message "Storing %s..." tempfile)
	(if (not (zerop (call-process "mkdir" nil nil nil "-p"
				      (file-name-directory tempfile))))
	    (error "Unable to create save directory: %s" tempfile))
	(write-file tempfile)
	(setf buffer-file-name oldfile)
	(setf default-directory olddir)
	(message "Storing %s..." filename)
	(setf command (format "cd '%s' && %s '%s' '%s'"
			      tempdir storecmd arcname
			      (file-relative-name tempfile tempdir)))
	(setf rc (call-process "sh" nil nil nil "-c" command))
	(cond ((or (not archive-delete-tempdirs) (string= topdir ""))
	       (call-process "rm" nil nil nil "-rf" tempfile))
	      (t
	       (setf pathname (concat tempdir "/" topdir))
	       (call-process "rm" nil nil nil "-rf" pathname) ))
	(if (not (= rc 0))
	    (error "Unable to update archive: %s" arcname))
	(set-buffer-modified-p nil)
	(save-excursion
	  (set-buffer archive-buffer)
	  (archive-revert))))	; update the archive buffer
    (message "")))

(defun archive-delete-file ()
  "Delete the file named on this line."
  (interactive)
  (when (and delete-cmd
	     (file-writable-p archive-filename))
    (let* ((filename (archive-get-filename t nil)))
      (cond ((or (not archive-confirm-changes)
		 (y-or-n-p (format "Delete %s from archive %s ? "
				   filename archive-filename)))
	     (message "Deleting %s..." filename)
	     (archive-delete-file-aux filename)
	     (archive-revert)
	     (message "Deleting %s...done" filename))
	    (t
	     (message ""))))))

(defun archive-mark-for-deletion (arg)
  "Mark the current (or next ARG) files for deletion."
  (interactive "P")
  (when (and delete-cmd
	     (file-writable-p archive-filename))
    (let (buffer-read-only)
      (setf arg (or arg 1))
      (while (and (plusp arg)
		  (< (point) (point-max)))
	(when (setf filename (archive-get-filename t t))
	  (beginning-of-line)
	  (delete-char 1) (insert ?D))
	(beginning-of-line)
	(archive-next-line 1)
	(setf arg (1- arg)))
      (set-buffer-modified-p nil))))

(defun archive-unmark (arg)
  "Unmark the current (or next ARG) files."
  (interactive "P")
  (let (buffer-read-only)
    (setf arg (or arg 1))
    (while (and (plusp arg)
		(< (point) (point-max)))
      (when (setf filename (archive-get-filename t t))
	(beginning-of-line)
	(delete-char 1) (insert 32))
      (beginning-of-line)
      (archive-next-line 1)
      (setf arg (1- arg)))
    (set-buffer-modified-p nil)))

(defun archive-do-deletions ()
  "Delete the files marked for deletion."
  (interactive)
  (cond ((and delete-cmd
	      (file-writable-p archive-filename)
	      (or (not archive-confirm-changes)
		  (y-or-n-p "Delete marked files ? ")))
	 (save-excursion
	   (message "Deleting files...")
	   (let (buffer-read-only filename)
	     (goto-char (point-min))
	     (while (< (point) (point-max))
	       (beginning-of-line)
	       (when (and (looking-at "D")
			  (setf filename (archive-get-filename t t)))
		 (beginning-of-line) (delete-char 1) (insert 32)
		 (archive-delete-file-aux filename))
	       (archive-next-line 1)))
	   (message "Deleting files...done")
	   (archive-revert)))
	(t
	 (message ""))))

(defun archive-delete-file-aux (filename &optional verbose)
  "Delete FILENAME from the archive."
  (let* ((arcbuff  (current-buffer))
	 (arcname  archive-filename)
	 (command  nil)
	 (tempfile (concat temporary-directory "/" filename)))
    (unless delete-cmd
      (error "Deleting from this archive is not possible"))
    (unless (file-writable-p archive-filename)
      (error "No write permission for archive %s" archive-filename))
    (archive-check-filename filename 'ok-if-directory)
    (archive-check-tempdir temporary-directory)
    (setf command (format "%s '%s' '%s'" delete-cmd archive-filename filename))
    (and verbose (message "Deleting %s..." filename))
    (if (not (zerop (call-process "sh" nil nil nil "-c" command)))
	(error "Unable to delete file %s" filename))
    (and verbose (message "Deleting %s...done" filename))))

(defun archive-copy-file (target)
  "Copy the file named on this line to TARGET."
  (interactive "FCopy to: ")
  (let* ((filename (archive-get-filename t nil))
	 (arcbuff  (current-buffer))
	 (arcname  archive-filename)
	 (command  nil)
	 (tempfile (concat temporary-directory "/" filename)))
    (archive-check-filename filename)
    (archive-check-tempdir temporary-directory)
    (setf target (expand-file-name target))
    (if (and (file-directory-p target) (not (string-match "/$" target)))
	(setf target (concat target "/")))
    (if (or (file-directory-p target) (string= target archive-filename))
	(setf target (concat (file-name-directory target)
			     (file-name-nondirectory filename))))
    (if (not (file-writable-p target))
	(error "No permission to write file %s" target))
    (cond ((or (not (file-exists-p target))
	       (y-or-n-p (format "File %s exists; overwrite? " target)))
	   (setf command (concat (format "cd '%s';" temporary-directory)
				 (format "rm -rf '%s';" tempfile)
				 (format "%s '%s' '%s';" 
					 extract-cmd archive-filename 
					 filename)
				 (format "rm -rf '%s';" target)
				 (format "mv '%s' '%s'" 
					 tempfile (expand-file-name target))))
	   (message "Writing %s..." target)
	   (if (not (zerop (call-process "sh" nil nil nil "-c" command)))
	       (error "Unable to copy file to %s" target))
	   (message "Writing %s...done" target))
	  (t
	   (message "")))))

(defun archive-define-handler (extensions list-cmd extract-cmd store-cmd
			       delete-cmd file-finder &optional major-mode)
  "Define a new handler for archive-mode."
  (unless (consp extensions) (setf extensions  (list extensions)))
  (let ((major-mode (or major-mode 'archive-mode))
	old-item
	new-item)
    (dolist (extension extensions)
      (setf old-item (assoc extension archive-handlers))
      (setf new-item (list extension list-cmd extract-cmd store-cmd delete-cmd
			   file-finder))
      (cond (old-item
	     (rplacd old-item (rest new-item)))
	    (t
	     (push new-item archive-handlers)))
      (push (cons (concat "\\." extension "\\'") major-mode) auto-mode-alist))
    (dolist (extension extensions)
      (setf extension (upcase extension))
      (setf old-item (assoc extension archive-handlers))
      (setf new-item (list extension list-cmd extract-cmd store-cmd delete-cmd
			   file-finder))
      (cond (old-item
	     (rplacd old-item (rest new-item)))
	    (t
	     (push new-item archive-handlers)))
      (push (cons (concat "\\." extension "\\'") major-mode) auto-mode-alist))
    (list extensions major-mode)))

;; Define handlers for known archive types
(archive-define-handler
 "tar"					; file extension(s), in lowercase
 "tar -tv --force-local -f"		; list command
 "tar -x --force-local -f"		; extract command
 "tar -r --force-local -f"		; store command
 "tar --delete --force-local -f"	; delete command
 ":[0-9][0-9] [0-9]+ ")			; regexp preceeding filenames

(archive-define-handler
 '("tar.gz" "tar.z" "tgz" "taz" "tpz")
 "tar -tzv --force-local -f"
 "tar -xz --force-local -f"
 "tgz -r --force-local -f"		; use helper script tgz
 "tgz --delete --force-local -f"	; use helper script tgz
 ":[0-9][0-9] [0-9]+ "
 'tgz-mode)				; major-mode function

(archive-define-handler
 "tar.Z"				; must be uppercase, see tarzan-mode
 "tar -tZv --force-local -f"
 "tar -xZ --force-local -f"
 "tgz -r --force-local -f"		; needs the helper script tgz
 "tgz --delete --force-local -f"
 ":[0-9][0-9] [0-9]+ "
 'tarzan-mode)				; major-mode function

(archive-define-handler
 '("tar.bz2" "tbz")
 "tar -tv --use-compress-program bzip2 --force-local -f"
 "tar -x --use-compress-program bzip2 --force-local -f"
 "tgz -r --use-compress-program bzip2 --force-local -f"
 "tgz --delete --use-compress-program bzip2 --force-local -f"
 ":[0-9][0-9] [0-9]+ "
 'tbz-mode)				; major-mode function

(archive-define-handler
 "zip"
 "unzip -l"
 "unzip -xqq"
 "zip"
 "zip -d"
 ":[0-9][0-9]  [ ^]")

(archive-define-handler
 "zoo"
 "zoo v"
 "zoo xOS//"
 "zoo a"
 "zoo DP"
 ":[0-9:+]+[ ]+[ C] ")

(archive-define-handler
 '("lha" "lzh")
 "lha v"
 "lha x"
 "lha a"
 "lha d"
 "[0-9a-f] ... .. ...[0-9][0-9] ")

(archive-define-handler
 "arc"
 "arc v"
 "arc x"
 "arc a"
 "arc d"
 'archive-find-arc-filename)		; filename finder function

(archive-define-handler
 "arj"
 "unarj l"
 "unarj x"
 nil					; can't store in .arj archives
 nil					; can't delete from .arj archives
 'archive-find-arc-filename)		; use the .arc file-finder function

(archive-define-handler
 "sit"
 "unsit -l"
 "unsit -d"
 nil
 nil
 'archive-find-sit-filename)

(archive-define-handler
 "cpio"
 "cpio --list --verbose --force-local -F"
 "cpio --extract --make-directories --force-local -F"
 nil					; needs helper script for storing
 nil
 "^  [^d].*[A-Z][a-z]+ [ 123][0-9] +[0-9:]+ ")

(archive-define-handler
 "deb"
 "debarc -t"				; needs the helper script debarc
 "debarc -x"
 nil
 nil
 ":[0-9][0-9] [0-9]+ ")			; like tar-mode

(archive-define-handler
 "rpm"
 "rpmarc -t"				; needs the helper script rpmarc
 "rpmarc -x"
 nil
 nil
 "^  [^d].*[A-Z][a-z]+ [ 123][0-9] +[0-9:]+ ")	; like cpio-mode

;;; end of file
