// ctfile.cpp
//
// Copyright (C) 1999  Robert Barron, KA5WSS
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "ctfile.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

/* Used to convert from CT file based band numbers to qso_data type band numbers. */
static unsigned char bands[] = {0, BAND_160M, BAND_80M, BAND_40M, BAND_20M, BAND_15M,
                                   BAND_10M, 0, 0, BAND_50MHZ, BAND_144MHZ, BAND_NOVICE,
                                   BAND_SATELLITE, BAND_PACKET};
static unsigned char vhfbands[] = {0, BAND_50MHZ, BAND_144MHZ, BAND_222MHZ, BAND_432MHZ,
                                      BAND_902MHZ, BAND_1GHZ, BAND_2GHZ, BAND_3GHZ,
                                      BAND_5GHZ, BAND_10GHZ, BAND_24GHZ};                                   
static char *ContestNames[] = {NULL, "SS", "ARRL", "CQWW", "WPX", " ", "AR10", "A160",
                                "C160", "FD", " ", "ARDX", "WAE", "VHF", "DXPN", NULL};
static char *modes[] = {NULL, "CW", "SSB", NULL};
static char *PortNames[] = {"NONE", "COM1", "COM2", "COM3", "COM4", "LPT1", "LPT2", NULL};
static char *dvkPortNames[] = {"NONE", "LPT1", "LPT2", "DVP", " ", " ", " ", NULL};
static char *categories[] = {"SO", "SU", "MS", " ", "MM", " ", " ", "HP", " ", " ", "SB", NULL};

RCtFile::RCtFile(char *name)
{
    RCtFile::open(name);
}

boolean RCtFile::open(char *name, int mode)
{
    char index;
    
#ifdef LINUX
    if (!RFile::open(name, mode))
        return FALSE;
#else
    // State ios::binary again just in case the user does not.
    if (!RFile::open(name, mode | ios::binary))
        return FALSE;
#endif

    if (!(mode & ios::out))          // Not an output file
    {
        /* Read in the header information and store in neutral format. */        
        RFile::read((void *)&ctheader, sizeof(CtHeader));
        strncpy(header.callsign, ctheader.call, CT_CALL_SIZE);
        strncpy(header.name, ctheader.name, CT_NAME_SIZE);
        strncpy(header.address, ctheader.address, CT_ADDRESS_SIZE);
        strncpy(header.city, ctheader.town, CT_TOWN_SIZE);
        strncpy(header.state, ctheader.state, CT_STATE_SIZE);
        strncpy(header.zip, ctheader.zip, CT_ZIP_SIZE);

        header.stationNumber = (unsigned char)atoi(ctheader.station);
        strcpy(header.radio, ctheader.radio);
        switch(ctheader.tnc[0]) {
            case 'L':
                header.tnc = TNC_LOCAL;
                break;
            case 'E':
            case 'R':
                header.tnc = TNC_REMOTE;
            case 'N':
            default:
                header.tnc = TNC_NONE;
        }
        for (unsigned char portCount = 0; PortNames[portCount] != NULL; portCount++)
        {
            if (strncmp(ctheader.cw_port, PortNames[portCount], 4) == 0)
                header.cwPort = portCount;
            if (strncmp(ctheader.dvk_port, dvkPortNames[portCount], 4) == 0)
                header.dvkPort = portCount;
        }
        
        if (strncmp(ctheader.timezone, "UTC", 3) != 0)
        {
            RFile::close();    // Might want to support this later
            return FALSE;     //  But for now,  use UTC!!!!!!!
        }
        
        header.zone = (unsigned char)atoi(ctheader.zone);
        strncpy(header.grid, ctheader.my_grid, CT_GRID_SIZE);
        strncpy(header.club, ctheader.club, CT_CLUB_SIZE);

        /* Look at contest name in CT file header.  Determine the */
        /*  appropriate neutral contest ID number based on that. */
        /*  The contest ID num is the offset in index in the array. */
        for (index = 1; ContestNames[index] != NULL; index++)
            if (strncmp(ContestNames[index], ctheader.contest_type,
                        strlen(ContestNames[index])) == 0)
            {
                header.contest = index;           // Found the contest number!
                break;
            }

        /* If can not identify this contest return with an error. */
        if (ContestNames[index] == NULL)
        {
            RFile::close();
            return FALSE;
        }
        strncpy(header.contestName, ctheader.contest_title, 9);
        for (unsigned char catCount = 0; catCount < CAT_SINGLE_OP_SINGLE_BAND; catCount++)
            if (strcmp(ctheader.category, categories[catCount]) == 0)
                header.category = catCount;

        // Store the CW messages
        strncpy(header.CQ_CW_Message, ctheader.cq_msg, 40);
        strncpy(header.QRZ_CW_Message, ctheader.qrz_msg, 20);
        strncpy(header.Exchange_CW_Message, ctheader.ex_msg, 20);
        strncpy(header.Dupe_CW_Message, ctheader.dupe_msg, 20);
        strncpy(header.F6_CW_Message, ctheader.F6_text, 10);
        strncpy(header.F7_CW_Message, ctheader.F7_text, 10);
        
        // Remember the COM: port information.
        for (int comm_count = 0; comm_count < 4; comm_count++)
        {
            header.comm[comm_count].device = ctheader.comm[comm_count].device;
            header.comm[comm_count].baud = ctheader.comm[comm_count].baud;
        }

        header.flags.m_stn = (unsigned char)ctheader.m_stn;
        header.flags.post = (unsigned char)ctheader.post_flag;
        header.flags.beep = (unsigned char)ctheader.beep;
        header.flags.band_rate = (unsigned char)ctheader.band_rate;
        header.flags.sound = (unsigned char)ctheader.sound;
        header.flags.cw_abbrev = (unsigned char)ctheader.cw_abbrev;
        header.flags.autosave = (unsigned char)ctheader.autosav;
        header.flags.nocompress = (unsigned char)ctheader.nocompress;
        
        // Don't bother with these fields unless they are specifically needed.
        if (header.contest == ARRL_SS_CONTEST)
        {
            header.ss.precedence = ctheader.ss.prec[0];
            header.ss.check = (unsigned char)atoi(ctheader.ss.chk);
            strncpy(header.ss.section, ctheader.sec, CT_SECTION_SIZE);
        }
        else if (header.contest == FD_CONTEST)
        {
            header.fd.category = ctheader.fd.category;
            header.fd.transmitters = ctheader.fd.tx_count;

            header.fd.ep_bonus = ctheader.fd.ep_bonus;
            header.fd.pr_bonus = ctheader.fd.pr_bonus;
            header.fd.loc_bonus = ctheader.fd.loc_bonus;
            header.fd.info_bonus = ctheader.fd.info_bonus;
            header.fd.msg_bonus = ctheader.fd.msg_bonus;
            header.fd.sat_bonus = ctheader.fd.sat_bonus;
            header.fd.nat_bonus = ctheader.fd.nat_bonus;
            header.fd.w1aw_bonus = ctheader.fd.w1aw_bonus;
            header.fd.pkt_bonus = ctheader.fd.pkt_bonus;
            header.fd.tfc_bonus = ctheader.fd.tfc_bonus;
        }
    }
    
    location = 0;

    return TRUE;
}

/* This routine takes as input the neutral format header from the input file */
/*  and stores it in this object.  It also writes out the header to its file */
/*  in its native format. */
boolean RCtFile::writeHeader(header_data newheader)
{
    // Not yet implemented
    newheader = newheader;              // Get rid of compiler warnings.
#if 0    
    if (headerWritten || (ContestNames[newheader.contest] == NULL))
        return FALSE;

    RQsoFile::storeHeader(newheader);

    memset(&ctheader, '\0', sizeof(CtHeader));

    strncpy(ctheader.call, header.callsign, CT_CALL_SIZE);
    strncpy(ctheader.name, header.name, CT_NAME_SIZE);
    strncpy(ctheader.address, header.address, CT_ADDRESS_SIZE);
    strncpy(ctheader.town, header.city, CT_TOWN_SIZE);
    strncpy(ctheader.state, header.state, CT_STATE_SIZE);
    strncpy(ctheader.zip, header.zip, CT_ZIP_SIZE);

    /* WARNING! Does write() work with binary data????? */
    /* Write the header to the file in native format. */
    RFile::write(&ctheader, sizeof(CtHeader));
    
    headerWritten = TRUE;
    return TRUE;
#endif
    return FALSE;    
}

boolean RCtFile::nextQso(qso_data *qso)
{
    ct_log_data CTqso;
    static unsigned char defaultMode = 0;
    unsigned short sentRST;

    if (defaultMode == 0)
        for (defaultMode = 1;
             (modes[defaultMode] != NULL
                && (strcmp(modes[defaultMode], ctheader.mode) != 0));
             defaultMode++);
             
    qso->init();
    if (RFile::read((void *)&CTqso, sizeof(ct_log_data)))
    {
        /* Some CT files somehow got spaces in their info field.  Remove them. */
        for (unsigned char count = 0; count < strlen(CTqso.info); count++)
            if (!isalnum(CTqso.info[count]))
                CTqso.info[count] = (char)NULL;
    
        qso->setCallsign(CTqso.call, strlen(CTqso.call));
        qso->setTime(CTqso.time);
        if (header.contest == ARRL_VHF_QSO_PARTY)
            qso->setBand(vhfbands[CTqso.band]);
        else if ((header.contest == DXPEDITION_MODE) && (CTqso.band == BAND_144MHZ))
            qso->setBand(BAND_SATELLITE);
        else if (header.contest == ARRL_10M_CONTEST)
            qso->setBand(BAND_10M);
        else
            qso->setBand(bands[CTqso.band]);
        if (CTqso.mode == 0)
            qso->setMode(defaultMode);
        else
            qso->setMode(CTqso.mode);
        switch (header.contest)
        {
            case ARRL_SS_CONTEST:
            case FD_CONTEST:
                // Don't bother checking RST for contests that do not exchange
                //  this info.  Also can't check because RST field is shared with
                //  SS and FD info!
                break;
            default:
                // Error check.  CT can allow an incorrect RST for a mode!
                if ((qso->getMode() != CW_MODE) && (CTqso.rst > 59))
                    CTqso.rst = (short)(CTqso.rst / 10);
                else if ((qso->getMode() == CW_MODE) && (CTqso.rst < 100))
                    CTqso.rst = (short)((CTqso.rst * 10) + 9);
                if ((qso->getMode() == CW_MODE) || (qso->getMode() == RTTY_MODE)
                    || (qso->getMode() == PACKET_MODE))
                    sentRST = 599;
                else
                    sentRST = 59;
                break;
        }
        if (CTqso.stn_num & 0x40)
            qso->setQslSent('Y');
        if (CTqso.stn_num & 0x80)
            qso->setQslReceived('Y');
        
        if (header.contest == ARRL_SS_CONTEST)
        {
            int code = CTqso.status & 0x44;
        
            qso->setSerialNumber(CTqso.serial);
            if (code == 0)
                qso->setSSPrecedence('Q');
            else if (code == 0x40)
                qso->setSSPrecedence('B');
            else
                qso->setSSPrecedence('A');
            qso->setSSCheck(CTqso.check);
            qso->setSection(CTqso.info, CT_SECTION_SIZE);
        }
        else if (header.contest == CQ_WW_CONTEST)
        {
            qso->setReceivedRST(CTqso.rst);
            qso->setSentRST(sentRST);
            qso->setCqZone(CTqso.info);
        }
        else if (header.contest == CQ_WPX_CONTEST)
        {
            qso->setReceivedRST(CTqso.rst);
            qso->setSentRST(sentRST);
            qso->setSerialNumber(CTqso.info);
        }
        else if (header.contest == FD_CONTEST)
        {
            qso->setFdTransmitters(CTqso.fd.tx_count);        // Transmitter count.
            qso->setFdCategory(CTqso.fd.category);            // A-E.
            qso->setSection(CTqso.info, 5);         // ARRL section.
        }
        else if (header.contest == WAE_CONTEST)
        {
            qso->setReceivedRST(CTqso.rst);
            qso->setSentRST(sentRST);
            qso->setInfo(CTqso.info, 5);
            qso->setQtcNum(CTqso.qtc_num);
        }
        else
        {
            /* This code handles ARRL DX, WPX, ARRL 10M, ARRL 160M, WAE, VHF QSO, */
            /*  CQ 160 and DXpedition mode */
            qso->setReceivedRST(CTqso.rst);
            qso->setSentRST(sentRST);
            qso->setInfo(CTqso.info, 5);
        }
        
        location++;
        return TRUE;
    }
    else
        return FALSE;
}

boolean RCtFile::writeQso(qso_data *qso)
{
    // Not yet implemented.
    qso = qso;                  // Get rid of compiler warnings.
#if 0    
    ct_log_data ct_qso;

    memset(&ct_qso, '\0', sizeof(ct_qso));

    strncpy(ct_qso.call, qso->getCallsign(), CT_CALL_SIZE);
    ct_qso.time = qso->getTime();
    for (char index = 1; bands[index] != (unsigned char)NULL; index++)
        if (bands[index] == qso->getBand())
            ct_qso.band = index;
    ct_qso.mode = qso->getMode();

    /* Now extract contest specific information. */
    if (header.contest == ARRL_SS_CONTEST)
    {
        ct_qso.serial = (short)qso->getSSSerialNumber();

        /* WARNING!!! No precedence code has been implemented yet!!!!! */
        
        ct_qso.check = qso->getSSCheck();
        strncpy(ct_qso.info, qso->getSSSection(), CT_SECTION_SIZE);
    }
    else
    {
        ct_qso.rst = (short)qso->getReceivedRST();
        strncpy(ct_qso.info, qso->getInfo(), 5);
    }
    
    return TRUE;
#endif
    return FALSE;    
}
