// nafile.cpp
//
// Copyright (C) 1999  Robert Barron, KA5WSS
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "nafile.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#define MAX_NA_CONTEST      15
unsigned char contests[MAX_NA_CONTEST + 1] = {NA_SPRINT_CONTEST, NA_QSO_PARTY_CONTEST,
                              IARU_CONTEST, ARRL_SS_CONTEST, ARRL_10M_CONTEST,
                              ARRL_160M_CONTEST, CQ_160M_CONTEST, FD_CONTEST, 0,
                              ARRL_DX_CONTEST, ARRL_DX_CONTEST_DX,  0, 0,
                              ARRL_VHF_QSO_PARTY, CQ_WPX_CONTEST, CQ_WW_CONTEST};

static int modes[] = {CW_MODE, SSB_MODE};
#define NA_MAX_MODE    1

static long bands[] = {0, 2, 4, 8, 11, 15, 19, 22, 25, 30, 54, 77, 148, 225, 450, 928, 1300, 2450,
                        3500, 6000, 10500, 24250, 47200, 77000, 120000, 150000, 250000};
#define NA_MAX_BAND    BAND_241GHZ                       

RNaFile::RNaFile(char *name)
{
    RNaFile::open(name);
}

boolean RNaFile::open(char *name, int mode)
{
    char tempStr[11];
    
#ifdef LINUX
    if (!RFile::open(name, mode))
        return FALSE;
#else
    // State ios::binary again just in case the user does not.
    if (!RFile::open(name, mode | ios::binary))
        return FALSE;
#endif

    if (!(mode & ios::out))          // Not an output file
    {
        /* Read in the header information and store in neutral format. */        
        RFile::read((void *)&naheader, sizeof(NaHeader));

        NaStringCopy(header.callsign, naheader.callsign, NA_CALL_SIZE);
        NaStringCopy(header.name, naheader.name, 30);
        
        NaStringCopy(header.address, naheader.address, 30);
        NaStringCopy(header.city, naheader.city, 30);
        NaStringCopy(header.state, naheader.state, 2);
        NaStringCopy(header.zip, naheader.zip, 5);

        header.stationNumber = (unsigned char)naheader.stationNumber;
        
        NaStringCopy(header.club, naheader.clubName, 30);

        if (naheader.contest > MAX_NA_CONTEST)
            return FALSE;
        else
            if ((header.contest = contests[naheader.contest]) == 0)
                return FALSE;

        // Store the CW messages
        NaStringCopy(header.CQ_CW_Message, naheader.cq_msg, 32);
        NaStringCopy(header.QRZ_CW_Message, naheader.qrz_msg, 32);
        NaStringCopy(header.Exchange_CW_Message, naheader.ex_msg, 32);
        NaStringCopy(header.Dupe_CW_Message, naheader.b4_msg, 32);

        // Don't bother with these fields unless they are specifically needed.
        switch (header.contest) {
            case ARRL_SS_CONTEST:
                header.ss.precedence = naheader.misc.ss.ss_prec;
                NaStringCopy(tempStr, naheader.misc.ss.ss_check, 2);
                header.ss.check = (unsigned char)atoi(tempStr);
                NaStringCopy(header.ss.section, naheader.location, 4);
                break;
            case FD_CONTEST:
                NaStringCopy(tempStr, naheader.misc.ex_info, 8);
                header.fd.category = tempStr[strlen(tempStr) - 1];
                tempStr[strlen(tempStr) - 1] = (char)NULL;
                header.fd.transmitters = (unsigned char)atoi(tempStr);
                break;
            case CQ_WW_CONTEST:
                NaStringCopy(tempStr, naheader.location, 4);
                header.zone = (unsigned char)atoi(tempStr);
            case ARRL_10M_CONTEST:
            case ARRL_160M_CONTEST:
                // Should save QTH somewhere!!!!
            case NA_QSO_PARTY_CONTEST:
                // Should save the nickname somewhere!!!!
            default:
                break;
        };
    }
    
    location = 0;

    return TRUE;
}

/* This routine takes as input the neutral format header from the input file */
/*  and stores it in this object.  It also writes out the header to its file */
/*  in its native format. */
boolean RNaFile::writeHeader(header_data newheader)
{
    // This capability not yet implemented.
    newheader = newheader;
    
#if 0    
    RQsoFile::storeHeader(newheader);

    memset(&naheader, '\0', sizeof(NaHeader));

    RFile::write(&naheader, sizeof(NaHeader));
    
    headerWritten = TRUE;
    return TRUE;
#endif
    return FALSE;    
}

boolean RNaFile::nextQso(qso_data *qso)
{
    NaQso na_qso;
    char tempStr[11];

    qso->init();
    memset(&na_qso, '\0', sizeof(NaQso));
    if (RFile::read((void *)&na_qso, sizeof(NaQso)))
    {
        time_t baseTime = 315532800;                // Unix time at midnight 1/1/1980
            
        long freq = (na_qso.freq >> 2) / 10000;     // Convert to freq in MHz
    
        NaStringCopy(tempStr, na_qso.callsign, NA_CALL_SIZE);
        qso->setCallsign(tempStr);
        qso->setTime((time_t)na_qso.date + baseTime);

        qso->setFreq(na_qso.freq >> 2);                   // Store full frequency.
        qso->setBand(1);                                  // Determine the proper Amateur band.
        while (freq > bands[qso->getBand()])
            qso->setBand(qso->getBand() + 1);

        if (na_qso.mode <= NA_MAX_MODE)
            qso->setMode((char)modes[na_qso.mode]);       // Convert to generic mode id.
        else
            return FALSE;                               // Not a supported mode!

        switch (header.contest) {
            case ARRL_SS_CONTEST:
                NaStringCopy(tempStr, na_qso.info, 4);
                qso->setSerialNumber(tempStr);
                qso->setSSPrecedence(na_qso.info[4]);
                NaStringCopy(tempStr, &na_qso.info[9], 2);
                qso->setSSCheck(tempStr);
                NaStringCopy(tempStr, na_qso.qth, 4);
                qso->setSection(tempStr);
                break;
            case CQ_WW_CONTEST:
                NaStringCopy(tempStr, na_qso.info, 3);
                qso->setReceivedRST(tempStr);
                NaStringCopy(tempStr, na_qso.qth, 4);
                qso->setCqZone(tempStr);
                break;
            case IARU_CONTEST:
                NaStringCopy(tempStr, na_qso.info, 3);
                qso->setReceivedRST(tempStr);
                NaStringCopy(tempStr, na_qso.qth, 4);
                if (isdigit(tempStr[0]))
                    qso->setItuZone(tempStr);
                else
                    qso->setInfo(tempStr);
                break;
            case CQ_WPX_CONTEST:
                NaStringCopy(tempStr, na_qso.info, 3);
                qso->setReceivedRST(tempStr);
                NaStringCopy(tempStr, &na_qso.info[3], 4);
                qso->setSerialNumber(tempStr);
                break;
            case FD_CONTEST:
                NaStringCopy(tempStr, na_qso.info, 8);
                qso->setFdCategory(tempStr[strlen(tempStr) - 1]);       // A-E.
                tempStr[strlen(tempStr) - 1] = (char)NULL;
                qso->setFdTransmitters(tempStr); // Transmitter count.
                NaStringCopy(tempStr, na_qso.qth, 4);          // ARRL section.
                qso->setSection(tempStr);
                break;
            case NA_QSO_PARTY_CONTEST:
                NaStringCopy(tempStr, na_qso.info, 11);
                qso->setName(tempStr);
                NaStringCopy(tempStr, na_qso.qth, 4);
                qso->setInfo(tempStr);
                break;
            case WAE_CONTEST:
//            qso->receivedRST = CTqso.rst;
//            strncpy(qso->info, CTqso.info, 5);
//            qso->qtc_num = CTqso.qtc_num;
                break;
            case NA_SPRINT_CONTEST:
                NaStringCopy(tempStr, na_qso.info, 3);
                qso->setSerialNumber(tempStr);
                NaStringCopy(tempStr, &na_qso.info[3], 8);
                qso->setName(tempStr);
                NaStringCopy(tempStr, na_qso.qth, 4);
                qso->setInfo(tempStr);
                break;
            case ARRL_DX_CONTEST:
            case ARRL_10M_CONTEST:
            case ARRL_160M_CONTEST:
            default:
                NaStringCopy(tempStr, na_qso.info, 3);
                qso->setReceivedRST(tempStr);
                NaStringCopy(tempStr, na_qso.qth, 4);
                qso->setInfo(tempStr);
                break;
        };
        
        location++;
        return TRUE;
    }
    else
        return FALSE;
}

boolean RNaFile::writeQso(qso_data *qso)
{
    NaQso na_qso;

    memset(&na_qso, '\0', sizeof(NaQso));

    NaStringCopy(na_qso.callsign, qso->getCallsign(), NA_CALL_SIZE);

    return TRUE;
}

void RNaFile::NaStringCopy(char *resultString, const char *naString, int length)
{
    char *buffer = new char[length + 1];
    char *tmpPtr;

    strncpy(buffer, naString, length);
    for (tmpPtr = &buffer[length - 1]; ((*tmpPtr == ' ') && (tmpPtr > buffer)); tmpPtr--)
        ;
    *(tmpPtr + 1) = (char)NULL;
    strcpy(resultString, buffer);

    delete buffer;
    return;
}

