/*
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include <ctype.h>
#include <errno.h>
#include "compat/strerror.h"
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include "compat/strftime.h"
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <netdb.h>
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#include <termios.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <netinet/in.h>
#include "compat/netinet_in.h"
#include <arpa/inet.h>
#include "compat/inet_ntoa.h"
#include "packet.h"
#include "init_protocol.h"
#include "udp_protocol.h"
#include "bnet_protocol.h"
#include "tag.h"
#include "bn_type.h"
#include "field_sizes.h"
#include "network.h"
#include "version.h"
#include "util.h"
#ifdef CLIENTDEBUG
#include "eventlog.h"
#endif
#include "bnettime.h"
#include "client.h"


#define COMPNAMELEN 128
#define SCREEN_WIDTH 80 /* FIXME need to determine real terminal width here */

#ifdef CLIENTDEBUG
#define dprintf printf
#else
#define dprintf if (0) printf
#endif

static char const * bnclass_get_str(unsigned int class); /* FIXME: this is also in command.c */
static void usage(char const * progname);


static char const * bnclass_get_str(unsigned int class)
{
    switch (class)
    {
    case PLAYERINFO_DRTL_CLASS_WARRIOR:
	return "warrior";
    case PLAYERINFO_DRTL_CLASS_ROGUE:
	return "rogue";
    case PLAYERINFO_DRTL_CLASS_SORCERER:
	return "sorcerer";
    default:
	return "unknown";
    }
}


static void usage(char const * progname)
{
    fprintf(stderr,"usage: %s [<options>] [<servername> [<TCP portnumber>]]\n"
            "    -b, --client=SEXP           report client as Brood Wars\n"
            "    -d, --client=DRTL           report client as Diablo Retail\n"
            "    --client=DSHR               report client as Diablo Shareware\n"
            "    -s, --client=STAR           report client as Starcraft (default)\n"
            "    --client=SSHR               report client as Starcraft Shareware\n"
	    "    -o NAME, --owner=NAME       report CD owner as NAME\n"
	    "    -k KEY, --cdkey=KEY         report CD key as KEY\n"
	    "    -p PLR, --player=PLR        print stats for player PLR\n"
	    "    --bnetd                     also print BNETD-specific stats\n"
	    "    --fsgs                      also print FSGS-specific stats\n"
            "    -h, --help, --usage         show this information and exit\n"
            "    -v, --version               print version number and exit\n",
	    progname);
    exit(1);
}


int main(int argc, char * argv[])
{
    int                a;
    int                sd;
    int                gotplayer=0;
    struct sockaddr_in saddr;
    t_packet *         packet;
    t_packet *         rpacket;
    char               compname[COMPNAMELEN];
    char const *       cdowner=NULL;
    char const *       cdkey=NULL;
    char const *       clienttag=NULL;
    char const *       servname=NULL;
    unsigned short     servport=0;
    char const *       username;
    char               text[MAX_MESSAGE_LEN]="";
    struct hostent *   host;
    unsigned int       commpos;
    struct termios     in_attr_old;
    struct termios     in_attr_new;
    int                changed_in;
    unsigned int       count;
    unsigned int       sessionkey;
    unsigned int       val;
    int                fd_stdin;
    int                use_bnetd=0;
    int                use_fsgs=0;
    
    if (argc<1 || !argv || !argv[0])
    {
	fprintf(stderr,"bad arguments\n");
	return 1;
    }
    
    for (a=1; a<argc; a++)
	if (servname && isdigit((int)argv[a][0]) && a+1>=argc)
	{
            if (str_to_ushort(argv[a],&servport)<0)
            {
                fprintf(stderr,"%s: \"%s\" should be a positive integer\n",argv[0],argv[a]);
                usage(argv[0]);
            }
	}
	else if (!servname && argv[a][0]!='-' && a+2>=argc)
	    servname = argv[a];
        else if (strcmp(argv[a],"-b")==0 || strcmp(argv[a],"--client=SEXP")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_BROODWARS;
	}
        else if (strcmp(argv[a],"-d")==0 || strcmp(argv[a],"--client=DRTL")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_DIABLORTL;
	}
        else if (strcmp(argv[a],"--client=DSHR")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_DIABLOSHR;
	}
        else if (strcmp(argv[a],"-s")==0 || strcmp(argv[a],"--client=STAR")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_STARCRAFT;
	}
        else if (strcmp(argv[a],"--client=SSHR")==0)
	{
	    if (clienttag)
	    {
		fprintf(stderr,"%s: client type was already specified as \"%s\"\n",argv[0],clienttag);
		usage(argv[0]);
	    }
	    clienttag = CLIENTTAG_SHAREWARE;
	}
	else if (strncmp(argv[a],"--client=",9)==0)
	{
	    fprintf(stderr,"%s: unknown client tag \"%s\"\n",argv[0],&argv[a][9]);
	    usage(argv[0]);
	}
        else if (strcmp(argv[a],"-o")==0)
        {
            if (a+1>=argc)
            {
                fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
                usage(argv[0]);
            }
            if (cdowner)
            {
                fprintf(stderr,"%s: CD owner was already specified as \"%s\"\n",argv[0],cdowner);
                usage(argv[0]);
            }
            cdowner = argv[++a];
        }
        else if (strncmp(argv[a],"--owner=",8)==0)
        {
            if (cdowner)
            {
                fprintf(stderr,"%s: CD owner was already specified as \"%s\"\n",argv[0],cdowner);
                usage(argv[0]);
            }
            cdowner = &argv[a][8];
        }
	else if (strcmp(argv[a],"-k")==0)
	{
	    if (a+1>=argc)
            {
                fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
                usage(argv[0]);
            }
	    if (cdkey)
	    {
		fprintf(stderr,"%s: CD key was already specified as \"%s\"\n",argv[0],cdkey);
		usage(argv[0]);
	    }
	    cdkey = argv[++a];
	}
	else if (strncmp(argv[a],"--cdkey=",8)==0)
	{
	    if (cdkey)
	    {
		fprintf(stderr,"%s: CD key was already specified as \"%s\"\n",argv[0],cdkey);
		usage(argv[0]);
	    }
	    cdkey = &argv[a][8];
	}
	else if (strcmp(argv[a],"-p")==0)
	{
	    if (a+1>=argc)
            {
                fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
                usage(argv[0]);
            }
	    if (gotplayer)
	    {
		fprintf(stderr,"%s: player was already specified as \"%s\"\n",argv[0],text);
		usage(argv[0]);
	    }
	    a++;
	    if (argv[a][0]=='\0')
	    {
		fprintf(stderr,"%s: player name can not be empty\n",argv[0]);
		usage(argv[0]);
	    }
	    strncpy(text,argv[a],MAX_MESSAGE_LEN);
	    text[MAX_MESSAGE_LEN-1] = '\0';
	    gotplayer = 1;
	}
	else if (strncmp(argv[a],"--player=",9)==0)
	{
	    if (gotplayer)
	    {
		fprintf(stderr,"%s: player was already specified as \"%s\"\n",argv[0],text);
		usage(argv[0]);
	    }
	    if (argv[a][9]=='\0')
	    {
		fprintf(stderr,"%s: player name can not be empty\n",argv[0]);
		usage(argv[0]);
	    }
	    strncpy(text,&argv[a][9],MAX_MESSAGE_LEN);
	    text[MAX_MESSAGE_LEN-1] = '\0';
	    gotplayer = 1;
	}
        else if (strcmp(argv[a],"--bnetd")==0)
	    use_bnetd = 1;
        else if (strcmp(argv[a],"--fsgs")==0)
	    use_fsgs = 1;
        else if (strcmp(argv[a],"--client")==0)
	{
	    fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
        else if (strcmp(argv[a],"--owner")==0)
	{
	    fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
        else if (strcmp(argv[a],"--cdkey")==0)
	{
	    fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
        else if (strcmp(argv[a],"--player")==0)
	{
	    fprintf(stderr,"%s: option \"%s\" requires an argument\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
	else if (strcmp(argv[a],"-v")==0 || strcmp(argv[a],"--version")==0)
	{
            printf("version "BNETD_VERSION"\n");
            return 0;
	}
	else if (strcmp(argv[a],"-h")==0 || strcmp(argv[a],"--help")==0 || strcmp(argv[a],"--usage")
==0)
            usage(argv[0]);
	else
	{
	    fprintf(stderr,"%s: unknown option \"%s\"\n",argv[0],argv[a]);
	    usage(argv[0]);
	}
    
    if (servport==0)
	servport = BNETD_SERV_PORT;
    if (!cdowner)
	cdowner = BNETD_DEFAULT_OWNER;
    if (!cdkey)
	cdkey = BNETD_DEFAULT_KEY;
    if (!clienttag)
	clienttag = CLIENTTAG_STARCRAFT;
    if (!servname)
	servname = BNETD_DEFAULT_HOST;
    
    if (!(host = gethostbyname(servname)))
    {
	fprintf(stderr,"%s: unknown host \"%s\"\n",argv[0],servname);
	return 1;
    }
    
    if (host->h_addrtype!=AF_INET)
    {
	fprintf(stderr,"%s: host is not in IPv4 address family\n",argv[0]);
	return 1;
    }
    
    if (gethostname(compname,COMPNAMELEN)<0)
    {
	fprintf(stderr,"%s: could not get host name (gethostname: %s)\n",argv[0],strerror(errno));
	return 1;
    }
    if (!(username = getlogin()))
	username = "detached";
    
    if (gotplayer)
	changed_in = 0; /* no need to change terminal attributes */
    else
    {
	fd_stdin = fileno(stdin);
	if (tcgetattr(fd_stdin,&in_attr_old)>=0)
	{
	    in_attr_new = in_attr_old;
	    in_attr_new.c_lflag &= ~(ECHO | ICANON); /* turn off ECHO and ICANON */
	    in_attr_new.c_cc[VMIN]  = 1; /* require reads to return at least one byte */
	    in_attr_new.c_cc[VTIME] = 0; /* no timeout */
	    tcsetattr(fd_stdin,TCSANOW,&in_attr_new);
	    changed_in = 1;
	}
	else
	{
	    fprintf(stderr,"%s: could not set terminal attributes for stdin\n",argv[0]);
	    changed_in = 0;
	}
    }
    
    if ((sd = socket(PF_INET,SOCK_STREAM,0))<0)
    {
	fprintf(stderr,"%s: could not create socket (socket: %s)\n",argv[0],strerror(errno));
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    
    memset(&saddr,0,sizeof(saddr));
    saddr.sin_family = AF_INET;
    saddr.sin_port   = htons(servport);
    memcpy(&saddr.sin_addr.s_addr,host->h_addr_list[0],host->h_length);
    if (connect(sd,(struct sockaddr *)&saddr,sizeof(saddr))<0)
    {
	fprintf(stderr,"%s: could not connect to server \"%s\" port %hu (connect: %s)\n",argv[0],servname,servport,strerror(errno));
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    
    printf("Connected to %s:%hu.\n",inet_ntoa(saddr.sin_addr),servport);
    
#ifdef CLIENTDEBUG
    eventlog_set(stderr);
#endif
    
    /* wait for udptest */
    {
	int                lsock;
	struct sockaddr_in laddr;
	char               buff[BNET_UDPTEST_SIZE];
	int                ok=1;
	time_t             start;
	
	if ((lsock = socket(PF_INET,SOCK_DGRAM,0))<0)
	{
	    fprintf(stderr,"%s: could not create UDP socket (socket: %s)\n",argv[0],strerror(errno));
	    ok = 0;
	}
	else
	{
	    if (fcntl(lsock,F_SETFL,O_NONBLOCK)<0)
		fprintf(stderr,"%s: could not set UDP socket to non-blocking mode (fcntl: %s)\n",argv[0],strerror(errno));
	    memset(&laddr,0,sizeof(laddr));
	    laddr.sin_family = AF_INET;
	    laddr.sin_port = htons(BNETD_TEST_PORT);
	    laddr.sin_addr.s_addr = htonl(INADDR_ANY);
	    if (bind(lsock,(struct sockaddr *)&laddr,sizeof(laddr))<0)
	    {
		fprintf(stderr,"%s: could bind to UDP port %d (bind: %s)\n",argv[0],BNETD_TEST_PORT,strerror(errno));
		close(lsock);
		ok = 0;
	    }
	}
	
	{
	    t_connect_header temp;
	    
	    bn_byte_set(&temp.class,CONNECT_CLASS_NORMAL);
	    if (write(sd,&temp,sizeof(temp))!=sizeof(temp))
	    {
		fprintf(stderr,"%s: could not write to server (write: %s)\n",argv[0],strerror(errno));
		if (changed_in)
		    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		return 1;
	    }
	}
	
	if (ok)
	{
	    start = time(NULL);
	    count = 0;
	    while (start+(time_t)5>=time(NULL))  /* wait 5 seconds */
	    {
#ifdef HAVE_RECV /* some recvfrom()s don't handle NULL, but recv is sorta depreciated */
	        if (recv(lsock,buff,BNET_UDPTEST_SIZE,0)!=BNET_UDPTEST_SIZE)
#else
	        if (recvfrom(lsock,buff,BNET_UDPTEST_SIZE,0,NULL,NULL)!=BNET_UDPTEST_SIZE)
#endif
		{
		    if (errno!=EAGAIN && errno!=EWOULDBLOCK)
			fprintf(stderr,"%s: failed to receive UDPTEST (recvfrom: %s)\n",argv[0],strerror(errno));
		    continue;
		}
	        start = time(NULL);
		if (memcmp(buff,BNET_UDPTEST,BNET_UDPTEST_SIZE)!=0)
		{
		    fprintf(stderr,"%s: got bad data in UDPTEST\n",argv[0]);
		    continue;
		}
		count++;
		if (count>=2)
		   break;
	    }
	    close(lsock);
	    if (count<2)
		printf("Could not receive UDP packets. Connection may be slow or firewalled.\n");
	}
    }
    dprintf("Got UDP data\n");
    
    /* reuse this same packet over and over */
    if (!(rpacket = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
   
    if (strcmp(clienttag,CLIENTTAG_DIABLOSHR)==0 ||
        strcmp(clienttag,CLIENTTAG_DIABLORTL)==0)
    {
	if (!(packet = packet_create(packet_class_normal)))
	{
	    fprintf(stderr,"%s: could not create packet\n",argv[0]);
	    if (changed_in)
		tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	    return 1;
	}
	packet_set_size(packet,sizeof(t_client_unknown_1b));
	packet_set_type(packet,CLIENT_UNKNOWN_1B);
	bn_short_set(&packet->u.client_unknown_1b.unknown1,CLIENT_UNKNOWN_1B_UNKNOWN3);
	bn_short_nset(&packet->u.client_unknown_1b.port,BNETD_GAME_PORT);
	bn_int_nset(&packet->u.client_unknown_1b.ip,0); /* FIXME */
	bn_int_set(&packet->u.client_unknown_1b.unknown2,CLIENT_UNKNOWN_1B_UNKNOWN3);
	bn_int_set(&packet->u.client_unknown_1b.unknown3,CLIENT_UNKNOWN_1B_UNKNOWN3);
	client_blocksend_packet(sd,packet);
	packet_del_ref(packet);
    }
    
    if (!(packet = packet_create(packet_class_normal)))
    {
        fprintf(stderr,"%s: could not create packet\n",argv[0]);
        if (changed_in)
            tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
        return 1;
    }
#ifdef OLD_PROT
    packet_set_size(packet,sizeof(t_client_compinfo1));
    packet_set_type(packet,CLIENT_COMPINFO1);
    bn_int_set(&packet->u.client_compinfo1.reg_version,CLIENT_COMPINFO1_REG_VERSION);
    bn_int_set(&packet->u.client_compinfo1.reg_auth,CLIENT_COMPINFO1_REG_AUTH);
    bn_int_set(&packet->u.client_compinfo1.client_id,CLIENT_COMPINFO1_CLIENT_ID);
    bn_int_set(&packet->u.client_compinfo1.client_token,CLIENT_COMPINFO1_CLIENT_TOKEN);
    packet_append_string(packet,compname);
    packet_append_string(packet,username);
#else
    packet_set_size(packet,sizeof(t_client_compinfo2));
    packet_set_type(packet,CLIENT_COMPINFO2);
    bn_int_set(&packet->u.client_compinfo2.unknown1,CLIENT_COMPINFO2_UNKNOWN1);
    bn_int_set(&packet->u.client_compinfo2.unknown2,CLIENT_COMPINFO2_UNKNOWN2);
    bn_int_set(&packet->u.client_compinfo2.reg_auth,CLIENT_COMPINFO2_REG_AUTH);
    bn_int_set(&packet->u.client_compinfo2.client_id,CLIENT_COMPINFO2_CLIENT_ID);
    bn_int_set(&packet->u.client_compinfo2.client_token,CLIENT_COMPINFO2_CLIENT_TOKEN);
    packet_append_string(packet,compname);
    packet_append_string(packet,username);
#endif
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_countryinfo));
    packet_set_type(packet,CLIENT_COUNTRYINFO);
    bn_long_set_a_b(&packet->u.client_countryinfo.timestamp1,CLIENT_COUNTRYINFO_TIMESTAMP1_A,CLIENT_COUNTRYINFO_TIMESTAMP1_B);
    bn_long_set_a_b(&packet->u.client_countryinfo.timestamp2,CLIENT_COUNTRYINFO_TIMESTAMP2_A,CLIENT_COUNTRYINFO_TIMESTAMP2_B);
    bn_int_set(&packet->u.client_countryinfo.unknown5,CLIENT_COUNTRYINFO_UNKNOWN5);
    bn_int_set(&packet->u.client_countryinfo.langid1,CLIENT_COUNTRYINFO_LANGID_USENGLISH);
    bn_int_set(&packet->u.client_countryinfo.langid2,CLIENT_COUNTRYINFO_LANGID_USENGLISH);
    bn_int_set(&packet->u.client_countryinfo.langid3,CLIENT_COUNTRYINFO_LANGID_USENGLISH);
    packet_append_string(packet,CLIENT_COUNTRYINFO_LANGSTR_USENGLISH);
    packet_append_string(packet,CLIENT_COUNTRYINFO_COUNTRYCODE_USA);
    packet_append_string(packet,CLIENT_COUNTRYINFO_COUNTRYABB_USA);
    packet_append_string(packet,CLIENT_COUNTRYINFO_COUNTRYNAME_USA);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_unknown_2b));
    packet_set_type(packet,CLIENT_UNKNOWN_2B);
    bn_int_set(&packet->u.client_unknown_2b.unknown1,CLIENT_UNKNOWN_2B_UNKNOWN1);
    bn_int_set(&packet->u.client_unknown_2b.unknown2,CLIENT_UNKNOWN_2B_UNKNOWN2);
    bn_int_set(&packet->u.client_unknown_2b.unknown3,CLIENT_UNKNOWN_2B_UNKNOWN3);
    bn_int_set(&packet->u.client_unknown_2b.unknown4,CLIENT_UNKNOWN_2B_UNKNOWN4);
    bn_int_set(&packet->u.client_unknown_2b.unknown5,CLIENT_UNKNOWN_2B_UNKNOWN5);
    bn_int_set(&packet->u.client_unknown_2b.unknown6,CLIENT_UNKNOWN_2B_UNKNOWN6);
    bn_int_set(&packet->u.client_unknown_2b.unknown7,CLIENT_UNKNOWN_2B_UNKNOWN7);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_progident));
    packet_set_type(packet,CLIENT_PROGIDENT);
    bn_int_tag_set(&packet->u.client_progident.archtag,ARCHTAG_INTEL);
    bn_int_tag_set(&packet->u.client_progident.clienttag,clienttag);
    bn_int_set(&packet->u.client_progident.versionid,0xBD);
    bn_int_set(&packet->u.client_progident.unknown1,CLIENT_PROGIDENT_UNKNOWN1);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	   fprintf(stderr,"%s: server closed connection\n",argv[0]);
	   if (changed_in)
	       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	   return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_COMPREPLY);
    dprintf("Got COMPREPLY\n");
    

#ifdef OLD_PROT
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
        {
           fprintf(stderr,"%s: server closed connection\n",argv[0]);
           if (changed_in)
               tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
           return 1;
        }
    while (packet_get_type(rpacket)!=SERVER_SESSIONKEY1);
    sessionkey = bn_int_get(rpacket->u.server_sessionkey1.sessionkey);
    dprintf("Got SESSIONKEY1 (0x%08x)\n",sessionkey);
#else
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
        {
           fprintf(stderr,"%s: server closed connection\n",argv[0]);
           if (changed_in)
               tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
           return 1;
        }
    while (packet_get_type(rpacket)!=SERVER_SESSIONKEY2);
    sessionkey = bn_int_get(rpacket->u.server_sessionkey2.sessionkey);
    dprintf("Got SESSIONKEY2 (0x%08x)\n",sessionkey);
#endif
    
    /* get authreply */
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_iconreq));
    packet_set_type(packet,CLIENT_ICONREQ);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
   
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	    fprintf(stderr,"%s: server closed connection\n",argv[0]);
	    if (changed_in)
		tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	    return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_ICONREPLY);
    dprintf("Got ICONREPLY\n");
    
    if (strcmp(clienttag,CLIENTTAG_STARCRAFT)==0 ||
	strcmp(clienttag,CLIENTTAG_BROODWARS)==0)
    {
	if (!(packet = packet_create(packet_class_normal)))
	{
	    fprintf(stderr,"%s: could not create packet\n",argv[0]);
	    if (changed_in)
		tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	    return 1;
	}
	packet_set_size(packet,sizeof(t_client_cdkey));
	packet_set_type(packet,CLIENT_CDKEY);
	bn_int_set(&packet->u.client_cdkey.unknown1,CLIENT_CDKEY_UNKNOWN1);
	packet_append_string(packet,cdkey);
	packet_append_string(packet,cdowner);
	client_blocksend_packet(sd,packet);
	packet_del_ref(packet);
	
	do
	    if (client_blockrecv_packet(sd,rpacket)<0)
	    {
		fprintf(stderr,"%s: server closed connection\n",argv[0]);
		if (changed_in)
		    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		return 1;
	    }
	while (packet_get_type(rpacket)!=SERVER_CDKEYREPLY);
	dprintf("Got CDKEYREPLY (%u)\n",bn_int_get(rpacket->u.server_cdkeyreply.message));
    }
    
    if (!(packet = packet_create(packet_class_normal)))
    {
	fprintf(stderr,"%s: could not create packet\n",argv[0]);
	if (changed_in)
	    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	return 1;
    }
    packet_set_size(packet,sizeof(t_client_tosreq));
    packet_set_type(packet,CLIENT_TOSREQ);
    bn_int_set(&packet->u.client_tosreq.unknown1,CLIENT_TOSREQ_UNKNOWN1);
    bn_int_set(&packet->u.client_tosreq.unknown2,CLIENT_TOSREQ_UNKNOWN2);
    packet_append_string(packet,CLIENT_TOSREQ_FILEUSA);
    client_blocksend_packet(sd,packet);
    packet_del_ref(packet);
    do
        if (client_blockrecv_packet(sd,rpacket)<0)
	{
	   fprintf(stderr,"%s: server closed connection\n",argv[0]);
	   if (changed_in)
	       tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	   return 1;
	}
    while (packet_get_type(rpacket)!=SERVER_TOSREPLY);
    dprintf("Got TOSREPLY\n");
    
    if (!gotplayer)
    {
	commpos = 0;
	text[0] = '\0';
    }
    
    for (;;)
    {
	unsigned int i;
	
	if (!gotplayer)
	{
	    switch (client_get_comm("player: ",text,MAX_MESSAGE_LEN,&commpos,1,SCREEN_WIDTH))
	    {
	    case -1: /* cancel or error */
                break;
		
	    case 0: /* timeout */
                continue;
		
	    case 1:
                if (text[0]=='\0')
            	    continue;
	    }
	    
	    if (text[0]=='\0')
	        break;
            printf("\r        ");
            for (i=0; i<strlen(text) && i<SCREEN_WIDTH-2; i++)
                printf(" ");
	    printf("\r");
	}
	
	if (!(packet = packet_create(packet_class_normal)))
	{
	    fprintf(stderr,"%s: could not create packet\n",argv[0]);
	    if (changed_in)
		tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
	    return 1;
	}
	packet_set_size(packet,sizeof(t_client_statsreq));
	packet_set_type(packet,CLIENT_STATSREQ);
	bn_int_set(&packet->u.client_statsreq.name_count,1);
	bn_int_set(&packet->u.client_statsreq.unknown1,CLIENT_STATSREQ_UNKNOWN1);
	packet_append_string(packet,text);
	count = 0;
	
	if (use_bnetd)
	{
	    packet_append_string(packet,"BNET\\acct\\username");
	    packet_append_string(packet,"BNET\\acct\\userid");
	    packet_append_string(packet,"BNET\\acct\\lastlogin_clienttag");
	    packet_append_string(packet,"BNET\\acct\\lastlogin_connection");
	    packet_append_string(packet,"BNET\\acct\\lastlogin_time");
	    packet_append_string(packet,"BNET\\acct\\firstlogin_clienttag");
	    packet_append_string(packet,"BNET\\acct\\firstlogin_connection");
	    packet_append_string(packet,"BNET\\acct\\firstlogin_time");
	    count += 8;
	}
	if (use_fsgs)
	{
	    packet_append_string(packet,"FSGS\\Created");
	    count += 1;
	}
	
	packet_append_string(packet,"profile\\sex");
	packet_append_string(packet,"profile\\age");
	packet_append_string(packet,"profile\\location");
	packet_append_string(packet,"profile\\description");
	count += 4;
	
	if (strcasecmp(clienttag,"STAR")==0)
	{
	    packet_append_string(packet,"Record\\STAR\\0\\last game");
	    packet_append_string(packet,"Record\\STAR\\0\\last game result");
	    packet_append_string(packet,"Record\\STAR\\0\\wins");
	    packet_append_string(packet,"Record\\STAR\\0\\losses");
	    packet_append_string(packet,"Record\\STAR\\0\\disconnects");
	    packet_append_string(packet,"Record\\STAR\\0\\draws");
	    count += 6;
	    
	    packet_append_string(packet,"Record\\STAR\\1\\last game");
	    packet_append_string(packet,"Record\\STAR\\1\\last game result");
	    packet_append_string(packet,"Record\\STAR\\1\\rating");
	    packet_append_string(packet,"Record\\STAR\\1\\active rating");
	    packet_append_string(packet,"Record\\STAR\\1\\high rating");
	    packet_append_string(packet,"Record\\STAR\\1\\rank");
	    packet_append_string(packet,"Record\\STAR\\1\\active rank");
	    packet_append_string(packet,"Record\\STAR\\1\\high rank");
	    packet_append_string(packet,"Record\\STAR\\1\\wins");
	    packet_append_string(packet,"Record\\STAR\\1\\losses");
	    packet_append_string(packet,"Record\\STAR\\1\\disconnects");
	    packet_append_string(packet,"Record\\STAR\\1\\draws");
	    count += 12;
	}
	else if (strcasecmp(clienttag,"SEXP")==0)
	{
	    packet_append_string(packet,"Record\\SEXP\\0\\last game");
	    packet_append_string(packet,"Record\\SEXP\\0\\last game result");
	    packet_append_string(packet,"Record\\SEXP\\0\\wins");
	    packet_append_string(packet,"Record\\SEXP\\0\\losses");
	    packet_append_string(packet,"Record\\SEXP\\0\\disconnects");
	    packet_append_string(packet,"Record\\SEXP\\0\\draws");
	    count += 6;
	    
	    packet_append_string(packet,"Record\\SEXP\\1\\last game");
	    packet_append_string(packet,"Record\\SEXP\\1\\last game result");
	    packet_append_string(packet,"Record\\SEXP\\1\\rating");
	    packet_append_string(packet,"Record\\SEXP\\1\\active rating");
	    packet_append_string(packet,"Record\\SEXP\\1\\high rating");
	    packet_append_string(packet,"Record\\SEXP\\1\\rank");
	    packet_append_string(packet,"Record\\SEXP\\1\\active rank");
	    packet_append_string(packet,"Record\\SEXP\\1\\high rank");
	    packet_append_string(packet,"Record\\SEXP\\1\\wins");
	    packet_append_string(packet,"Record\\SEXP\\1\\losses");
	    packet_append_string(packet,"Record\\SEXP\\1\\disconnects");
	    packet_append_string(packet,"Record\\SEXP\\1\\draws");
	    count += 12;
	}
	else if (strcasecmp(clienttag,"SSHR")==0)
	{
	    packet_append_string(packet,"Record\\SSHR\\0\\last game");
	    packet_append_string(packet,"Record\\SSHR\\0\\last game result");
	    packet_append_string(packet,"Record\\SSHR\\0\\wins");
	    packet_append_string(packet,"Record\\SSHR\\0\\losses");
	    packet_append_string(packet,"Record\\SSHR\\0\\disconnects");
	    packet_append_string(packet,"Record\\SSHR\\0\\draws");
	    count += 6;
	}
	else if (strcasecmp(clienttag,"DSHR")==0 ||
		 strcasecmp(clienttag,"DRTL")==0)
	{
	    if (use_bnetd)
	    {
		packet_append_string(packet,"BNET\\Record\\DRTL\\0\\level");
		packet_append_string(packet,"BNET\\Record\\DRTL\\0\\class");
		packet_append_string(packet,"BNET\\Record\\DRTL\\0\\strength");
		packet_append_string(packet,"BNET\\Record\\DRTL\\0\\magic");
		packet_append_string(packet,"BNET\\Record\\DRTL\\0\\dexterity");
		packet_append_string(packet,"BNET\\Record\\DRTL\\0\\vitality");
		packet_append_string(packet,"BNET\\Record\\DRTL\\0\\gold");
		packet_append_string(packet,"BNET\\Record\\DRTL\\0\\diablo kills");
		count += 8;
	    }
	}
	bn_int_set(&packet->u.client_statsreq.key_count,count);
	client_blocksend_packet(sd,packet);
	packet_del_ref(packet);
	
	do
	    if (client_blockrecv_packet(sd,rpacket)<0)
	    {
		fprintf(stderr,"%s: server closed connection\n",argv[0]);
		if (changed_in)
		    tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
		return 1;
	    }
	while (packet_get_type(rpacket)!=SERVER_STATSREPLY);
	dprintf("Got STATSREPLY\n");
	
	{
	    unsigned int   names,keys;
	    unsigned int   j;
	    unsigned int   strpos;
	    char const *   temp;
	    unsigned int   uid;
	    struct in_addr laddr;
	    time_t         tm;
	    char           timestr[STAT_TIME_MAXLEN];
	    
	    names = bn_int_get(rpacket->u.server_statsreply.name_count);
	    keys = bn_int_get(rpacket->u.server_statsreply.key_count);
	    
	    printf("----\n");
	    
	    strpos = sizeof(t_server_statsreply);
	    for (i=0; i<names; i++)
	    {
		j = 0;
		if (use_bnetd)
		{
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (temp[0]=='\0')
			printf("               Username: UNKNOWN\n");
		    else
			printf("               Username: %s\n",temp);
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&uid)<0 || uid<1)
			printf("                Account: UNKNOWN\n");
		    else
			printf("                Account: #%06u\n",uid);
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    printf("      Last login client: %s\n",temp);
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&val)<0)
			printf("        Last login host: UNKNOWN\n");
		    else
		    {
			laddr.s_addr = htonl(val);
			printf("        Last login host: %s\n",inet_ntoa(laddr));
		    }
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&val)<0 || val==0)
			printf("        Last login time: NEVER\n");
		    else
		    {
			tm = (time_t)val;
			strftime(timestr,STAT_TIME_MAXLEN,STAT_TIME_FORMAT,localtime(&tm));
			printf("        Last login time: %s\n",timestr);
		    }
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    printf("     First login client: %s\n",temp);
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&val)<0)
			printf("       First login host: UNKNOWN\n");
		    else
		    {
			laddr.s_addr = htonl(val);
			printf("       First login host: %s\n",inet_ntoa(laddr));
		    }
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&val)<0 || val==0)
			printf("       First login time: NEVER\n");
		    else
		    {
			tm = (time_t)val;
			strftime(timestr,STAT_TIME_MAXLEN,STAT_TIME_FORMAT,localtime(&tm));
			printf("       First login time: %s\n",timestr);
		    }
		}
		else
		    printf("               Username: %s\n",text);
		if (use_fsgs)
		{
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&val)<0 || val==0)
			printf("          Creation time: NEVER\n");
		    else
		    {
			tm = (time_t)val;
			strftime(timestr,STAT_TIME_MAXLEN,STAT_TIME_FORMAT,localtime(&tm));
			printf("          Creation time: %s\n",timestr);
		    }
		}
		
		if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
		    strpos += strlen(temp)+1;
		else
		    temp = "";
		j++;
    		printf("                    Sex: %s\n",temp);
		
		if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
		    strpos += strlen(temp)+1;
		else
		    temp = "";
		j++;
    		printf("                    Age: %s\n",temp);
		
		if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
		    strpos += strlen(temp)+1;
		else
		    temp = "";
		j++;
    		printf("               Location: %s\n",temp);
		
		if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
		    strpos += strlen(temp)+1;
		else
		    temp = "";
		j++;
    		printf("            Description: %s\n",temp);
		
		if (strcasecmp(clienttag,"STAR")==0 ||
		    strcasecmp(clienttag,"SSHR")==0 ||
		    strcasecmp(clienttag,"SEXP")==0)
		{
		    t_bnettime   bntime;
		    unsigned int wins,losses,disconnects,draws;
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (bnettime_set_str(&bntime,temp)<0)
			strcpy(timestr,"NEVER");
		    else
		    {
			tm = bnettime_to_time(bntime);
			strftime(timestr,STAT_TIME_MAXLEN,STAT_TIME_FORMAT,localtime(&tm));
		    }
	    	    printf("         Last game time: %s\n",timestr);
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
	    	    printf("       Last game result: %s\n",temp);
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&wins)<0)
			wins = 0;
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&losses)<0)
			losses = 0;
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&disconnects)<0)
			disconnects = 0;
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&draws)<0)
			draws = 0;
	            printf("                 Record: (%u/%u/%u) %u draws\n",wins,losses,disconnects,draws);
		}
		    
		if (strcasecmp(clienttag,"STAR")==0 ||
		    strcasecmp(clienttag,"SEXP")==0)
		{
		    t_bnettime   bntime;
		    unsigned int wins,losses,disconnects,draws;
		    unsigned int rating,rank;
		    unsigned int active_rating,active_rank;
		    unsigned int high_rating,high_rank;
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (bnettime_set_str(&bntime,temp)<0)
			strcpy(timestr,"NEVER");
		    else
		    {
			tm = bnettime_to_time(bntime);
			strftime(timestr,STAT_TIME_MAXLEN,STAT_TIME_FORMAT,localtime(&tm));
		    }
	    	    printf("  Last ladder game time: %s\n",timestr);
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
	    	    printf("Last ladder game result: %s\n",temp);
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&rating)<0)
			rating = 0;
		    if (rating)
	    		printf("  Current ladder rating: %u\n",rating);
		    else
	    		printf("  Current ladder rating: UNRATED\n");
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&active_rating)<0)
			active_rating = 0;
		    if (active_rating)
	    		printf("   Active ladder rating: %u\n",active_rating);
		    else
	    		printf("   Active ladder rating: UNRATED\n");
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&high_rating)<0)
			high_rating = 0;
		    if (high_rating)
	    		printf("     High ladder rating: %u\n",high_rating);
		    else
	    		printf("     High ladder rating: UNRATED\n");
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&rank)<0)
			rank = 0;
		    if (rank)
	  		printf("    Current ladder rank: #%u\n",rank);
		    else
			printf("    Current ladder rank: UNRANKED\n");
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&active_rank)<0)
			active_rank = 0;
		    if (active_rank)
	    		printf("     Active ladder rank: #%u\n",active_rank);
		    else
			printf("     Active ladder rank: UNRANKED\n");
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&high_rank)<0)
			high_rank = 0;
		    if (high_rank)
	    		printf("       High ladder rank: #%u\n",high_rank);
		    else
			printf("       High ladder rank: UNRANKED\n");
		    
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&wins)<0)
			wins = 0;
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&losses)<0)
			losses = 0;
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&disconnects)<0)
			disconnects = 0;
		    if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			strpos += strlen(temp)+1;
		    else
			temp = "";
		    j++;
		    if (str_to_uint(temp,&draws)<0)
			draws = 0;
	            printf("          Ladder record: (%u/%u/%u) %u draws\n",wins,losses,disconnects,draws);
		}
		else if (strcasecmp(clienttag,"DSHR")==0 ||
			 strcasecmp(clienttag,"DRTL")==0)
		{
		    unsigned int level;
		    unsigned int class;
		    unsigned int diablo_kills;
		    unsigned int strength;
		    unsigned int magic;
		    unsigned int dexterity;
		    unsigned int vitality;
		    unsigned int gold;
		    
		    if (use_bnetd)
		    {
			if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			    strpos += strlen(temp)+1;
			else
			    temp = "";
			j++;
			if (str_to_uint(temp,&level)<0)
			    level = 0;
			if (level>0)
			    printf("                  Level: %u\n",level);
			else
			    printf("                  Level: NONE\n");
			
			if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			    strpos += strlen(temp)+1;
			else
			    temp = "";
			j++;
			if (str_to_uint(temp,&class)<0)
			    class = 99;
			printf("                  Class: %s\n",bnclass_get_str(class));
			
			if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			    strpos += strlen(temp)+1;
			else
			    temp = "";
			j++;
			if (str_to_uint(temp,&strength)<0)
			    strength = 0;
			
			if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			    strpos += strlen(temp)+1;
			else
			    temp = "";
			j++;
			if (str_to_uint(temp,&magic)<0)
			    magic = 0;
			
			if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			    strpos += strlen(temp)+1;
			else
			    temp = "";
			j++;
			if (str_to_uint(temp,&dexterity)<0)
			    dexterity = 0;
			
			if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			    strpos += strlen(temp)+1;
			else
			    temp = "";
			j++;
			if (str_to_uint(temp,&vitality)<0)
			    vitality = 0;
			
			if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			    strpos += strlen(temp)+1;
			else
			    temp = "";
			j++;
			if (str_to_uint(temp,&gold)<0)
			    gold = 0;
			
			printf("                  Stats: %u str  %u mag  %u dex  %u vit  %u gld\n",
			       strength,
			       magic,
			       dexterity,
			       vitality,
			       gold);
			
			if (j<keys && (temp = packet_get_str_const(rpacket,strpos,256)))
			    strpos += strlen(temp)+1;
			else
			    temp = "";
			j++;
			if (str_to_uint(temp,&diablo_kills)<0)
			    diablo_kills = 0;
			printf("           Diablo Kills: %u\n",diablo_kills);
		    }
		}
	    }
	    printf("----\n");
	}
	
	if (gotplayer)
	    break;
	commpos = 0;
	text[0] = '\0';
    }
    
    if (changed_in)
	tcsetattr(fd_stdin,TCSAFLUSH,&in_attr_old);
    return 0;
}
