// synthfunction.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "synthfunction.h"
#include "wavetable.h"

SynthFunction::SynthFunction(Data* output, double amplitude)
	: QueuedOutputFunction(output, outQSize), myAmp(amplitude) {}

int
SynthFunction::doProcessing() {
	return addToOutQueue(doSynthesis());
}

////////

Buzz::Buzz(Data* output, double freq, int nHarmonics, double amp)
		: SynthFunction(output, amp), mySineTable(buzzTableSize), myPhase(0) {
	initialize();
	set(freq, nHarmonics, amp);
}

Buzz::Buzz(Data* output) : SynthFunction(output, 0.0),
	mySineTable(buzzTableSize), myPhase(0), mySamplingIncr(0) {}

inline double
Buzz::tableLength() {
	return double(buzzTableSize);
}

void
Buzz::initialize() {
	myPhase = 0;
	lengthOverHalfSR = tableLength() / (sampRate() * 2.0);
	Super::initialize();
}

void
Buzz::restoreState() {
	myPhase = 0;
	Super::restoreState();
}

void
Buzz::set(double freq, int nHarmonics, double amp) {
	mySamplingIncr = freq * lengthOverHalfSR;
	harmonicsX2 = nHarmonics * 2;
	myAmp = amp;
}

double
Buzz::doSynthesis() {
	int intPhase = int(myPhase);
	double h2np1 = harmonicsX2 + 1.0;
	double fraction = int((myPhase - intPhase) * h2np1)/h2np1;
	double sineVal = mySineTable[double(intPhase + fraction)];	// interpolates
	double output = 0.0;
	if(sineVal == 0.0)
		output = myAmp;
	else { 
		int k = int(h2np1 * myPhase);
		output = myAmp * (mySineTable[k]/sineVal - 1.)/harmonicsX2;
	} 
	myPhase += mySamplingIncr;
	double tableSize = tableLength();
	while(myPhase >= tableSize)		// wrap phase to prevent huge values
		myPhase -= tableSize;	
	return output;
}
