/*
 * GSIOB.C - interface object handling routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

#define SET_VAL(t, fnc)                                                     \
    {t **p;                                                                 \
     p     = (t **) (hp)->def;                                              \
     *p[0] = fnc(val);}

#define RESET_VAL(t, hp)                                                    \
    {t **p;                                                                 \
     p     = (t **) (hp)->def;                                              \
     *p[0] = (1.0 - f)*(*p[1]) + f*(*p[2]);}

#define GET_VAL(t, hp, xo, dx)                                              \
    {t **p, p1, p2, p3;                                                     \
     double f;                                                              \
     p   = (t **) (hp)->def;                                                \
     p1  = *p[0];                                                           \
     p2  = *p[1];                                                           \
     p3  = *p[2];                                                           \
     f   = (p1 - p2)/(p3 - p2);                                             \
     xo += f*dx;}

int
 PG_edit_mode = FALSE;

HASHTAB
 *PG_callback_tab = NULL;

char
 *PG_TEXT_OBJECT_S = "Text",
 *PG_VARIABLE_OBJECT_S = "Variable",
 *PG_BUTTON_OBJECT_S = "Button",
 *PG_CONTAINER_OBJECT_S = "Container";

extern PG_interface_object
 *SC_DECLARE(PG_find_object, (PG_device *dev, char *s,
			   PG_interface_object *parent));

PFVoid
 SC_DECLARE(PG_lookup_callback, (char *name));

#ifdef HAVE_WINDOW_DEVICE

static int
 SC_DECLARE(_PG_string_value, (int ityp, hashel *hp, char *s));

#endif

static int
 SC_DECLARE(_PG_value_match, (int ityp, hashel *hp, char *val)),
 SC_DECLARE(_PG_value_string, (int ityp, hashel *hp, char *s)),
 SC_DECLARE(_PG_get_val, (hashel *hp, int xo, int dx)),
 SC_DECLARE(_PG_set_text_background_color,
	 (PG_interface_object *iob, int flag));

static PG_interface_object
 *SC_DECLARE(_PG_parent_is_variable, (PG_interface_object *iob)),
 *SC_DECLARE(PG_event_select_visual, 
             (PG_interface_object *iob, PG_event *ev)),
 *SC_DECLARE(PG_event_select_logical, 
             (PG_interface_object *iob, PG_event *ev));

static void
 SC_DECLARE(_PG_shift_curve, (PG_curve *crv, int xo, int yo)),
 SC_DECLARE(PG_edit_interface_object, (PG_interface_object *iob)),
 SC_DECLARE(_PG_set_val, (hashel *hp, int dxn, int dxd)),
 SC_DECLARE(PG_slider, (PG_interface_object *iob, PG_event *ev)),
 SC_DECLARE(PG_toggle, (PG_interface_object *iob, PG_event *ev)),
 SC_DECLARE(PG_handle_variable, 
            (PG_interface_object *iob, PG_event *ev)),
 SC_DECLARE(_PG_find_registered, 
            (PG_interface_object *iob, hashel **php,
	     int *pityp)),
 SC_DECLARE(_PG_draw_text_object, (PG_interface_object *iob)),
 SC_DECLARE(_PG_draw_variable_object, (PG_interface_object *iob)),
 SC_DECLARE(_PG_draw_slider_button, (PG_interface_object *iob)),
 SC_DECLARE(_PG_draw_button_object, (PG_interface_object *iob));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DEFINE_REGION - graphically define a region
 *                  - COORD defines the meaning of the return values
 */

void PG_define_region(dev, coord, pxmn, pxmx, pymn, pymx)
   PG_device *dev;
   int coord;
   REAL *pxmn, *pxmx, *pymn, *pymx;
   {REAL xmn, xmx, ymn, ymx;
    int ix1, ix2, ixmn, ixmx;
    int iy1, iy2, iymn, iymx;
    int lnc, lns, lop, btn, mod, ix, iy, first;

    PG_get_line_color(dev, &lnc);
    PG_get_line_style(dev, &lns);
    PG_get_logical_op(dev, &lop);

    if (dev->xor_parity)
       {PG_set_color_line(dev, dev->WHITE, FALSE);}
    else
       PG_set_color_line(dev, dev->BLACK, FALSE);

    PG_set_line_style(dev, LINE_DOTTED);
    PG_set_logical_op(dev, GS_XOR);

/* wait for the left button to be pressed */
    for (btn = FALSE; btn != MOUSE_LEFT; )
        PG_query_pointer(dev, &ix, &iy, &btn, &mod);

    ix1 = ix;
    iy1 = iy;

    PtoS(dev, ix1, iy1, xmn, ymn);
    StoW(dev, xmn, ymn);

    xmx = xmn;
    ymx = ymn;
    first = TRUE;
    while (btn == MOUSE_LEFT)
       {PG_query_pointer(dev, &ix, &iy, &btn, &mod);

	if (first)
	   first = FALSE;
	else
	   PG_draw_box(dev, xmn, xmx, ymn, ymx);

	ix2 = ix;
	iy2 = iy;
	PtoS(dev, ix2, iy2, xmx, ymx);
	StoW(dev, xmx, ymx);

/* draw the new one */
	PG_draw_box(dev, xmn, xmx, ymn, ymx);};

    PG_set_line_color(dev, lnc);
    PG_set_line_style(dev, lns);
    PG_set_logical_op(dev, lop);

    ixmn = min(ix1, ix2);
    ixmx = max(ix1, ix2);
    switch (dev->quadrant)
       {case QUAD_ONE :
	     iymn = min(iy1, iy2);
	     iymx = max(iy1, iy2);
             break;
        case QUAD_FOUR :
	     iymx = min(iy1, iy2);
	     iymn = max(iy1, iy2);
             break;};

    switch (coord)
       {case WORLDC :
	     PtoS(dev, ixmn, iymn, xmn, ymn);
	     StoW(dev, xmn, ymn);
	     if (pxmn != NULL)
	        *pxmn = xmn;
	     if (pymn != NULL)
	        *pymn = ymn;
	     PtoS(dev, ixmx, iymx, xmx, ymx);
	     StoW(dev, xmx, ymx);
	     if (pxmx != NULL)
                *pxmx = xmx;
	     if (pymx != NULL)
	        *pymx = ymx;
	     break;

        case NORMC :
	     PtoS(dev, ixmn, iymn, xmn, ymn);
	     if (pxmn != NULL)
	        *pxmn = xmn;
	     if (pymn != NULL)
	        *pymn = ymn;
	     PtoS(dev, ixmx, iymx, xmx, ymx);
	     if (pxmx != NULL)
	        *pxmx = xmx;
             *pymx = ymx;
	     break;

        case PIXELC :
	     if (pxmn != NULL)
	        *pxmn = ixmn;
	     if (pymn != NULL)
	        *pymn = iymn;
	     if (pxmx != NULL)
	        *pxmx = ixmx;
	     if (pymx != NULL)
	        *pymx = iymx;
	     break;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_PRINT_POINTER_LOCATION - print the location of the mouse
 *                           - if COORD is TRUE print in WC else
 *                           - print in NDC
 */

void PG_print_pointer_location(dev, cx, cy, coord)
   PG_device *dev;
   double cx, cy;
   int coord;
   {int wx, wy;
    int btn, mod;
    REAL dx, dy;
    char bf[MAXLINE];
    static int owx = 0, owy = 0;

    PG_query_pointer(dev, &wx, &wy, &btn, &mod);
    if ((owx != wx) || (owy != wy))
       {owx = wx;
        owy = wy;

        if (coord)
           sprintf(bf, "x = -0.000e000, y = -0.000e000");
        else
           sprintf(bf, "x = 0000, y = 0000");
        PG_get_text_ext_NDC(dev, bf, &dx, &dy);

        dx += cx;
        dy += cy;

        PG_clear_region_NDC(dev, cx, dx, cy, dy, 2);

        StoW(dev, cx, cy);
        if (coord)
           {PtoS(dev, wx, wy, dx, dy);
            StoW(dev, dx, dy);
            PG_set_text_color(dev, dev->WHITE);
            PG_write_WC(dev, cx, cy,
                         "x = %10.3e, y = %10.3e",
                         dx, dy);}
        else
           PG_write_WC(dev, cx, cy, "x = %d, y = %d", wx, wy);

        PG_update_vs(dev);};

    return;}

/*--------------------------------------------------------------------------*/

/*                        CALLBACK MANAGEMENT ROUTINES                      */

/*--------------------------------------------------------------------------*/

/* _PG_GET_VAL - get the value of the hashel def given
 *             - only numeric types are applicable here
 */

static int _PG_get_val(hp, xo, dx)
   hashel *hp;
   int xo, dx;
   {char *type;

    type = hp->type;
    if (strcmp(type, SC_INTEGER_S) == 0)
       {GET_VAL(int, hp, xo, dx);}

    else if (strcmp(type, SC_LONG_S) == 0)
       {GET_VAL(long, hp, xo, dx);}

    else if (strcmp(type, SC_DOUBLE_S) == 0)
       {GET_VAL(double, hp, xo, dx);}

    else if (strcmp(type, SC_FLOAT_S) == 0)
       {GET_VAL(float, hp, xo, dx);};

    return(xo);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SET_VAL - set the value of the hashel def given
 *             - only numeric types are applicable here
 */

static void _PG_set_val(hp, dxn, dxd)
   hashel *hp;
   int dxn, dxd;
   {char *type;
    double f;

    f = ((double) dxn) / ((double) dxd);

    type = hp->type;
    if (strcmp(type, SC_INTEGER_S) == 0)
       {RESET_VAL(int, hp);}

    else if (strcmp(type, SC_LONG_S) == 0)
       {RESET_VAL(long, hp);}

    else if (strcmp(type, SC_DOUBLE_S) == 0)
       {RESET_VAL(double, hp);}

    else if (strcmp(type, SC_FLOAT_S) == 0)
       {RESET_VAL(float, hp);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_VALUE_MATCH - return TRUE iff the given text string value
 *                 - matches the value in the specified hashel
 */

static int _PG_value_match(ityp, hp, val)
   int ityp;
   hashel *hp;
   char *val;
   {int match;
    char *ps;

    match = FALSE;
    switch (ityp)
       {case SC_INTEGER_I :
	     match = (**(int **) hp->def == SC_stoi(val));
	     break;

	case SC_LONG_I :
	     match = (**(long **) hp->def == SC_stoi(val));
	     break;

	case SC_DOUBLE_I :
	     match = (**(double **) hp->def == SC_stof(val));
	     break;

	case SC_FLOAT_I :
	     match = (**(float **) hp->def == SC_stof(val));
	     break;

	case SC_STRING_I :
	     ps = **(char ***) hp->def;
	     match = (ps == NULL) ?
	             FALSE :
		     (strcmp(ps, val) == 0);
	     break;

	default :
	     match = FALSE;
	     break;};

    return(match);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_VALUE_STRING - return an ASCII representation of the given value */

static int _PG_value_string(ityp, hp, s)
   int ityp;
   hashel *hp;
   char *s;
   {int match;

    match = TRUE;
    switch (ityp)
       {case SC_INTEGER_I :
             sprintf(s, "%d", **(int **) hp->def);
	     break;

	case SC_LONG_I :
             sprintf(s, "%ld", **(long **) hp->def);
	     break;

	case SC_DOUBLE_I :
             sprintf(s, "%.1f", **(double **) hp->def);
	     break;

	case SC_FLOAT_I :
             sprintf(s, "%.1f", **(float **) hp->def);
	     break;

	case SC_STRING_I :
             sprintf(s, "%s", **(char ***) hp->def);
	     break;

	default :
	     match = FALSE;
	     break;};

    return(match);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FIND_REGISTERED - get the call back function info for this object */

static void _PG_find_registered(iob, php, pityp)
   PG_interface_object *iob;
   hashel **php;
   int *pityp;
   {hashel *hp;
    int ityp;
    char *name, *type;

    if (iob != NULL)
       {name = iob->name;
	hp   = SC_lookup(name, PG_callback_tab);
	if (hp != NULL)
	   {type = hp->type;
	    ityp = -1;
	    if (strcmp(type, SC_INTEGER_S) == 0)
	       ityp = SC_INTEGER_I;

	    else if (strcmp(type, SC_LONG_S) == 0)
	       ityp = SC_LONG_I;

	    else if (strcmp(type, SC_DOUBLE_S) == 0)
	       ityp = SC_DOUBLE_I;

	    else if (strcmp(type, SC_FLOAT_S) == 0)
	       ityp = SC_FLOAT_I;

	    else if (strcmp(type, SC_STRING_S) == 0)
	       ityp = SC_STRING_I;};};

    *php = hp;
    *pityp = ityp;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_CALLBACK_TABLE - initialize the call back table */

static void PG_setup_callback_table()
   {

    PG_callback_tab = SC_make_hash_table(HSZLARGE, NODOC);

/* action methods */
    PG_register_callback("slider", PG_slider);
    PG_register_callback("toggle", PG_toggle);

/* draw methods */
    PG_register_callback("draw-button", _PG_draw_button_object);
    PG_register_callback("draw-container", _PG_draw_button_object);
    PG_register_callback("draw-slider", _PG_draw_slider_button);
    PG_register_callback("draw-text", _PG_draw_text_object);
    PG_register_callback("draw-variable", _PG_draw_variable_object);

/* select methods */
    PG_register_callback("select-visual",
			 (PFVoid) PG_event_select_visual);
    PG_register_callback("select-logical",
			 (PFVoid) PG_event_select_logical);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_REGISTER_CALLBACK - assign a name to a call back function
 *                      - this lets the read and write connect names
 *                      - with call back functions
 */

void PG_register_callback(name, fnc)
   char *name;
   PFVoid fnc;
   {long *ip;
    SC_address addr;

    if (PG_callback_tab == NULL)
       PG_setup_callback_table();

    addr.funcaddr = (PFInt) fnc;

    ip  = FMAKE(long, "PG_REGISTER_CALLBACK:ip");
    *ip = addr.diskaddr;

    SC_install(name, ip, "procedure", PG_callback_tab);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_LOOKUP_CALLBACK - lookup and return the function pointer registered
 *                    - under name
 */

PFVoid PG_lookup_callback(name)
   char *name;
   {long *ip;
    SC_address addr;

    ip = (long *) SC_def_lookup(name, PG_callback_tab);
    if (ip == NULL)
       addr.memaddr = NULL;
    else
       addr.diskaddr = *ip;

    return((PFVoid) addr.funcaddr);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_REGISTER_VARIABLE - assign a name to a call back variable
 *                      - this lets the read and write connect names
 *                      - with call back variables
 */

void PG_register_variable(name, type, var, vn, vx)
   char *name, *type;
   byte *var, *vn, *vx;
   {char **s;

    if (PG_callback_tab == NULL)
       PG_setup_callback_table();

    s = FMAKE_N(char *, 3, "PG_REGISTER_VARIABLE:s");
    s[0] = (char *) var;
    s[1] = (char *) vn;
    s[2] = (char *) vx;

    SC_install(name, s, type, PG_callback_tab);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_LOOKUP_VARIABLE - lookup the named VARIABLE object */

static hashel *PG_lookup_variable(name)
   char *name;
   {hashel *hp;
    char **sp;

    hp = SC_lookup(name, PG_callback_tab);

/* if the variable doesn't exist then implicitly define a string
 * this is useful for controlling transient objects at least
 */
    if (hp == NULL)
       {sp = FMAKE(char *, "PG_LOOKUP_VARIABLE:sp");
	PG_register_variable(name, SC_STRING_S, sp, NULL, NULL);
	hp = SC_lookup(name, PG_callback_tab);};

    return(hp);}

/*--------------------------------------------------------------------------*/

/*                           INTERFACE OBJECT ROUTINES                      */

/*--------------------------------------------------------------------------*/

/* PG_ALIGN_INTERFACE_OBJECT - adjust the absolute origins of the interface
 *                           - object tree
 */

static void PG_align_interface_object(iob, xo, yo)
   PG_interface_object *iob;
   int xo, yo;
   {int i, n;
    REAL x, y, dx, dy;
    PG_interface_object **ch;
    PG_device *dev;
    PG_curve *crv;

    dev = iob->device;
    crv = iob->curve;

    _PG_parent_limits_NDC(dev, iob, &x, &y, &dx, &dy);
    StoP(dev, x, y, xo, yo);

    n  = iob->n_children;
    ch = iob->children;
    for (i = 0; i < n; i++)
        {crv = ch[i]->curve;

	 crv->x_origin = xo;
	 crv->y_origin = yo;

	 PG_align_interface_object(ch[i], xo, yo);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_ALIGN_INTERFACE_OBJECTS - align all of the interface objects for
 *                            - the given device
 */

#if 0

static void PG_align_interface_objects(dev)
   PG_device *dev;
   {int i, niobs, ix, iy;
    PG_interface_object **iobs, *iob;
    PG_curve *crv;

/* align the interface objects, setting the absolute origins of each */
    iobs  = dev->interface_objects;
    niobs = dev->n_interface_objects;

    for (i = 0; i < niobs; i++)
        {iob = iobs[i];
         crv = iob->curve;
         ix  = crv->x_origin;
         iy  = crv->y_origin;
         PG_align_interface_object(iobs[i], ix, iy);};

    return;}

#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_EDIT_INTERFACE_OBJECT - edit an interface object
 *                          - options are: move it around the screen
 *                          -              change the size of the object
 */

static void PG_edit_interface_object(iob)
   PG_interface_object *iob;
   {PG_move_object(iob, 0, 10000, 0, 10000, TRUE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MOVE_OBJECT - move an interface object */

void PG_move_object(iob, ixmn, ixmx, iymn, iymx, redraw)
   PG_interface_object *iob;
   int ixmn, ixmx, iymn, iymx, redraw;
   {int ix, iy, xo, yo, dx, dy, btn, mod, first, lop, lclr, fclr, lns;
    int jx, jy, bc;
    PG_curve *crv;
    PG_device *dev;

    dev = iob->device;
    crv = iob->curve;

    PG_make_device_current(dev);

    PG_get_logical_op(dev, &lop);
    PG_get_line_color(dev, &lclr);
    PG_get_line_style(dev, &lns);
    PG_get_fill_color(dev, &fclr);

    if (ixmn > ixmx)
       SC_SWAP_VALUE(int, ixmn, ixmx);

    if (iymn > iymx)
       SC_SWAP_VALUE(int, iymn, iymx);

#if 0
    if (iob->parent != NULL)
       bc = iob->parent->background;
    else
       bc = dev->BLACK;
#else
    bc = dev->BLACK;
#endif

    PG_set_color_line(dev, bc, FALSE);
    PG_draw_curve(dev, crv, TRUE);
    PG_set_fill_color(dev, bc);
    PG_fill_curve(dev, crv);

/*    fc = iob->foreground; */
    if (dev->xor_parity)
       {PG_set_color_line(dev, dev->WHITE, FALSE);}
    else
       PG_set_color_line(dev, dev->BLACK, FALSE);

    PG_set_line_style(dev, LINE_DOTTED);
    PG_set_logical_op(dev, GS_XOR);

    first = TRUE;
    btn   = TRUE;
    while (btn || first)
       {PG_query_pointer(dev, &ix, &iy, &btn, &mod);

        if (first)
           {dx = ix - crv->x_origin;
            dy = iy - crv->y_origin;
            first = FALSE;}
	else
/* erase the old one */
	   PG_draw_curve(dev, crv, TRUE);

	jx = ix - dx;
	jx = min(jx, ixmx);
	jx = max(jx, ixmn);
	crv->x_origin = jx;

	jy = iy - dy;
	jy = min(jy, iymx);
	jy = max(jy, iymn);
	crv->y_origin = jy;

/* draw the new one */
	PG_draw_curve(dev, crv, TRUE);};

    PG_draw_curve(dev, crv, TRUE);
    PG_set_logical_op(dev, lop);
    PG_set_fill_color(dev, fclr);
    PG_set_line_color(dev, lclr);
    PG_set_line_style(dev, lns);

    xo = crv->x_origin;
    yo = crv->y_origin;

    PG_align_interface_object(iob, xo, yo);
    if (redraw)
       PG_draw_interface_object(iob);

    return;}

/*--------------------------------------------------------------------------*/

/*                        INTERFACE OBJECT DRAW ROUTINES                    */

/*--------------------------------------------------------------------------*/

/* PG_DRAW_INTERFACE_OBJECT - draw a tree of interface objects starting with
 *                          - the given one
 */

void PG_draw_interface_object(iob)
   PG_interface_object *iob;
   {int i, n;
    PG_interface_object **ch;
    PG_device *dev;
    REAL xlog, ylog;

    if (iob->is_visible)
       {dev = iob->device;
        if (iob->draw != NULL)
           {PG_make_device_current(dev);
	    xlog = dev->ifxlog;
	    ylog = dev->ifylog;
	    dev->ifxlog = 0;
	    dev->ifylog = 0;
            (*iob->draw)(iob);
	    dev->ifxlog = xlog;
	    dev->ifylog = ylog;};

        n  = iob->n_children;
        ch = iob->children;
        for (i = 0; i < n; i++)
            {PG_draw_interface_object(ch[i]);};};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_INTERFACE_OBJECTS - draw all of the visible interface objects
 *                           - which belong to the given device
 */

void PG_draw_interface_objects(dev)
   PG_device *dev;
   {int i, niobs;
    PG_interface_object **iobs;

    iobs  = dev->interface_objects;
    niobs = dev->n_interface_objects;

    for (i = 0; i < niobs; i++)
        PG_draw_interface_object(iobs[i]);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_REGISTER_INTERFACE_OBJECT - add the given interface object to the
 *                              - list which the specified device maintains
 */

void PG_register_interface_object(dev, iob)
   PG_device *dev;
   PG_interface_object *iob;
   {int n, nmax;
    PG_interface_object **lst;

    lst  = dev->interface_objects;
    n    = dev->n_interface_objects;
    nmax = dev->max_interface_objects;

    SC_REMEMBER(PG_interface_object *, iob, lst, n, nmax, 50);

    SC_mark(iob, 1);

    dev->interface_objects     = lst;
    dev->n_interface_objects   = n;
    dev->max_interface_objects = nmax;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PARENT_IS_VARIABLE - is an ancestor of this interface_object a
 *                        - variable?
 */

static PG_interface_object *_PG_parent_is_variable(iob)
   PG_interface_object *iob;
   {

    for (; iob != NULL; iob = iob->parent)
        if (strcmp(iob->type, PG_VARIABLE_OBJECT_S) == 0)
           return(iob);

    return(NULL);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SHIFT_CURVE - shift the curve to the specified origin */

static void _PG_shift_curve(crv, xo, yo)
   PG_curve *crv;
   int xo, yo;
   {int i, n, dx, dy;
    int *x, *y;

    x = crv->x;
    y = crv->y;
    n = crv->n;

    dx = x[0] - xo;
    dy = y[0] - yo;

    for (i = 0; i < n; i++)
        {*x++ -= dx;
         *y++ -= dy;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SET_TEXT_BACKGROUND_COLOR - fixup the background color of the
 *                               - text object if it is unset (-1)
 */

static int _PG_set_text_background_color(iob, flag)
   PG_interface_object *iob;
   int flag;
   {int clr;
    PG_text_box *b;
    PG_interface_object *piob;

    clr = iob->device->BLACK;
    if (strcmp(iob->type, PG_TEXT_OBJECT_S) == 0)
       {b   = (PG_text_box *) iob->obj;
        clr = flag ? -1 : b->background;
        for (piob = iob; (piob != NULL) && (clr == -1); piob = piob->parent)
            clr = piob->background;

        b->background = clr;};

    return(clr);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_TEXT_OBJECT - the draw function for TEXT interface objects */

static void _PG_draw_text_object(iob)
   PG_interface_object *iob;
   {PG_text_box *b;

    if ((strcmp(iob->type, PG_TEXT_OBJECT_S) != 0) || !iob->is_visible)
       return;

    b = (PG_text_box *) iob->obj;

    _PG_set_text_background_color(iob, TRUE);
    if (b->foreground != b->background)
       PG_refresh_text_box(b);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_VARIABLE_OBJECT - the draw function for
 *                          - VARIABLE interface objects
 */

static void _PG_draw_variable_object(iob)
   PG_interface_object *iob;
   {int i, nch, ityp, match, isa;
    REAL x, y, dx, dy, tx, ty;
    char s[MAXLINE];
    char *val;
    hashel *hp;
    PG_device *dev;
    PG_text_box *b;
    PG_interface_object **chn, *ch;

    if ((strcmp(iob->type, PG_VARIABLE_OBJECT_S) != 0) ||
	!iob->is_visible)
       return;

    dev = iob->device;

    _PG_find_registered(iob, &hp, &ityp);
    if (hp != NULL)
       {nch = iob->n_children;
	chn = iob->children;
	for (i = 0; i < nch; i++)
	    {ch = chn[i];
	     if (ch->is_selectable)
	        {if (strcmp(ch->type, PG_BUTTON_OBJECT_S) == 0)
		    {val   = ch->action_name;
		     match = _PG_value_match(ityp, hp, val);
                     isa   = ch->is_active;
		     if (match && !isa)
		        {SC_SWAP_VALUE(int, ch->foreground, ch->background);}
		     else if (!match && isa)
		        {SC_SWAP_VALUE(int, ch->foreground, ch->background);};

		     ch->is_active = match;}

	         else if (strcmp(ch->type, PG_TEXT_OBJECT_S) == 0)
                    {b = (PG_text_box *) ch->obj;
		     if (_PG_value_string(ityp, hp, s))
		        strcpy(b->text_buffer[0], s);};};};};

    _PG_draw_text_object(iob);

    if (iob->foreground != iob->background)
       {sprintf(s, "%s", iob->name);
        PG_get_text_ext_NDC(dev, s, &tx, &ty);
        _PG_parent_limits_NDC(dev, iob, &x, &y, &dx, &dy);

	x += 0.5*(dx - tx);
	y += 0.5*(dy - ty);

	StoW(dev, x, y);

	PG_set_color_text(dev, iob->foreground, TRUE);
	PG_write_WC(dev, x, y, "%s", s);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_SLIDER_BUTTON - the draw function for SLIDERS */

static void _PG_draw_slider_button(iob)
   PG_interface_object *iob;
   {int dx, dy;
    int i, nv, nch;
    REAL rxmn, rxmx, rymn, rymx;
    REAL pxmn, pxmx, pymn, pymx;
    char *name;
    hashel *hp;
    PG_curve *pcrv, *crv;
    PG_interface_object *piob, **chn, *ch;
    PG_device *dev;

    if (((strcmp(iob->type, PG_BUTTON_OBJECT_S) != 0) &&
	(strcmp(iob->type, PG_CONTAINER_OBJECT_S) != 0)) ||
        !iob->is_visible)
       return;

    dev  = iob->device;
    crv  = iob->curve;
    piob = iob->parent;
    pcrv = piob->curve;

    PG_get_curve_extent(dev, crv, PIXELC,
			&rxmn, &rxmx, &rymn, &rymx);

    PG_get_curve_extent(dev, pcrv, PIXELC,
			&pxmn, &pxmx, &pymn, &pymx);

    dx = (pxmx - pxmn) - (rxmx - rxmn);
    switch (dev->quadrant)
       {case QUAD_ONE :
	     dy = (pymx - pymn) - (rymx - rymn);
	     break;

	default :
	case QUAD_FOUR :
	     dy = (rymx - rymn) - (pymx - pymn);
	     break;};

    piob = iob->parent->parent;
    nch  = piob->n_children;
    chn  = piob->children;
    nv   = 0;
    for (i = 0; i < nch; i++)
        {ch = chn[i];
	 if ((ch != NULL) &&
	     (strcmp(ch->type, PG_VARIABLE_OBJECT_S) == 0))
	    {name = ch->name;
	     hp   = PG_lookup_variable(name);
	     if ((dx != 0) && (nv == 0))
	        {crv->x_origin = _PG_get_val(hp, pcrv->x_origin, dx);
		 if (dy != 0)
		    {nv++;
		     continue;};
		 break;}
	     else if (dy != 0)
	        {crv->y_origin = _PG_get_val(hp, pcrv->y_origin, dy);
		 break;};};};

    _PG_draw_button_object(iob);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_BUTTON_OBJECT - the draw function for BUTTON interface objects */

static void _PG_draw_button_object(iob)
   PG_interface_object *iob;
   {int lclr, fclr;
    PG_device *dev;
    PG_curve *crv;

    if (((strcmp(iob->type, PG_BUTTON_OBJECT_S) != 0) &&
	(strcmp(iob->type, PG_CONTAINER_OBJECT_S) != 0)) ||
        !iob->is_visible)
       return;

    dev = iob->device;
    crv = iob->curve;

    PG_get_fill_color(dev, &fclr);
    PG_set_color_fill(dev, iob->background, TRUE);
    PG_fill_curve(dev, crv);
    PG_set_color_fill(dev, fclr, TRUE);

    PG_get_line_color(dev, &lclr);
    PG_set_color_line(dev, dev->WHITE, TRUE);
    PG_draw_curve(dev, crv, TRUE);
    PG_set_color_line(dev, lclr, TRUE);

    return;}

/*--------------------------------------------------------------------------*/

#if 0

/*--------------------------------------------------------------------------*/

/* _PG_DRAW_BND_OBJECT - the draw function for boundary interface objects */

static void _PG_draw_bnd_object(iob)
   PG_interface_object *iob;
   {PG_curve *crv;
    PG_device *dev;

    if (iob->is_visible)
       {dev = iob->device;
        crv = iob->curve;

        PG_draw_curve(dev, crv, TRUE);};

    return;}

/*--------------------------------------------------------------------------*/

#endif

/*                       INTERFACE OBJECT ACTION ROUTINES                   */

/*--------------------------------------------------------------------------*/

/* PG_SLIDER - handle a slider button */

static void PG_slider(iob, ev)
   PG_interface_object *iob;
   PG_event *ev;
   {PG_device *dev;

    if (ev != NULL)
       {int ixmn, ixmx, iymn, iymx, dx, dy;
        int i, nv, nch;
        REAL rxmn, rxmx, rymn, rymx;
        REAL pxmn, pxmx, pymn, pymx;
        char *name;
        hashel *hp;
        PG_curve *pcrv, *crv;
        PG_interface_object *piob, **chn, *ch;

        dev  = iob->device;
        crv  = iob->curve;
        piob = iob->parent;
        pcrv = piob->curve;

        PG_get_curve_extent(dev, crv, PIXELC,
			    &rxmn, &rxmx, &rymn, &rymx);

        PG_get_curve_extent(dev, pcrv, PIXELC,
			    &pxmn, &pxmx, &pymn, &pymx);

        dx   = (pxmx - pxmn) - (rxmx - rxmn);
        ixmn = pcrv->x_origin;
        ixmx = ixmn + dx;

	switch (dev->quadrant)
	   {case QUAD_ONE :
	         dy = (pymx - pymn) - (rymx - rymn);
		 break;

	    default :
	    case QUAD_FOUR :
	         dy = (rymx - rymn) - (pymx - pymn);
		 break;};

        iymn = pcrv->y_origin;
        iymx = iymn + dy;

	PG_move_object(iob, ixmn, ixmx, iymn, iymx, FALSE);

        if ((dx != 0) || (dy != 0))
	   {piob = piob->parent;
	    nch  = piob->n_children;
	    chn  = piob->children;
	    nv   = 0;
	    for (i = 0; i < nch; i++)
	        {ch = chn[i];
		 if ((ch != NULL) &&
		     (strcmp(ch->type, PG_VARIABLE_OBJECT_S) == 0))
		    {name = ch->name;
		     hp   = PG_lookup_variable(name);
		     if ((dx != 0) && (nv == 0))
		        {ixmx = crv->x_origin;
			 _PG_set_val(hp, (ixmx - ixmn), dx);
			 if (dy != 0)
			    {nv++;
			     continue;};
			 break;}
		     else if (dy != 0)
		        {iymx = crv->y_origin;
			 _PG_set_val(hp, (iymx - iymn), dy);
			 break;};};};};

        PG_draw_interface_object(piob);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SWITCH_OBJECT_STATE - change the objects state from that given
 *                         - if FLAG is TRUE toggle the item iff
 *                         - the corresponding argument is TRUE
 *                         - if FLAG is FALSE set the item to the value of
 *                         - the corresponding argument
 */

static void _PG_switch_object_state(iob, isv, iss, isa, flag, redraw)
   PG_interface_object *iob;
   int isv, iss, isa, flag, redraw;
   {PG_device *dev;
    int iv, is, ia;

    if (flag)
       {iv = iob->is_visible;
        is = iob->is_selectable;
        ia = iob->is_active;
        iv = isv ? !iv : iv;
        is = iss ? !is : is;
        ia = isa ? !ia : ia;}
    else
       {iv = isv;
        is = iss;
        ia = isa;};

    iob->is_visible    = iv;
    iob->is_selectable = is;
    iob->is_active     = ia;

    if (iv)
       PG_draw_interface_object(iob);
    else
       {dev = iob->device;

        PG_set_fill_color(dev, iob->background);
	PG_fill_curve(dev, iob->curve);
	PG_draw_curve(dev, iob->curve, TRUE);

        if (redraw)
	   PG_draw_interface_objects(dev);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_TOGGLE - handle a toggle button */

static void PG_toggle(iob, ev)
   PG_interface_object *iob;
   PG_event *ev;
   {PG_interface_object *trans, **iobs, *ch, *ths;
    char *name, *type, *val;
    int i, niobs;
    hashel *hp;

    type = iob->type;
    if (strcmp(type, PG_BUTTON_OBJECT_S) == 0)
       {PG_handle_button_press(iob, ev);

        trans = (PG_interface_object *) iob->obj;
        _PG_switch_object_state(trans, TRUE, TRUE, FALSE, TRUE, TRUE);}

    else if (strcmp(type, PG_VARIABLE_OBJECT_S) == 0)
       {PG_draw_interface_object(iob);

        hp = PG_lookup_variable(iob->name);
        if (hp == NULL)
           return;
        val   = **(char ***) hp->def;

        iobs  = iob->children;
        niobs = iob->n_children;
        ths   = NULL;
        for (i = 0; i < niobs; i++)
            {ch = iobs[i];
             if (strcmp(ch->type, PG_BUTTON_OBJECT_S) == 0)
                {name  = ch->action_name;
		 trans = PG_find_object(iob->device, name, NULL);
		 if (trans != NULL)
                    {if (strcmp(val, name) == 0)
		        ths = trans;
		    else
		        _PG_switch_object_state(trans,
						FALSE, FALSE, FALSE,
						FALSE, FALSE);};};};
	if (ths != NULL)
	   _PG_switch_object_state(ths,
				   TRUE, TRUE, FALSE,
				   TRUE, TRUE);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_HANDLE_VARIABLE - handle the value assignment of the variable bound
 *                    - to the button
 */

static void PG_handle_variable(iob, ev)
   PG_interface_object *iob;
   PG_event *ev;
   {char *name, *type, *val;
    hashel *hp;
    PG_interface_object *piob;

    if (PG_edit_mode)
       {PG_edit_interface_object(iob);
	PG_edit_mode = FALSE;}

/* find the ancestor containing the variable reference */
    else
       {for (piob = iob; piob != NULL; piob = piob->parent)
            if (strcmp(piob->type, PG_VARIABLE_OBJECT_S) == 0)
	       break;

	name = piob->name;
        hp   = PG_lookup_variable(name);
	if (hp != NULL)
	   {type = hp->type;
	    val  = iob->action_name;
	    if (strcmp(type, SC_INTEGER_S) == 0)
	       {SET_VAL(int, SC_stoi);}

	    else if (strcmp(type, SC_LONG_S) == 0)
	       {SET_VAL(long, SC_stoi);}

	    else if (strcmp(type, SC_DOUBLE_S) == 0)
	       {SET_VAL(double, SC_stof);}

	    else if (strcmp(type, SC_FLOAT_S) == 0)
	       {SET_VAL(float, SC_stof);}

	    else if (strcmp(type, SC_STRING_S) == 0)
	       {SFREE((**(char ***) hp->def));
		SET_VAL(char *, SC_strsave);};};

        if (piob->action != NULL)
           (*piob->action)(piob, ev);
        else
	   PG_draw_interface_object(piob);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_HANDLE_BUTTON_PRESS - handle a button press
 *                        - return the device associate with a non-NULL event
 *                        - or interpet D as a device
 */

PG_device *PG_handle_button_press(d, ev)
   byte *d;
   PG_event *ev;
   {PG_device *dev;

    if (ev == NULL)
       dev = (PG_device *) d;
    else
       {int ix, iy, btn, mod, ityp;
        PG_interface_object *iob;

        iob  = (PG_interface_object *) d;
        dev  = iob->device;
        ityp = (strcmp(iob->type, PG_BUTTON_OBJECT_S) == 0);

	if (PG_edit_mode)
	   {PG_edit_interface_object(iob);
	    PG_edit_mode = FALSE;
            return(NULL);};

        if (ityp)
           {SC_SWAP_VALUE(int, iob->foreground, iob->background);
            PG_draw_interface_object(iob);};

        btn = TRUE;
        while (btn)
           PG_query_pointer(dev, &ix, &iy, &btn, &mod);

        if (ityp)
           {SC_SWAP_VALUE(int, iob->foreground, iob->background);};

        PG_draw_interface_object(iob);};

    return(dev);}

/*--------------------------------------------------------------------------*/

#ifdef HAVE_WINDOW_DEVICE

/*--------------------------------------------------------------------------*/

/* _PG_STRING_VALUE - reset the value in the hashel from the value in
 *                  - the string
 */

static int _PG_string_value(ityp, hp, s)
   int ityp;
   hashel *hp;
   char *s;
   {int match;
    char *ps;

    match = TRUE;
    switch (ityp)
       {case SC_INTEGER_I :
	     **(int **) hp->def = SC_stoi(s);
	     break;

	case SC_LONG_I :
	     **(long **) hp->def = SC_stoi(s);
	     break;

	case SC_DOUBLE_I :
	     **(double **) hp->def = SC_stof(s);
	     break;

	case SC_FLOAT_I :
	     **(float **) hp->def = SC_stof(s);
	     break;

	case SC_STRING_I :
	     ps = **(char ***) hp->def;
             SFREE(ps);
             **(char ***) hp->def = SC_strsavef(s,
                                    "char*:_PG_STRING_VALUE:string");
	     break;

	default :
	     match = FALSE;
	     break;};

    return(match);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_HANDLE_KEYBOARD_EVENT - helper for PG_HANDLE_KEY_PRESS */

static void _PG_handle_keyboard_event(iob, ev)
   PG_interface_object *iob;
   PG_event *ev;
   {char bf[80], *s;
    int x, y, mod, ityp;
    hashel *hp;
    PG_text_box *b;
    PG_device *dev;
    PG_interface_object *par;

    dev = iob->device;

    for (bf[0] = '\0'; TRUE; )
        {PG_KEY_EVENT_INFO(dev, ev, x, y, bf, 80, mod);
	 if ((bf[0] == '\0') && (mod != 0))
	    PG_GET_NEXT_EVENT(*ev);
	 else
	    break;};
    bf[1] = '\0';

    if (bf[0] != '\0')
       {b   = (PG_text_box *) iob->obj;
	par = _PG_parent_is_variable(iob);
	if (par != NULL)
	   switch (bf[0])
	      {case '\r' :
	       case '\n' :
		    _PG_find_registered(par, &hp, &ityp);
		    if (hp != NULL)
		       {s = b->text_buffer[0];
			_PG_string_value(ityp, hp, s);
			if (par->parent != NULL)
			   PG_draw_interface_object(par->parent);};

		    return;

	       default :
		    break;};
                   
	KEY_ACTION(b, bf);};

    return;}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* PG_HANDLE_KEY_PRESS - handle key down events */

PG_device *PG_handle_key_press(d, ev)
   byte *d;
   PG_event *ev;
   {PG_device *dev;

#ifdef HAVE_WINDOW_DEVICE

    if (ev == NULL)
       dev = (PG_device *) d;
    else
       {PG_interface_object *iob;
	int type;

        type = PG_EVENT_TYPE(*ev);

/* mouse events are used in edit mode so handle them */
	if (type == MOUSE_DOWN_EVENT)
           PG_handle_button_press(d, ev);

        else if (type == KEY_DOWN_EVENT)
	   {iob = (PG_interface_object *) d;
	    dev = iob->device;
	    if (iob != NULL)
	       _PG_handle_keyboard_event(iob, ev);};};

#else

    dev = (PG_device *) d;

#endif

    return(dev);}

/*--------------------------------------------------------------------------*/

/*                     INTERFACE OBJECT SELECTION ROUTINES                  */

/*--------------------------------------------------------------------------*/

/* PG_EVENT_SELECT_VISUAL - check the event to see if it occured in
 *                        - the given interface object or
 *                        - one of its children
 *                        - return the interface object if any
 *                        - and NULL otherwise
 *                        - this is based on visual containment
 *                        - other event selectors would be appropriate to the
 *                        - specific interface object (e.g. a PG_graph)
 */

static PG_interface_object *PG_event_select_visual(iob, ev)
   PG_interface_object *iob;
   PG_event *ev;
   {

#ifdef HAVE_WINDOW_DEVICE

    int i, n, nch, button, qualifier, iev_x, iev_y;
    PG_interface_object *ch, **chn;
    PG_device *dev;
    PG_curve *crv;

    if (!iob->is_selectable)
       return(NULL);

    dev = iob->device;

    PG_MOUSE_EVENT_INFO(dev, ev, iev_x, iev_y, button, qualifier);

    crv = iob->curve;

    iev_x -= crv->x_origin;
    iev_y -= crv->y_origin;
    n      = crv->n - 1;

    if (PM_inside_fix(iev_x, iev_y, crv->x, crv->y, n, -1))
       {nch = iob->n_children;
        chn = iob->children;
        for (i = 0; i < nch; i++)
            {ch = PG_event_select_visual(chn[i], ev);
             if (ch != NULL)
                return(ch);};

        return(iob);}

    else

#endif

       return(NULL);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_EVENT_SELECT_LOGICAL - check the event to see if it occured in
 *                         - the given interface object or
 *                         - one of its children
 *                         - return the interface object if any
 *                         - and NULL otherwise
 *                         - this is based on logical containment
 */

static PG_interface_object *PG_event_select_logical(iob, ev)
   PG_interface_object *iob;
   PG_event *ev;

#ifdef HAVE_WINDOW_DEVICE

   {int i, n, nch, button, qualifier, iev_x, iev_y, ev_x, ev_y;
    PG_interface_object *ch, *lch, **chn;
    PG_device *dev;
    PG_curve *crv;

    if (!iob->is_selectable)
       return(NULL);

    dev = iob->device;

    PG_MOUSE_EVENT_INFO(dev, ev, ev_x, ev_y, button, qualifier);

    nch = iob->n_children;
    chn = iob->children;
    for (i = 0; i < nch; i++)
        {ch = chn[i];
         if (ch->is_selectable)
            {lch = PG_event_select_logical(ch, ev);
             if (lch == NULL)
	        {crv   = ch->curve;
		 iev_x = ev_x - crv->x_origin;
		 iev_y = ev_y - crv->y_origin;
		 n     = crv->n - 1;

		 if (PM_inside_fix(iev_x, iev_y, crv->x, crv->y, n, -1))
		    return(ch);}

	     else
	       return(lch);};};

    crv   = iob->curve;
    iev_x = ev_x - crv->x_origin;
    iev_y = ev_y - crv->y_origin;
    n     = crv->n - 1;
    return(PM_inside_fix(iev_x, iev_y, crv->x, crv->y, n, -1) ?
	   iob : NULL);}

#else

   {return(NULL);}

#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_OBJECT_EVENT - return the interface object in which the 
 *                     - given event occured
 */

PG_interface_object *PG_get_object_event(dev, ev)
   PG_device *dev;
   PG_event *ev;
   {int i, niobs;
    PG_interface_object *tiob, *iob, **iobs;

    iob   = NULL;
    iobs  = dev->interface_objects;
    niobs = dev->n_interface_objects;
    for (i = 0; i < niobs; i++)
        {tiob = iobs[i];
         if (tiob->is_selectable && (tiob->select != NULL))
            {iob = (*tiob->select)(tiob, ev);
             if (iob != NULL)
                break;};};

    return(iob);}

/*--------------------------------------------------------------------------*/

/*                          MEMORY MANAGEMENT ROUTINES                      */

/*--------------------------------------------------------------------------*/

/* PG_MAKE_INTERFACE_OBJECT - create and return an graphical interface
 *                          - object
 */

PG_interface_object *PG_make_interface_object(dev, name, type, obj,
					      align, ang, crv, flags, fc, bc,
					      select, draw, action,
                                              parent, children)
   PG_device *dev;
   char *name, *type;
   byte *obj;
   int align;
   double ang;
   PG_curve *crv;
   int *flags, fc, bc;
   char *select, *draw, *action;
   PG_interface_object *parent, **children;
   {PG_interface_object *iob;
    PFVoid fnc;

    iob = FMAKE(PG_interface_object, "PG_MAKE_INTERFACE_OBJECT:iob");
   
    iob->device        = dev;
    iob->curve         = crv;
    iob->name          = SC_strsavef(name,
                         "char*:PG_MAKE_INTERFACE_OBJECT:name");
    iob->type          = SC_strsavef(type,
                         "char*:PG_MAKE_INTERFACE_OBJECT:type");
    iob->obj           = obj;
    iob->foreground    = fc;
    iob->background    = bc;
    iob->is_visible    = flags[0];
    iob->is_active     = flags[2];
    iob->is_selectable = flags[1];

    iob->draw_name     = NULL;
    iob->action_name   = NULL;
    iob->select_name   = NULL;

    iob->draw          = NULL;
    iob->action        = NULL;
    iob->select        = NULL;

    iob->parent        = parent;
    iob->n_children    = 0;
    iob->max_children  = 0;
    iob->children      = children;

    if (strcmp(type, PG_TEXT_OBJECT_S) == 0)
       {PG_text_box *b;

	b = PG_open_text_rect(dev, "IOB", FALSE, NULL, crv);

	SFREE(b->text_buffer[0]);
	b->text_buffer[0] = SC_strsavef(name,
                            "char*:PG_MAKE_INTERFACE_OBJECT:name");
	b->n_chars_line = max(b->n_chars_line, strlen(name));
	b->align      = align;
	b->angle      = ang;
        b->foreground = fc;
        b->background = bc;

        if (b->n_lines > 2)
           b->type = SCROLLING_WINDOW;
        else
           {b->type   = FALSE;
	    b->border = 0.0;};

        iob->obj    = (byte *) b;
        iob->action = (PFVoid) PG_handle_key_press;
	iob->draw   = _PG_draw_text_object;

        _PG_set_text_background_color(iob, FALSE);}

    else if (strcmp(type, PG_VARIABLE_OBJECT_S) == 0)
       iob->draw = _PG_draw_variable_object;

    else
       iob->draw = _PG_draw_button_object;

    if (children != NULL)
       iob->n_children = SC_arrlen(children)/sizeof(PG_interface_object *);
    else
       iob->n_children = 0;

    if (select != NULL)
       {iob->select_name = SC_strsavef(select,
                           "char*:PG_MAKE_INTERFACE_OBJECT:select");
        iob->select      = (PFPPG_interface_object) PG_lookup_callback(select);};

    if (iob->select == NULL)
       {if (strcmp(type, PG_CONTAINER_OBJECT_S) == 0)
           iob->select = PG_event_select_logical;
	else
           iob->select = PG_event_select_visual;};

    if (draw != NULL)
       {iob->draw_name = SC_strsavef(draw,
                         "char*:PG_MAKE_INTERFACE_OBJECT:draw");
        iob->draw      = PG_lookup_callback(draw);};

    if (action != NULL)
       {iob->action_name = SC_strsavef(action,
                           "char*:PG_MAKE_INTERFACE_OBJECT:action");
        fnc              = PG_lookup_callback(action);
        if (fnc == PG_slider)
           _PG_shift_curve(iob->curve, 0, 0);

        if (fnc != NULL)
           iob->action = fnc;

        else if (_PG_parent_is_variable(parent) != NULL)
           iob->action = PG_handle_variable;};

/* make things at least editable by default */
    if (iob->action == NULL)
       iob->action = (PFVoid) PG_handle_button_press;

    return(iob);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_RL_INTERFACE_OBJECT - release an interface object
 *                         - if flag is TRUE recursively release children
 */

void _PG_rl_interface_object(iob, flag)
   PG_interface_object *iob;
   int flag;
   {int i, n;
    PG_interface_object **ch;

    if (flag)
       {n  = iob->n_children;
        ch = iob->children;
        for (i = 0; i < n; i++)
            _PG_rl_interface_object(ch[i], TRUE);

        SFREE(ch);};

    if (strcmp(iob->type, PG_TEXT_OBJECT_S) == 0)
       PG_close_text_box((PG_text_box *) (iob->obj));

    else if (strcmp(iob->type, PG_VARIABLE_OBJECT_S) == 0)
       SFREE(iob->obj);

    SFREE(iob->name);
    SFREE(iob->type);
    SFREE(iob);
   
    return;}

/*--------------------------------------------------------------------------*/

/*                       SIMPLE OBJECTS FOR APPLICATIONS                    */

/*--------------------------------------------------------------------------*/

/* PG_ADD_BUTTON - register a BUTTON interface object with the device */

void PG_add_button(dev, xmn, xmx, ymn, ymx, s, fnc)
   PG_device *dev;
   double xmn, xmx, ymn, ymx;
   char *s;
   PFVoid fnc;
   {PG_interface_object *tiob, *biob;
    PG_curve *bcrv, *tcrv;
    REAL dx, dy, txmn, txmx, tymn, tymx;
    int flags[3];

    if (fnc == NULL)
       return;

    PG_register_callback(s, fnc);

    bcrv = PG_make_box_curve(dev, NORMC, 0.0, 0.0, xmn, xmx, ymn, ymx);

    PG_get_text_ext_NDC(dev, s, &dx, &dy);

    txmn = xmn + 0.5*(xmx - xmn - dx);
    txmx = txmn + dx;
    tymn = ymn + 0.5*(ymx - ymn - dy);
    tymx = tymn + dy;

    flags[0] = TRUE;
    flags[1] = TRUE;
    flags[2] = FALSE;

    tcrv = PG_make_box_curve(dev, NORMC, xmn, ymn, txmn, txmx, tymn, tymx);
    biob = PG_make_interface_object(dev, "BUTTON", PG_BUTTON_OBJECT_S, s,
				    CENTER, 0.0, bcrv, flags,
				    dev->BLACK, dev->GRAY,
				    NULL, "draw-button", s,
                                    NULL, NULL);

    flags[0] = TRUE;
    flags[1] = FALSE;
    flags[2] = FALSE;

    tiob = PG_make_interface_object(dev, s, PG_TEXT_OBJECT_S, NULL,
				    CENTER, 0.0, tcrv, flags,
				    dev->DARK_BLUE, -1,
				    NULL, "draw-text", NULL,
                                    biob, NULL);

    PG_PUSH_CHILD_IOB(biob, tiob);

    PG_register_interface_object(dev, biob);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
