/*
 * SHENVR.C - environment and eval control routines for Scheme
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "scheme.h"

char
 *SS_OBJECT_P_S = "object *";

void
 SC_DECLARE(_SS_eval_err, (byte));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_EXP_EVAL - make an explicit interpreter level call to eval */

object *SS_exp_eval(obj)
   object *obj;
   {SS_Assign(SS_Exn, obj);

    _SS_eval();

    return(SS_Val);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_EVAL - make an implicit C level call to eval */

object *SS_eval(obj)
   object *obj;
   {SS_Assign(SS_Exn, obj);
    SS_Assign(SS_Env, SS_Global_Env);

    _SS_eval();

    return(SS_Val);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SS_EVAL_ERR - handle error returns during an eval */

void _SS_eval_err()
   {SS_Assign(SS_Val, SS_f);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_SAVE_REGISTERS - save the Scheme register set on the stack */

void SS_save_registers(vp)
   int vp;
   {if (vp)
       {SS_Save(SS_Val);};

    SS_Save(SS_Exn);
    SS_Save(SS_Env);
    SS_Save(SS_Fun);
    SS_Save(SS_This);
    SS_Save(SS_Unev);
    SS_Save(SS_Argl);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_RESTORE_REGISTERS - restore the Scheme register set from the stack */

void SS_restore_registers(vp)
   int vp;
   {SS_Restore(SS_Argl);
    SS_Restore(SS_Unev);
    SS_Restore(SS_This);
    SS_Restore(SS_Fun);
    SS_Restore(SS_Env);
    SS_Restore(SS_Exn);

    if (vp)
       {SS_Restore(SS_Val);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_IDENT - dummy function necessary to make begin and cond work */

object *SS_Ident(obj)
   object *obj;
   {return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_TRUE - return TRUE if object is not boolean #f or () */

int SS_true(obj)
   object *obj;
   {if (SS_booleanp(obj))
       return(SS_BOOLEAN_VALUE(obj));
    else if (SS_nullobjp(obj))
       return(FALSE);
    else
       return(TRUE);}

/*--------------------------------------------------------------------------*/

/*                               HANDLERS                                   */

/*--------------------------------------------------------------------------*/

/* SS_ZARGS - zero argument macro/procedure handler */

object *SS_zargs(pr, argl)
   object *(*pr)();
   object *argl;
   {return((*pr)());}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_SARGS - single argument macro/procedure handler */

object *SS_sargs(pr, argl)
   object *(*pr)();
   object *argl;
   {return((*pr)(SS_car(argl)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_NARGS - n argument macro/procedure handler */

object *SS_nargs(pr, argl)
   object *(*pr)();
   object *argl;
   {return((*pr)(argl));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_ZNARGS - zero or more arguments macro/procedure handler */

object *SS_znargs(pr, argl)
   object *(*pr)();
   object *argl;
   {return((*pr)(argl));}

/*--------------------------------------------------------------------------*/

/*                         PROCEDURE ACCESSORS                              */

/*--------------------------------------------------------------------------*/

/* SS_PARAMS - returns a list of the formal parameters of the procedure */

object *SS_params(fun)
   object *fun;
   {return(SS_cadr(SS_COMPOUND_PROCEDURE_FUNCTION(fun)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_PROC_BODY - returns a list of the body or the procedure */

object *SS_proc_body(fun)
   object *fun;
   {return(SS_cddr(SS_COMPOUND_PROCEDURE_FUNCTION(fun)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_PROC_ENV - returns the environment frame index of the procedure */

object *SS_proc_env(fun)
   object *fun;
   {object *cp;

    cp = SS_COMPOUND_PROCEDURE_FUNCTION(fun);
    return(SS_car(cp));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_DO_BINDINGS - bind the parameters to the values in a new frame
 *                - and return the resulting environment
 */

object *SS_do_bindings(pp, argp)
   object *pp, *argp;
   {if (!SS_procedurep(pp))
       SS_error("BAD PROCEDURE TO DO-BINDINGS", pp);

    return(SS_xtnd_env(SS_params(pp), argp, SS_proc_env(pp)));}

/*--------------------------------------------------------------------------*/

/*                     ENVIRONMENT CONTROL ROUTINES                         */

/*--------------------------------------------------------------------------*/

/* SS_DEFP - defined? predicate in SCHEME */

object *SS_defp(vr)
   object *vr;
   {object *b;

    b = SS_bind_env(vr, SS_Env);

    return((b == NULL) ? SS_f : SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_LK_VAR_VAL - look up the variable in the environment */

object *SS_lk_var_val(vr, penv)
   object *vr, *penv;
   {Register char *token;
    Register object *b, *obj;

    obj = SS_null;
    b   = SS_bind_env(vr, penv);
    if (b != NULL)
       obj = b;

/* look in the various hash tables - it may be a variable holding a primitive
 * procedure object
 */
    else
       {token = SS_VARIABLE_NAME(vr);
        b     = SS_INQUIRE_OBJECT(token);
        if (b != NULL)
           obj = SS_VARIABLE_VALUE(b);

        if (SS_nullobjp(obj))
           SS_error("UNBOUND VARIABLE - SS_LK_VAR_VAL", vr);};

    if ((obj->val == NULL) || (obj->eval_type == 0))
       SS_error("FREED OBJECT - SS_LK_VAR_VAL", SS_null);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_BIND_ENV - return the binding of the variable
 *             - in the given environment
 */

object *SS_bind_env(vr, penv)
   object *vr, *penv;
   {Register object *b;
    char *name;
    HASHTAB *tab;

    name = SS_VARIABLE_NAME(vr);
    if (name != NULL)
       for ( ; !SS_nullobjp(penv); penv = SS_cdr(penv))
	   {tab = SS_GET(HASHTAB, SS_car(penv));
	    b   = (object *) SC_def_lookup(name, tab);
	    if (b != NULL)
	       return(b);};

    return(NULL);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_XTND_ENV - A and S environment control */

object *SS_xtnd_env(vrs, vls, base)
   object *vrs, *vls, *base;
   {object *env;

    env = SS_mk_cons(SS_mk_frame(vrs, vls), base);

    return(env);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_SET_VAR - set the specified variable to the given value
 *            - in the specified environment
 */

void SS_set_var(vr, vl, penv)
   object *vr, *vl, *penv;
   {Register object *b;
    char *name;
    HASHTAB *tab;

    name = SS_VARIABLE_NAME(vr);
    for ( ; !SS_nullobjp(penv); penv = SS_cdr(penv))
        {tab = SS_GET(HASHTAB, SS_car(penv));
         b   = (object *) SC_def_lookup(name, tab);
         if (b != NULL)
	    {SS_GC(b);
	     SC_install(name, vl, SS_OBJECT_P_S, tab);
	     return;};};

    SS_error("UNBOUND VARIABLE - SS_SET_VAR", vr);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_DEF_VAR - define the specified variable to the given value
 *            - in the specified environment
 */

void SS_def_var(vr, vl, penv)
   object *vr, *vl, *penv;
   {Register object *b, *lst;
    char *name;
    HASHTAB *tab;

    name = SS_VARIABLE_NAME(vr);
    for (lst = penv ; !SS_nullobjp(lst); lst = SS_cdr(lst))
        {tab = SS_GET(HASHTAB, SS_car(lst));
         b   = (object *) SC_def_lookup(name, tab);
         if (b != NULL)
	    {SS_GC(b);
	     SC_install(name, vl, SS_OBJECT_P_S, tab);
	     return;};};

    tab = SS_GET(HASHTAB, SS_car(penv));
    SC_install(name, vl, SS_OBJECT_P_S, tab);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SS_MK_FRAME - make a new environmental frame */

object *SS_mk_frame(vrs, vls)
   object *vrs, *vls;
   {Register object *fr, *var, *val;
    char *name;
    HASHTAB *tab;

    tab = SC_make_hash_table(HSZSMALL, NODOC);

    if (SS_variablep(vrs))
       {name = SS_VARIABLE_NAME(vrs);
	SC_install(name, vls, SS_OBJECT_P_S, tab);}

    else
       {for (; SS_consp(vrs); vrs = SS_cdr(vrs))
            {if (SS_consp(vls))
                {val = SS_car(vls);
                 vls = SS_cdr(vls);}
             else
                val = SS_null;

             var  = SS_car(vrs);
	     name = SS_VARIABLE_NAME(var);
	     SC_install(name, val, SS_OBJECT_P_S, tab);};

/* if end of vrs is not NULL we had dotted arg list */
        if (!SS_nullobjp(vrs))
	   {name = SS_VARIABLE_NAME(vrs);
	    SC_install(name, vls, SS_OBJECT_P_S, tab);};};

    fr = SS_mk_hash_table(tab);

    return(fr);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
