/*
 * TRASCII - ASCII translation spoke for SX
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#ifndef HAVE_ASCII

#define HAVE_ASCII

#include "cpyright.h"
 
#include "../../sxtrans.h"
#include "../trconfig.h"

int
 _TR_table_n = 0;

long
 _TR_table_ln = 0;

char
 *TR_table_name = NULL;

static PM_matrix
 *TR_current_table = NULL;

static int
 SC_DECLARE(TR_find_ascii_table, (FILE *fp, int n,
			       int *pfn, int *pnr, int *pnc,
			       int nl, long *paddrt,
			       int nlab, long *paddrl));

static REAL
 *SC_DECLARE(TR_extract_vector, (PM_matrix *a,
                              long o, long s, long n));

static PM_set
 *SC_DECLARE(TR_table_set, (object *specs, PM_matrix *table));

static object
 *SC_DECLARE(TR_read_ascii_table, (object *argl)),
 *SC_DECLARE(TR_table_map, (object *argl)),
 *SC_DECLARE(TR_wrt_ascii_table, (object *argl));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* TR_INSTALL_ASCII_FUNCS - install the ASCII extensions to Scheme */
 
void TR_install_ascii_funcs()
   {

    SS_install("read-table*",
               "Procedure: Read nth table that starts at or after specified line in ASCII file\n     Usage: read-table* <file> [<n> [<label-line> [<line>]]]",
               SS_nargs,
               TR_read_ascii_table, SS_PR_PROC);

    SS_install("read-table",
               "Macro: Read nth table that starts at or after specified line in ASCII file\n     Usage: read-table* <file> [<n> [<label-line> [<line>]]]",
               SS_nargs,
               TR_read_ascii_table, SS_UR_MACRO);

    SS_install("table->pm-mapping",
               "Procedure: Extract a mapping from current table\n     Usage: table->pm-mapping <name> <domain-list> <range-list>",
               SS_nargs,
               TR_table_map, SS_PR_PROC);

    SS_install("pm-mapping->table",
               "Procedure: Write a mapping to the given file\n     Usage: pm-mapping->table <file> [<mapping>]+",
               SS_nargs,
               TR_wrt_ascii_table, SS_PR_PROC);

    return;}
 
/*--------------------------------------------------------------------------*/

/*                                INPUT ROUTINES                            */

/*--------------------------------------------------------------------------*/

/* TR_READ_ASCII_TABLE - read a table of numbers from an  ASCII input file */

static object *TR_read_ascii_table(argl)
   object *argl;
   {int i, j, n, nc, nr, nl, fn, nlabel;
    long addrt, addrl;
    char *name, *token, label[MAXLINE];
    FILE *fp;

    memset(label, 0, MAXLINE);

    if (TR_current_table != NULL)
       {PM_destroy(TR_current_table);
        TR_current_table = NULL;};

    name   = NULL;
    n      = 1;
    nl     = 1;
    nlabel = 0;
    SS_args(argl,
            SC_STRING_I, &name,
            SC_INTEGER_I, &n,
            SC_INTEGER_I, &nlabel,
            SC_INTEGER_I, &nl,
            0);

    TR_table_name = SC_strsavef(name, "char*:TR_READ_ASCII_TABLE:tablename");
    name = SC_search_file(NULL, TR_table_name);

    if (name == NULL)
       SS_error("CAN'T FIND FILE - TR_READ_ASCII_TABLE", argl);

    fp = io_open(name, "r");
    if (fp == NULL)
       SS_error("CAN'T OPEN FILE - TR_READ_ASCII_TABLE", argl);

    if (!TR_find_ascii_table(fp, n, &fn, &nr, &nc, nl, &addrt, nlabel, &addrl))
       SS_error("REQUESTED TABLE NOT FOUND - TR_READ_ASCII_TABLE", argl);

    TR_current_table = PM_create(nr, nc);

    if (addrl != -1)
       {if (io_seek(fp, addrl, SEEK_SET))
	   return(FALSE);

	GETLN(label, MAXLINE, fp);}
    else
        label[0] = '\0';

    if (io_seek(fp, addrt, SEEK_SET))
       return(FALSE);

    for (i = 1; i <= nr; i++)
        {GETLN(SC_line, MAXLINE, fp);

         for (j = 1; j <= nc; j++)
             {token = SC_firsttok(SC_line, " \t\n\r");
              if ((j == 1) && !fn)
                 token = SC_firsttok(SC_line, " \t\n\r");

              PM_element(TR_current_table, i, j) = ATOF(token);};};

    io_close(fp);

    if (SS_interactive == ON)
       {if (label[0] == '\0')
           PRINT(stdout,
                 "\n Table %d : %d rows and %d columns\n\n",
                 n, nr, nc);
        else
           PRINT(stdout,
                 "\n Table %d : %d rows and %d columns\n Label: %s\n\n",
                 n, nr, nc, label);};

    _TR_table_n = n;
    _TR_table_ln = nl;

    return(SS_make_list(SC_INTEGER_I, &nr,
                        SC_INTEGER_I, &nc,
                        0));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TR_FIND_ASCII_TABLE - find the specified table in the given file
 *                     - return TRUE if successfTR, otherwise return FALSE
 *
 * table-row :  line of one or more tokens, all numeric
 *                or
 *              line of two or more tokens, all numeric except the first
 * first-row :  table-row
 *                and
 *              (previous line not table-row
 *                 or
 *               number-of-tokens(this line) != number-of-tokens(previous line)
 *                 or
 *               number-numeric-tokens(this line) != number-numeric-tokens(previous line))
 * continuation-row :  table-row
 *                       and
 *                     not first-row
 * table :  first-row followed by zero or more continuation-rows
 * nc    :  number of numeric tokens per table-row
 * nr    :  one plus number of continuation-rows
 * 
 */

static int TR_find_ascii_table(fp, n, pfn, pnr, pnc, nl, paddrt, nlab, paddrl)
   FILE *fp;
   int n;
   int *pfn, *pnr, *pnc;
   int nl;
   long *paddrt;
   int nlab;
   long *paddrl;
   {int i, j, nc, nr, nt, firstnum, nbefore, nafter, nlb;
    long *addr, naddr, addrt, addrl;
    char *token;

    addrl   = -1;
    nbefore = 0;
    nafter  = 0;
    if (nlab > 0)
       nbefore = nlab;
    else
       nafter = -nlab;

    addr = FMAKE_N(long, nbefore+1,
           "TR_FIND_ASCII_TABLE:addr");

    nr  = 0;
    nc  = 0;
    nlb = 0;

/* skip to specified line */
    for (i = 1; i < nl; i++)
	{addr[nlb++] = io_tell(fp);
	 if (nlb > nbefore)
	    nlb = 0;
	 if (GETLN(SC_line, MAXLINE, fp) == NULL)
            return(FALSE);};

/* loop over tables */
    for (j = 0; j < n; j++)

/* loop over lines until first-row found or EOF */
        {while (TRUE)
            {nc       = 0;
             firstnum = FALSE;
             addrt    = io_tell(fp);

             addr[nlb++] = addrt;
             if (nlb > nbefore)
                nlb = 0;

             if (GETLN(SC_line, MAXLINE, fp) == NULL)
                break;

/* loop over tokens */
             for (nt = 0; TRUE; nt++)
                 {token = SC_firsttok(SC_line, " \t\n\r");
                  if (token == NULL)
                     break;
                  if (nt == 0)
                     {if (SC_numstrp(token))
                         firstnum = TRUE;}
                  else if (!SC_numstrp(token))
                     {nt = 0;
                      break;};};

             nc = firstnum ? nt : nt - 1;
             if (nc > 0)
                break;};
            
/* loop over lines of table */
         for (nr = 1; TRUE; nr++)
             {naddr = io_tell(fp);
              if (GETLN(SC_line, MAXLINE, fp) == NULL)
                 break;

              for (nt = 0; TRUE; nt++)
                  {token = SC_firsttok(SC_line, " \t\n\r");
                   if (token == NULL)
                       break;
                   if (nt == 0)
                      {if ((SC_numstrp(token) && !firstnum) ||
                          (!SC_numstrp(token) && firstnum))
                          break;}
                   else
                      if (!SC_numstrp(token))
                         {nt = 0;
                          break;};};

              nt = firstnum ? nt : nt - 1;
              if (nt != nc)
                 break;

/* if this table is before the one we want, buffer up addresses. */
	      if (j < n-1)
		{addr[nlb++] = naddr;
		 if (nlb > nbefore)
		   nlb = 0;};};

         io_seek(fp, naddr, SEEK_SET);};
         
    if (nbefore > 0)
       {nlb--;
        if (nlb < 0)
           nlb += nbefore + 1;
        addrt = addr[nlb];

        nlb -= nbefore;
        if (nlb < 0)
           nlb += nbefore + 1;
        addrl = addr[nlb];};

    if (nafter > 0)
       {addrt = addr[0];
        for (i = 0; i < nafter; i++)
            {addrl = io_tell(fp);
             if (GETLN(SC_line, MAXLINE, fp) == NULL)
                return(FALSE);};};

    SFREE(addr);

    if ((j == n) && (nc > 0))
       {*pfn    = firstnum;
        *pnc    = nc;
        *pnr    = nr;
        *paddrt = addrt;
        *paddrl = addrl;
        return(TRUE);}
    else
       return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TR_EXTRACT_VECTOR - extract a vector from the given array */

static REAL *TR_extract_vector(a, o, s, n)
   PM_matrix *a;
   long o, s, n;
   {long i;
    REAL *val, *src;

    src = a->array + o;
    val = FMAKE_N(REAL, n, "TR_EXTRACT_VECTOR:val");
    for (i = 0L; i < n; i++)
        val[i] = src[i*s];

    return(val);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TR_TABLE_SET - build a set from information in the TABLE using
 *              - specifications in SPECS as follows:
 *              -
 *              - <set-list>  := (name <dims> <component-1> <component-2> ...)
 *              - <dims>      := (d1 d2 ...)
 *              - <component> := (start [n-pts] [stride])
 */

static PM_set *TR_table_set(specs, table)
   object *specs;
   PM_matrix *table;
   {object *sp, *dims, *comps;
    int i, nd, nde, dv;
    int *maxes;
    long start, step, ne, npts;
    char *name;
    REAL **elem;
    PM_set *set;

    name = NULL;
    SS_args(specs,
	    SC_STRING_I, &name,
	    SS_OBJECT_I, &dims,
	    0);

    nd    = _SS_length(dims);
    maxes = FMAKE_N(int, nd, "TR_TABLE_SET:maxes");
    for (i = 0; dims != SS_null; dims = SS_cdr(dims), i++)
        {SS_args(dims,
		 SC_INTEGER_I, &dv,
		 0);
	 maxes[i] = dv;};

    comps = SS_cddr(specs);
    nde   = _SS_length(comps);
    elem  = FMAKE_N(REAL *, nde, "TR_TABLE_SET:elem");

    for (i = 0; comps != SS_null; comps = SS_cdr(comps), i++)
        {sp = SS_car(comps);
         start = 0L;
         npts  = TR_current_table->nrow;
         step  = TR_current_table->ncol;
	 SS_args(sp,
		 SC_LONG_I, &start,
		 SC_LONG_I, &npts,
		 SC_LONG_I, &step,
		 0);

         if ((i > 0) && (npts != ne))
	    SS_error("BAD SPECIFICATION - TR_TABLE_SET", sp);

         elem[i] = TR_extract_vector(TR_current_table, start, step, npts);
         ne = npts;};

    set = _PM_make_set(name, SC_REAL_S, FALSE, ne, nd, nde,
		       maxes, elem, NULL,
		       NULL, NULL, NULL,
		       NULL, NULL, NULL, NULL, NULL);

    return(set);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TR_TABLE_MAP - create and return a graph from the current table using
 *              - the given specifications
 *              -
 *              - (table-map <name> <domain-list> <range-list>)
 *              - <set-list>  := (name <dims> <component-1> <component-2> ...)
 *              - <dims>      := (d1 d2 ...)
 *              - <component> := (start [stop] [stride])
 */

static object *TR_table_map(argl)
   object *argl;
   {char *name, bf[MAXLINE];
    object *dmlst, *rnlst;
    PM_set *domain, *range;
    PM_mapping *f;

    name = NULL;
    SS_args(argl,
	    SC_STRING_I, &name,
	    SS_OBJECT_I, &dmlst,
	    SS_OBJECT_I, &rnlst,
            0);

    domain = TR_table_set(dmlst, TR_current_table);
    range  = TR_table_set(rnlst, TR_current_table);

    if (name == NULL)
       {static int tm = 1;

	sprintf(bf, "TableMap%d", tm++);
	name = bf;};
    
    f = PM_make_mapping(name, PM_LR_S, domain, range, N_CENT, NULL);

    return(SX_mk_mapping(f));}

/*--------------------------------------------------------------------------*/

/*                          MAPPING OUTPUT ROUTINES                         */

/*--------------------------------------------------------------------------*/

/* TR_WRT_ASCII_TABLE - write mapping to an ASCII file */

static object *TR_wrt_ascii_table(argl)
   object *argl;
   {long i, j, ne, ndd, ndr, nded, nder;
    int *maxes;
    object *outprt;
    REAL **delem, **relem;
    PM_set *domain, *range;
    PM_mapping *f;
    FILE *fp;

    outprt = SS_car(argl);
    argl   = SS_cdr(argl);

    fp = SS_OUTSTREAM(outprt);

    for ( ; SS_consp(argl); argl = SS_cdr(argl))
        {SS_args(argl,
		 G_MAPPING, &f,
		 0);

         domain = f->domain;
         range  = f->range;

         ndd  = domain->dimension;
         ndr  = range->dimension;
         nded = domain->dimension_elem;
         nder = range->dimension_elem;
         ne   = domain->n_elements;

         delem = (REAL **) domain->elements;
         relem = (REAL **) range->elements;

/* print the dimension info */
	 io_printf(fp, "# %s (dimension", f->name);
         maxes = domain->max_index;
	 for (i = 0; i < ndd; i++)
	     io_printf(fp, " %d", maxes[i]);
	 io_printf(fp, ")\n");

/* print the column labels */
         io_printf(fp, "# Indx");
	 for (i = 0; i < nded; i++)
	     io_printf(fp, "     D%d    ", i+1);
	 for (i = 0; i < nder; i++)
	     io_printf(fp, "     R%d    ", i+1);
	 io_printf(fp, "\n");

/* print the columns */
	 for (j = 0L; j < ne; j++)
	     {io_printf(fp, "%6ld", j+1);
	      for (i = 0; i < nded; i++)
		  io_printf(fp, " %10.3e", delem[i][j]);
	      for (i = 0; i < nder; i++)
		  io_printf(fp, " %10.3e", relem[i][j]);
	      io_printf(fp, "\n");};

	 io_printf(fp, "\n");};

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

#endif
