/*
 * SXPDBD.C - PDB diff functionality in SX
 *
 * Source Version: 3.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "sx.h"

#define LINE_SIZE       90
#define LINE_SIZ2      120

#define NOT_PRESENT    -10
#define CANT_SEEK      -20
#define CANT_ALLOC     -30
#define CANT_READ      -40
#define NOT_SAME_TYPE  -50
#define NOT_SAME_SIZE  -60
#define NOT_PRIMITIVE  -70
#define BAD_UNIT_SIZE  -80

/*--------------------------------------------------------------------------*/

#define SX_SKIP_ITAG(fpa)  _PD_rfgets(bf, MAXLINE, fpa->stream)

/*--------------------------------------------------------------------------*/

#define SX_READ_ITAG(fpa, bf1, na, ta, ada, fla)                             \
    _PD_rfgets(bf1, MAXLINE, fpa);                                           \
    na = atoi(SC_strtok(bf1, "\001", s));                                    \
    ta = SC_strtok(NULL, "\001\n", s);                                       \
    if ((token = SC_strtok(NULL, "\001\n", s)) == NULL)                      \
       {ada = -1L;                                                           \
        fla = TRUE;}                                                         \
    else                                                                     \
       {ada = atol(token);                                                   \
        if ((token = SC_strtok(NULL, "\001\n", s)) == NULL)                  \
           fla = TRUE;                                                       \
        else                                                                 \
           fla = atoi(token);}

/*--------------------------------------------------------------------------*/

#define SKIP_UNIT(file_a, ada, skip)                                         \
    io_seek(file_a->stream, ada, SEEK_SET);                                  \
    _PD_skip_over(file_a, skip, FALSE);

/*--------------------------------------------------------------------------*/

#define DIFF_FIX_ARRAY(ret, indx, a, b)                                      \
    if (SX_disp_individ_diff)                                                \
       {for (i = 0L; i < nitems; i++)                                        \
            {if (a[i] != b[i])                                               \
                {ret    &= FALSE;                                            \
                 indx[i] = TRUE;};};}                                        \
    else                                                                     \
       {for (i = 0L; i < nitems; i++)                                        \
            ret &= (a[i] == b[i]);}

/*--------------------------------------------------------------------------*/

#define DIFF_FLOAT_ARRAY(ret, indx, a, b, tol)                               \
    if (SX_disp_individ_diff)                                                \
       {for (i = 0L; i < nitems; i++)                                        \
            {del  = 2.0*(a[i] - b[i])/(ABS(a[i]) + ABS(b[i]) + SMALL);       \
             if ((del < -tol) || (tol < del))                                \
                {ret    &= FALSE;                                            \
                 indx[i] = TRUE;};};}                                        \
    else                                                                     \
       {for (i = 0L; i < nitems; i++)                                        \
            {del  = 2.0*(a[i] - b[i])/(ABS(a[i]) + ABS(b[i]) + SMALL);       \
             ret &= ((-tol < del) && (del < tol));};}

/*--------------------------------------------------------------------------*/

#define DISP_ARRAY(fmt, indx, a, b)                                          \
    for (i = 0L; i < length; i++)                                            \
        {if (indx[i])                                                        \
            {if ((dims == NULL) && (length == 1))                            \
                sprintf(tmp, "%s", name1);                                   \
             else                                                            \
                sprintf(tmp, "%s(%s)", name1,                                \
                        PD_index_to_expr(bf1, i, dims, mjr, def_off));       \
             memcpy(bf, tmp, strlen(tmp));                                   \
             sprintf(tmp, fmt, a[i], b[i]);                                  \
             memcpy(&bf[nn], tmp, strlen(tmp) + 1);                          \
             PRINT(stdout, "%s\n", bf);                                      \
             memset(bf, ' ', LINE_SIZE);};}

/*--------------------------------------------------------------------------*/

double
 SX_float_tol,
 SX_double_tol;

int
 SX_disp_individ_diff = FALSE,
 SX_promote_flag      = 0,
 SX_promote_fixed     = FALSE,
 SX_promote_float     = FALSE;

static int
 err_fpe = FALSE;

static jmp_buf
 diff_err;

static int
 SC_DECLARE(_SX_display_diff,
         (PDBfile *file, char *name1, char *name2, char *pv1, char *pv2,
          char *indx, long length, char *type, dimdes *dims)),
 SC_DECLARE(_SX_diff_tree,
         (char *name1, char *name2, PDBfile *file_a, PDBfile *file_b,
          syment *epa, syment *epb, int flag)),
 SC_DECLARE(_SX_diff_var,
         (PDBfile *file_a, PDBfile *file_b, char *name1, char *name2,
          long *size_a, long *size_b)),
 SC_DECLARE(_SX_diff_leaf,
         (char *name1, char *name2, PDBfile *file_a, PDBfile *file_b,
          syment *epa, syment *epb)),
 SC_DECLARE(_SX_diff_leaf_indirects,
	 (PDBfile *file_a, PDBfile *file_b, PDBfile *file_c,
	  char *bf_a, char *bf_b, long na, char *name1, 
          char *name2, defstr *dp)),
 SC_DECLARE(_SX_diff_structs,
         (PDBfile *fp, char *name1, char *name2, char *bf_a, char *bf_b,
          char *type, long nitems)),
 SC_DECLARE(_SX_diff_primitives,
         (PDBfile *fp, char *name1, char *name2, char *bf_a, char *bf_b,
          char *type, long nitems, dimdes *dims)),
 SC_DECLARE(_SX_diff_indirection,
         (char *name1, char *name2, PDBfile *file_a, PDBfile *file_b)),
 SC_DECLARE(_SX_diff_casts,
         (PDBfile *file_a, PDBfile *file_b,
          char *name1, char *name2, char **pa, char **pb)),
 SC_DECLARE(_SX_check_pointers,
         (PDBfile *file_a, PDBfile *file_b,
          char *name1, char *name2, char *ta, long na,
	  long ada, char *tb, long nb, long adb)),
 SC_DECLARE(_SX_rd_leaf_t,
         (PDBfile *file, syment *ep, char *vr, char *type,
          long nitems, long bytepitem, char *tc));

static void
 SC_DECLARE(_SX_print_individ_diff,
         (PDBfile *file, char *name1, char *name2, char *pv1, char *pv2,
          char *indx, long length, char *type,
          dimdes *dims, int mjr)),
 SC_DECLARE(_SX_diff_signal, (int sig));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DIFF_VAR - two variables from different files */

object *SX_diff_var(argl)
   object *argl;
   {int ret, same_name;
    object *obj;
    PDBfile *file_a, *file_b;
    char *name1, *name2, bf[MAXLINE];
    long size_a, size_b;

    if (SS_consp(argl))
       {SX_GET_OBJECT_FROM_LIST(SX_ipdbfilep(obj), file_a,
                                FILE_STREAM(PDBfile, obj), argl,
                                "FIRST ARGUMENT NOT PDBFILE - SX_DIFF_VAR");}
    else
       SS_error("BAD FIRST ARGUMENT - SX_DIFF_VAR", SS_null);

    if (SS_consp(argl))
       {SX_GET_OBJECT_FROM_LIST(SX_ipdbfilep(obj), file_b,
                                FILE_STREAM(PDBfile, obj), argl,
                                "SECOND ARGUMENT NOT PDBFILE - SX_DIFF_VAR");}
    else
       SS_error("BAD SECOND ARGUMENT - SX_DIFF_VAR", SS_null);

    if (SS_consp(argl))
       {SX_GET_STRING_FROM_LIST(name1, argl,
                                "THIRD ARGUMENT NOT NAME - SX_DIFF_VAR");}
    else
       SS_error("BAD THIRD ARGUMENT - SX_DIFF_VAR", SS_null);

    if (SS_consp(argl))
       {SX_GET_STRING_FROM_LIST(name2, argl,
                                "FOURTH ARGUMENT NOT NAME - SX_DIFF_VAR");}
    else
       SS_error("BAD FOURTH ARGUMENT - SX_DIFF_VAR", SS_null);

    if (SS_consp(argl))
       {SX_GET_INTEGER_FROM_LIST(PD_tolerance, argl,
                                 "FIFTH ARGUMENT NOT INTEGER - SX_DIFF_VAR");};

    if (SS_consp(argl))
       {SX_GET_INTEGER_FROM_LIST(SX_disp_individ_diff, argl,
                                 "SIXTH ARGUMENT NOT INTEGER - SX_DIFF_VAR");};

    if (SS_consp(argl))
       {SX_GET_INTEGER_FROM_LIST(SX_promote_flag, argl,
                                 "SEVENTH ARGUMENT NOT INTEGER - SX_DIFF_VAR");};
    
    if ((SX_promote_flag == 1) || (SX_promote_flag == 3))
       SX_promote_fixed = TRUE;
    else
       SX_promote_fixed = FALSE;

    if ((SX_promote_flag == 2) || (SX_promote_flag == 3))
       SX_promote_float = TRUE;
    else
       SX_promote_float = FALSE;

    ret = _SX_diff_var(file_a, file_b, name1, name2, &size_a, &size_b);
    same_name = (strcmp (name1, name2) == 0);
    switch (ret)
       {case NOT_PRESENT   : if (same_name)
                                sprintf(bf,
                                        "Variable %s not present in both files",
                                        name1);
                             else
                                sprintf(bf,
                                        "Variable not present: %s in File 1 or %s in File 2",
                                        name1, name2);
                             obj = SS_mk_string(bf);
                             break;
        case CANT_SEEK   :   if (same_name)
                                sprintf(bf,
                                        "Can't seek variable %s in both files",
                                        name1);
                             else
                                sprintf(bf,
                                        "Can't seek variable: %s in File 1 or %s in File 2",
                                        name1, name2);
                             obj = SS_mk_string(bf);
                             break;
        case CANT_ALLOC  :   if (same_name)
                                sprintf(bf,
                                        "Can't allocate variable %s from both files",
                                        name1);
                             else
                                sprintf(bf,
                                        "Can't allocate variable: %s from File 1 or %s from File 2",
                                        name1, name2);
                             obj = SS_mk_string(bf);
                             break;
        case CANT_READ   :   if (same_name)
                                sprintf(bf,
                                        "Can't read variable %s in both files",
                                        name1);
                             else
                                sprintf(bf,
                                        "Can't read variable: %s in File 1 or %s in File 2",
                                        name1, name2);
                             obj = SS_mk_string(bf);
                             break;
        case NOT_SAME_TYPE : if (same_name)
                                sprintf(bf,
                                        "Variable %s not same type in both files",
                                        name1);
                             else
                                sprintf(bf,
                                        "Variable %s in File 1 not same type as %s in File 2",
                                        name1, name2);
                             obj = SS_mk_string(bf);
                             break;
        case NOT_SAME_SIZE : if (same_name)
                                sprintf(bf,
                                        "Variable %s not same size: %ld in File 1 and %ld in File 2",
                                        name1, size_a, size_b);
                             else
                                sprintf(bf,
                                        "Variables not same size: %s in File 1 is %ld and %s in File 2 is %ld",
                                        name1, size_a, name2, size_b);
                             obj = SS_mk_string(bf);
                             break;
        case NOT_PRIMITIVE : if (same_name)
                                sprintf(bf,
                                        "Variable %s not primitive type",
                                        name1);
                             else
                                sprintf(bf,
                                        "Variable not primitive type: %s in File 1 or %s in File 2",
                                        name1, name2);
                             obj = SS_mk_string(bf);
                             break;
        case BAD_UNIT_SIZE : if (same_name)
                                sprintf(bf,
                                        "Bad unit size for variable %s",
                                        name1);
                             else
                                sprintf(bf,
                                        "Bad unit size for variable %s in File 1 or %s in File 2",
                                        name1, name2);
                             obj = SS_mk_string(bf);
                             break;
        case TRUE          : obj = SS_t;
                             break;
        default            : if (err_fpe)
                                {if (same_name)
                                    sprintf(bf,
                                            "Variable %s contains bad numbers",
                                            name1);
                                 else
                                    sprintf(bf,
                                            "Variable %s in File 1 or %s in File 2 contains bad numbers",
                                            name1, name2);
                                 obj = SS_mk_string(bf);
			         err_fpe = FALSE;}
                             else
			        obj = SS_f;
                             break;};

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DIFF_VAR - compare a variable from each of two files
 *              - return TRUE iff successful
 */

static int _SX_diff_var(file_a, file_b, name1, name2, nitemsa, nitemsb)
   PDBfile *file_a, *file_b;
   char *name1, *name2;
   long *nitemsa, *nitemsb;
   {int ret;
    syment *epa, *epb;
    char *s, *typea, *typeb;
    char fullpath_a[MAXLINE], fullpath_b[MAXLINE], *fullpath;

    switch (setjmp(diff_err))
       {case ABORT    : return(FALSE);
        case ERR_FREE : return(TRUE);
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

    SIGNAL(SIGFPE, _SX_diff_signal);

    strcpy(fullpath_a, _PD_fixname(file_a, name1));
    s = _PD_expand_hyper_name(file_a, fullpath_a);
    if (s == NULL)
       return(NOT_PRESENT);
    strcpy(fullpath_a, s);
    SFREE(s);

    strcpy(fullpath_b, _PD_fixname(file_b, name2));
    s = _PD_expand_hyper_name(file_b, fullpath_b);
    if (s == NULL)
       return(NOT_PRESENT);
    strcpy(fullpath_b, s);
    SFREE(s);

    epa = _PD_effective_ep(file_a, fullpath_a, FALSE, NULL);
    epb = _PD_effective_ep(file_b, fullpath_b, FALSE, NULL);

    if ((epa == NULL) || (epb == NULL))
       {_PD_rl_syment_d(epa);
        _PD_rl_syment_d(epb);
        return(NOT_PRESENT);};
    if (strlen(fullpath_a) < strlen(fullpath_b))
       fullpath = fullpath_a;
    else
       fullpath = fullpath_b;

/* position the file pointers */
    if (io_seek(file_a->stream, PD_entry_address(epa), SEEK_SET) ||
        io_seek(file_b->stream, PD_entry_address(epb), SEEK_SET))
       {_PD_rl_syment_d(epa);
        _PD_rl_syment_d(epb);
        return(CANT_SEEK);};

    typea = PD_entry_type(epa);
    typeb = PD_entry_type(epb);

    if (_SX_type_equal(file_a, file_b, typea, typeb) == 0)
       {_PD_rl_syment_d(epa);
        _PD_rl_syment_d(epb);
        return(NOT_SAME_TYPE);};

    *nitemsa = PD_entry_number(epa);
    *nitemsb = PD_entry_number(epb);

    if (*nitemsa != *nitemsb)
       {_PD_rl_syment_d(epa);
        _PD_rl_syment_d(epb);
        return(NOT_SAME_SIZE);};

    _PD_digits_tol(file_a, file_b);
    _PD_set_format_defaults();
    _PD_set_user_formats();

    ret = _SX_diff_tree(fullpath_a, fullpath_b, file_a, file_b, epa, epb, FALSE);
    _PD_rl_syment_d(epa);
    _PD_rl_syment_d(epb);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DIFF_TREE - compare two data trees
 *               - return TRUE iff the contents of the two trees are identical
 *               - (indirections need not point to the same addresses
 *               - but their contents must compare)
 */

static int _SX_diff_tree(name1, name2, file_a, file_b, epa, epb, flag)
   char *name1, *name2;
   PDBfile *file_a, *file_b;
   syment *epa, *epb;
   int flag;
   {long i, nitems;
    char namea[MAXLINE], nameb[MAXLINE];
    int ret;

    if (!_PD_indirection(PD_entry_type(epa)))
       ret = _SX_diff_leaf(name1, name2, file_a, file_b, epa, epb);
    else
       {nitems = PD_entry_number(epa);
        ret    = TRUE;
        for (i = 0L; i < nitems; i++)
	    {if (flag)
	        {sprintf(namea, "%s[%ld]", name1, i + file_a->default_offset);
		 sprintf(nameb, "%s[%ld]", name2, i + file_b->default_offset);}
	     else
		{sprintf(namea, "%s", name1);
		 sprintf(nameb, "%s", name2);};
	     ret &= _SX_diff_indirection(namea, nameb, file_a, file_b);};};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DIFF_INDIRECTION - read the information about an indirection,
 *                      - allocate the space, connect the pointer, and
 *                      - read in the data and compare it
 */

static int _SX_diff_indirection(name1, name2, file_a, file_b)
   char *name1, *name2;
   PDBfile *file_a, *file_b;
   {long na, nb, ada, adb, oada, oadb;
    int fla, flb, ret;
    char *ta, *tb, *token, bfa[MAXLINE], bfb[MAXLINE], bfc[MAXLINE], *s;
    FILE *fpa, *fpb;
    syment *epa, *epb;

    fpa = file_a->stream;
    fpb = file_b->stream;

    SX_READ_ITAG(fpa, bfa, na, ta, ada, fla);
    SX_READ_ITAG(fpb, bfb, nb, tb, adb, flb);

    if (!_SX_check_pointers(file_a, file_b, name1, name2,
			    ta, 1L, ada, tb, 1L, adb))
       return(FALSE);

    if (ta == NULL)
       return(TRUE);

    if ((na != nb) || (strcmp(ta, tb) != 0) ||
        (((ada == -1L) || (adb == -1L)) && (ada != adb)))
       {if (strcmp (name1, name2) == 0)
           PRINT(stdout, "\nPointer, %s, differs in type or number of elements\n", name1);
        else
           PRINT(stdout, "\nPointer %s differs in type or number of elements from pointer %s\n", name1, name2);
        if (ada != -1L)
           SKIP_UNIT(file_a, ada, 1L);
        if (adb != -1L)
           SKIP_UNIT(file_b, adb, 1L);
        return(FALSE);};

    if (!_SX_type_equal(file_a, file_b, ta, tb))
       {PRINT(stdout, "\nType, %s, differs between the two files\n", ta);
        longjmp(diff_err, ABORT);};

    if ((ada != -1L) && (na != 0L))
       {oada = io_tell(fpa);
        if (oada == -1L)
           SS_error("CAN'T FIND ADDRESS A - _SX_DIFF_INDIRECTION", SS_null);

        oadb = io_tell(fpb);
        if (oadb == -1L)
           SS_error("CAN'T FIND ADDRESS B - _SX_DIFF_INDIRECTION", SS_null);

        if (!fla)
           {io_seek(fpa, ada, SEEK_SET);
            _PD_rfgets(bfc, MAXLINE, fpa);};

        if (!flb)
           {io_seek(fpb, adb, SEEK_SET);
            _PD_rfgets(bfc, MAXLINE, fpb);};

        if ((ada = io_tell(fpa)) == -1L)
           SS_error("CAN'T FIND NEW ADDRESS A - _SX_DIFF_INDIRECTION",
                    SS_null);
        if ((adb = io_tell(fpb)) == -1L)
           SS_error("CAN'T FIND NEW ADDRESS B - _SX_DIFF_INDIRECTION",
                    SS_null);

        epa = _PD_mk_syment(ta, na, ada, NULL, NULL);
        epb = _PD_mk_syment(tb, nb, adb, NULL, NULL);

        ret = _SX_diff_tree(name1, name2, file_a, file_b, epa, epb, TRUE);

        _PD_rl_syment_d(epa);
        _PD_rl_syment_d(epb);

        if (!fla)
           if (io_seek(fpa, oada, SEEK_SET))
              SS_error("FAILED TO FIND OLD ADDRESS A - _SX_DIFF_INDIRECTION",
                       SS_null);

        if (!flb)
           if (io_seek(fpb, oadb, SEEK_SET))
              SS_error("FAILED TO FIND OLD ADDRESS B - _SX_DIFF_INDIRECTION",
                       SS_null);};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DIFF_LEAF - compare two arrays of numbers */

static int _SX_diff_leaf(name1, name2, file_a, file_b, epa, epb)
   char *name1, *name2;
   PDBfile *file_a, *file_b;
   syment *epa, *epb;
   {int ret;
    long na, nb, bpi, bpa, bpb;
    char *ta, *tb, *tc, *bf_a, *bf_b;
    memdes *mem_lst;
    defstr *dp;
    dimdes *dims;
    HASHTAB *hc, *fa, *fb;
    PDBfile *file_c;

    na = PD_entry_number(epa);
    ta = PD_entry_type(epa);
    nb = PD_entry_number(epb);
    tb = PD_entry_type(epb);
    fa = file_a->chart;
    fb = file_b->chart;

    if (!_SX_type_equal(file_a, file_b, ta, tb))
       longjmp(diff_err, ABORT);

/* find the bytes per item */
    bpa = _PD_lookup_size(ta, fa);
    if (bpa == -1L)
       return(BAD_UNIT_SIZE);

    bpb = _PD_lookup_size(tb, fb);
    if (bpb == -1L)
       return(BAD_UNIT_SIZE);

    if (bpa >= bpb)
       {tc = ta;
        hc = file_a->host_chart;
        file_c = file_a;}
    else
       {tc = tb;
        hc = file_b->host_chart;
        file_c = file_b;}

    if ((bpi = _PD_lookup_size(tc, hc)) == -1L)
       return(BAD_UNIT_SIZE);

    bf_a = (char *) SC_alloc(na, bpi, "_SX_DIFF_LEAF:bf_a");
    bf_b = (char *) SC_alloc(nb, bpi, "_SX_DIFF_LEAF:bf_b");

    if ((bf_a == NULL) || (bf_b == NULL))
       {if ((bf_a == NULL) && (bf_b == NULL) &&
	    (na == 0) && (nb == 0))
	   return(TRUE);
	else
	   return(CANT_ALLOC);};

/* bring in the leaf from file a */
    if (!_SX_rd_leaf_t(file_a, epa, bf_a, ta, na, bpa, tc))
       return(CANT_READ);

/* bring in the leaf from file b */
    if (!_SX_rd_leaf_t(file_b, epb, bf_b, tb, nb, bpb, tc))
       return(CANT_READ);

    dp = PD_inquire_table_type(hc, tc);
    if (dp == NULL)
       SS_error("VARIABLE NOT IN STRUCTURE CHART - _SX_DIFF_LEAF", SS_mk_string(tc));

    dims = PD_entry_dimensions(epa);

    mem_lst = dp->members;
    if (mem_lst == NULL)
        ret = _SX_diff_primitives(file_c, name1, name2, bf_a, bf_b, tc, na, dims);
    else
       {ret = _SX_diff_structs(file_c, name1, name2, bf_a, bf_b, tc, na);

        if (dp->n_indirects)
           ret &= _SX_diff_leaf_indirects(file_a, file_b, file_c,
					  bf_a, bf_b, na, name1, name2, dp);};

    SFREE(bf_a);
    SFREE(bf_b);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DIFF_LEAF_INDIRECTS - compare the indirects in structs */

static int _SX_diff_leaf_indirects(file_a, file_b, file_c,
				   bf_a, bf_b, na, name1, name2, dp)
   PDBfile *file_a, *file_b, *file_c;
   char *bf_a, *bf_b;
   long na;
   char *name1, *name2;
   defstr *dp;
   {int ret;
    long i, size;
    char *sva, *svb, *dsva, *dsvb;
    char bf[MAXLINE], bf1[MAXLINE], bf2[MAXLINE], *mta, *mtb, *mtype;
    char mname1[MAXLINE], mname2[MAXLINE];
    memdes *desc, *mem_lst;

    size    = dp->size;
    mem_lst = dp->members;

    sva  = bf_a;
    svb  = bf_b;
    ret  = TRUE;
    for (i = 0L; i < na; i++)
        {for (desc = mem_lst; desc != NULL; desc = desc->next)
	     {sprintf(mname1, "%s[%ld].%s", name1,
		      i + file_c->default_offset, desc->name);
	      sprintf(mname2, "%s[%ld].%s", name2,
		      i + file_c->default_offset, desc->name);
/*	     {sprintf(mname, "%s.%s", name, desc->name); */

	      if (desc->cast_offs < 0L)
		 {mtype = desc->type;
		  if (_PD_indirection(mtype))
		     ret &= _SX_diff_indirection(mname1, mname2, file_a, file_b);}
	      else
		 {dsva = DEREF(sva + desc->member_offs);
		  dsvb = DEREF(svb + desc->member_offs);

		  if ((dsva == NULL) && (dsvb == NULL))
		     {SX_SKIP_ITAG(file_a);
		      SX_SKIP_ITAG(file_b);}

		  else if (!_SX_check_pointers(file_a, file_b, mname1, mname2,
					       dsva, 0L, 0L, dsvb, 0L, 0L))
		     longjmp(diff_err, ABORT);

		  else
		     {sprintf(bf1, "%s[%ld].%s", name1,
			      i + file_c->default_offset, desc->cast_memb);
		      sprintf(bf2, "%s[%ld].%s", name2,
			      i + file_c->default_offset, desc->cast_memb);
		      _SX_diff_casts(file_a, file_b, bf1, bf2, &mta, &mtb);

		      if (_PD_indirection(mta))
			 ret &= _SX_diff_indirection(mname1,
						     mname2,
						     file_a,
						     file_b);};};};
	 sva += size;
	 svb += size;};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DIFF_CASTS - read the casts from two files
 *                - compare them and return TRUE iff they are OK
 *                - SUBTLE POINT: PDBLib will allocate space for mta and mtb
 *                - It will also remember that it has read those pointers
 *                - Don't SFREE them or they will be lost
 *                - (PD_reset_pointer_list would have to be invoked)
 */

static int _SX_diff_casts(file_a, file_b, name1, name2, pa, pb)
   PDBfile *file_a, *file_b;
   char *name1, *name2, **pa, **pb;
   {long oa, ob;
    char *mta, *mtb;
    int same_name;

    oa = io_tell(file_a->stream);
    ob = io_tell(file_b->stream);

    if (!PD_read(file_a, name1, pa))
       SS_error("CAN'T READ CAST A - _SX_DIFF_CASTS",
                SS_null);
    if (!PD_read(file_b, name2, pb))
       SS_error("CAN'T READ CAST B - _SX_DIFF_CASTS",
                SS_null);

    io_seek(file_a->stream, oa, SEEK_SET);
    io_seek(file_b->stream, ob, SEEK_SET);

    mta = *pa;
    mtb = *pb;

    same_name = (strcmp (name1,name2) == 0);
    if ((mta != NULL) && (mtb != NULL))
       {if (!_SX_type_equal(file_a, file_b, mta, mtb))
           {if (same_name)
               PRINT(stdout, "\nInconsistent casts on %s: ", name1);
            else
               PRINT(stdout, "\nInconsistent casts on %s and %s: ", name1, name2);
            PRINT(stdout, "%s in first file and %s in second file\n",
                  mta, mtb);
            longjmp(diff_err, ABORT);};}
    else if ((mta == NULL) && (mtb == NULL))
       {if (same_name)
           PRINT(stdout, "\nNull casts from %s in both files\n", name1);
        else
           PRINT(stdout, "\nNull casts from %s in File 1 and %s in File 2\n", name1, name2);
        longjmp(diff_err, ABORT);}
    else if (mta == NULL)
       {PRINT(stdout, "\nNo cast from %s in first file\n", name1);
        longjmp(diff_err, ABORT);}
    else if (mtb == NULL)
       {PRINT(stdout, "\nNo cast from %s in second file\n", name2);
        longjmp(diff_err, ABORT);};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DIFF_STRUCTS - compare the non-pointered members of
 *                  - two structures in memory
 *                  - return TRUE if they are the same and
 *                  - FALSE otherwise
 */

static int _SX_diff_structs(fp, name1, name2, bf_a, bf_b, type, nitems)
   PDBfile *fp;
   char *name1, *name2, *bf_a, *bf_b, *type;
   long nitems;
   {int ret;
    long i, mitems, sz;
    memdes *desc;
    char *mtype, *sva, *svb, *tva, *tvb, mname1[MAXLINE], mname2[MAXLINE];
    defstr *dp;
    dimdes *mdims;
    HASHTAB *hc;

    ret = TRUE;
    hc  = fp->host_chart;
    dp  = PD_inquire_table_type(hc, type);
    if (dp == NULL)
       SS_error("BAD TYPE - _SX_DIFF_STRUCTS", SS_null);

    sz  = dp->size;
    sva = bf_a;
    svb = bf_b;
    for (i = 0L; i < nitems; i++)
        {for (desc = dp->members; desc != NULL; desc = desc->next)
             {tva = sva + desc->member_offs;
              tvb = svb + desc->member_offs;

              mtype = desc->type;
              if (_PD_indirection(mtype))
                 continue;

              mitems = desc->number;

	      sprintf(mname1, "%s[%ld].%s", name1, i + fp->default_offset, desc->name);
	      sprintf(mname2, "%s[%ld].%s", name2, i + fp->default_offset, desc->name);
/*
              if (nitems > 1L)
                 sprintf(mname, "%s[%ld].%s", name, i, desc->name);
              else
                 sprintf(mname, "%s.%s", name, desc->name);
*/
              mdims = desc->dimensions;

              if (_PD_prim_typep(mtype, hc, PD_READ))
                 ret &= _SX_diff_primitives(fp, mname1, mname2, tva, tvb,
                                            mtype, mitems, mdims);
              else
                 ret &= _SX_diff_structs(fp, mname1, mname2, tva, tvb,
                                         mtype, mitems);};

         sva += sz;
         svb += sz;};
    

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DIFF_PRIMITIVES - compare two arrays of primitive data types */

static int _SX_diff_primitives(fp, name1, name2, bf_a, bf_b, type, nitems, dims)
   PDBfile *fp;
   char *name1, *name2, *bf_a, *bf_b, *type;
   long nitems;
   dimdes *dims;
   {int ret;
    long i, *lp_a, *lp_b;
    int *ip_a, *ip_b;
    short *sp_a, *sp_b;
    char *cp_a, *cp_b, *indx;
    double del;
    double *dp_a, *dp_b;
    float *fp_a, *fp_b;

    if (SX_disp_individ_diff)
       indx = FMAKE_N(char, nitems, "_SX_DIFF_PRIMITIVES:indx");
    else
       indx = NULL;

/* find the type and compare */
    ret = TRUE;
    if (strcmp(type, "char") == 0)
       {cp_a = (char *) bf_a;
        cp_b = (char *) bf_b;
        DIFF_FIX_ARRAY(ret, indx, cp_a, cp_b);}

    else if (strcmp(type, "short") == 0)
       {sp_a = (short *) bf_a;
        sp_b = (short *) bf_b;
        DIFF_FIX_ARRAY(ret, indx, sp_a, sp_b);}

    else if (strcmp(type, "int") == 0)
       {ip_a = (int *) bf_a;
        ip_b = (int *) bf_b;
        DIFF_FIX_ARRAY(ret, indx, ip_a, ip_b);}

    else if (strcmp(type, "integer") == 0)
       {ip_a = (int *) bf_a;
        ip_b = (int *) bf_b;
        DIFF_FIX_ARRAY(ret, indx, ip_a, ip_b);}

    else if (strcmp(type, "long") == 0)
       {lp_a = (long *) bf_a;
        lp_b = (long *) bf_b;
        DIFF_FIX_ARRAY(ret, indx, lp_a, lp_b);}

    else if (strcmp(type, "REAL") == 0)
       {if (sizeof(REAL) == sizeof(double))
           {dp_a = (double *) bf_a;
            dp_b = (double *) bf_b;
            DIFF_FLOAT_ARRAY(ret, indx, dp_a, dp_b, PD_double_tol);}
        else if (sizeof(REAL) == sizeof(float))
           {fp_a = (float *) bf_a;
            fp_b = (float *) bf_b;
            DIFF_FLOAT_ARRAY(ret, indx, fp_a, fp_b, PD_float_tol);};}

    else if (strcmp(type, "float") == 0)
       {fp_a = (float *) bf_a;
        fp_b = (float *) bf_b;
        DIFF_FLOAT_ARRAY(ret, indx, fp_a, fp_b, PD_float_tol);}

    else if (strcmp(type, "double") == 0)
       {dp_a = (double *) bf_a;
        dp_b = (double *) bf_b;
        DIFF_FLOAT_ARRAY(ret, indx, dp_a, dp_b, PD_double_tol);};

    if (!ret)
       _SX_display_diff(fp, name1, name2, bf_a, bf_b, indx, nitems, type, dims);

    if (SX_disp_individ_diff)
       SFREE(indx);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_CHECK_POINTERS - check that two pointers are both commensurate
 *                    - either both are NULL or both are non-NULL
 *                    - skip over apropriate data in files to preserve
 *                    - data flow through diff process
 *                    - return TRUE iff they are commensurate
 */

static int _SX_check_pointers(file_a, file_b, name1, name2,
			      ta, na, ada, tb, nb, adb)
   PDBfile *file_a, *file_b;
   char *name1, *name2, *ta;
   long na, ada;
   char *tb;
   long nb, adb;
   {if (((ta == NULL) && (tb == NULL)) ||
        ((ta != NULL) && (tb != NULL)))
       return(TRUE);

    if (ta == NULL)
       {PRINT(stdout, "\nPointer, %s, in first file is NULL\n", name1);
        SKIP_UNIT(file_b, adb, nb);}
    else
       {PRINT(stdout, "\nPointer, %s, in second file is NULL\n", name2);
        SKIP_UNIT(file_a, ada, na);};

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_RD_LEAF_T - read a leaf into a temporary space passed in */

static int _SX_rd_leaf_t(file, ep, vr, in_type, nitems, bytepitem, out_type)
   PDBfile *file;
   syment *ep;
   char *vr, *in_type;
   long nitems, bytepitem;
   char *out_type;
   {char *buf, *vbuf, *svr;
    long i, n;
    long in_offs, out_offs, nrd;
    int convert;
    defstr *dpf;
    symblock *sp;
    FILE *fp;

    switch (setjmp(_PD_trace_err))
       {case ABORT    : return(FALSE);
        case ERR_FREE : return(TRUE);
        default       : memset(PD_err, 0, MAXLINE);
                        break;};

    dpf = _PD_lookup_type(out_type, file->chart);
    fp  = file->stream;

    sp = PD_entry_blocks(ep);
    n  = PD_n_blocks(ep);
    if (n == 1)
       sp[0].number = PD_entry_number(ep);

    convert = FALSE;
    if ((dpf->convert > 0) || (strcmp(in_type, out_type) != 0))
       convert = TRUE;

    if (convert)
       {buf = (char *) SC_alloc(nitems, bytepitem, "_SX_RD_LEAF_T:buf");
        if (buf == NULL)
           return(FALSE);}
    else
       buf = vr;       

    nrd = 0L;
    for (i = 0; i < n; i++)
        {if (io_read(buf, (size_t) bytepitem, (size_t) sp[i].number, fp) != sp[i].number)
            break;
         nrd += sp[i].number;
         if ((i+1) < n) 
            if (io_seek(fp, sp[i+1].diskaddr, SEEK_SET))
               break;}

    if (nrd != nitems)
       {if (convert)
          SFREE(buf);
        return(FALSE);}

    if (convert)
       {vbuf     = buf;
        svr      = vr;
        in_offs  = 0L;
        out_offs = 0L;
        PD_convert(&svr, &vbuf, in_type, out_type, nitems,
                   file->std, file->host_std, file->host_std,
                   &in_offs, &out_offs,
                   file->chart, file->host_chart, 0, PD_TRACE);
        SFREE(buf);}

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_TYPE_EQUAL - return TRUE if the two types in the two files are the same
 *                - or in the same class and that class' promote flag is set.
 */

int _SX_type_equal(file_a, file_b, typea, typeb)
   PDBfile *file_a, *file_b;
   char *typea, *typeb;
   {int ret;
    char *token;
    defstr *dpa, *dpb;
    memdes *dea, *deb;
    char suffixa[MAXLINE], suffixb[MAXLINE], bf[MAXLINE];

    strcpy(bf, typea);
    strcpy(suffixa, SC_lasttok(bf, " "));
    strcpy(bf, typeb);
    strcpy(suffixb, SC_lasttok(bf, " "));

    if ((strcmp(typea, suffixa) != 0) && (strcmp(suffixa, suffixb) != 0))
       return(FALSE);

    strcpy(bf, typea);
    token = SC_firsttok(bf, " ");
    dpa   = PD_inquire_type(file_a, token);

    strcpy(bf, typeb);
    token = SC_firsttok(bf, " ");
    dpb   = PD_inquire_type(file_b, token);

    if ((dpa == NULL) || (dpb == NULL))
       return(FALSE);

    if (SX_promote_flag && (dpa->members == NULL) && (dpb->members == NULL))
        {if (SX_promote_float && (dpa->format != NULL) && (dpb->format != NULL))
           return(TRUE); 
         if (SX_promote_fixed && (dpa->format == NULL) && (dpb->format == NULL)
             && (dpa->order_flag != -1) && (dpb->order_flag != -1))
           return(TRUE);}

    if (strcmp(typea, typeb) != 0)
       return(FALSE);

    ret = TRUE;
    ret &= (dpa->n_indirects == dpb->n_indirects);

    for (dea = dpa->members, deb = dpb->members;
         (dea != NULL) && (deb != NULL);
         dea = dea->next, deb = deb->next)
        {ret &= (dea->number == deb->number);

/*  this test prevents comparison of structs with promoted members */
         ret &= (strcmp(dea->type, deb->type) == 0);};

/*  cannot require cast_offs to match because they may differ
 *  if files were created on or targeted for different machines
 *       ret &= (dea->cast_offs == deb->cast_offs);};
 */

    if (dea != deb)
       return(FALSE);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DISPLAY_DIFF - display a diff list object */

object *SX_display_diff(argl)
   object *argl;
   {

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DISPLAY_DIFF - print out the given differences */

static int _SX_display_diff(file, name1, name2, pv1, pv2, indx, length, type, dims)
   PDBfile *file;
   char *name1, *name2;
   char *pv1, *pv2, *indx;
   long length;
   char *type;
   dimdes *dims;
   {int mjr, def_off;

    mjr     = file->major_order;
    def_off = file->default_offset;
    if (!SX_disp_individ_diff)
       {PRINT(stdout, "\nValues from the first file:\n");

        if (pv1 != NULL)
           _PD_print_leaf(stdout,
                          "", "", "", name1, file, pv1, length,
                          type, dims, mjr, def_off, FALSE);

        PRINT(stdout, "\nValues from the second file:\n");

        if (pv2 != NULL)
           _PD_print_leaf(stdout,
                          "", "", "", name2, file, pv2, length,
                          type, dims, mjr, def_off, FALSE);}

    else if ((pv1 != NULL) && (pv2 != NULL) && (indx != NULL))
       _SX_print_individ_diff(file, name1, name2, pv1, pv2, indx, length, type,
                              dims, mjr);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_PRINT_INDIVID_DIFF - print individual differences side by side */

static void _SX_print_individ_diff(file, name1, name2, pv1, pv2, indx, length, type,
				   dims, mjr)
   PDBfile *file;
   char *name1, *name2, *pv1, *pv2, *indx;
   long length;
   char *type;
   dimdes *dims;
   int mjr;
   {int nn, def_off, same_name, len1, len2;
    long i, isz, msz, *lp_a, *lp_b;
    int *ip_a, *ip_b;
    short *sp_a, *sp_b;
    char *cp_a, *cp_b;
    double *dp_a, *dp_b;
    float *fp_a, *fp_b;
    char bf[LINE_SIZE+1], bf1[80], bf2[LINE_SIZ2+1], tmp[80], fmt[80];

    same_name = (strcmp(name1, name2) == 0);

    nn = strlen(name1);
    memset(bf, ' ', LINE_SIZE);
    bf[LINE_SIZE] = '\0';

    def_off = file->default_offset;
    msz = 0L;
    for (i = 0L; i < length; i++)
        {if (indx[i])
            {isz = strlen(PD_index_to_expr(bf1, i, dims, mjr, def_off));
             msz = max(msz, isz);};};
    nn += (msz + 3);

    bf[nn] = '\0';
    PRINT(stdout, "\n%s%s", bf, "          File 1                    File 2\n\n");
    if (!same_name)
       {memset(bf2, ' ', LINE_SIZ2+1);
        bf2[LINE_SIZ2] = '\0';
        len1 = strlen(name1);
        len2 = strlen(name2);
/* NOTE: the hard-wired numbers in the strncpy calls below are based on
 *       the "File 1..." line above. If that line is changed, these
 *       numbers will need to be changed as well.
 */                                                 
        strncpy(&bf2[nn+10], name1, len1);
        strncpy(&bf2[nn+36], name2, len2);
        bf2[nn+36+len2] = '\0';
        PRINT(stdout, "%s%s", bf2, "\n\n");};
    memset(bf, ' ', LINE_SIZE);

    strcpy(fmt, ":   ");

/* find the type and compare */
    if (strcmp(type, "char") == 0)
       {cp_a = (char *) pv1;
        cp_b = (char *) pv2;
/*      Old version tries to print each character as a string and gets into trouble */
/*      strcat(strcat(strcat(fmt, PD_print_formats1[5]), "   "), PD_print_formats1[5]);     */
        strcat(strcat(strcat(fmt, "        %c"), "                         "), "%c");
        DISP_ARRAY(fmt, indx, cp_a, cp_b);}

    else if (strcmp(type, "short") == 0)
       {sp_a = (short *) pv1;
        sp_b = (short *) pv2;
        strcat(strcat(strcat(fmt, PD_print_formats1[4]), "   "),
               PD_print_formats1[4]);
        DISP_ARRAY(fmt, indx, sp_a, sp_b);}

    else if (strcmp(type, "int") == 0)
       {ip_a = (int *) pv1;
        ip_b = (int *) pv2;
        strcat(strcat(strcat(fmt, PD_print_formats1[0]), "   "),
               PD_print_formats1[0]);
        DISP_ARRAY(fmt, indx, ip_a, ip_b);}

    else if (strcmp(type, "integer") == 0)
       {ip_a = (int *) pv1;
        ip_b = (int *) pv2;
        strcat(strcat(strcat(fmt, PD_print_formats1[0]), "   "),
               PD_print_formats1[0]);
        DISP_ARRAY(fmt, indx, ip_a, ip_b);}

    else if (strcmp(type, "long") == 0)
       {lp_a = (long *) pv1;
        lp_b = (long *) pv2;
        strcat(strcat(strcat(fmt, PD_print_formats1[1]), "   "),
               PD_print_formats1[1]);
        DISP_ARRAY(fmt, indx, lp_a, lp_b);}

    else if (strcmp(type, "REAL") == 0)
       {if (sizeof(REAL) == sizeof(double))
           {dp_a = (double *) pv1;
            dp_b = (double *) pv2;
            strcat(strcat(strcat(fmt, PD_print_formats1[3]), "   "),
                   PD_print_formats1[3]);
            DISP_ARRAY(fmt, indx, dp_a, dp_b);}

        else if (sizeof(REAL) == sizeof(float))
           {fp_a = (float *) pv1;
            fp_b = (float *) pv2;
            strcat(strcat(strcat(fmt, PD_print_formats1[2]), "   "),
                   PD_print_formats1[2]);
            DISP_ARRAY(fmt, indx, fp_a, fp_b);};}

    else if (strcmp(type, "float") == 0)
       {fp_a = (float *) pv1;
        fp_b = (float *) pv2;
        strcat(strcat(strcat(fmt, PD_print_formats1[2]), "   "),
               PD_print_formats1[2]);
        DISP_ARRAY(fmt, indx, fp_a, fp_b);}

    else if (strcmp(type, "double") == 0)
       {dp_a = (double *) pv1;
        dp_b = (double *) pv2;
        strcat(strcat(strcat(fmt, PD_print_formats1[3]), "   "),
               PD_print_formats1[3]);
        DISP_ARRAY(fmt, indx, dp_a, dp_b);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_DIFF_SIGNAL - handle signals during comparisons */

static void _SX_diff_signal(sig)
   int sig;
   {err_fpe = TRUE;
    longjmp(diff_err, ABORT);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
