/* cp_io.c = input and output routines for circle packing */

#include "cp_head.h"
#include <xview/seln.h>
#include <xview/alert.h>
#include <xview/notice.h>
#include <X11/Xutil.h>
#include <sys/stat.h>
#include <unistd.h>

extern char *get_selection();
extern Menu print_menu;
static	FILE *fp,*sysfp;
static float fill,linewidth;
static char next[BUFSIZE];
extern complex ss_view(),s_pt_to_visual_plane();
extern float radius();
extern struct Pathlist *sph_poly_list();
extern void free_overlaps();
int dum_int;

int
readcall(p,filename) /* read file into p. Assumed that overwrite has
	already been approved if p is nonempty. filename has
	complete path included. */
struct p_data *p;
char *filename;
{

	if (!stat_file(filename))
	 {
		sprintf(msgbuf,"The file `%s' cannot be found. Check the path.",filename);
		emsg();
		return 0;
	 }
	if (open_and_read(p,filename)>0)
	 {
		read_consolidate(p,filename);
 		return 1;
	 }
	sprintf(msgbuf,"Read into pack %d has failed.",p);
	emsg();
	return 0;
} /* readcall */

int
open_and_read(p,filename) /* open input file and call for read, return 1 if
successful */
struct p_data *p;
char *filename;
{
	int i;

	if ( (strlen(filename)>NAME_MAX) || ((fp=fopen(filename,"r"))==NULL) )
		return 0;
	i=readpack(p,fp);
	fclose(fp);
	return i;
} /* open_and_read */

read_consolidate(p,filename) /* Update after successful read */
struct p_data *p;
char *filename;
{
	int n,pnum=pack_num(p);
	float halfwidth;
	char *name;

	sprintf(msgbuf,"Read into %d succeeded.",pnum); msg();
	n=strlen(filename);
	while (n>0 && filename[n-1]!='/') n--;
	name=filename+n;  /* cut off any directory names */
	strcpy(p->file_name,name);
	sprintf(buf,"%s",p->file_name);
	if (p->hes<0) strcat(buf," (hyp)\0");
	else if (p->hes==0) strcat(buf," (eucl)\0");
	else strcat(buf," (sph)\0");
	pmsg(pnum,buf);
	p->status=1;
		/* reset screen data */
	p->screen->display_opt=1;
	p->screen->box=std_real_box;
	halfwidth=(p->screen->box.ry-p->screen->box.ly)
		*p->screen->pix_box.rx/p->screen->pix_box.ry/2.0;
	p->screen->box.lx=(-1)*halfwidth;
	p->screen->box.rx=halfwidth;
		/* must fix so it fits current 
		   canvas pixel size */
	if (p->hes>=0)
	 {
		p->screen->unitcircle=0;
		xv_set(tog[pnum],PANEL_TOGGLE_VALUE,0,FALSE,0);
	 }
	else 
	 {
		p->screen->unitcircle=1;
		xv_set(tog[pnum],PANEL_TOGGLE_VALUE,0,TRUE,0);
	 }
	set_cursor(p->screen,0);
	live_node[pnum]=live_face[pnum]=0;
} /* read_consolidate */

int
readpack(p,fp) /* Assume fp is open for reading. Read into pack p, 
return 1 if successful */
struct p_data *p;
FILE *fp;
{
	int i,j,k,count,new_flag=0,name_flag=0,K_flag=0;
	int angsum_flag=0,aim_flag=0,rad_flag=0,cent_flag=0;
	float dum;
	char command[64],geometry[64],filename[NAME_MAX],*name;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern int alloc_pack_space();

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	while ( (fscanf(fp,"%63s",command)!=EOF)
		&& (strcmp(command,"NODECOUNT:")!=0)
		&& (strcmp(command,"CHECKCOUNT:")!=0)
		&& (strcmp(command,"END")!=0) )
	 {		/* in case PACKNAME apprears first */
		if (strcmp(command,"PACKNAME:")==0)
		 {
		      fgets(buf,1024,fp);
		      sscanf(buf,"%127s",filename);
		      name_flag=strlen(filename);
		 }
	 }
	if (strcmp(command,"")==0) return -1; /* no data */
	if (strcmp(command,"NODECOUNT:")==0) /* new packing */
	 {
		new_flag=1;
		p->hes=0; /* default */
		if (fscanf(fp,"%d",&count)!=1
			|| count<3 || !alloc_pack_space(p,count,0) )
		 {
			sprintf(msgbuf,"Read aborted: out of memory");
			emsg(); 
			return -1;
		 }
		pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr; /* update ptr's */
		p->nodecount=count;
	 }
	else if (strcmp(command,"CHECKCOUNT:")==0) /* partial data */
	 {
		if (!p->status || fscanf(fp,"%d",&count)!=1 
			|| count!=p->nodecount)
		 {
			sprintf(msgbuf,
			   "Read aborted: CHECKCOUNT=NODECOUNT failed.");
			emsg(); 
			return -1;
		 }
	 }
	else return -1; /* unrecognized stuff */
	while ( (fscanf(fp,"%63s",command)!=EOF) 
		&& (strcmp(command,"END")!=0) )
	 {
		if (strcmp(command,"ALPHA/BETA/GAMMA:")==0)
		 		fscanf(fp,"%d %d %d",&p->alpha,
				&p->beta,&p->gamma);
		else if (strcmp(command,"GEOMETRY:")==0)
		 {
		 	fscanf(fp,"%63s",geometry);
		 	if (toupper(geometry[0])=='H') p->hes=-1;
			else if (toupper(geometry[0])=='E') p->hes=0;
			else if (toupper(geometry[0])=='S') p->hes=1;
			else 
			 {
				dum=atof(geometry);
				if (dum>okerr) p->hes=1;
				else if (dum<(-1.0)*okerr) p->hes=-1;
				else p->hes=0;
			 }
		 }
		else if (strcmp(command,"FLOWERS:")==0)
		 {
		      K_flag=1;
		      if (!new_flag) 
		       {
			   sprintf(msgbuf,
			     "Read aborted: FLOWERS given w/o NODECOUNT.");
			   emsg();
			   return (0);
		       }
		      for (i=1;i<=count && new_flag;i++)
		       {
		         if (!fscanf(fp,"%d",&k)) new_flag=0;
			 if ( !fscanf(fp,"%d",&pK_ptr[k].num)
			    || pK_ptr[k].num>MAX_PETALS ) 
				new_flag=0;
			 if (pK_ptr[k].flower) free(pK_ptr[k].flower);
			 pK_ptr[k].flower=(int *)
			   calloc((size_t)(pK_ptr[k].num+1),sizeof(int));
		         for (j=0;new_flag && j<=pK_ptr[k].num;j++)
			   if (!fscanf(fp,"%d",&pK_ptr[k].flower[j])) 
				new_flag=0;
			 if ( new_flag 
			    && pK_ptr[k].flower[0]
			    !=pK_ptr[k].flower[pK_ptr[k].num] 
			    && pK_ptr[k].num>(MAX_PETALS-1) ) 
			 	new_flag=0; 
		       }
		      if (!new_flag || !complex_count(p,TRUE))
		       {
			   sprintf(msgbuf,
			     "Read aborted: error in reading complex, v%d.",
			     k);
			   emsg();
			   return (0);
		       }
		      free_overlaps(p);
		 }
		else if (strcmp(command,"PACKNAME:")==0)
		 {
		      fgets(buf,1024,fp);
		      sscanf(buf,"%127s",filename);
		      name_flag=strlen(filename);
		 }
		else if (strcmp(command,"RADII:")==0)
		 {
		      rad_flag=1;
		      if (p->hes<0) for (i=1;i<=count;i++)
		       {
				fscanf(fp,"%lf",&dum);
				if (dum>0) pR_ptr[i].rad=exp(-dum);
				else pR_ptr[i].rad=dum;
		       }
		      else for (i=1;i<=count;i++)
			 fscanf(fp,"%lf",&pR_ptr[i].rad);
		 }
			/* old method for aims: all */
		else if (strcmp(command,"AIMS:")==0) 
		 {
		      aim_flag=1;
		      for (i=1;i<=count;i++)
			 fscanf(fp,"%lf",&pR_ptr[i].aim);
		 }
			/* new method: design'd aims (vert, aim) */
		else if (strcmp(command,"ANGLE-AIMS:")==0)
		 {
			set_aim_default(p);
			aim_flag=1;
			while ((fscanf(fp,"%d %lf",&i,&dum)==2)
				&& i>0 && i<=count) 
				pR_ptr[i].aim=dum;
		 }
		else if (strcmp(command,"OVERLAPS:")==0) 
			/* old version, read all. NOTE: depends on FLOWERS */
		 {
		      if (alloc_overlaps(p))
		       {
		    	  for (i=1;i<=count && (fscanf(fp,"%d",&j)==1);i++)
			   {
			     		/* read but disregard j */
		  	     for (j=0;j<=pK_ptr[i].num 
				&& (fscanf(fp,"%lf",&dum)==1);j++)
				set_overlap(p,i,j,dum);
			   }
		       }
		 }
		else if (strcmp(command,"INV_DISTANCES:")==0) 
			/* new version. NOTE: depends on FLOWERS */
		 {
			if (alloc_overlaps(p))
			 {
				k=1;
				while (fscanf(fp,"%d %d",&i,&k)==2 
				   && (j=nghb(p,i,k))>=0
				   && fscanf(fp,"%lf",&dum) )
					set_overlap(p,i,j,dum);
			 }
		 }
		else if (strcmp(command,"CENTERS:")==0)
		 {
		      cent_flag=1;
		      for (i=1;i<=count;i++)
			 fscanf(fp,"%lf %lf",
				&pR_ptr[i].center.re,&pR_ptr[i].center.im);
		 }
		else if (strcmp(command,"ANGLESUMS:")==0)
    	            {
		      angsum_flag=1;
		      for (i=1;i<=count;i++)
			 fscanf(fp,"%lf",&pR_ptr[i].curv);
		 }
		else if ( strcmp(command,"CIRCLE-COLORS:")==0) 
		 {
			while (fscanf(fp,"%d %d",&i,&j)==2
				&& i>0 && i<=count
				&& j>=0 && j<=255)
				pK_ptr[i].color=j;
		 }
		else if ( strcmp(command,"FACE-COLORS:")==0) 
		 {
			while (fscanf(fp,"%d %d",&i,&j)==2
				&& i>0 && i<=p->facecount
				&& j>=0 && j<=255)
				p->faces[i].color=j;
		 }
	} /* end of while */
	if (new_flag)
	 {
		if (!K_flag)
		 {
		   sprintf(msgbuf,"Read aborted: no FLOWERS given.");
		   emsg();
		   return (0);
		 }
		if (!rad_flag) for (i=1;i<=count;i++) pR_ptr[i].rad=0.5;
		if (!cent_flag) for (i=1;i<=count;i++) 
			pR_ptr[i].center.re=pR_ptr[i].center.im=0.0;
	 }	
	if (name_flag)
	 {
		i=strlen(filename);
		while (i>0 && filename[i-1]!='/') i--;
		name=filename+i;  /* cut off any directory names */
		strcpy(p->file_name,name);
		sprintf(buf,"%s",p->file_name);
		if (p->hes<0) strcat(buf," (hyp)\0");
		else if (p->hes==0) strcat(buf," (eucl)\0");
		else strcat(buf," (sph)\0");
		pmsg(pack_num(p),buf);
	 }
	choose_alpha(p);choose_beta(p);choose_gamma(p);
	if (new_flag)
	 {
		facedraworder(p,NULL);
		if (!angsum_flag) fillcurves(p);
		if (!aim_flag) set_aim_default(p);
	 }
	p->active_node=p->beta;
	return (1);
} /* readpack */

int
writecall(p,filename,act) /* calls routine for writing file from pack p.
	Assumed that overwrite has already been approved if filename
	already exists. filename includes full path.*/
struct p_data *p;
int act;
char *filename;
{
	if (p->status && open_and_write(p,filename,act))
	 {
		write_consolidate(p,filename);
		return 1;
	 }
	sprintf(msgbuf,"Write from %d has failed.",pack_num(p));
	emsg();
	return 0;
} /* writecall */

write_consolidate(p,filename) /* Update after successful write */
struct p_data *p;
char *filename;
{
	int n,pnum=pack_num(p);
	char *name;

	sprintf(buf,"\nSENT pack %d to disc as %s.  ",pnum,filename);
	n=strlen(filename);
	while (filename[n-1]!='/' && n>0) n--;
	name=filename+n;  /* cut off any directory names */
	strcpy(p->file_name,name); /* update file_name */
		sprintf(buf,"%s",p->file_name);
		if (p->hes<0) strcat(buf," (hyp)\0");
		else if (p->hes==0) strcat(buf," (eucl)\0");
		else strcat(buf," (sph)\0");
		pmsg(pnum,buf);canvmsg(pnum);
} /* write_consolidate */

int
open_and_write(p,filename,act) /* open input file, call write, return 1 if
successful */
struct p_data *p;
char *filename;
int act;
{
	int i;

	if ( (strlen(filename)>NAME_MAX) || ((fp=fopen(filename,"w"))==NULL) )
		return 0;
	i=writepack(p,fp,act);
	fclose(fp);
	return i;
} /* open_and_write */

int
writepack(p,fp,act) /* Assume fp is open for writing. Write from pack p, 
return 1 if okay. act: 0=combinatorics, 1=minimal, 2=standard,3=complete, */
struct p_data *p;
FILE *fp;
int act;
{
	int i,j,k,n,count,flag,colorflag,jj;
	float angle;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;
	pR_ptr=p->packR_ptr;
	count=p->nodecount;
	fprintf(fp,"NODECOUNT:   %d\n\n",count);
	if (act>2)
	 {
		buf[0]='\0';
		sscanf(p->file_name,"%s",buf);
		stripsp(buf);
		if (strlen(buf)==0)
			sprintf(buf,"NoName ");
		fprintf(fp,"PACKNAME: %s\n\n",buf);
	 }
	if (act>0)
	 {
		fprintf(fp,"ALPHA/BETA/GAMMA:   %d  %d  %d\n\n",
		p->alpha,p->beta,p->gamma);
		if (p->hes<(-1.0)*okerr) sprintf(buf,"hyperbolic\n");
		else if (p->hes>okerr) sprintf(buf,"spherical\n");
		else sprintf(buf,"euclidean\n");
		fprintf(fp,"GEOMETRY:   %s\n",buf);
	 }
	fprintf(fp,"FLOWERS: \n");
	for (n=1;n<=count;n++)
	 {
        	fprintf(fp,"\n%d  %d     ",n,pK_ptr[n].num);
		for (i=0;i<=pK_ptr[n].num;i++) 
			fprintf(fp,"%d  ",pK_ptr[n].flower[i]);
	 }
	fprintf(fp,"\n\n");
	if (act>0)
	 {
		if (p->overlap_status)
		 {
	   		fprintf(fp,"INV_DISTANCES:\n");
	   		for (i=1;i<=count;i++)
			   for (j=0;j<(pK_ptr[i].num+pK_ptr[i].bdry_flag);j++)
			     if (i<(k=pK_ptr[i].flower[j])
				&& (angle=pK_ptr[i].overlaps[j]) != 1.0)
				   fprintf(fp,"\n%d %d  %.6e ",
					i,k,angle);
			fprintf(fp,"\n  (done)\n\n");
		 }
		flag=0;
		for (i=1;i<=count;i++)
		 {
		   if (pK_ptr[i].bdry_flag 
			&& pR_ptr[i].aim>=0.0) flag++;
		   else if ( !pK_ptr[i].bdry_flag 
			&& ( (pR_ptr[i].aim-2.0*M_PI)>toler 
			|| (2.0*M_PI-pR_ptr[i].aim)>toler ) ) flag++;
		 }
		if (flag)
		 {
		   jj=j=0;
		   for (i=1;i<=count && jj==0;i++) 
		    {
			if ( (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0) 
			   || (!pK_ptr[i].bdry_flag 
			   && (pR_ptr[i].aim<(2.0*M_PI+okerr)
			   || pR_ptr[i].aim>(2.0*M_PI-okerr))) )
				jj++;
		    }
		   if (jj>0) /* at least one non-default aim */
		    {
			fprintf(fp,"ANGLE-AIMS:\n");
		     	for (i=1;i<=count;i++)
			   if ( (pK_ptr[i].bdry_flag && pR_ptr[i].aim>=0) 
			     || (!pK_ptr[i].bdry_flag 
			     && (pR_ptr[i].aim<(2.0*M_PI-okerr)
			     || pR_ptr[i].aim>(2.0*M_PI+okerr))) )
			    {
			       fprintf(fp," %d % .10e  ",i,pR_ptr[i].aim);
			       j++;
			       if ((j % 3)==0) fprintf(fp,"\n");
			    }
		    }
		   fprintf(fp,"\n  (done)\n\n");
		 }
	 }
	if (act>1)
	 {
		fprintf(fp,"RADII: \n");
		for (i=1;i<=count;i++)
		 {
		   fprintf(fp,"% .10e  ",radius(p,i));
		   if ((i % 4)==0) fprintf(fp,"\n");
		 }
		fprintf(fp,"\n\n");
		fprintf(fp,"CENTERS:\n");
		for (i=1;i<=count;i++)
		 {
		   fprintf(fp,"  % .6e % .6e  ",pR_ptr[i].center.re,
			pR_ptr[i].center.im);
		   if ((i % 2)==0) fprintf(fp,"\n");
		 }
		fprintf(fp,"\n\n");
	 }
	if (act>2) 
	 {
		fprintf(fp,"ANGLESUMS: \n");
		for (i=1;i<=count;i++)
		 {
                	fprintf(fp," %.6e  ",pR_ptr[i].curv);
			if ((i % 5)==0) fprintf(fp,"\n");
		 }
		fprintf(fp,"\n\n");
		colorflag=0;
		for (i=1;i<=count && colorflag==0;i++) 
			if (pK_ptr[i].color != FG_COLOR) colorflag++;
		if (colorflag) /* found some non-default colors */
		 {
			fprintf(fp,"CIRCLE-COLORS:\n");
			j=0;
			for (i=1;i<=count;i++)
			 {
				if (pK_ptr[i].color!=FG_COLOR)
				 {
					fprintf(fp," %d %d  ",
						i,pK_ptr[i].color);
					j++;
				 }
				if ((j % 10)==0) fprintf(fp,"\n");
			 }
			fprintf(fp,"\n  (done)\n\n");
		 }
		colorflag=0;
		for (i=1;i<=p->facecount && colorflag;i++) 
			if (p->faces[i].color != FG_COLOR) colorflag++;
		if (colorflag) /* found some non-default colors */
		 {
			fprintf(fp,"FACE-COLORS:\n");
			j=0;
			for (i=1;i<=p->facecount;i++)
			 {
				if (p->faces[i].color!=FG_COLOR)
				 {
					fprintf(fp," %d %d  ",
						i,p->faces[i].color);
					j++;
				 }
				if ((j % 10)==0) fprintf(fp,"\n");
			 }
			fprintf(fp,"\n  (done)\n\n");
		 }
	 }
	fprintf(fp,"END\n");fflush(fp);
	return 1;
} /* writepack */

int
read_path_file(filename) /* read a closed path from a file. */
char *filename;
{
	int count;

	if (!stat_file(filename))
	 {
	   sprintf(msgbuf,"The file `%s' cannot be found",filename);
	   emsg();
	   return 0;
	 }
	if ((fp=fopen(filename,"r"))!=NULL) 
	 {
	   count=readpath(fp);
	   fclose(fp);
	   return count;
	 }
	else return 0;
} /* read_path_file */

int
readpath(fp) /* assume file open for reading path */
FILE *fp;
{
	int count=0;
	float xx,yy;
	char command[50];
	struct Pathlist *trace;

	while ( (fscanf(fp,"%s",command)!=EOF) && (strcmp(command,"PATH")!=0) 
		&& (strcmp(command,"END")!=0) );
	if (strcmp(command,"END")==0 || fscanf(fp,"%lf %lf",&xx,&yy)!=2) 
		return 0; /* no points */
	path_free(&pathlist); /* free the pathlist */
	pathlength=0;
	if ( (pathlist=(struct Pathlist *)calloc(1,sizeof(struct Pathlist)))
		!=NULL ) 
	 {
		pathlist->x=xx;pathlist->y=yy;
		count++;
	 }
	else return 0; 
	trace=pathlist;
	while 
	   ( fscanf(fp,"%lf %lf",&xx,&yy)==2 && count<2000
	  	 && ((trace->next=(struct Pathlist *)calloc(1,sizeof
		 (struct Pathlist)))!=NULL ) )
	 {
		trace=trace->next;
		trace->x=xx;trace->y=yy;
		count++;
	 }
	if (count<3)	/* didn't get enough points */
	 {path_free(&pathlist); pathlength=0;return 0;} 
	trace=pathlist;
	while (trace->next!=NULL) trace=trace->next;
	if (trace->x!=pathlist->x || trace->y!=pathlist->y)
	 {
		trace->next=(struct Pathlist *)calloc
			(1,sizeof(struct Pathlist));
		trace->next->x=pathlist->x;
		trace->next->y=pathlist->y;
		count++;
	 } /* make sure path closes up */
	pathlength=count;
	return count;
} /* readpath */

int
write_path_file(filename) /* write a closed path to a file.*/
char *filename;
{
	int count;

	if (pathlist!=NULL && (fp=fopen(filename,"w"))!=NULL
		&& strlen(filename)<=NAME_MAX)
	 {
		count=writepath(fp);
		fclose(fp);
		return count;
	 }
	else return 0;
} /* write_path_file */

int
writepath(fp) /* assume file open for writing path */
FILE *fp;
{
	int count=0;
	struct Pathlist *trace;

	fprintf(fp,"PATH\n");
	trace=pathlist;
	while (trace!=NULL)
	 {
		fprintf(fp,"%lf %lf \n",trace->x,trace->y);
		trace=trace->next;
		count++;
	 }
	fprintf(fp,"END\n");
	return count;
} /* writepath */

int
copy_packcall(datastr) /* copy p1 into p2; */
char *datastr;
{
	int p1,p2,width,height;
	float midpt,halfwidth;
	Pixmap xpm_hold;
	XID xid_hold;

	if ( sscanf(datastr,"%d %d",&p1,&p2) ==2 && p1>=0 && p2!=p1
		&& p1<NUM_PACKS && p2>=0 && p2<NUM_PACKS &&
		packdata[p1].status && !packdata[p2].locks
		&& copy_pack(&packdata[p1],&packdata[p2]) )
	 {
	 		/* save pointers */
	 	xid_hold=screendata[p2].xid;
	 	xpm_hold=screendata[p2].xpm;		
		screendata[p2]=screendata[p1];
		screendata[p2].xid=xid_hold;
		screendata[p2].xpm=xpm_hold;
			/* move screen parameters */
		screendata[p2].pix_box.rx=width=
			(int)xv_get(canvas_frame[p2],XV_WIDTH);
		screendata[p2].pix_box.ry=height=
			(int)xv_get(canvas_frame[p2],XV_HEIGHT);
		midpt=(screendata[p2].box.lx+screendata[p2].box.rx)/2.0;
		halfwidth=(screendata[p2].box.ry-screendata[p2].box.ly)
			*width/height/2.0;
		screendata[p2].box.lx=midpt-halfwidth;
		screendata[p2].box.rx=midpt+halfwidth;
		set_cursor(&screendata[p2],0);
		sprintf(buf,"%s",packdata[p2].file_name);
		if (packdata[p2].hes<0) strcat(buf," (hyp)\0");
		else if (packdata[p2].hes==0) strcat(buf," (eucl)\0");
		else strcat(buf," (sph)\0");
		pmsg(p2,buf);
		return 1;
	 }
	return 0;
} /* copy_packcall */

int
copy_pack(p1,p2) /* copy p1 into p2. */
struct p_data *p1,*p2;
{
	int i,j,size;
	struct K_data *pK_ptr1,*pK_ptr2;
	struct R_data *pR_ptr1,*pR_ptr2;
	extern int alloc_pack_space();

	empty_pack(p2); /* throw out old stuff */
	if (!alloc_pack_space(p2,(p1->nodecount + 1),0))
	 {
		sprintf(msgbuf,"Error allocating pack space for p%d.",
			pack_num(p2));
		emsg();
		return 0;
	 }
	pK_ptr1=p1->packK_ptr;pR_ptr1=p1->packR_ptr;
	pK_ptr2=p2->packK_ptr;pR_ptr2=p2->packR_ptr;
	size=p2->sizelimit;
	*p2=*p1;
 		/* reset some data and pointers */
	p2->sizelimit=size;
	p2->faces=NULL;
	p2->overlap_status=0;
	p2->packK_ptr=pK_ptr2; 
	p2->packR_ptr=pR_ptr2;
	p2->screen=&screendata[pack_num(p2)];
		/* move data over */
	for (i=1;i<=p1->nodecount;i++)
	 {
		pK_ptr2[i]=pK_ptr1[i];
		pR_ptr2[i]=pR_ptr1[i];
		pK_ptr2[i].flower=(int *)
			malloc((pK_ptr2[i].num+1)*sizeof(int));
		for (j=0;j<=pK_ptr2[i].num;j++)
			pK_ptr2[i].flower[j]=pK_ptr1[i].flower[j];
	 }
	if (!alloc_faces_space(p2))
	 {
		sprintf(msgbuf,"Error allocating face data space for p%d.",
			pack_num(p2));
		emsg();
		return 0;
	 }
	for (i=1;i<=p1->facecount;i++) p2->faces[i]=p1->faces[i];
	if (p1->overlap_status && alloc_overlaps(p2))
	   for (i=1;i<=p1->nodecount;i++)
		for (j=0;j<=pK_ptr1[i].num;j++)
			pK_ptr2[i].overlaps[j]=pK_ptr1[i].overlaps[j];
	p2->locks=0;
	set_pack_labels();
	return 1;
} /* copy_pack */

int
alloc_pack_space(p,new_size,flag) /* Enlarge pack data space, step of 
5000. Allocate space for pK and pR data, free old space. 
If flag=1, adjust size of current pack, else this is new pack. 
CAUTION: after return, be sure to update any corrupted pointers, e.g.,
pK_ptr and pR_ptr. */
struct p_data *p;
int new_size,flag;
{
	int v,size,oldsize;
	struct K_data *newK,*pK_ptr;
	struct R_data *newR,*pR_ptr;
	
	if (!p) return 0;
	oldsize=p->sizelimit;
	size=((int)((new_size-1)/5000))*5000+5000;
	if (flag && size==oldsize) return 1; /* no action needed */
	if (flag && (p->nodecount > (size-1))) return 0; 
		/* pack too big for given new_size */

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	p->sizelimit=size;
	if ((newK=(struct K_data *)
	   calloc((size_t)(size+1),sizeof(struct K_data)))==NULL)
		return 0;
	if ((newR=(struct R_data *)
	   malloc((size+1)*sizeof(struct R_data)))==NULL)
	 {free(newK);p->sizelimit=oldsize;return 0;}
	if (flag) for (v=1;v<=p->nodecount;v++) /* copy old data */
	 {
		newK[v]=pK_ptr[v];
		newR[v]=pR_ptr[v];
	 }
	else /* empty out pack and reset */
	 {
	 	if (p->faces) 
		 {free(p->faces);p->faces=NULL;}
		if (pK_ptr) for (v=1;v<=p->nodecount;v++)
		 {
			if (pK_ptr[v].flower) free(pK_ptr[v].flower);
			if (pK_ptr[v].overlaps) free(pK_ptr[v].overlaps);
		 }
		p->overlap_status=p->status=0;
		p->nodecount=p->first_red_face=p->status=p->locks=0;
	 }
	if (pK_ptr) free(pK_ptr);
	if (pR_ptr) free(pR_ptr);
	p->packK_ptr=newK;
	p->packR_ptr=newR;
	return 1;
} /* alloc_pack_space */

empty_pack(p) /* reset pack, default size =5000. */
struct p_data *p;
{
	int pnum=pack_num(p);
	extern int alloc_pack_space();

	alloc_pack_space(p,5000,0);
	sprintf(packlabels[pnum],"Pack %d: empty             ",pnum);
	strcpy(p->file_name,"None\0");
	set_pack_labels();
} /* empty_pack */

int
writeeucldata(p,filename) /* write triples of x,y,z, of eucl data 
	on circle of p. filename should already have complete path. */
struct p_data *p;
char *filename;
{
	int i,node;
	float e_rad,s_rad;
	complex h_center,e_center;
	struct R_data *pR_ptr;

	if (!p->status || (node=p->nodecount)<=0) return 0;
	pR_ptr=p->packR_ptr;
	if (strlen(filename)<=NAME_MAX && (fp=fopen(filename,"w"))!=NULL)
	 {
		fprintf(fp,
		  "CHECKCOUNT: %d\n\nGEOMETRY: euclidean\n\nRADII:\n",
			node);
		if (p->hes<0)
		 {
			for (i=1;i<=node;i++)
			 {
			   h_center=pR_ptr[i].center;
			   s_rad=pR_ptr[i].rad;
			   h_to_e_data(h_center,s_rad,&e_center,&e_rad);
			   fprintf(fp,"% .10e ",e_rad);
		           if ((i % 5)==0) fprintf(fp,"\n");
			 }
		 }
		else if (p->hes==0)
		 {
			for (i=1;i<=node;i++)
			 {
			   fprintf(fp," % .10e ",pR_ptr[i].rad);
		           if ((i % 5)==0) fprintf(fp,"\n");
			 }
		 }
		fprintf(fp,"\n\nCENTERS:\n");
		if (p->hes<0)
		 {
			for (i=1;i<=node;i++)
			 {
			   h_center=pR_ptr[i].center;
			   s_rad=pR_ptr[i].rad;
			   h_to_e_data(h_center,s_rad,&e_center,&e_rad);
			   fprintf(fp,"  % .6e % .6e  ",
					e_center.re,e_center.im);
		           if ((i % 2)==0) fprintf(fp,"\n");
			 }
		 }
		else 
		 {
			for (i=1;i<=node;i++)
			 {
			   fprintf(fp,"  % .6e % .6e  ",
				pR_ptr[i].center.re,pR_ptr[i].center.im);
		           if ((i % 2)==0) fprintf(fp,"\n");
			 }
		 }
		fprintf(fp,"\nEND\n");
	 	fclose(fp);
		return 1;
	 }
	else return 0;
} /* writeeucldata */

int
seed_start(p,datastr) /* start a flower in pack p with n petals. */
struct p_data *p;
char *datastr;
{
	int n,i,pnum;
	float halfwidth;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;
	extern void free_overlaps();
	extern int alloc_pack_space();
	
	if (!alloc_pack_space(p,100,0)) return 0;
	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	pnum=pack_num(p);
	p->screen->box=std_real_box;
	halfwidth=(p->screen->box.ry-p->screen->box.ly)
		*p->screen->pix_box.rx/p->screen->pix_box.ry/2.0;
	p->screen->box.lx=(-1)*halfwidth;
	p->screen->box.rx=halfwidth;
	set_cursor(p->screen,0);
	live_node[pnum]=live_face[pnum]=0;	
	if (sscanf(datastr,"%d",&n)!=1 || n<3 || n>MAX_PETALS) n=6;
	p->nodecount=n+1;
	p->alpha=1;p->beta=p->gamma=2;
	pK_ptr[1].num=n;
	pK_ptr[1].bdry_flag=0;
	if (pK_ptr[1].flower) free(pK_ptr[1].flower);
	pK_ptr[1].flower=(int *)malloc((n+1)*sizeof(int));
	for (i=0;i<n;i++) pK_ptr[1].flower[i]=i+2;
	pK_ptr[1].flower[n]=2;
	pR_ptr[1].rad=.4;
	for (i=2;i<=(n+1);i++)
	 {
		if (pK_ptr[i].flower) free(pK_ptr[i].flower);
		pK_ptr[i].flower=(int *)malloc(3*sizeof(int));
		pK_ptr[i].num=2;
		pK_ptr[i].flower[0]=i+1;
		pK_ptr[i].flower[1]=1;
		pK_ptr[i].flower[2]=i-1;
		pK_ptr[i].bdry_flag=1;
		pR_ptr[i].rad=.4;
	 }
	pK_ptr[2].flower[2]=n+1;
	pK_ptr[n+1].flower[0]=2;
	p->status=1;
	p->hes=-1;
	p->active_node=1;
	free_overlaps(p);
	if (p->faces) {free(p->faces);p->faces=NULL;}
	sprintf(buf,"Seed pack");
	strcpy(p->file_name,buf);
	sprintf(buf,"no header");
	strcpy(p->fhead,buf);
	p->screen->display_opt=1;
	p->screen->unitcircle=1;
	panel_set(tog[pnum],PANEL_TOGGLE_VALUE,0,TRUE,0);
	sprintf(buf,"Seed pack (hyp)\0",pnum);
	pmsg(pnum,buf);canvmsg(pnum);
	complex_count(p,TRUE);
	facedraworder(p,NULL);
	set_aim_default(p);
	fillcurves(p);
	sprintf(buf,"repack -p%d",pnum);
	handle_cmd(buf);
	sprintf(buf,"fix -p%d -c",pnum);
	handle_cmd(buf);
	sprintf(buf,"disp -p%d -w -c",pnum);
	handle_cmd(buf);
	return 1;
} /* seed_start */

int
confirm_action(message) /* alert package for confirming a read or write */
char *message;
{
	int result;
	Event event;
	result=notice_prompt(base_frame,&event,
		NOTICE_MESSAGE_STRINGS,message,0,
		NOTICE_BUTTON_YES,"confirm",
		NOTICE_BUTTON_NO,"Cancel",
		0);
	switch (result)
	{
		case NOTICE_YES: return 1;
		case NOTICE_NO: return 0;
		case NOTICE_FAILED: 
		 {
			strcpy(msgbuf,"Alert failed for some reason");
			emsg();
			return 0;
		 }
	 }
	return 1;
} /* confirm_action */

int
alloc_faces_space(p) /* allocate face data space (knowing facecount) */
struct p_data *p;
{
	if (p->faces!=NULL) free(p->faces);
	p->faces=(f_data *)malloc((p->facecount+1)*sizeof(f_data));
	if ((p->faces)==NULL) return 0;
	return 1;
} /* alloc_faces_space */

int
alloc_overlaps(p) /* Allocate overlaps ptr space and initialize. 
Return 1 if succeeds or already set. */
struct p_data *p;
{
	int v,i;

	if (p->overlap_status) return 1; /* already have space */
	for (v=1;v<=p->nodecount;v++)
	 {
		if ( (p->packK_ptr[v].overlaps=(float *)
		   malloc((p->packK_ptr[v].num+1)*sizeof(float)))==NULL )
		 {
		   strcpy(msgbuf,"Failed to allocate memory for edge data.");
		   emsg();
		   free_overlaps(p);
		   return 0;
		 }
		else for (i=0;i<=p->packK_ptr[v].num;i++)
			p->packK_ptr[v].overlaps[i]=1.0;
	 }
	p->overlap_status=1;
	return 1;
} /* alloc_overlaps */

void
free_overlaps(p) /* free pointers to overlaps */
struct p_data *p;
{
	int v;

	for (v=1;v<=p->nodecount;v++)
		if (p->packK_ptr[v].overlaps) 
		 {
			free(p->packK_ptr[v].overlaps);
			p->packK_ptr[v].overlaps=NULL;
		 }
	p->overlap_status=0;
} /* free_overlaps */

int
set_overlap(p,v,j,angle) /* set overlap or inv. dist value in all
approp places. Return 0 on error or if space not allocated. */
struct p_data *p;
int v,j;
float angle;
{
	int w,indx;
	struct K_data *pK_ptr;
	extern int nghb();

	pK_ptr=p->packK_ptr;
	if (!p->overlap_status) return 0;
	w=p->packK_ptr[v].flower[j];
	indx=nghb(p,v,w);
	pK_ptr[v].overlaps[indx]=angle;
	if (indx==0 && !pK_ptr[v].bdry_flag) 
		pK_ptr[v].overlaps[pK_ptr[v].num]=angle;
	indx=nghb(p,w,v);
	pK_ptr[w].overlaps[indx]=angle;
	if (indx==0 && !pK_ptr[w].bdry_flag) 
		pK_ptr[w].overlaps[pK_ptr[w].num]=angle;
} /* set_overlap */	

int
print_inq_data(p,f,datastr)
struct p_data *p;
int f;
char *datastr;
{
	int i,j,v,intp=0,bdryp=0,count=0,n1,n2,vv[3];
	int num,pnum=pack_num(p),hits,flag,incompat_flag;
	float interr=0.0,bdryerr=0.0,dum,b_dist,nc,dumf,duml,tf,tl,tc;
	float b_vert_dist,datum,rad1,rad2,inv_d,act_inv_d,diff,over_err;
	complex ctr,e_cent;
	char buff[256],buff2[256],*endptr;
	struct s_data *scr;
	extern struct Vertlist *node_link_parse(),*face_link_parse();
	extern struct Edgelist *node_pair_link();
	extern float Cvw(),node_conductance(),twin_conductance(),
		radius(),curv_aim_error(),comp_inv_dist(),
		comp_single_angle();
	struct Vertlist *vertlist,*trace;
	struct Edgelist *edgelist,*track;
	struct p_data *p1,*p2;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	if (f==14) /* program parameters */
	 {
		sprintf(msgbuf,"Param: Packing: accuracy=%e  (okerr=%e);\n  Packing algorithm cycles=%d;  iterations=%d; \n  Misc: post_size=%d; brush=%d;act pack=%d; act node=%d",
			toler,okerr, totalpasses,iterates,
			line_thick,pnum,p->active_node);
		msg();
			/* give remote routine status */
		for (i=0;i<NUM_PROC;i++)
		   if (remote[i].pid)
		    {
			sprintf(msgbuf,
			   "  Remote routine %d active: pid=%d, program is '%s', moniker = %s.",
		     i,remote[i].pid,remote[i].program,remote[i].moniker);
			msg();
		    }
		return 1;
	 }
	if (!p->status)
	 {
		sprintf(msgbuf,"Pack %d is empty.",pnum);
		emsg();
		return 0;
	 }
	scr=&screendata[pnum];
	switch (f)
	 {
case 1: /* vertex data */
 {
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		do {
		    v=trace->v;
		    sprintf(msgbuf,
"\nData on vert#%d: rad=% .5e, ang sum=% .5ePi, aim=% .5ePi, bdry?=%d, #star=%d; p=%d.",
		    v,radius(p,v),
		    p->packR_ptr[v].curv/M_PI,p->packR_ptr[v].aim/M_PI,
		    p->packK_ptr[v].bdry_flag,p->packK_ptr[v].num,pnum);
		    msg();
		    count++;
		    trace=trace->next;
		 } while (trace!=NULL && count < 5);
		vert_free(&vertlist); /* free any excess */
		return count;
	 }
	else v=p->active_node;
	 {
		sprintf(msgbuf,
"\nData on vert#%d: rad=% .5e, ang sum=% .5ePi, aim=% .5ePi, bdry?=%d, #star=%d; p=%d.",
		v,radius(p,v),
		p->packR_ptr[v].curv/M_PI,p->packR_ptr[v].aim/M_PI,
		p->packK_ptr[v].bdry_flag,p->packK_ptr[v].num,pnum);
		msg();
		return 1;
	 }
 }
case 2: /* pack data */
 {
	if (sscanf(datastr,"%d",&pnum)!=1 || pnum<0 || pnum>=NUM_PACKS
	   || !packdata[pnum].status)
		pnum=current_p;
	p1=&packdata[pnum];
	sprintf(msgbuf,
	  "Pack %d: %d nodes, (alpha beta gamma)=(%d %d %d), first_face=%d\n Euler=%d; genus=%d; %d bdry comps;",
		pnum,p1->nodecount,p1->alpha,p1->beta,p1->gamma,p1->first_face,
		p1->euler,p1->genus,p1->num_bdry_comp);
	if (packdata[pnum].num_bdry_comp>0)
		sprintf(msgbuf+strlen(msgbuf),"bdry_starts = ");
	for (i=1;i<=packdata[pnum].num_bdry_comp;i++)
	   sprintf(msgbuf+strlen(msgbuf),"%d ",packdata[pnum].bdry_starts[i]);
	msg();
	return 1;
 }
case 4: /* flower */
case 5: /* selected radii */
case 6: /* selected angle sums */
case 7: /* selected aims */
case 12: /* selected model curvatures */
 {
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		if (f==4) sprintf(msgbuf,"Flowers:");
		else if (f==5) sprintf(msgbuf,"Radii:");
		else if (f==6) sprintf(msgbuf,"Angle sums/Pi:");
		else if (f==7) sprintf(msgbuf,"Aims/Pi:");
		else if (f==11) sprintf(msgbuf,"Kappa:");
		trace=vertlist;
		do {
		   if (f==5) datum=radius(p,trace->v);
		   else if (f==6) datum=p->packR_ptr[trace->v].curv/M_PI;
		   else if (f==7) datum=p->packR_ptr[trace->v].aim/M_PI;
		   else if (f==11)
		    {
			kappa_calc(p,trace->v,&datum);
			datum=datum*(fabs(datum));
		    }
		   else if (f==4)
		    {
			sprintf(buff,"\np%d,v%d,deg=%d,flower: ",
				pnum,trace->v,
				p->packK_ptr[trace->v].num);
			strcat(msgbuf,buff);
			for (i=0;i<=p->packK_ptr[trace->v].num;i++)
			 {
				sprintf(buff,"%d  ",
					p->packK_ptr[trace->v].flower[i]);
				strcat(msgbuf,buff);
			 }
			*buff='\0';
		    }
		   if (f!=4) sprintf(buff,"%d % .5e; ",trace->v,datum);
		   strcat(msgbuf,buff);
		   count++;
		   trace=trace->next;
		 } while (trace!=NULL && count < 5);
		vert_free(&vertlist); /* free any excess */
		msg();
	 }
	return count;
 }
case 8: /* data on selected faces */
 {
	if ( (vertlist=face_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		do {
		   incompat_flag=0.0;
		   for (j=0;j<3;j++) vv[j]=p->faces[trace->v].vert[j];
		   sprintf(msgbuf,"Pack %d, face %d data: verts (%d,%d,%d), plot %d.\n",
			pnum,trace->v,vv[0],vv[1],vv[2],
			vv[p->faces[trace->v].index_flag]);
		   msg();
		   sprintf(msgbuf,"  Rad: (%lf,%lf,%lf).\n  Angles: (%lf,%lf,%lf).\n",
			pR_ptr[vv[0]].rad,pR_ptr[vv[1]].rad,
			pR_ptr[vv[2]].rad,
			comp_single_angle(p,vv[0],vv[1],vv[2],&incompat_flag),
			comp_single_angle(p,vv[1],vv[2],vv[0],&incompat_flag),
			comp_single_angle(p,vv[2],vv[0],vv[1],&incompat_flag));
		   msg();
		   if (incompat_flag)
		    {
			sprintf(msgbuf," NOTE: radii/overlaps incompatible.\n");
			emsg();
		    }
		   sprintf(msgbuf,"  Overlaps (0->1->2->0): assigned=");
		   if (!p->overlap_status)
			sprintf(buff,"1,1,1\n   errors (assigned-actual)=%lf,%lf,%lf.\n",
			   1.0-comp_inv_dist(p,vv[0],vv[1]),
			   1.0-comp_inv_dist(p,vv[1],vv[2]),
			   1.0-comp_inv_dist(p,vv[2],vv[0]));
		   else
			sprintf(buff,"%lf,%lf,%lf\n   errors (assigned-actual)=%lf,%lf,%lf.\n",
			   pK_ptr[vv[0]].overlaps[nghb(p,vv[0],vv[1])],
			   pK_ptr[vv[1]].overlaps[nghb(p,vv[1],vv[2])],
			   pK_ptr[vv[2]].overlaps[nghb(p,vv[2],vv[0])],
			   pK_ptr[vv[0]].overlaps[nghb(p,vv[0],vv[1])]
				-comp_inv_dist(p,vv[0],vv[1]),
			   pK_ptr[vv[1]].overlaps[nghb(p,vv[1],vv[2])]
				-comp_inv_dist(p,vv[1],vv[2]),
			   pK_ptr[vv[2]].overlaps[nghb(p,vv[2],vv[0])]
				-comp_inv_dist(p,vv[2],vv[0]));
		   strcat(msgbuf,buff);
		   msg();
		   count++;
		   trace=trace->next;
		 } while (trace!=NULL && count < 5);
		vert_free(&vertlist); /* free any excess */
	 }
	return count;
 }
case 9: /* overlaps */
 {
	sprintf(msgbuf,"Overlaps, p%d:",pnum);
	if (!p->overlap_status)
	 {
		strcat(msgbuf," All set to 0.0, (default=tangency case).");
		msg();
		return count;
	 }
	if ((edgelist=node_pair_link(p,datastr,&endptr,&hits))!=NULL)
	 {
		track=edgelist;
		do {
		   inv_d=pK_ptr[track->v].
			overlaps[nghb(p,track->v,track->w)];
		   if (inv_d>=0 && inv_d<=1) /* want overlap */
		    {
			act_inv_d=comp_inv_dist(p,track->v,track->w);
			if (act_inv_d<=1)
			  sprintf(buff," (%d,%d): aim=cos(% .5e Pi); actual=cos(% .5e Pi).\n",
				track->v,track->w,
				acos(inv_d)/M_PI,acos(act_inv_d)/M_PI);
			else
			  sprintf(buff," (%d,%d): aim=cos(% .5e Pi); actual=*% .5e.\n",
				track->v,track->w,
				acos(inv_d)/M_PI,act_inv_d);
			strcat(msgbuf,buff);
			count++;
		    }
		   else if (inv_d>1) /* want separated */
		    {
			act_inv_d=comp_inv_dist(p,track->v,track->w);
			if (act_inv_d>1)
			  sprintf(buff," (%d,%d): aim=*% .5e; actual=*% .5e.\n",
				track->v,track->w,inv_d,act_inv_d);
			else
			  sprintf(buff," (%d,%d): aim=*% .5e; actual=cos(% .5e Pi).\n",
				track->v,track->w,inv_d,acos(act_inv_d)/M_PI);
			strcat(msgbuf,buff);
			count++;
		    }
		   else /* assigned overlaps exceeding Pi/2 */
		    {
			act_inv_d=comp_inv_dist(p,track->v,track->w);
			if (act_inv_d>1)
			  sprintf(buff," (%d,%d): aim=cos(% .5e Pi) (NOTE: > Pi/2); actual=*% .5e.\n",
				track->v,track->w,acos(inv_d)/M_PI,act_inv_d);
			else
			  sprintf(buff," (%d,%d): aim=cos(% .5e Pi) (NOTE: > Pi/2); actual=cos(% .5e Pi).\n",
				track->v,track->w,
				acos(inv_d)/M_PI,acos(act_inv_d)/M_PI);

			strcat(msgbuf,buff);
			count++;
		    }
		   track=track->next;
		 } while (track!=NULL);
		edge_free(&edgelist);
	 }
	msg();
	return count;
 }		
case 11: /* altn_rad alternative radii: eucl, hyp. */
 {
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		if (p->hes<0) sprintf(msgbuf,"Radii, p%d (v  eucl, hyp):",
			pnum);
		else if (p->hes==0) sprintf(msgbuf,"Eucl:");
		else sprintf(msgbuf,"Sph:");
		trace=vertlist;
		do {
			if (p->hes<0)		   
			   h_to_e_data(p->packR_ptr[trace->v].center,
				   p->packR_ptr[trace->v].rad,&e_cent,&datum);
			else datum=p->packR_ptr[trace->v].rad;
			sprintf(buff," %d  % .5e, ",trace->v,datum);
			strcat(msgbuf,buff);
			if (p->hes<0)
			 {
				datum=p->packR_ptr[trace->v].rad;
			   	if (datum<=0) sprintf(buff,"inf ");
			   	else {
					datum=(-log(datum));
					sprintf(buff,"% .5e ",datum);}
			   	strcat(msgbuf,buff);
			 }
			count++;
		   	trace=trace->next;
		 } while (trace!=NULL && count < 5);
		vert_free(&vertlist); /* free any excess */
		msg();
	 }
	return count;
 }
case 10: /* selected centers */
 {
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		do {
		   ctr=p->packR_ptr[trace->v].center;
		   sprintf(buff,"%d % .12e % .12e; ",trace->v,ctr.re,ctr.im);
		   strcat(msgbuf,buff);
		   count++;
		   trace=trace->next;
		 } while (trace!=NULL && count < 3);
		vert_free(&vertlist); /* free any excess */
		msg();
	 }
	return count;
 }
case 13: /* screendata */
 {
	sprintf(msgbuf,
		"Screen %d: lower left=(%d, %d), upper right=(%d,%d).  ",
		pnum,(int)(scr->box.lx),(int)(scr->box.ly),
		(int)(scr->box.rx),(int)(scr->box.ry));
	msg();
	return 1;
 }
case 16: /* curvature/overlap errors */
 {
	sprintf(msgbuf,"Pack %d curvature error/Pi = % .5e.",
		pnum,curv_aim_error(p)/M_PI);
	msg();
	for (i=1;i<=p->nodecount;i++)
	 {
		diff=fabs(p->packR_ptr[i].curv-p->packR_ptr[i].aim);
		if (p->packK_ptr[i].bdry_flag 
		   && p->packR_ptr[i].rad >= 0 && p->packR_ptr[i].aim >= 0
		   && diff > bdryerr)
		 {
		   bdryerr=diff;
		   bdryp=i;
		 }
		else if (!p->packK_ptr[i].bdry_flag 
		   && diff >interr)
		 {
			interr=diff;
			intp=i;
		 }
	 }
	if (intp)
	 {
		sprintf(msgbuf,"Pack %d: Worst int ang sum err=% .5e at v=%d.",
			pnum,interr,intp);
		msg();
	 }
	if (bdryp)
	 {
		sprintf(msgbuf,"     Worst bdry ang sum err=% .5e at v=%d.",
			bdryerr,bdryp);
		msg();
	 }
	if (p->hes<okerr && p->overlap_status) 
		/* check for worst overlap error; color 
		faces with incompatibility; color circles with unreliable
		angle sums. */
	 {
		incompat_flag=0;
		for (i=1;i<=p->nodecount;i++)
		 {
			p->packK_ptr[i].mark=0;
			p->packK_ptr[i].color=BG_COLOR;
			if (p->hes<-okerr) 
			   h_anglesum_overlap(p,i,
				p->packR_ptr[i].rad,&dum,&flag);
			else e_anglesum_overlap(p,i,
				p->packR_ptr[i].rad,&dum,&flag);
			if (flag)
			 {
				incompat_flag++;
				p->packK_ptr[i].mark=1;
				p->packK_ptr[i].color=FG_COLOR;
			 }
		 }	
		if (incompat_flag)
		 {
			sprintf(msgbuf,"     %d vertices (marked) with incompatibilities; their ang sums are unreliable.",incompat_flag);
			msg();
		 }			
		over_err=0.0;
		for (i=1;i<p->nodecount;i++)
		for (j=0;j<p->packK_ptr[i].num;j++)
		 {
			if (i<(v=p->packK_ptr[i].flower[j]))
			 {
			   diff=fabs(p->packK_ptr[i].overlaps[j]-
				comp_inv_dist(p,i,v));
			   if (diff > okerr && diff > over_err)
			    {n1=i;n2=v;over_err=diff;} 
					/* worst, so far */
			 }
		 }
	 	if (over_err>0.0)
		 {
		   sprintf(msgbuf,"     Worst overlap error=% .5e, edge (%d,%d).",
			over_err,n1,n2);
		   msg();
		 }
		else
		 {
		   sprintf(msgbuf,"Overlap errors within tolerance.");
		   msg();
		 }
	 }
	return 1;
 }
case 17: /* distance to bdry */
 {
	h_to_e_data(p->packR_ptr[p->alpha].center,
		p->packR_ptr[p->alpha].rad,&ctr,&dum);
	bdry_dist(p,ctr,&b_dist,&b_vert_dist);
	sprintf(msgbuf,"Euclidean distance from alpha to the boundary polygon is % .5e.",b_dist);
	msg();
	return 1;
 }
case 18: /* ratio_ftn <p,q>. Gives (eucl radius of alpha vert of q)
divided by (eucl rad of alpha vert of p) */
 {
	if (sscanf(datastr,"%d %d",&n1,&n2)!=2 
		|| n1< 0 || n1>=NUM_PACKS || !packdata[n1].status 
		|| n2< 0 || n2>=NUM_PACKS || !packdata[n2].status)
	 {sprintf(msgbuf,"Specify two active packs.");emsg();return 0;}
	p1=&packdata[n1];p2=&packdata[n2];
	if (p1->hes>0 || p2->hes>0)
	 {sprintf(msgbuf,"One of packs is spherical.");emsg();return 0;}
	if (p1->hes<0) h_to_e_data(p1->packR_ptr[p1->alpha].center,
				   p1->packR_ptr[p1->alpha].rad,&e_cent,&rad1);
	else rad1=p1->packR_ptr[p1->alpha].rad;
	if (p2->hes<0) h_to_e_data(p2->packR_ptr[p2->alpha].center,
				   p2->packR_ptr[p2->alpha].rad,&e_cent,&rad2);
	else rad2=p2->packR_ptr[p2->alpha].rad;
	sprintf(msgbuf,"Radii comparisons (for alpha vert):\n  Eucl:  ratio p%d/p%d = % .5e;     difference p%d-p%d = % .5e.",
		n2,n1,rad2/rad1,n2,n1,rad2-rad1);
	msg();
	if (p->hes<0)
	 {
		rad2=-log(p2->packR_ptr[p2->alpha].rad);
		rad1=-log(p1->packR_ptr[p1->alpha].rad);
		sprintf(msgbuf,"  Hyp:   ratio p%d/p%d = % .5e;   difference p%d-p%d = % .5e.",
		n2,n1,rad2/rad1,n2,n1,rad2-rad1);
		msg();
	 }
	return 1;
 }
case 20: /* give conductances, transition probabilities for nodes. */
 {
	if (p->hes>=0) return 0;  /* Only for hyp case at this time. */
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		do {
		   v=trace->v;
		   num=pK_ptr[v].num;
		   sprintf(msgbuf,"Random walk data, pack %d, vert %d: neighbors = ",
		   	pnum,v);
		   for (i=0;i<=num;i++) 
		    {
			sprintf(buff," %d",pK_ptr[v].flower[i]);
			strcat(msgbuf,buff);
		    }
		   msg();
		   sprintf(msgbuf,"  Conductances:  total node = % .5e; twin = % .5e",
			nc=node_conductance(p,v),tc=twin_conductance(p,v));
		   msg();
		   dumf=Cvw(pR_ptr[v].rad,0.0,
			pR_ptr[pK_ptr[v].flower[0]].rad,
			pR_ptr[pK_ptr[v].flower[1]].rad,1);
		   tf=dumf/nc;
		   duml = Cvw(pR_ptr[v].rad,
			pR_ptr[pK_ptr[v].flower[num-1]].rad,
			pR_ptr[pK_ptr[v].flower[num]].rad,0.0,2);
		   tl=duml/nc;
		   if (!pK_ptr[v].bdry_flag) {dumf += duml;tf += tl;}
		   sprintf(msgbuf,"  % .5e",dumf);
		   sprintf(buff2,"  % .5e",tf);
		   for (i=1;i<num;i++)
		    {
			dum=Cvw(pR_ptr[v].rad,
				pR_ptr[pK_ptr[v].flower[i+1]].rad,
				pR_ptr[pK_ptr[v].flower[i]].rad,
				pR_ptr[pK_ptr[v].flower[i-1]].rad,0);
			tf=dum/nc;
			sprintf(buff,"  % .5e",dum);
			strcat(msgbuf,buff);
			sprintf(buff,"  % .5e",tf);
			strcat(buff2,buff);
		    }
		   if (pK_ptr[v].bdry_flag)
		    {
			sprintf(buff,"  % .5e",duml);
			strcat(msgbuf,buff);
			sprintf(buff,"  % .5e",tl);
			strcat(buff2,buff);
		    }
		   msg(); 	/* print conductances */
		   sprintf(msgbuf,"  Transition Probabilities:  leakage = % .5e\n",
			tc/nc);
		   strcat(msgbuf,buff2);
		   msg(); 	/* print transition probabilities */
		   count++;
		   trace=trace->next;
		 } while (trace!=NULL && count < 5);
		vert_free(&vertlist); /* free any excess */
	 }
	return count;
 }
case 21: /* status of remote programs */
 {
	for (i=0;i<NUM_PROC;i++) if (remote[i].pid)
	 {
		sprintf(msgbuf,"Remote[%d] is %s, moniker = %s.",
			i,remote[i].program,remote[i].moniker);
		msg();
	 }
	return 1;
 }
	 } /* end of switch */
	return 0;
} /* print_inq_data */

float
radius(p,v) /* give radius. In hyp case, converts s-rad to usual hyp rad.  */
struct p_data *p;
int v;
{
	if ( (p->hes<0) && (p->packR_ptr[v].rad > 0) ) 
		return (-log(p->packR_ptr[v].rad));
	return (p->packR_ptr[v].rad);
} /* radius */

close_script_proc(item,value,event)
Panel_item item;int value;Event *event;
{xv_set(script_frame,WIN_SHOW,FALSE,0);}

reset_script_proc(item,value,event)
Panel_item item;int value;Event *event;
{reset_script();}

reset_script()
{
	Textsw_index first;

	first=(Textsw_index)0;	
	xv_set(script_sw,TEXTSW_INSERTION_POINT,first,0);
	textsw_possibly_normalize(script_sw,first);
	script_file_data_mark=script_file_cmd_mark
		=script_file_path_mark=(Textsw_mark)
		textsw_add_mark(script_sw,first,TEXTSW_MARK_DEFAULTS);
} /* reset_script */

find_script_proc(item,value,event) 
Panel_item item;int value;Event *event;
{find_next_script_cmd(0,0);}

next_script_proc(item,value,event) /* get next full line of commands from
the script command file */
Panel_item item;int value;Event *event;
{
	if (find_next_script_cmd(0,1)==(-1)) return;
	sort_cmd(next_script_cmd);
	free(next_script_cmd);next_script_cmd=NULL;
	return;
} /* next_script_proc */

int
read_infile(p,datastr) /* read pack data from script file */
struct p_data *p;
char *datastr;
{
	int Pid;
	char *databuf,*ptr;
	char tmpfile[NAME_MAX],filename[NAME_MAX];
	FILE *tmp_fp;
	extern char *find_script_data();

	if ((sscanf(datastr,"%s",filename)!=1) 
	   || (databuf=ptr=find_script_data(filename,1))==NULL ) 
	 {
		sprintf(msgbuf,"No pack data in script file with given label.");
		emsg();
		free(databuf);
		return 0;
	 }
		/* first have to put data in tmp file */
	Pid=(int)getpid();
	sprintf(tmpfile,"/tmp/%d%s",Pid,filename);
	if ( (tmp_fp=fopen(tmpfile,"w"))==NULL )
	 {
		sprintf(msgbuf,"Temporary store of script file failed.");
		emsg();
		free(databuf);
		return 0;
	 }
	while ( *ptr!='\0' && (fputc(*ptr,tmp_fp)!=EOF) ) ptr++;
	fclose(tmp_fp);
/*	sysfp=popen("/bin/sh","w");
	fprintf(sysfp,"chmod 666 /tmp/%s",filename);
	fclose(sysfp);
*/
	if ( (tmp_fp=fopen(tmpfile,"r"))==NULL ) 
	 {
		free(databuf);
		return 0;
	 }
	if (readpack(p,tmp_fp))	read_consolidate(p,filename);
	fclose(tmp_fp);
	free(databuf);
	return 1;
} /* read_infile */

int
cmds_infile(datastr) /* read string of cmds from script file */
char *datastr;
{
	int Pid;
	char *databuf,*ptr;
	char tmpfile[NAME_MAX],filename[NAME_MAX];
	FILE *tmp_fp;
	extern char *find_script_data();

	if ((sscanf(datastr,"%s",filename)!=1) 
	   || (databuf=ptr=find_script_data(filename,3))==NULL ) 
	 {
		sprintf(msgbuf,"Didn't find labeled cmds stream in script file.");
		emsg();
		free(databuf);
		return 0;
	 }
		/* first have to put cmds in tmp file */
	Pid=(int)getpid();
	sprintf(tmpfile,"/tmp/%d%s",Pid,filename);
	if ( (tmp_fp=fopen(tmpfile,"w"))==NULL )
	 {
		sprintf(msgbuf,"Temporary store of script data failed.");
		emsg();
		free(databuf);
		return 0;
	 }
	while ( *ptr!='\0' && (fputc(*ptr,tmp_fp)!=EOF) ) ptr++;
	fclose(tmp_fp);
	fexec_call(tmpfile);
	free(databuf);
	return 1;
} /* cmds_infile */

int
path_infile(datastr) /* read path data from script file */
char *datastr;
{
	int Pid;
	char *databuf,*ptr;
	char tmpfile[NAME_MAX],filename[NAME_MAX];
	FILE *tmp_fp;
	extern char *find_script_data();

	if ((sscanf(datastr,"%s",filename)!=1) 
	   || (databuf=ptr=find_script_data(filename,2))==NULL ) 
	 {
		sprintf(msgbuf,"No path data in script file with given label.");
		emsg();
		free(databuf);
		return 0;
	 }
		/* first have to put data in tmp file */
	Pid=(int)getpid();
	sprintf(tmpfile,"/tmp/%d%s",Pid,filename);
	if ( (tmp_fp=fopen(tmpfile,"w"))==NULL )
	 {
		sprintf(msgbuf,"Temporary store of path data failed.");
		emsg();
		free(databuf);
		return 0;
	 }
	while ( *ptr!='\0' && (fputc(*ptr,tmp_fp)!=EOF) ) 
		ptr++;
	fclose(tmp_fp);
/*	sysfp=popen("/bin/sh","w");
	fprintf(sysfp,"chmod 666 /tmp/%s",filename);
	fclose(sysfp);
*/
	if ( (tmp_fp=fopen(tmpfile,"r"))==NULL ) 
	 {
		free(databuf);
		return 0;
	 }
	readpath(tmp_fp); 
	fclose(tmp_fp);
	free(databuf);
	return 1;
} /* path_infile */

int
fexec_call(datastr) /* open file, execute commands */
char *datastr;
{
	int count=0;
	char filename[256],command[BUFSIZE];
	FILE *fp;
	
	sscanf(datastr,"%s",filename);
	if ((fp=fopen(filename,"r"))==NULL)
	 {
	 	sprintf(msgbuf,"fexec error opening %s.",
	 		filename);
	 	emsg();
	 	return count;
	 }
	while (fgets(command,BUFSIZE-1,fp) && command!='\0')
	 	count += handle_cmd(command);
	fclose(fp);
	return count;
} /* fexec_call */
	
/* ==================== PostScript output routines ================== */

int
post_canvas(p) /* circles/triangles/edges/labels/etc. set from 
current display options. 0=filled cir,1=open
cir,4=open compl,7=filled compl, 8=open cir&compl. */
struct p_data *p;
{
	int opt;
	char buff[32];

	if (!p->status) return 0;
	if (!open_psfile(p,1)) return 0; /* open */
	switch (fill_mode)
	 {
		case 0: {fill=.5;break;/* grey */}
		case 1: {fill=1;break; /* white */}
		case 2: {fill=0;break; /* black */}
	 }
	linewidth=0.25*line_thick;
	opt=p->screen->display_opt;
	*buff='\0';
	if (p->screen->unitcircle) strcat(buff," -u");
	if (opt==1 || opt==8) /* open circles */
		strcat(buff," -c a");
	if (opt==0) /* filled circles */
		strcat(buff," -cf a");
	if (opt==4 || opt==8) /* open complex */
		strcat(buff," -f a");
	if (opt==7) /* filled complex */
		strcat(buff," -ff a");
	if (opt==5) /* label vertices */
		strcat(buff," -nc a");
	if (opt==6) /* label faces */
		strcat(buff," -nf a");
	print_call(post_fp,p,buff);
	close_psfile(1);
	print_psfile(1);
	return 1;
} /* post_canvas */

sel_print_proc(item,event) /* bring up print_menu */
Panel_item item;
Event *event;
{if (event_action(event)==ACTION_MENU && event_is_down(event))
		menu_show(print_menu,panel,event,0); }

void 
print_proc(menu,menuitem) /* act on print_menu */
Menu menu;
Menu_item menuitem;
{
	int pnum,opt,act = (int)xv_get(menuitem,MENU_VALUE)-1;
	char *datastr;
	struct p_data *p;

	p=&packdata[current_p];
	if (act>1 && act<11 && !custom_status)
	 {
		strcpy(msgbuf,"The 'custom' postscript file has not been opened.");
		emsg();
		return;
	 }
	switch (act)
	 {
case 1: open_psfile(p,2); return;/* open file (lose previous contents) */
case 3: sprintf(buf," -u"); break;	/* unit circle */ 
case 4: /* selected open circles */
 {
	if( strlen(datastr=get_selection(1)) ) 
		sprintf(buf," -c %s",datastr);
	else sprintf(buf," -c a");
	break;
 }
case 5:  /* selected filled circles */
 {
	if ( strlen(datastr=get_selection(1)) ) 
		sprintf(buf," -cf %s",datastr);
	else sprintf(buf," -cf a");
	break;
 }
case 6: /* selected open faces */
 {
	if ( strlen(datastr=get_selection(1)) )
		sprintf(buf," -f %s",datastr);
	else sprintf(buf," -f a");
	break;
 }
case 7: /* selected filled faces */
 {
	if ( strlen(datastr=get_selection(1)) )
		sprintf(buf," -ff %s",datastr);
	else sprintf(buf," -ff a");
	break;
 }
case 8: /* selected edges */
 {
	if (!strlen(datastr=get_selection(1)))
	 {strcpy(msgbuf,"You must select vertices. ");emsg();break;}
	else sprintf(buf," -e %s",datastr);
	break;
 }	
case 9: /* selected vertex labels */
 {
	if ( strlen(datastr=get_selection(1)) )
		sprintf(buf," -nc %s",datastr);
	else sprintf(buf," -nc a");
	break;
 }
case 10: /* selected face labels */
 {
	if ( strlen(datastr=get_selection(1)) ) 
		sprintf(buf," -nf %s",datastr);
	else sprintf(buf," -nf a");
	break;
 }
case 11: /* caption */
 {
	if ( strlen(datastr=get_selection(1)) ) 
		sprintf(buf," -t %s",datastr);
	else sprintf(buf," -t");
	break;
 } 
case 12: /* record display option objects from selected pack. */
 {
	if (!strlen(datastr=get_selection(1)) || sscanf(datastr,"%d",&pnum)!=1
		|| pnum<0 || pnum>NUM_PACKS || !packdata[pnum].status) return;
	opt=screendata[pnum].display_opt;
	*buf='\0';
	if (opt==1 || opt==8) /* open circles */
		strcat(buf," -c a");
	if (opt==0) /* filled circles */
		strcat(buf," -cf a");
	if (opt==4 || opt==8) /* open complex */
		strcat(buf," -f a");
	if (opt==7) /* filled complex */
		strcat(buf," -ff a");
	if (opt==5) /* label vertices */
		strcat(buf," -nc a");
	if (opt==6) /* label faces */
		strcat(buf," -nf a");
	break;
 }
case 13: sprintf(buf," -g");break; 	/* print path */
case 14: /* wrap up the postscript file and close it */
 {
	close_psfile(2);
	return;
 }
case 15: /* print the special postscript file */
 {
	print_psfile(2);
	return;
 }
case 17: /* print the canvas postscript file. */
 {
	post_canvas(p);
	return;
 }
	 } /* end of switch */
	print_call(custom_fp,p,buf);
	return;
} /* print_proc */

int
post_num(fp,pt,n,box) /* store postscript text */
FILE *fp;
int n;
complex pt;
Box box;
{
	if (cir_ck(pt,okerr,box)) 
	 {
		sprintf(buf,"%d",n);
		fprintf(fp,"%lf %lf moveto (%s) show\nnewpath\n",
			pt.re,pt.im,buf);
	 }
	return 1;
} /* post_num */

int
post_h_geo(fp,ctr,rad,arg1,arg2) /* store postscript description of 
geodesic. Assume it is on the screen. */
FILE *fp;
complex ctr;
float rad,arg1,arg2;
{
	fprintf(fp,"%lf %lf %lf %lf %lf arc\n0 setgray\nstroke\n",
		ctr.re,ctr.im,rad,arg1*degPI,arg2*degPI);
	return 1;
} /* post_h_geo */

int
post_s_geo(fp,p,z1,z2) /* store spherical geo */
FILE *fp;
struct p_data *p;
complex z1,z2;
{
	int length=1,k;
	struct Pathlist *trace,*plist=NULL;

	if ((plist=s_geodesic(ss_view(p->screen,z1,1,&dum_int),
		ss_view(p->screen,z2,1,&dum_int),num_plot))==NULL) 
		return 0;
	trace=plist;
	while ((trace->next)!=NULL)
	 {trace=trace->next;length++;}
	trace=plist;
	fprintf(fp,"newpath\n %lf %lf moveto\n",trace->x,trace->y);
	for (k=1;k<=length && (trace=trace->next)!=NULL;k++)
		fprintf(fp,"%lf %lf lineto\n",trace->x,trace->y);
	fprintf(fp,"stroke\n");
	path_free(&plist);
	return 1;
} /* post_s_geo */

int
post_line(fp,p1,p2) /* store postscript description of straight line.
Assume on screen. */
FILE *fp;
complex p1,p2;
{
	fprintf(fp,"%lf %lf moveto\n%lf %lf lineto\nstroke\n",
		p1.re,p1.im,p2.re,p2.im);
	return 1;
}

int
post_cir(fp,ctr,rad,ff) /* store postscript description of eucl circle. 
Assume on screen. ff=fill flag, 1->foreground, 2->background,3->color.
(not yet set for color). */
FILE *fp;
int ff;
complex ctr;
float rad;
{
	float grey;

	if (ff)
	 {
		if (ff==2) grey=1.0; /* background(=white) */
		else grey=1.0-fill;
		fprintf(fp,"%lf     %lf %lf %lf disc\n",
		grey,ctr.re,ctr.im,rad); /* filled disc */
	 }
	else fprintf(fp,"%lf %lf %lf circle\n",
		ctr.re,ctr.im,rad);	/* circle */
	return 1;
} /* post_cir */

int 
post_s_cir(fp,ctr,rad,ff) /* post spherical circle. */
FILE *fp;
int ff;
complex ctr;
float rad;
{
	struct Pathlist *list;
	extern struct Pathlist *s_circle_list();

	if ((list=s_circle_list(ctr,rad,num_plot))==NULL) return 0;
	return post_s_convex(fp,list,ff);
} /* post_s_cir */

int
post_s_tri(fp,p1,p2,p3,ff) /* post sph tri, ff is fill flag */
FILE *fp;
int ff;
complex p1,p2,p3;
{
	struct Pathlist *list;
	extern struct Pathlist *sph_tri_list();

	if ((list=sph_tri_list(p1,p2,p3,num_plot))==NULL) return 0;
	return post_s_convex(fp,list,ff);
} /* s_triangle */

int
post_s_convex(fp,slist,ff) /* sph tri or convex circle */
FILE *fp;
struct Pathlist *slist;
int ff;
{
	int i=1,k;
	struct Pathlist *trace;
	extern struct Pathlist *fix_convex_list();

	if (slist==NULL || (slist=fix_convex_list(slist))==NULL) return 0;
	trace=slist;
	while (trace->next!=NULL && i<10000)
	 {
		i++;
		trace=trace->next;
	 }
	trace=slist;
	fprintf(fp,"newpath\n %lf %lf moveto\n",trace->x,trace->y);
	for (k=2;k<=i;k++)
	 {
		trace=trace->next;
		fprintf(fp,"%lf %lf lineto\n",trace->x,trace->y);
	 }
	if (ff) fprintf(fp,"\ngsave gry setgray fill grestore\n");
			/*fill*/
	fprintf(fp,"stroke\n");
	path_free(&slist);
	return 1;
} /* post_s_convex */

int
post_equator(fp,q) /* spherical unit circle, screen q */
FILE *fp;
struct s_data *q;
{
	complex ctr;

	ctr.re=0.0;ctr.im=M_PI;
	return post_s_cir(fp,ss_view(q,ctr,1,&dum_int),M_PI/2.0,0);
} /* post_equator */
	
int
handle_geo(fp,p,a,b) /* do the right thing with geodesic to postscript */
FILE *fp;
struct p_data *p;
complex a,b;
{
	hyp_geodesic hg;

	if (p->hes==0)
	 {
		if (line_ck(a,b,p->screen->box)) return (post_line(fp,a,b));
	 }
	else if (p->hes<0)
	 {
		h_geodesic(a,b,&hg);
		if (hg.line_flag)
		 {
			if (!line_ck(a,b,p->screen->box)) return 1;
			return(post_line(fp,a,b));
		 }
		else 
		 {
			if (!geo_ck(hg,p->screen->box))
				return 1; /* circle is off screen */
			return (post_h_geo(fp,hg.c,hg.rad,hg.arg1,hg.arg2));
		 }
	 }
	return(post_s_geo(fp,p,a,b));
} /* handle_geo */

int
open_psfile(p,flag) /* open either the ps_file (flag=1) or print_file
(flag=2) and store the preliminary postscript stuff. */
struct p_data *p;
int flag;
{
	float size,factor;
	char filename[NAME_MAX];
	FILE *fp;

	if (flag==1) /* canvas postscript file */
	 {
		fclose(post_fp);
		strcpy(filename,(char *)xv_get(ps_file_item,PANEL_VALUE));
		if ((post_fp=fopen(filename,"w"))==NULL) 
		 { sprintf(msgbuf,"Can't open %s. ",filename);emsg();return 0; }
		fp=post_fp;
	 }
	else if (flag==2) /* custom file */
	 {
		if (custom_status) fclose(custom_fp);
		strcpy(filename,(char *)xv_get(print_file_item,PANEL_VALUE));
		if ((custom_fp=fopen(filename,"w"))==NULL) 
		 {sprintf(msgbuf,"Can't open %s. ",filename);emsg();return 0;}
		fp=custom_fp;
		custom_status=1;
	 }
	else return 0;
	size= (float) sc_print_size;
	factor= size/(p->screen->box.ry-p->screen->box.ly);
/* -------- preamble 
   -------- define 'ourlinewidth' at 1 point, 'circle', 'disc' */

	fprintf(fp,"%%!\n%% CirclePack preamble ===============\n     /ourlinewidth\n        { %lf mul setlinewidth}  def\n     /circle\n        { 0 360 arc stroke}  def\n     /disc\n        { 0 360 arc gsave setgray fill grestore stroke}  def\n",1/(factor*72));

/* -------- font */
	fprintf(fp,"        /wht 1.0 def %% gray levels \n        /gry 0.8 def\n        /drk 0.5 def\n        /blck 0.0 def\n        /Times-Roman findfont .04 scalefont setfont\n");

/* -------- scaling, centering, lines */
	fprintf(fp,"     72 72 scale %% inches\n     %lf %lf translate\n     1 setlinecap\n     1 setlinejoin\n%% end preamble ================\n",
		4.25,5.5);

	fprintf(fp,"save  %% global\n");
/* -------- scaling for screen and desired output size */
	fprintf(fp,"%% -----------\n     %lf %lf scale \n     .25 ourlinewidth\n     0 setgray\n",
		factor,(1.01)*factor);
		/* now scaled based on screens size and desired output size */


	fprintf(fp,"newpath\n%lf %lf moveto\n%lf %lf lineto\n%lf %lf lineto\n%lf %lf lineto\nclosepath\n%%gsave stroke grestore\nclip\nnewpath\n%% ------------\n",
		p->screen->box.lx,p->screen->box.ry,
		p->screen->box.lx,p->screen->box.ly,
		p->screen->box.rx,p->screen->box.ly,
		p->screen->box.rx,p->screen->box.ry); /* clip box */

	return 1;
} /* open_psfile */

int
close_psfile(flag) /* close canvas file (flag=1) or custom file (flag=2). */
int flag;
{
	if (flag==1)
	 {
		fprintf(post_fp,"restore %% global\nshowpage\n");
		fclose(post_fp);
		return 1;
	 }
	if (flag==2)
	 {
		if (!custom_status) return 0;
		fprintf(custom_fp,"restore %% global\nshowpage\n");
		fclose(custom_fp);
		custom_status=0;
		return 1;
	 }
	return 0;
} /* close_psfile */

int 
print_psfile(flag) /* print canvas (flag=1) or custom (flag=2) file */
int flag;
{
	char buff[64];

	if (flag==1)
	 {
		close_psfile(1);
		strcpy(buff,(char *)xv_get(ps_file_item,PANEL_VALUE));
	 }
	else if (flag==2)
	 {
		if (custom_status) close_psfile(2);
		strcpy(buff,(char *)xv_get(print_file_item,PANEL_VALUE));
	 }
	else return 0;
	strcpy(buf,(char *)xv_get(print_com_item,PANEL_VALUE));
	strcat(buf," ");
	strcat(buf,buff);
	sysfp=popen("/bin/sh","w");
	fprintf(sysfp,"%s\n",buf);
	pclose(sysfp);
	sprintf(msgbuf,"Have sent `%s' to the printer.",buff);
	msg();
	return 1;
} /* print_psfile */

int
post_face_label(fp,p,face)
FILE *fp;
struct p_data *p;
int face;
{
	int i0,i1,i2,front;
	complex ctr,p0,p1,p2;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	i0=p->faces[face].vert[0];p0=pR_ptr[i0].center;
	i1=p->faces[face].vert[1];p1=pR_ptr[i1].center;
	i2=p->faces[face].vert[2];p2=pR_ptr[i2].center;
	if (p->hes<=0)
	 {
		ctr.re=(p0.re+p1.re+p2.re)*.33333;
		ctr.im=(p0.im+p1.im+p2.im)*.33333;
	 }
	else if (p->hes>0)
	 {
		ctr=sph_tri_center(p0,p1,p2);
		ctr=ss_view(p->screen,ctr,1,&front);
		if (!front) return 0;
		ctr=s_pt_to_visual_plane(ctr);
	 }
	return (post_num(fp,ctr,face,p->screen->box));
} /* post_face_label */

int
post_cir_label(fp,p,v)
FILE *fp;
struct p_data *p;
int v;
{
	int front;
	float stretch;
	complex ctr;
	struct R_data *pR_ptr;
	
	pR_ptr=p->packR_ptr;
	stretch=1.0+(p->screen->box.rx-p->screen->box.lx)*10/
		  (p->screen->pix_box.rx-p->screen->pix_box.lx); 
	    /* to shift horocycle labels a little to miss circle*/
	if (p->hes<(-okerr) && pR_ptr[v].rad<0)
	 {
		ctr.re=pR_ptr[v].center.re*stretch;
		ctr.im=pR_ptr[v].center.im*stretch;
	 } 
	else if (p->hes>okerr) 
	 {
		ctr=ss_view(p->screen,pR_ptr[v].center,1,&front);
		if (!front) return 0;
		ctr=s_pt_to_visual_plane(ctr);
	}
	else ctr=pR_ptr[v].center;
	return (post_num(fp,ctr,v,p->screen->box));
} /* post_cir_label */

int
post_caption(fp,datastr) /* not operational; puts commented-out
string in postscript file */
FILE *fp;
char *datastr;
{
	fprintf(fp,"\n%% ----- (%s) show\nnewpath\n",datastr);
	return 1;
}

int
print_call(fp,p,datastr)
FILE *fp;
struct p_data *p;
char *datastr;
{
	int k,vert,count=0,fill_flag=0,i,dflag,hits,front,v,w;
	char *nextpoint,*endptr,buff[BUFSIZE];
	extern struct Vertlist *node_link_parse(),*face_link_parse();
	struct Vertlist *vertlist=NULL,*trace,*facelist=NULL;
	extern struct Edgelist *node_pair_link(),*get_extended_edge();
	struct Edgelist *edgelist=NULL,*track,*eelist=NULL,*eetrack;
	complex ctr;
	struct Pathlist *ptrace;
	
	nextpoint=datastr;
	if (!grab_next(&nextpoint,next) || next[0]!='-') return 0;
	switch (fill_mode)
	 {
		case 0: {fill=.5;break;/* grey */}
		case 1: {fill=1;break; /* white */}
		case 2: {fill=0;break; /* black */}
	 }
	linewidth=0.25*line_thick;
	do
	 {
for (i=strlen(next);i<5;i++) next[i]='\0'; /* clear old info */
if (next[0]!='-') return count;
if (next[1]=='o') 
	{
		count += open_psfile(p,2);fp=custom_fp;
		if (fp && p->hes> okerr)
		 {
			ctr.re=ctr.im=0.0;
			post_cir(fp,ctr,1.0,0);
		 }
	 }
else if (next[1]=='x' && fp==custom_fp) 
	{count += close_psfile(2);fp=NULL;}
else if (next[1]=='l' && fp==custom_fp) 
	{count += print_psfile(2);fp=NULL;}
else if (fp==NULL || ftell(fp)<=0) /* file not open */
	 {
		sprintf(msgbuf,"Check whether file has been opened.");
		emsg();
		break;
	 }
else if (next[1]=='u') /* unit circle */
 {
	fprintf(fp,"%lf ourlinewidth\n",linewidth);
	if (p->hes>0) post_equator(fp,p->screen);
	else {ctr.re=ctr.im=0.0;post_cir(fp,ctr,1.0,0);}
	count++;
 }
else if (next[1]=='c') /* post circles */
 {
	fprintf(fp,"%lf ourlinewidth\n",linewidth);
	if (next[2]=='f')
	 {
		fill_flag=1; /* want filled faces, default=foreground */
		if (next[3]=='b') fill_flag=2; /* background fill */
/* eventually want colors */
	 }
	else fill_flag=0;
	if ((vertlist=
		node_link_parse(p,nextpoint,&endptr,&hits))!=NULL)
	 {
		nextpoint=endptr;
		trace=vertlist;
		do {
			vert=trace->v;
			count +=
			   post_any_circle(fp,p,vert,fill_flag);
			trace=trace->next;
	  	 } while (trace!=NULL);
		vert_free(&vertlist);
	 }
	else for (k=1;k<=p->nodecount;k++) /* default to all */
	 {
		count += post_any_circle(fp,p,k,fill_flag);
	 }
 }
else if (next[1]=='d' && p->hes>okerr) /* post only small dot
instead of circle, sphere only.*/
 {
	for (i=1;i<=p->nodecount;i++)
	 {
		ctr=ss_view(p->screen,p->packR_ptr[i].center,1,&dum_int);
		if (cos(ctr.re)>0) /* on front */
		 {
			ctr.re=sin(ctr.im)*sin(ctr.re);
			ctr.im=cos(ctr.im);
			post_cir(fp,ctr,.005,1);
		 }
	 }
 }
else if (next[1]=='C') /* all circles, recomputed as posted */
 {
	if (next[2]=='f' || next[3]=='f') fill_flag=1;
	else fill_flag=0;
	if (next[2]=='n' || next[3]=='n') dflag=6; /* with labels */
	else dflag=1;
	count += post_in_order(fp,p,dflag,fill_flag,0);
 } /* finished with circle cases */
else if (next[1]=='e') /* draw edges */
 {
	fprintf(fp,"%lf ourlinewidth\n",linewidth);
	if (next[2]=='e' && (vertlist=node_link_parse(p,nextpoint,
		&endptr,&hits))!=NULL) /* hex extended edges */
	 {
		nextpoint=endptr;
		trace=vertlist;
		while (trace && (v=trace->v)
		   && trace->next && (w=trace->next->v)) /* get pairs */
		 {
			trace=trace->next;
			if ((eelist=get_extended_edge(p,v,w,16))!=NULL)
			 {
				eetrack=eelist;
				do {
				   count += handle_geo(fp,p,
					p->packR_ptr[eetrack->v].center,
					p->packR_ptr[eetrack->w].center);
				   eetrack=eetrack->next;
				 } while (eetrack!=NULL);
				edge_free(&eelist);
			 }
		 }
		vert_free(&vertlist);
	 }
	else if ((edgelist=node_pair_link(p,nextpoint,&endptr,&hits))
		!=NULL)
	 {
		nextpoint=endptr;
		track=edgelist;
		do {
			count += handle_geo(fp,p,
				p->packR_ptr[track->v].center,
				p->packR_ptr[track->w].center);
			track=track->next;
		 } while (track!=NULL);
		edge_free(&edgelist);
	 }
 } /* finished with edges */
else if (next[1]=='f' || next[1]=='F') /* draw faces */
 {
	if (next[2]=='f')
	 {
		fill_flag=1; /* want filled faces, default=foreground */
		if (next[3]=='b') fill_flag=2; /* background fill */
/* eventually want color faces */
	 }
	else fill_flag=0;
	if ( (next[1]=='F' && next[2]=='n')
	   || (next[1]=='F' && next[2]=='C' && next[3]=='n') ) 
		dflag=1; /* with labels */
	else dflag=0;
	if ((facelist=
		face_link_parse(p,nextpoint,&endptr,&hits))!=NULL)
	 {
		nextpoint=endptr;
		if (next[1]=='F')  /* locate faces as they're drawn */
			count += post_facelist(fp,
				p,fill_flag,facelist,1,0,dflag);
		else count += post_facelist(fp,
				p,fill_flag,facelist,0,0,dflag);
	 }
	else if (next[1]=='F') /* use draworder, but recompute cents */
	 {
		if (next[2]=='C') /* also circles */
		 {
			if (next[3]=='n') dflag=8; /* with f/c labels */
			else dflag=3;
		 }
		else if (next[2]=='n') dflag=7; /* with labels */
		else dflag=2;
		count += post_in_order(fp,p,dflag,fill_flag,0);
	 }				
	else /* default to all */
	 {
		for (k=1;k<=p->facecount;k++) 
			post_any_face(fp,p,k,fill_flag);
		count++;
	 }
 } /* finished with faces */
else if (next[1]=='s') /* shaded polygon about vertlist */
 {
	if (next[2]=='f') fill_flag=1;
	else fill_flag=0;
	if ((vertlist=node_link_parse(p,nextpoint,
		&endptr,&hits))!=NULL)
	 {
		nextpoint=endptr;
		trace=vertlist;
		count +=
		   post_any_polygon(fp,p,vertlist,fill_flag);
		vert_free(&vertlist);
	 }
 }
else if (next[1]=='n') /* labels */
 {
	if (next[2]=='f')
	 {
	   if ((facelist=
		face_link_parse(p,nextpoint,&endptr,&hits))!=NULL)
	    {
		nextpoint=endptr;
		trace=facelist;
		do {
			count += post_face_label(fp,p,trace->v);
			trace=trace->next;
		 } while (trace!=NULL);
		vert_free(&facelist);
	    }
	   else for (k=1;k<=p->facecount;k++) 
		count += post_face_label(fp,p,k);
	 }
	else if (next[2]=='c') 
	 {
	   if ((vertlist=
		node_link_parse(p,nextpoint,&endptr,&hits))!=NULL)
	    {
		nextpoint=endptr;
		trace=vertlist;
		do {
			count += post_cir_label(fp,p,trace->v);
			trace=trace->next;
		 } while (trace!=NULL);
		vert_free(&vertlist);
	    }
	   else for (k=1;k<=p->nodecount;k++) 
		count += post_cir_label(fp,p,k);
	 }
	else if ( 			/* 'l' or 'z' */
	   (
	    (
	     next[2]=='z'  		/* given point */
		&& grab_next(&nextpoint,next)
		   && sscanf(next,"%lf",&(ctr.re))
		&& grab_next(&nextpoint,next)
		   && sscanf(next,"%lf",&(ctr.im))
		&& !(i=0)
	    )
	    ||
	    (
	     next[2]=='l'  		/* or center of circle */
		&& grab_next(&nextpoint,next)
		   && (i=atoi(next))>0 && i<=p->nodecount
	    )
	   )
	   && grab_next(&nextpoint,next) 
	   && strlen(strncpy(buf,next,32))!=0 
		)
		/* put string either at cent of i or at given ctr. */
	 {
		if (i>0) ctr=p->packR_ptr[i].center;
		strcat(buf,"\0");
		if (p->hes>okerr) 
	 	 {
			ctr=ss_view(p->screen,ctr,1,&front);
			if (front)
			 {
			   ctr=s_pt_to_visual_plane(ctr);
			   fprintf(fp,"%lf %lf moveto (%s) show\nnewpath\n",
				ctr.re,ctr.im,buf);
			   count++;
			 }
		 }
		else
		 {
			fprintf(fp,"%lf %lf moveto (%s) show\nnewpath\n",
				ctr.re,ctr.im,buf);
			count++;
		 }
	 }
 } /* finished with labels */
else if (next[1]=='g') /* print path */
 {
	if (pathlength<4) break;
	fprintf(fp,"newpath\n");
	fprintf(fp,"%lf ourlinewidth\n",linewidth);
	fprintf(fp,"%lf %lf moveto\n",pathlist->x,pathlist->y);
	ptrace=pathlist;
	while ((ptrace=ptrace->next)!=NULL) 
		fprintf(fp,"%lf %lf lineto\n",ptrace->x,ptrace->y);
	fprintf(fp,"closepath\nstroke\n");
 }
else if (next[1]=='t') /* print caption (filename=default); must be
	last command of this print_call. */
 {
	if (strlen(strcpy(buff,nextpoint))==0)
		strcpy(buff,p->file_name);
	count += post_caption(fp,buff);
	return count;
 } 
		 } /* end of do */
		while (nextpoint!=NULL && grab_next(&nextpoint,next) );
		return count;		
} /* print_call */

int
post_facelist(fp,p,flag,facelist,fix,l_face,nflag) 
/* post faces from facelist. 
flag: 0=open, 1=background, 2=foreground, 3=color. fix=1 means to 
locate faces successively as drawn (changing stored
centers). facelist is cleared. l_face="live face" may be given; nflag true
means to draw face numbers. */ 
FILE *fp;
int flag,fix,l_face,nflag;
struct Vertlist *facelist;
struct p_data *p;
{
	int col=0,n,v1,v2,v3,vv=0;
	float o1,o2,o3;
	struct Vertlist *trace;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	if (facelist==NULL) return 0;
	switch(flag)
	 { case 1:{fill=1;col=BG_COLOR;break;}
	   case 2:{fill=1;col=FG_COLOR;break;}
	   case 3:{fill=1;break;}
	 }
	trace=facelist;
		/* post first */
	if (l_face>0 && l_face<=p->facecount) vv=l_face;
	do {
			/* if fix, draw successive ones as contig. */
		if (fix)
		 {
		   if ( (n=nghb_tri(p,vv,trace->v))<0 )
			n=p->faces[trace->v].index_flag;
			/* if not contig, use index_flag */
		   v1=p->faces[trace->v].vert[n];
		   v2=p->faces[trace->v].vert[(n+1)%3];
		   v3=p->faces[trace->v].vert[(n+2)%3];

	if (p->overlap_status) 
	 {
		o1=pK_ptr[v1].overlaps[nghb(p,v1,v2)];
		o2=pK_ptr[v2].overlaps[nghb(p,v2,v3)];
		o3=pK_ptr[v3].overlaps[nghb(p,v3,v2)];
	 }
	else o1=o2=o3=1.0;
	any_compcenter(p->hes,
		pR_ptr[v1].center,pR_ptr[v2].center,&pR_ptr[v3].center,
		pR_ptr[v1].rad,pR_ptr[v2].rad,&pR_ptr[v3].rad,o1,o2,o3);
			/* compute and store new center */
		 }

		while (trace!=NULL && trace->v==vv) trace=trace->next;
		if (trace==NULL) break;
		if ( (vv=trace->v)>0 && vv<=(p->facecount) )
		 {
		   if (flag==3) col=p->faces[vv].color;
		   post_any_face(fp,p,vv,flag);
		   if (nflag) post_face_label(fp,p,vv);
		 }
		trace=trace->next;
	 } while (trace!=NULL);
	vert_free(&facelist);
	return vv;
} /* post_facelist */

int
post_in_order(fp,p,flag,fill,pl) /* post faces and/or circles in drawingorder,
but post and recompute as you go along. E.g., will give 'ghosts'.
flag: 1=circles,2=faces,3=both. +5 ==> also labels. If pl set, place first
face; else, leave it where it is. */
FILE *fp;
struct p_data *p;
int flag,fill,pl;
{
	int nf,ind,a,b,c;

	nf=p->first_face;
	if (pl) place_face(p,nf,p->faces[nf].index_flag);
	if (flag==1 || flag==3 || flag==6 || flag==8)
	 {
		ind=p->faces[nf].index_flag;
		a=p->faces[nf].vert[ind];
		b=p->faces[nf].vert[(ind+1) % 3];
		c=p->faces[nf].vert[(ind+2) % 3];
		post_any_circle(fp,p,a,fill);
		post_any_circle(fp,p,b,fill);
		post_any_circle(fp,p,c,fill);
		if (flag==6 || flag==8)
		 {
			post_cir_label(fp,p,a);
			post_cir_label(fp,p,b);
			post_cir_label(fp,p,c);
		 }
	 }
	if (flag==2 || flag==3 || flag==7 || flag==8)
		post_any_face(fp,p,nf,fill);
	if (flag==7 || flag==8) post_face_label(fp,p,nf);
	while ( (nf=p->faces[nf].next_face)!=p->first_face )
	 {
		comp_center_face(p,nf,-1);
		if (flag==1 || flag==3|| flag==6 || flag==8)
		 {
			c=p->faces[nf].vert[(p->faces[nf].index_flag + 2) % 3];
			post_any_circle(fp,p,c,fill); /* new circle */
			if (flag==6 || flag==8)	post_cir_label(fp,p,c);
		 }
		if (flag==7 || flag==8) post_face_label(fp,p,nf);
		if (flag==2 || flag==3 || flag==7 || flag==8)
			post_any_face(fp,p,nf,fill);
	 }
	return 1;
} /* post_in_order */

int
post_any_circle(fp,p,vert,fill_flag) /* posts face in approp geometry */
FILE *fp;
struct p_data *p;
int vert,fill_flag;
{
	complex ctr;
	float r,rad;

	if (vert<1 || vert > p->nodecount) return 0;
	ctr=p->packR_ptr[vert].center;
	r=p->packR_ptr[vert].rad;
	if (p->hes < 0) 
	 {
		h_to_e_data(ctr,r,&ctr,&rad);
		if (cir_ck(ctr,rad,p->screen->box))
			return post_cir(fp,ctr,rad,fill_flag);
	 }
	else if (p->hes==0)
	 {
		if (cir_ck(ctr,r,p->screen->box))
			return post_cir(fp,ctr,r,fill_flag);
	 }
	else if (p->hes>0)
	 {
		return post_s_cir(fp,ss_view(p->screen,ctr,1,&dum_int),r,fill_flag);
	 }
	return 0;
} /* post_any_circle */

int
post_any_face(fp,p,face,ff)
FILE *fp;
struct p_data *p;
int face,ff;
{
	int front;
	complex p1,p2,p3;
	hyp_geodesic hg1,hg2,hg3;
	struct R_data *pR_ptr;

	pR_ptr=p->packR_ptr;
	if (face<1 || face > p->facecount) return 0;
	p1=pR_ptr[p->faces[face].vert[0]].center;
	p2=pR_ptr[p->faces[face].vert[1]].center;
	p3=pR_ptr[p->faces[face].vert[2]].center;
  if (p->hes< (-okerr))
   {
	h_geodesic(p1,p2,&hg1);
	h_geodesic(p2,p3,&hg2);
	h_geodesic(p3,p1,&hg3); /* set triangle structures */
	if (!geo_ck(hg1,p->screen->box) && 
		!geo_ck(hg2,p->screen->box) && 
		!geo_ck(hg3,p->screen->box)) 
		 return;  /* off screen */
	fprintf(fp,"%lf %lf moveto\n",p1.re,p1.im);
	if (hg1.line_flag)
		fprintf(fp,"%lf %lf lineto\n",p2.re,p2.im);
	else 
	 {
		if (hg1.z1.re==p1.re && hg1.z1.im==p1.im) /* right direction*/
			fprintf(fp,"%lf %lf %lf %lf %lf arc\n",
				hg1.c.re,hg1.c.im,hg1.rad,
				hg1.arg1*degPI,hg1.arg2*degPI);
		else fprintf(fp,"%lf %lf %lf %lf %lf arcn\n",
				hg1.c.re,hg1.c.im,hg1.rad,
				hg1.arg2*degPI,hg1.arg1*degPI);
	 }
	if (hg2.line_flag)
		fprintf(fp,"%lf %lf lineto\n",p3.re,p3.im);
	else 
	 {
		if (hg2.z1.re==p2.re && hg2.z1.im==p2.im) /* right direction*/
			fprintf(fp,"%lf %lf %lf %lf %lf arc\n",
				hg2.c.re,hg2.c.im,hg2.rad,
				hg2.arg1*degPI,hg2.arg2*degPI);
		else fprintf(fp,"%lf %lf %lf %lf %lf arcn\n",
				hg2.c.re,hg2.c.im,hg2.rad,
				hg2.arg2*degPI,hg2.arg1*degPI);
	 }
	if (hg3.line_flag)
		fprintf(fp,"%lf %lf lineto\n",p1.re,p1.im);
	else 
	 {
		if (hg3.z1.re==p3.re && hg3.z1.im==p3.im) /* right direction*/
			fprintf(fp,"%lf %lf %lf %lf %lf arc\n",
				hg3.c.re,hg3.c.im,hg3.rad,
				hg3.arg1*degPI,hg3.arg2*degPI);
		else fprintf(fp,"%lf %lf %lf %lf %lf arcn\n",
				hg3.c.re,hg3.c.im,hg3.rad,
				hg3.arg2*degPI,hg3.arg1*degPI);
	 }
	if (ff)  /* filled? */
		fprintf(fp,"gsave gry  setgray fill grestore\n"); 
		/* ?? need to put color in, if desired */
	fprintf(fp,"stroke\n"); /* draw border */
	return;
   } /* end of hyp case */
  else if (p->hes < okerr)
   {
	if (line_ck(p1,p2,p->screen->box) || 
		line_ck(p2,p3,p->screen->box) || 
		line_ck(p3,p1,p->screen->box))
	 {
		fprintf(fp,"%lf %lf moveto\n%lf %lf lineto\n%lf %lf lineto\n%lf %lf lineto\n",
			p1.re,p1.im,p2.re,p2.im,p3.re,p3.im,p1.re,p1.im);
		if (ff)
		   fprintf(fp,"gsave gry  setgray fill grestore\n");
			 /* filled */
		fprintf(fp,"stroke\n"); /* draw border */
	 }
	return;
   } /* end of eucl case */
  else if (p->hes>=okerr) 
	post_s_tri(fp,
		ss_view(p->screen,p1,1,&front),
		ss_view(p->screen,p2,1,&front),
		ss_view(p->screen,p3,1,&front),ff);
} /* post_any_face */

int
post_any_polygon(fp,p,vertlist,ff)
FILE *fp;
struct p_data *p;
struct Vertlist *vertlist;
int ff;
{
	int v,w;
	complex a,b;
	hyp_geodesic hg;
	struct Vertlist *vtrace;
	struct Pathlist *plist;
	struct R_data *pR_ptr;

	if (!(vtrace=vertlist) || !(vtrace->next)
		|| (v=vtrace->v)<1 || v>p->nodecount) return 0;
	pR_ptr=p->packR_ptr;
	a=pR_ptr[v].center;
	fprintf(fp,"%lf %lf moveto\n",a.re,a.im);
  if (p->hes<0)
   {
	while (vtrace && vtrace->next
	   && (v=vtrace->v)>0 && v<=p->nodecount
	   && (w=vtrace->next->v)>0 && w<=p->nodecount)
	 {
	   if (v!=w) /* eat repeats */
	    {
		a=pR_ptr[v].center;	
		b=pR_ptr[w].center;
		h_geodesic(a,b,&hg);
		if (hg.line_flag) /* use straight line */
		 {
			fprintf(fp,"%lf %lf lineto\n",b.re,b.im);
		 }
		else 
		 {
			if (hg.z1.re==a.re && hg.z1.im==a.im) 
				/* right direction*/
				fprintf(fp,"%lf %lf %lf %lf %lf arc\n",
				  hg.c.re,hg.c.im,hg.rad,
				  hg.arg1*degPI,hg.arg2*degPI);
			else fprintf(fp,"%lf %lf %lf %lf %lf arcn\n",
				  hg.c.re,hg.c.im,hg.rad,
				  hg.arg2*degPI,hg.arg1*degPI);
		 }
	    }
	   vtrace=vtrace->next;
	 } /* end of while */
	fprintf(fp,"closepath\n");
	if (ff)  /* filled? */
		fprintf(fp,"gsave gry   setgray fill grestore\n"); 
		/* ?? need to put color in, if desired */
	fprintf(fp,"stroke\n"); /* draw border */
	return 1;
   } /* end of hyp case */
  else if (p->hes==0)
   {
	while (vtrace && vtrace->next
	   && (v=vtrace->v)>0 && v<=p->nodecount
	   && (w=vtrace->next->v)>0 && w<=p->nodecount)
	 {
	   if (v!=w) /* eat repeats */
	    {
		b=pR_ptr[w].center;
		fprintf(fp,"%lf %lf lineto\n",b.re,b.im);
	    }
	   vtrace=vtrace->next;
	 } /* end of while */
	fprintf(fp,"closepath\n");
	if (ff)  /* filled? */
		fprintf(fp,"gsave gry   setgray fill grestore\n"); 
		/* ?? need to put color in, if desired */
	fprintf(fp,"stroke\n"); /* draw border */
	return 1;
   } /* end of eucl case */
  else if (p->hes>okerr) 
   {
	if (!(plist=sph_poly_list(p,vertlist))) return 0;
	post_s_convex(fp,plist,ff);
	return 1;
   }
  return 0;
} /* post_any_polygon */

/* ---------------------------------- some technical routines ---------- */
	
char *
get_selection(kill) /* gets current primary selection; remove
highlighting in scratch subwindow if kill true. */
int kill;
{
	static char datastr[BUFSIZE];
	Seln_holder holder;
	Seln_request *buffer;
	holder=seln_inquire(SELN_PRIMARY);
	buffer=seln_ask(&holder, SELN_REQ_CONTENTS_ASCII,0,0);
	strncpy(datastr,buffer->data+sizeof(Seln_attribute),BUFSIZE-10);
	if (kill) textsw_set_selection(scratch_sw,0,0,1);
	return (datastr);
} /* get_selection */

int
stat_file(filename) /* returns 1 if filename exists */
char *filename;
{
	struct stat statbuf;
	return( (strlen(filename)<=NAME_MAX) && (stat(filename, &statbuf)>=0) );
} /* stat_file */
