// datamodifier.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "localdefs.h"
#include "controller.h"
#include "datamodifier.h"
#include "data.h"
#include "editor.h"
#include "formatrequester.h"
#include "lpcdata.h"
#include "processfun.h"
#include "range.h"
#include "request.h"
#include "sound.h"
#include "valuerequester.h"

DataModifier::DataModifier(Data* data) : myData(data), myEditor(nil) {
	myData->ref();
}

DataModifier::DataModifier(Data* data, DataEditor* editor)
		: myData(data), myEditor(editor) {
	myData->ref();
}

DataModifier::~DataModifier() {
	Resource::unref(myData);
}

int
DataModifier::apply() {
	int status = false;
	if(ok()) {
		Application::inform(message());
		status = doApply(myData);
	}
	return status;
}

void
DataModifier::doPostModifyCommands() {
	if(myEditor != nil)
		doEditorCommands(myEditor);
}

int
DataModifier::sampRate() { return myData->sRate(); }

//********

Modifier *
Reverser::create(DataEditor* ed) {
	return new Reverser(ed->currentSelection());
}

int
Reverser::doApply(Data* data) {
	data->reverse();
	return true;
}

Modifier *
Reverser::createUnDo() {
	return new Reverser(target());
}

//********

class NormalizeRequester : public TitledRequester {
	friend Normalizer;
protected:
	NormalizeRequester(Normalizer *n)
		: TitledRequester("Normalize Current Selection:"), client(n) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	Normalizer* client;
};

void
NormalizeRequester::configureRequest(Request *request) {
	request->appendLabel("This will rescale the current selection's amplitudes");
	request->appendLabel("to fit between -1 and 1.");
}

boolean
NormalizeRequester::confirmValues() {
	return (
		(client->target()->dataType() > IntData)
		||
		Application::confirm(
			"Warning:  Performing this operation on non-floating-point samples",
			"will irreversibly destroy the data.",
			"Continue anyway?",
			Cancel
		)
	);
}

//********

Modifier *
Normalizer::create(DataEditor* ed) {
	return new Normalizer(ed->currentSelection());
}

Requester *
Normalizer::createRequester() {
	return new NormalizeRequester(this);
}

int
Normalizer::doApply(Data* data) {
	reverseFactor = data->maxValue();		// store inverse for undo
	data->normalize();
	return true;
}

// undo is only possible when data type is float or double

Modifier *
Normalizer::createUnDo() {
	return (target()->dataType() >= FloatData) ?
		new Scaler(target(), reverseFactor) : nil;
}

//********

class TimeInsertRequester : public TitledRequester {
	friend SpaceInsert;
protected:
	TimeInsertRequester(SpaceInsert *);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	SpaceInsert* client;
	double timeToInsert;
};

TimeInsertRequester::TimeInsertRequester(SpaceInsert* s)
	: TitledRequester("Insert Blank Space:"), client(s), timeToInsert(0) {}

void
TimeInsertRequester::configureRequest(Request* request) {
	request->appendValue("Amount To Insert (seconds):",
	                     &timeToInsert, NonNegativeNumbers);
}

boolean
TimeInsertRequester::confirmValues() {
	client->nFrames = round(timeToInsert * client->sampRate());
	return true;
}

//********

SpaceInsert::SpaceInsert(Data* d, int nframes)
		: DataModifier(d), nFrames(nframes), insertTime(false) {
	initialize();
}

SpaceInsert::SpaceInsert(Data* d, boolean asTime)
	: DataModifier(d), nFrames(1), insertTime(asTime) {}

Modifier *
SpaceInsert::create(DataEditor* ed) {
	return new SpaceInsert(ed->currentSelection());
}

Requester *
SpaceInsert::createRequester() {
	if(insertTime)
		return new TimeInsertRequester(this);
	else
		return new RangedValueRequester<int>(
			"Insert Blank Space:", "Amount To Insert (frames):",
			this->nFrames, NonNegativeIntegers);
}

int
SpaceInsert::doApply(Data *data) {
	Range edit(0, nFrames);
	boolean was = Data::deferRescan(true);	// inserting zeros so no rescan
	int status = data->spliceIn(edit);
	Data::deferRescan(was);
	return status;
}

//********

OutSplicer::OutSplicer(Data* d, const Range& r) : DataModifier(d), region(r) {
	initialize();
}

Modifier *
OutSplicer::create(DataEditor* de) {
	return nil;
}

int
OutSplicer::doApply(Data* data) {
	if(region == data->frameRange()) {
		Application::alert("Entire file may not be spliced out.",
			"Select only a portion of the file.");
		return false;
	}
	else
		return data->spliceOut(region);
}

Modifier *
OutSplicer::createUnDo() {
	return nil;
}

//********

class LengthChangeRequester : public TitledRequester {
	friend class LengthChanger;
protected:
	LengthChangeRequester(const char* title, LengthChanger* l)
		: TitledRequester(title), client(l) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
protected:
	LengthChanger* client;
};

void
LengthChangeRequester::configureRequest(Request* request) {
	Range lengths = PositiveIntegers;
	lengths.set(8, lengths.intMax());	// min length 8
	request->appendValue("New Length (in frames):",
	                     &client->newLength, lengths);
}

boolean
LengthChangeRequester::confirmValues() {
	boolean status = false;
	status = (client->newLength >= client->oldLength ||
		Application::confirm(
			"This operation will destroy any data past",
			"the new file endpoint, and is not reversible.",
			"Continue?",
			Cancel)
	);
	return status;
}

//********

class DurChangeRequester : public LengthChangeRequester {
	friend LengthChanger;
protected:
	DurChangeRequester(const char* title, LengthChanger *);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	double newDuration;
};

DurChangeRequester::DurChangeRequester(const char* title, LengthChanger* client)
		: LengthChangeRequester(title, client) {
	newDuration = double(client->newLength)/client->sampRate();
}

void
DurChangeRequester::configureRequest(Request* request) {
	Range durations = AllNumbers;
	durations.set(8.0/client->sampRate(), durations.max());
	request->appendValue("New Duration (seconds):", &newDuration,
			     durations);
}

boolean
DurChangeRequester::confirmValues() {
	client->newLength = round(newDuration * client->sampRate());
	return LengthChangeRequester::confirmValues();
}

//********

LengthChanger::LengthChanger(Data* d, boolean asDur)
	: DataModifier(d),
	  oldLength(d->length()), newLength(oldLength), asDuration(asDur) {}

LengthChanger::LengthChanger(Data* d, int newlen)
		: DataModifier(d),
		  oldLength(d->length()), newLength(newlen), asDuration(false) {
	initialize();
}

Modifier *
LengthChanger::create(DataEditor* ed) {
	return new LengthChanger(ed->model());
}

Requester *
LengthChanger::createRequester() {
	if(asDuration)
		return new DurChangeRequester("Change File Duration:", this);
	else
		return new LengthChangeRequester("Change File Length:", this);
}

int
LengthChanger::doApply(Data *data) {
	int status = data->changeLength(newLength);
	if(status == true) {
		// defer rescan if increasing length since extending with zeros
		boolean was = Data::deferRescan(newLength >= oldLength);
		data->Notify();		// data does not automatically do this 
		Data::deferRescan(was);
	}
	return status;
}

Modifier *
LengthChanger::createUnDo() {
	return (newLength > oldLength) ?
		new LengthChanger(target(), oldLength) : nil;
}

//********

class SampRateRequester : public TitledRequester {
	friend SampRateChanger;
protected:
	SampRateRequester(SampRateChanger *s)
		: TitledRequester("Change Sampling Rate:"), client(s) {}
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	SampRateChanger* client;
};

void
SampRateRequester::configureRequest(Request* request) {
	request->appendValue("New Sampling Rate:",
	                     &client->newSrate, PositiveIntegers);
	if(client->target()->isA(Sound_Data))
		request->appendChoice(
			"Interpolate sound to preserve pitch level:",
			"|No|Yes|", &client->doInterp, true);
	else client->doInterp = False;
}

boolean
SampRateRequester::confirmValues() {
	if(client->doInterp == Requester::True)
		return Application::confirm("This operation is not reversible.",
			"Continue?");
	else
		return true;
}

//********

SampRateChanger::SampRateChanger(Data* d)
	: DataModifier(d), newSrate(sampRate()), doInterp(Requester::True) {}
		
SampRateChanger::SampRateChanger(Data* d, int newsrate, boolean interp)
		: DataModifier(d), newSrate(newsrate),
		doInterp(interp ? Requester::True : Requester::False) {
	initialize();
}

Modifier *
SampRateChanger::create(DataEditor* ed) {
	return new SampRateChanger(ed->model());
}

Requester *
SampRateChanger::createRequester() {
	return new SampRateRequester(this);
}

int
SampRateChanger::doApply(Data *data) {
	return data->changeSRate(newSrate, doInterp == Requester::True);
}

//********

class FormatChangeRequester : public FormatRequester {
	friend FormatChanger;
protected:
	FormatChangeRequester(FormatChanger *);
	redefined boolean confirmValues();
private:
	FormatChanger* client;
};

FormatChangeRequester::FormatChangeRequester(FormatChanger* f)
	: FormatRequester(nil, "Convert format to:", f->newType),
	  TitledRequester("Change Sound Sample Format:"),
	  client(f) {}

boolean
FormatChangeRequester::confirmValues() {
	boolean status = true;
	if(client->newType == client->oldType) {
		Application::alert("Samples are already in this format.");
		status = false;
	}
	return status;
}

//********

FormatChanger::FormatChanger(Data* d)
		: DataModifier(d), oldType(d->dataType()) {
	newType = (oldType == ShortData) ? FloatData : ShortData;
}
		
FormatChanger::FormatChanger(Data* d, DataType newtype)
		: DataModifier(d), oldType(d->dataType()), newType(newtype) {
	initialize();
}

Modifier *
FormatChanger::create(DataEditor* ed) {
	return new FormatChanger(ed->model());
}

Requester *
FormatChanger::createRequester() {
	return new FormatChangeRequester(this);
}

int
FormatChanger::doApply(Data *data) {
	return ((Sound *) data)->changeDataType(DataType(newType));
}

Modifier *
FormatChanger::createUnDo() {
	return (newType > oldType) ?
		new FormatChanger(target(), DataType(oldType)) : nil;
}

//********

Modifier *
FrameStabilizer::create(DataEditor* ed) {
	return new FrameStabilizer(ed->model());
}

int
FrameStabilizer::applyToLPCData(LPCData* lpc) {
	lpc->stabilizeFrames();
	return true;
}
