(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Franois Pessaux, projet Cristal, INRIA Rocquencourt     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*            Jun Furuse, projet Cristal, INRIA Rocquencourt           *)
(*                                                                     *)
(*  Copyright 1999,2000,2001,2002,2001,2002                            *)
(*  Institut National de Recherche en Informatique et en Automatique.  *)
(*  Distributed only by permission.                                    *)
(*                                                                     *)
(***********************************************************************)

(* Definition of colormaps, i.e. mappings from real RGB colors to
 integers. The integer corresponding to a color [c] is an index [i]
 into a vector of colors whose [i]th element is [c]. *)

exception Too_many_colors;;

type rgb = { mutable r : int; mutable g : int; mutable b : int };;
(* This is copied in Image also *)
 (* R(ed), G(reen), B(lue) representation of colors. *)

type rgba = { color: rgb; mutable alpha: int }
 (* RGB with alpha (transparent) information *)

type cmyk = { mutable c : int; mutable m : int; mutable y : int;
	      mutable k : int }
 (* Cyan Magenta Yellow blacK color model *)
	
type map = { mutable max : int; mutable map : rgb array };;
(* This is copied in Image also *)
(* Colormaps.
   The natural encoding of colors needs triples of integers, each
   component measuring the amount of each of 3 fundamental colors
   in the given color.
   A popular encoding is to consider a color as a point in a 3D space
   with R(ed), G(reen), B(lue) axes. The coordinates of the color is
   its encoding.
   Representing colors as their decomposition in a R(ed),
   G(reen), B(lue) basis is fairly general but is also wasting memory
   space. So colors in data structures recording images are very often
   represented as mere integers: their index into a vector that
   records the colors used in the image at hand. This colormap
   contains the ``true'' RGB denotations of colors, that are stored
   only once for each color. *)

(* Color manipulation functions. *)

val rgb_square_distance : rgb -> rgb -> int;;
    (* Compute the distance between two colours. *)

(* Colormap manipulation functions *)

val size : map -> int;;
    (* Returns the size of a colormap. *)

val plus : rgb -> rgb -> rgb;;
val minus : rgb -> rgb -> rgb;;

val find_exact : map -> rgb -> int;;
    (* Finds a color in the colormap and returns its color index.
       Raises exception [Not_found] if the color is not in the colormap. *)

val find_nearest : map -> rgb -> int;;
    (* [find_nearest m c] finds the color [co] that is the nearest to
    [c] color into the colormap [m].
    Returns the color index of [co] and the distance between [c]
    and [co]. *)

val add_color : map -> rgb -> int;;
    (* Add a new color into the given colormap and return its index.
       If the color is already in the colormap, it is not
       added again, and the corresponding color index is returned. *)

val add_colors : map -> rgb list -> int list;;
    (* Add the list of new colors into the given colormap and return 
       their indices. If a color is already in the colormap, it is not
       added again, and the corresponding color index is returned. *)

val copy : map -> map
    (* Copy a colormap *)

val brightness : rgb -> int

(********************************************************* Color name parser *)

val color_parse : string -> rgb
val colormap_parse : string array -> rgb array * int
