/*
    Eiciel - GNOME editor of ACL file permissions.
    Copyright (C) 2004-2005 Roger Ferrer Ibez

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "config.hpp"
#include <iostream>
#include <pwd.h>
#include <grp.h>
#include "eiciel_main_controler.hpp"
#include "tipus_element_acl.hpp"
#include <gettext.h>


using namespace std;

EicielMainControler::EicielMainControler() 
	: gestorACL(NULL), hiHaFitxerObert(false), 
	ultimMissatgeError(""), calActualitzarLlista(true), 
	mostrarSystem(false)
{
	generarLlistes();
}

EicielMainControler::~EicielMainControler()
{
	delete gestorACL;
}

void EicielMainControler::generarLlistes()
{
	if (!calActualitzarLlista) 
		return;
	
	llistatUsuaris.clear();
	struct passwd* u;
	setpwent();
	while((u = getpwent()) != NULL)
	{
		if (mostrarSystem || (u->pw_uid >= 1000))
		{
			llistatUsuaris.insert(u->pw_name);
		}
	}
	endpwent();

    // Creem la llista de grups
	llistatGrups.clear();
	struct group* g;
	setgrent();
	while ((g = getgrent()) != NULL)
	{
		if (mostrarSystem || (g->gr_gid >= 1000))
		{
			llistatGrups.insert(g->gr_name);
		}
	}
	endgrent();

	calActualitzarLlista = false;
}

void EicielMainControler::mostrarParticipantsSistema(bool b)
{
	if (b != mostrarSystem)
	{
		mostrarSystem = b;
		calActualitzarLlista = true;
	}
}

void EicielMainControler::obreFitxer(string s)
{
    try
    {
        GestorACL* nouGestor;
        nouGestor = new GestorACL(s);

        delete gestorACL;
        gestorACL = nouGestor;
        actualitzarLlistaACL();
        finestra->establirNomFitxer(s);
		finestra->activacio(true);

		// Aixo comprova si hom pot editar el fitxer
		comprovarEditable();

		hiHaFitxerObert = true;
    }
    catch (GestorACLException e)
    {
		hiHaFitxerObert = false;
		finestra->activacio(false);
		finestra->buidaLlistaACL();
		ultimMissatgeError = e.getMessage();
    }
}

void EicielMainControler::actualitzarLlistaACL()
{
	actualitzantFinestra = true;
	finestra->buidaLlistaACL();
	// L'usuari
	permisos_t perms = gestorACL->donaUsuari();
	bool mostrarAdmiracio(false);
	
	permisos_t permisosEfectius(7);
	if (gestorACL->teMascara())
	{
		permisosEfectius = gestorACL->donaMascara();
	}
	
	vector<entrada_acl> vACL;
	finestra->afegeixNoSeleccionable(Glib::locale_to_utf8(gestorACL->donaNomPropietari()), perms.lectura,
			perms.escriptura, perms.execucio, USUARI);
	vACL = gestorACL->donaACLAccessUsuari();
	for (vector<entrada_acl>::iterator i = vACL.begin(); 
			i != vACL.end(); i++)
	{
		finestra->afegeixSeleccionable(Glib::locale_to_utf8(i->nom), i->lectura, 
				i->escriptura, i->execucio, ACL_USUARI,
				permisosEfectius.lectura,
				permisosEfectius.escriptura,
				permisosEfectius.execucio);
		mostrarAdmiracio |= (!permisosEfectius.lectura && i->lectura) ||
			(!permisosEfectius.escriptura && i->escriptura) ||
			(!permisosEfectius.execucio && i->execucio);
	}

	perms = gestorACL->donaGrup();
	finestra->afegeixNoSeleccionable(Glib::locale_to_utf8(gestorACL->donaNomGrup()), perms.lectura,
			perms.escriptura, perms.execucio, GRUP,
			permisosEfectius.lectura,
			permisosEfectius.escriptura,
			permisosEfectius.execucio);
	mostrarAdmiracio |= (!permisosEfectius.lectura && perms.lectura) ||
		(!permisosEfectius.escriptura && perms.escriptura) ||
		(!permisosEfectius.execucio && perms.execucio);

	vACL = gestorACL->donaACLAccessGrup();
	for (vector<entrada_acl>::iterator i = vACL.begin(); 
			i != vACL.end(); i++)
	{
		finestra->afegeixSeleccionable(Glib::locale_to_utf8(i->nom), i->lectura, 
				i->escriptura, i->execucio, ACL_GRUP,
				permisosEfectius.lectura,
				permisosEfectius.escriptura,
				permisosEfectius.execucio);
		mostrarAdmiracio |= (!permisosEfectius.lectura && i->lectura) ||
			(!permisosEfectius.escriptura && i->escriptura) ||
			(!permisosEfectius.execucio && i->execucio);
	}

	if (gestorACL->teMascara())
	{
		perms = gestorACL->donaMascara();
		finestra->afegeixNoSeleccionable(_("Mask"), perms.lectura,
				perms.escriptura, perms.execucio, MASCARA);
	}

	perms = gestorACL->donaAltres();
	finestra->afegeixNoSeleccionable(_("Other"), perms.lectura,
			perms.escriptura, perms.execucio, ALTRES);

	finestra->activarBotoDefaultACL(gestorACL->esDirectori());
	finestra->hiHaDefaultACL(false);

	if (gestorACL->esDirectori())
	{
		bool hiHaACLDefault = false;
		permisos_t permisosDefaultEfectius(7);
		if (gestorACL->teDefaultMascara())
		{
			permisosDefaultEfectius = gestorACL->donaDefaultMascara();
		}

		if (gestorACL->teDefaultUsuari())
		{
			perms = gestorACL->donaDefaultUsuari();
			finestra->afegeixNoSeleccionable(Glib::locale_to_utf8(gestorACL->donaNomPropietari()), perms.lectura,
					perms.escriptura, perms.execucio, DEFAULT_USUARI);
			hiHaACLDefault = true;
		}

		vACL = gestorACL->donaACLDefaultUsuari();

		hiHaACLDefault |= (vACL.size() > 0);
		for (vector<entrada_acl>::iterator i = vACL.begin(); 
				i != vACL.end(); i++)
		{
			finestra->afegeixSeleccionable(Glib::locale_to_utf8(i->nom), i->lectura, 
					i->escriptura, i->execucio, DEFAULT_ACL_USUARI,
					permisosDefaultEfectius.lectura,
					permisosDefaultEfectius.escriptura,
					permisosDefaultEfectius.execucio);
			mostrarAdmiracio |= (!permisosDefaultEfectius.lectura && i->lectura) ||
				(!permisosDefaultEfectius.escriptura && i->escriptura) ||
				(!permisosDefaultEfectius.execucio && i->execucio);
		}


		if (gestorACL->teDefaultGrup())
		{
			perms = gestorACL->donaDefaultGrup();
			finestra->afegeixNoSeleccionable(Glib::locale_to_utf8(gestorACL->donaNomGrup()), perms.lectura,
					perms.escriptura, perms.execucio, DEFAULT_GRUP,
					permisosDefaultEfectius.lectura,
					permisosDefaultEfectius.escriptura,
					permisosDefaultEfectius.execucio);
			mostrarAdmiracio |= (!permisosDefaultEfectius.lectura && perms.lectura) ||
				(!permisosDefaultEfectius.escriptura && perms.escriptura) ||
				(!permisosDefaultEfectius.execucio && perms.execucio);
			hiHaACLDefault |= true;
		}

		vACL = gestorACL->donaACLDefaultGrup();

		hiHaACLDefault |= (vACL.size() > 0);
		for (vector<entrada_acl>::iterator i = vACL.begin(); 
				i != vACL.end(); i++)
		{
			finestra->afegeixSeleccionable(Glib::locale_to_utf8(i->nom), i->lectura, 
					i->escriptura, i->execucio, DEFAULT_ACL_GRUP,
					permisosDefaultEfectius.lectura,
					permisosDefaultEfectius.escriptura,
					permisosDefaultEfectius.execucio);
			mostrarAdmiracio |= (!permisosDefaultEfectius.lectura && i->lectura) ||
				(!permisosDefaultEfectius.escriptura && i->escriptura) ||
				(!permisosDefaultEfectius.execucio && i->execucio);
		}

		if (gestorACL->teDefaultMascara())
		{
			perms = gestorACL->donaDefaultMascara();
			finestra->afegeixNoSeleccionable(_("Default Mask"), perms.lectura,
					perms.escriptura, perms.execucio, DEFAULT_MASCARA);
			hiHaACLDefault |= true;
		}

		if (gestorACL->teDefaultAltres())
		{
			perms = gestorACL->donaDefaultAltres();
			finestra->afegeixNoSeleccionable(_("Default Other"), perms.lectura,
					perms.escriptura, perms.execucio, DEFAULT_ALTRES);
			hiHaACLDefault |= true;
		}
		finestra->hiHaDefaultACL(hiHaACLDefault);
	}

	finestra->mostrarAdmiracio(mostrarAdmiracio);
	
	actualitzantFinestra = false;
}

void EicielMainControler::afegirEntradaACL(string s, TipusElement e, bool esDefault)
{
	permisos_t p(7);

	if (esDefault)
	{
		if (e == ACL_USUARI)
		{
			e = DEFAULT_ACL_USUARI;
		}
		else if (e == ACL_GRUP)
		{
			e = DEFAULT_ACL_GRUP;
		}
	}

	try
	{
		switch(e)
		{
			case ACL_USUARI :
				{
					gestorACL->modificarACLUsuari(s, p);
					break;
				}
			case ACL_GRUP :
				{
					gestorACL->modificarACLGrup(s, p);
					break;
				}
			case DEFAULT_ACL_USUARI :
				{
					gestorACL->modificarDefaultACLUsuari(s, p);
					break;
				}
			case DEFAULT_ACL_GRUP :
				{
					gestorACL->modificarDefaultACLGrup(s, p);
					break;
				}
			default:
				break;
		}

		actualitzarLlistaACL();

		finestra->seleccionarACL(s, e);
	} 
	catch (GestorACLException e)
	{
		// cout << "Error afegir: " << e.getMessage() << endl;
		Glib::ustring s = _("Could not add ACL entry: ") + e.getMessage();
		Gtk::MessageDialog afegirACL(s, false,
				Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK);
		afegirACL.run();
		ultimMissatgeError = s;
		return;
	}
}

void EicielMainControler::eliminarACL(string nomEntrada, TipusElement e)
{
	bool actualitzat = true;
	try
	{
		switch(e)
		{
			case ACL_USUARI :
				{
					gestorACL->eliminarACLUsuari(nomEntrada);
					break;
				}
			case ACL_GRUP :
				{
					gestorACL->eliminarACLGrup(nomEntrada);
					break;
				}
			case DEFAULT_ACL_USUARI :
				{
					gestorACL->eliminarDefaultACLUsuari(nomEntrada);
					break;
				}
			case DEFAULT_ACL_GRUP :
				{
					gestorACL->eliminarDefaultACLGrup(nomEntrada);
					break;
				}
			default:
				actualitzat = false;
				break;
		}

		if (actualitzat)
		{
			actualitzarLlistaACL();
		}
	}
	catch(GestorACLException e)
	{
		Glib::ustring s = _("Could not remove ACL entry: ") + e.getMessage();
		Gtk::MessageDialog borrarACL(s, false,
				Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK);
		borrarACL.run();
		ultimMissatgeError = s;
	}
}

void EicielMainControler::actualitzaEntradaACL(TipusElement e, string s, 
		bool lectura, bool escriptura, bool execucio)
{
	permisos_t p(lectura, escriptura, execucio);

	try
	{
		switch(e)
		{
			case MASCARA :
				{
					gestorACL->modificarMascara(p);
					break;
				}
			case DEFAULT_MASCARA :
				{
					gestorACL->modificarMascaraDefault(p);
					break;
				}
			case USUARI :
				{
					gestorACL->modificarPermisosPropietari(p);
					break;
				}
			case GRUP :
				{
					gestorACL->modificarPermisosGrup(p);
					break;
				}
			case ALTRES :
				{
					gestorACL->modificarPermisosAltres(p);
					break;
				}
			case DEFAULT_USUARI :
				{
					gestorACL->modificarPermisosDefaultPropietari(p);
					break;
				}
			case DEFAULT_GRUP :
				{
					gestorACL->modificarPermisosDefaultGrup(p);
					break;
				}
			case DEFAULT_ALTRES :
				{
					gestorACL->modificarPermisosDefaultAltres(p);
					break;
				}
			case ACL_USUARI :
				{
					gestorACL->modificarACLUsuari(s, p);
					break;
				}
			case ACL_GRUP :
				{
					gestorACL->modificarACLGrup(s, p);
					break;
				}
			case DEFAULT_ACL_USUARI :
				{
					gestorACL->modificarDefaultACLUsuari(s, p);
					break;
				}
			case DEFAULT_ACL_GRUP :
				{
					gestorACL->modificarDefaultACLGrup(s, p);
					break;
				}
			default:
				break;
		}
		actualitzarLlistaACL();
	}
	catch (GestorACLException e)
	{
		Glib::ustring s = _("Could not modify ACL entry: ") + e.getMessage();
		Gtk::MessageDialog modificarACL(s, false,
				Gtk::MESSAGE_ERROR, Gtk::BUTTONS_OK);
		modificarACL.run();
		ultimMissatgeError = s;
	}
}

void EicielMainControler::canviACLDefault()
{
	if (actualitzantFinestra) return;

	try
	{
		if (!finestra->donaACLDefault())
		{
			Glib::ustring s(_("Are you sure you want to remove all ACL default entries?"));
			Gtk::MessageDialog borrarACL(s, false,
					Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_YES_NO);
			int resultat = borrarACL.run();
			if (resultat == Gtk::RESPONSE_YES)
			{
				gestorACL->buidarACLDefecte();
			}
		}
		else
		{
			gestorACL->creaACLDefecte();
		}
		actualitzarLlistaACL();
	}
	catch (GestorACLException e)
	{
		ultimMissatgeError = e.getMessage();
	}
}

// La grcia s q retornin la diferncia
// entre el "working" set i la llista real
// pero per sortir del pas est b ja xD

set<string> EicielMainControler::donaLlistaUsuaris()
{
	generarLlistes();
    return llistatUsuaris;
}

set<string> EicielMainControler::donaLlistaGrups()
{
	generarLlistes();
    return llistatGrups;
}

bool EicielMainControler::fitxerObert()
{
	return hiHaFitxerObert;
}

void EicielMainControler::comprovarEditable()
{
	/*
	 * A Linux hauriem de comprovar la capability CAP_FOWNER pero a FreeBSD
	 * aixo no existeix per tant farem una comprovaci inexacta basada en
	 * ownership || root
	 */
	uid_t real_user = getuid();
	if ((real_user != 0) && (real_user != gestorACL->donaUidPropietari()))
	{
		finestra->establirNomesLectura(true);
	}
	else
	{
		finestra->establirNomesLectura(false);
	}
}

Glib::ustring EicielMainControler::ultimError()
{
	return ultimMissatgeError;
}
