/*
 * Hydrogen
 * Copyright(c) 2002-2004 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 *
 * http://hydrogen.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: JackDriver.cpp,v 1.3 2004/07/19 09:46:46 comix Exp $
 *
 */

#include "JackDriver.h"
#ifdef JACK_SUPPORT

#include "../Hydrogen.h"

unsigned long jack_server_sampleRate = 0;
jack_nframes_t jack_server_bufferSize = 0;
JackDriver *jackDriverInstance = NULL;

int jackDriverSampleRate(jack_nframes_t nframes, void *arg) {
	char msg[100];
	sprintf( msg, "Jack SampleRate changed: the sample rate is now %d/sec", (int)nframes );
	jackDriverInstance->infoLog(msg);
	jack_server_sampleRate = nframes;
	return 0;
}




void jackDriverShutdown(void *arg) {
//	jackDriverInstance->deactivate();
	jackDriverInstance->client = NULL;
	( Hydrogen::getInstance() )->raiseError( Hydrogen::JACK_SERVER_SHUTDOWN );
}




JackDriver::JackDriver(JackProcessCallback processCallback) : GenericDriver( "JackDriver" )
{
	infoLog( "INIT" );
	jackDriverInstance = this;
	track_out_flag = true;
	this->processCallback = processCallback;

}





JackDriver::~JackDriver()
{
	infoLog( "DESTROY" );
	disconnect();
}






// return 0: ok
// return 1: cannot activate client
// return 2: cannot connect output port
// return 3: Jack server not running
// return 4: output port = NULL
int JackDriver::connect()
{
	infoLog("[connect]");

	if (jack_activate (client)) {
		( Hydrogen::getInstance() )->raiseError( Hydrogen::JACK_CANNOT_ACTIVATE_CLIENT );
		return 1;
	}

	if ( connect_out_flag ) {
	// connect the ports
		if ( jack_connect ( client, jack_port_name( output_port_1 ), output_port_name_1.c_str() ) ) {
			( Hydrogen::getInstance() )->raiseError( Hydrogen::JACK_CANNOT_CONNECT_OUTPUT_PORT );
			return 2;
		}
		if ( jack_connect ( client, jack_port_name( output_port_2 ), output_port_name_2.c_str() ) ) {
			( Hydrogen::getInstance() )->raiseError( Hydrogen::JACK_CANNOT_CONNECT_OUTPUT_PORT );
			return 2;
		}
	}

	return 0;
}





void JackDriver::disconnect()
{
	infoLog( "[disconnect]" );

	deactivate();
	jack_client_t *oldClient = client;
	client = NULL;
	if ( oldClient ) {
		infoLog( "calling jack_client_close" );
		int res = jack_client_close( oldClient );
		if ( res ) {
			errorLog("Error in jack_client_close");
			// FIXME: raise exception
		}
	}
	client = NULL;
}




void JackDriver::deactivate()
{
	infoLog( "[deactivate]" );
	if ( client ) {
		infoLog( "jack_deactivate" );
		int res = jack_deactivate( client );
		if ( res ) {
			errorLog("Error in jack_deactivate");
		}
	}
}




uint JackDriver::getBufferSize()
{
	return jack_server_bufferSize;
}



uint JackDriver::getSampleRate()
{
	return jack_server_sampleRate;
}




void JackDriver::updateTransportInfo()
{
	if (Preferences::getInstance()->getJackTransportMode() ==  Preferences::USE_JACK_TRANSPORT) {
		m_transportState = jack_transport_query( client, &m_transportPos );
		// update m_transport with jack-transport data
		switch ( m_transportState ) {
		case JackTransportStopped:
			m_transport.m_status = TransportInfo::STOPPED;
			break;

		case JackTransportRolling:
			m_transport.m_status = TransportInfo::ROLLING;
			break;

		case JackTransportStarting:
			m_transport.m_status = TransportInfo::STOPPED;
			break;

		default:
			errorLog( "[updateTransportInfo] Unknown jack transport state" );
		}


		// FIXME
		// TickSize and BPMi
		if ( m_transportPos.valid & JackPositionBBT ) {
//			if ( m_transport.m_nBPM != m_transportPos.beats_per_minute ) {
				m_transport.m_nBPM = m_transportPos.beats_per_minute;
				(Hydrogen::getInstance())->getSong()->setBpm( m_transportPos.beats_per_minute );
//				warningLog( "[updateTransportInfo] new bpm from jack-transport = " + toString(m_transportPos.beats_per_minute) );
//			}
//		double resolution = m_pSong->getResolution();
//		double sampleRate = (float)m_pAudioDriver->getSampleRate();
//		m_transport->m_nTickSize = (uint)( sampleRate * 60.0 /  bpm / resolution );
		}

		m_transport.m_nFrames = m_transportPos.frame;
	}
	else {
//		if (m_transport.m_status == TransportInfo::ROLLING) {
//			m_transport.m_nFrames += jack_server_bufferSize;
//		}

	}

}



float* JackDriver::getOut_L()
{
	jack_default_audio_sample_t *out = (jack_default_audio_sample_t *) jack_port_get_buffer (output_port_1, jack_server_bufferSize);
	return out;
}

float* JackDriver::getOut_R()
{
	jack_default_audio_sample_t *out = (jack_default_audio_sample_t *) jack_port_get_buffer (output_port_2, jack_server_bufferSize);
	return out;
}



float* JackDriver::getTrackOut_L(uint nTrack)
{
	jack_default_audio_sample_t *out = (jack_default_audio_sample_t *) jack_port_get_buffer (track_output_ports_L[nTrack], jack_server_bufferSize);
	return out;
}

float* JackDriver::getTrackOut_R(uint nTrack)
{
	jack_default_audio_sample_t *out = (jack_default_audio_sample_t *) jack_port_get_buffer (track_output_ports_R[nTrack], jack_server_bufferSize);
	return out;
}


int JackDriver::init(uint nBufferSize)
{
	output_port_name_1 = ( Preferences::getInstance() )->getJackPortName1();
	output_port_name_2 = ( Preferences::getInstance() )->getJackPortName2();


/*
	// check if another hydrogen instance is connected to jack
	infoLog("creating temp port");
	if ( (client = jack_client_new( "hydrogen-tmp" ) ) == 0 ) {
		errorLog("Jack server not running?");
		return 3;
	}
	const char **readports = jack_get_ports( client, NULL, NULL, JackPortIsOutput );
	int i = 0;
	int c = 0;
	while (readports && readports[i]) {
		string sPort = string(readports[i]);
		int nPos = sPort.find( "Hydrogen" );
		if (nPos) {
			infoLog( "non trovata" );
		}
		else {
			infoLog( "Trovata" );
		}
		i++;
	}
	infoLog("closing temp port");
	jack_client_close( client );
	infoLog("closed temp port");
*/

	static uint nInstance = 0;
	nInstance++;

	char clientName[50];
//	sprintf(clientName, "Hydrogen-%d", nInstance);
	sprintf(clientName, "Hydrogen-%d", getpid());

	/* try to become a client of the JACK server */
	if ( ( client = jack_client_new( clientName ) ) == 0 ) {
		errorLog( "[JackDriver::init] Jack server not running? (jack_client_new)" );
		return 3;
	}

	jack_server_sampleRate = jack_get_sample_rate (client);
	jack_server_bufferSize = jack_get_buffer_size (client);


	/* tell the JACK server to call `process()' whenever
	   there is work to be done.
	*/
	jack_set_process_callback (client, this->processCallback, 0);


	/* tell the JACK server to call `srate()' whenever
	   the sample rate of the system changes.
	*/
	jack_set_sample_rate_callback (client, jackDriverSampleRate, 0);


	/* tell the JACK server to call `jack_shutdown()' if
	   it ever shuts down, either entirely, or if it
	   just decides to stop calling us.
	*/
	jack_on_shutdown (client, jackDriverShutdown, 0);


	/* create two ports */
	output_port_1 = jack_port_register ( client, "out_L", JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0 );
	output_port_2 = jack_port_register ( client, "out_R", JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0 );

	if ( (output_port_1 == NULL) || ( output_port_2 == NULL ) ) {
		( Hydrogen::getInstance() )->raiseError( Hydrogen::JACK_ERROR_IN_PORT_REGISTER );
		return 4;
	}


	/* create MAX_INSTRUMENTS dedicated channel output ports */
	if (track_out_flag) {
		char tmpbuf[255];

		for (unsigned int n=0; n < MAX_INSTRUMENTS; ++n) {
			snprintf (tmpbuf, sizeof(tmpbuf), "track_out_%d_L", n+1);
			track_output_ports_L[n] = jack_port_register ( client, tmpbuf, JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0 );
			if (track_output_ports_L[n] == NULL) {
				( Hydrogen::getInstance() )->raiseError( Hydrogen::JACK_ERROR_IN_PORT_REGISTER );
			}

			snprintf (tmpbuf, sizeof(tmpbuf), "track_out_%d_R", n+1);
			track_output_ports_R[n] = jack_port_register ( client, tmpbuf, JACK_DEFAULT_AUDIO_TYPE, JackPortIsOutput, 0 );
			if (track_output_ports_R[n] == NULL) {
				( Hydrogen::getInstance() )->raiseError( Hydrogen::JACK_ERROR_IN_PORT_REGISTER );
			}
		}
	}

	// clear buffers
//	jack_default_audio_sample_t *out_L = (jack_default_audio_sample_t *) jack_port_get_buffer (output_port_1, jack_server_bufferSize);
//	jack_default_audio_sample_t *out_R = (jack_default_audio_sample_t *) jack_port_get_buffer (output_port_2, jack_server_bufferSize);
//	memset( out_L, 0, nBufferSize * sizeof( float ) );
//	memset( out_R, 0, nBufferSize * sizeof( float ) );


	return 0;
}



void JackDriver::play()
{
	if (( Preferences::getInstance() )->getJackTransportMode() ==  Preferences::USE_JACK_TRANSPORT) {
		if (client) {
			infoLog( "jack_transport_start()" );
			jack_transport_start( client );
		}
	}
	else {
		m_transport.m_status = TransportInfo::ROLLING;
	}
}



void JackDriver::stop()
{
	if (( Preferences::getInstance() )->getJackTransportMode() ==  Preferences::USE_JACK_TRANSPORT) {
		if (client) {
			infoLog( "jack_transport_stop()" );
			jack_transport_stop( client );
		}
	}
	else {
		m_transport.m_status = TransportInfo::STOPPED;
	}
}



void JackDriver::locate( unsigned long nFrame )
{
	if (( Preferences::getInstance() )->getJackTransportMode() ==  Preferences::USE_JACK_TRANSPORT) {
		if (client) {
			infoLog( "jack_transport_locate(" + toString(nFrame) + ")" );
			jack_transport_locate( client, nFrame );
		}
	}
	else {
		m_transport.m_nFrames = nFrame;
	}
}



void JackDriver::setBpm(float fBPM)
{
	infoLog( "[setBpm] " + toString(fBPM) );
	m_transport.m_nBPM = fBPM;
}



#endif // JACK_SUPPORT


