/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IDATAREADER_H
#define IDATAREADER_H


#include "iextendableobject.h"


#include "iarray.h"

class iBasicParticlesFileLoader;
class iDataInfo;
class iDataLimits;
class iDataReaderHelper;
class iDataSubject;
class iDataType;
class iDirectory;
class iFile;
class iFileLoader;
class iProgressEventObserver;
class iUniformScalarsFileLoader;
class iUniformVectorsFileLoader;
class iUniformTensorsFileLoader;

class vtkDataSet;
class vtkPolyData;
class vtkStructuredPoints;


class iDataReader : public iExtendableObject
{

	friend class iDataReaderHelper;

	IPOINTER_AS_PART(ViewModule);

public:

	struct VisitedFile
	{
		iString Name;
		const iFileLoader *Loader;
	};

	vtkTypeMacro(iDataReader,iExtendableObject);
	static iDataReader* New(iViewModule *vm = 0);
	static const iObjectType& Type();
	
	//
	//  Non-static members
	//
	IOBJECT_DECLARE_GETSET(ScaledDimension,mScaledDim,int);
	//virtual void SetScaledDimension(int v);
	//inline int GetScaledDimension() const { return mScaledDim; }

	IOBJECT_DECLARE_GETSET1(VoxelLocation,int);
	//virtual void SetVoxelLocation(int s);
	//inline bool GetVoxelLocation() const { return mVoxelLocation; }

	IOBJECT_DECLARE_GETSET1(BoundaryConditions,int);
	//virtual void SetBoundaryConditions(int s);
	//inline int GetBoundaryConditions() const { return mBoundaryConditions; }

	void SetDataShift(int d, double f); 
	inline double GetDataShift(int d) const { if(d>=0 && d<3) return mDataShift[d]; else return 0.0; }
	static const iObjectKey& KeyShiftData();

	//
	//  Special "action" keys
	//
	static const iObjectKey& KeyEraseData(); 
	static const iObjectKey& KeyLoadData(); 

	//
	//  Special "query" keys 
	//
	static const iObjectKey& KeyErrorMessage();

	inline bool IsSet() const { return mSetLoaders.Size() > 0; }
	static const iObjectKey& KeyIsSet();

	//
	//  Loading of data
	//
	void LoadFile(const iDataType &type, const iString &fname, bool dryrun = false, bool savename = true);
    void LoadRecord(int rec, int skip = 0, bool dryrun = false, bool savename = false);

	//
	//  Operation on data
	//
	bool IsThereData(const iDataType &type) const;
	bool IsThereData(const iDataInfo &info) const;

	void EraseData(const iDataType &type);
	void ReloadData(const iDataType &type);

	void SetTwoCopies(bool s);

	void ResetPipeline();

	iDataSubject* GetSubject(const iDataType &type) const;
	iDataLimits* GetLimits(const iDataType &type) const;

	//
	//  Generic GetData
	//
	vtkDataSet* GetData(const iDataType &type) const;

	inline bool IsFileAnimatable() const { return (mCurrentRecord >= 0); }
	inline int GetRecordNumber() const { return mCurrentRecord; }

	virtual void ShiftData(double *dx = 0);

	virtual float GetMemorySize() const;
	
	//
	//  Functions for file manipulation
	//
	const iString& GetLastFileSetName() const;
	bool IsAnotherSetLeader(const iString &fname) const;
	bool IsCurrentSetMember(const iString &fname) const;
	const iString GetFileSetName(int rec) const;
	const iDataType& GetFileSetDataType(int member = 0) const;

	const iString& GetLastFileName(const iDataType &type) const;
	inline const iString& GetLastFileName() const { return mLastFileName; }
	inline const iString& GetLastAttemptedFileName() const { return mLastAttemptedFileName; }

	inline const iArray<VisitedFile>& GetVisitedFilesList() const { return mVisitedFilesList; }

	virtual iProgressEventObserver* GetProgressEventObserver(const iDataType &type);

	virtual void PackCompleteState(iString &s) const;
	virtual void UnPackCompleteState(const iString &s);

	//
	//  Hidden (undocumented) keys
	//
	static const iObjectKey& KeyRecordLength(); 
	inline int GetRecordLength() const { return mRecordLength; }

protected:

	virtual ~iDataReader();

	virtual void ExtendableObjectPackStateBody(iString &s) const;
	virtual void ExtendableObjectUnPackStateBody(const iString &s);

private:
	
	iDataReader(iViewModule *vm);

	void LoadFile(iFileLoader *loader, const iString &fname, bool dryrun, bool savename);

	void InsertIntoVisitedFilesList(const iFileLoader *loader, const iString &name);
	void BreakSet();
	bool IsAborted(const iDataType &type) const;
	bool IsSetLeader(const iFileLoader *loader) const;

	bool mTwoCopies, mInExtensionUpdate;
	int mCellToPointMode;

	double mDataShift[3];
	iString mMode;
	int mScaledDim, mVoxelLocation, mBoundaryConditions;

	iFileLoader *mPeriodicityLeader;
	iSearchableArray<iFileLoader*> mSetLoaders;
	iArray<iFileLoader*> mLoaders;
	iUniformScalarsFileLoader *mUniformScalarsLoader;
	iUniformVectorsFileLoader *mUniformVectorsLoader;
	iUniformTensorsFileLoader *mUniformTensorsLoader;
	iBasicParticlesFileLoader *mBasicParticlesLoader;

	int mCurrentRecord, mRecordLength;

	iString mLastFileName, mLastAttemptedFileName;
	iArray<VisitedFile> mVisitedFilesList;
	iDirectory *mDirectory;
};


class iDataReaderExtension : public iObjectExtension
{

	IOBJECTEXTENSION_DECLARE_ABSTRACT(iDataReader);

public:

	inline iDataReader* GetReader() const { return mRealOwner; }

	virtual iFileLoader* GetLoader(int n) const = 0;

	//
	//  It is a good idea to keep track of the last file read.
	//
	inline const iString& GetLastFileName() const { return mLastFileName; }

	void AfterLoadFile(iFileLoader *loader, const iString &fname, bool dryrun);

protected:

	virtual void AfterLoadFileBody(iFileLoader *loader, const iString &fname, bool dryrun);

	iString mLastFileName;
};

#endif

