/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#ifndef IDATASTRETCH_H
#define IDATASTRETCH_H


#include "imath.h"
#include "iparameter.h"

class iString;


namespace iDataStretch
{
	template<class T> T ApplyStretch(T f, int mode, bool up);
	template<class T> T ResetStretch(T f, int mode);

	const iString& GetStretchNames();
};


class iDataStretchUser
{

public:

	void SetStretch(int s);
	inline int GetStretch() const { return mStretch; }

	float ApplyStretch(float f, bool up) const;
	float ResetStretch(float f) const;

	double ApplyStretch(double f, bool up) const;
	double ResetStretch(double f) const;

protected:
	
	iDataStretchUser();
	virtual ~iDataStretchUser();

	virtual void OnStretchChanged(){}

private:

	int mStretch; // use Set/GetStretch for access
};


inline float iDataStretchUser::ApplyStretch(float f, bool up) const
{
	return iDataStretch::ApplyStretch(f,mStretch,up);
}


inline float iDataStretchUser::ResetStretch(float f) const
{
	return iDataStretch::ResetStretch(f,mStretch);
}


inline double iDataStretchUser::ApplyStretch(double f, bool up) const
{
	return iDataStretch::ApplyStretch(f,mStretch,up);
}


inline double iDataStretchUser::ResetStretch(double f) const
{
	return iDataStretch::ResetStretch(f,mStretch);
}


//
//  stretch and restore the data
//
namespace iDataStretch
{
	template<class T>
	inline T ApplyStretch(T f, int mode, bool up)
	{
		const T zero = (T)0.0;
		const T tiny = (T)1.0e-36;
		const T invalid = (T)36.0;

		switch(mode)
		{
		case iParameter::_StretchLog:
			{
				if(f > zero) 
				{
					return log10(tiny+f);
				}
				else 
				{
					return (up ? invalid : -invalid);
				}
				break;
			}
		default: return f;
		}
	}


	template<class T>
	inline T ResetStretch(T f, int mode)
	{
		switch(mode)
		{
		case iParameter::_StretchLog: 
			{
				return pow10(f);
				break;
			}
		default: return f;
		}
	}
};

#endif // IDATASTRETCH_H

