/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iparticlefileloader.h"


#include "ibuffer.h"
#include "icommoneventobservers.h"
#include "idatalimits.h"
#include "idatasubject.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "iparallel.h"
#include "iparallelmanager.h"
#include "iparallelworker.h"
#include "iparticledensityestimator.h"
#include "iviewmodule.h"

#include <vtkCellArray.h>
#include <vtkDoubleArray.h>
#include <vtkFloatArray.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>

//
//  Templates
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"


//
//  Helper classes for parallel execution & templates
//
class iParticleHelper : protected iParallelWorker
{

public:

	iParticleHelper(iParticleFileLoader *loader);

	void ShiftData(bool paf, int dim, long n, bool per[3], float dr, float *pf, double *pd);
	void FindRange(int dim, long n, float *pf, float *amin, float *amax);
	void AddIndexAsAttribute(int att, int nc, long n, float *pf);

	template<class T, class D>
	bool ReadFortranRecordWithMaskTemplate(iFile &F, int dest, int inNum, int outOffset, float updateStart, float updateDuration, T *scale, T *offset);

	template<class T>
	void AutoScalePositions();

protected:

	virtual int ExecuteStep(int step, iParallel::ProcessorInfo &p);

	iParticleFileLoader *mLoader;

	iBuffer<float> mMin, mMax;

	bool *mPeriodic;
	float *mFarr;
	double *mDarr;

	int mDim, mCom, mNumProcs;
	bool mPaf;
	long mTot;
	float mDr;
};


//
//  Main class
//
iParticleFileLoader::iParticleFileLoader(iDataReader *r, int priority) : iFileLoader(r,priority)
{
	this->Define();
}


iParticleFileLoader::iParticleFileLoader(iDataReaderExtension *ext, int priority) : iFileLoader(ext,priority)
{
	this->Define();
}


void iParticleFileLoader::Define()
{
	mHaveNormals = false;

	mHelper = new iParticleHelper(this); IERROR_ASSERT(mHelper);
	mDensityEstimator = iParticleDensityEstimator::New(this); IERROR_ASSERT(mDensityEstimator);
	mDensityEstimator->AddObserver(vtkCommand::StartEvent,this->GetObserver());
	mDensityEstimator->AddObserver(vtkCommand::ProgressEvent,this->GetObserver());
	mDensityEstimator->AddObserver(vtkCommand::EndEvent,this->GetObserver());
}


iParticleFileLoader::~iParticleFileLoader()
{
	int i;
	for(i=0; i<this->NumStreams(); i++)
	{
		if(this->GetStream(i)->Data != 0) this->GetStream(i)->Data->Delete();
	}

	delete mHelper;
	mDensityEstimator->Delete();
}


void iParticleFileLoader::SetDownsampleMode(int v)
{
	mIterator.SetDownsampleMode(v);
}


int iParticleFileLoader::GetDownsampleMode() const
{
	return mIterator.GetDownsampleMode();
}


void iParticleFileLoader::SetNumNeighbors(int num)
{
	mDensityEstimator->SetNumNeighbors(num);
}


int iParticleFileLoader::GetNumNeighbors() const
{
	return mDensityEstimator->GetNumNeighbors();
}


void iParticleFileLoader::SetTypeIncluded(int type, bool s)
{
	if(type>=0 && type<this->NumStreams())
	{
		this->GetStream(type)->Included = s;
	}
}


void iParticleFileLoader::SetDownsampleFactor(int type, int value)
{
	if(value > 0)
	{
		if(type>=0 && type<this->NumStreams())
		{
			this->GetStream(type)->DownsampleFactor = value;
		}
		else if(type == -1)
		{
			int i;
			for(i=0; i<this->NumStreams(); i++) this->GetStream(i)->DownsampleFactor = value;
		}
	}
}


void iParticleFileLoader::SetDensityAttribute(int type, int attr)
{
	if(type>=0 && type<this->NumStreams())
	{
		this->GetStream(type)->Extras.DensityAttribute = (attr > -2) ? attr : -2;
	}
}


void iParticleFileLoader::SetOrderIsAttribute(int type, bool s)
{
	if(type>=0 && type<this->NumStreams())
	{
		this->GetStream(type)->Extras.OrderIsAttribute = s;
	}
}


void iParticleFileLoader::AddExtras(ParticleStream *stream)
{
	int natt = stream->NumAttributesInFile;

	//
	//  If we save the order in file as an attribute, fill it in.
	//
	if(stream->Extras.OrderIsAttribute && natt<stream->NumAttributesInData)
	{
		mHelper->AddIndexAsAttribute(natt,stream->Scalars->GetNumberOfComponents(),stream->Scalars->GetNumberOfTuples(),stream->Scalars->GetPointer(0));
		natt++;
	}

	//
	//  Check if we need to add density
	//
	if(stream->Extras.DensityAttribute>-2 && natt<stream->NumAttributesInData && stream->Extras.DensityAttribute<natt)
	{
		this->GetObserver()->SetMode(iProgressEventObserver::_Operating);
		mDensityEstimator->ComputeDensity(stream->Points,stream->Scalars,stream->Extras.DensityAttribute,natt);
		natt++;
	}

	//
	//  Final check
	//
	if(natt != stream->NumAttributesInData)
	{
		IERROR_HIGH("iParticleFileLoader is configured icorrectly: unfilled  attributes remained.");
		this->GetErrorStatus()->Set("Unable to read the data due to a bug.");
	}
}


void iParticleFileLoader::FinalizeBody()
{
	int i;

	//
	//  Apply extras
	//
	for(i=0; i<this->NumStreams(); i++) this->AddExtras(this->GetStream(i));
		

	for(i=0; i<this->NumStreams(); i++)
	{
		this->GetStream(i)->Data->SetPoints(this->GetStream(i)->Points);
		this->GetStream(i)->Data->SetVerts(this->GetStream(i)->Verts);
		this->GetStream(i)->Data->GetPointData()->SetScalars(this->GetStream(i)->Scalars);
		this->AttachDataToStream(i,this->GetStream(i)->Data);
	}

	this->ReleaseStreams();

	//
	//  Attribute limits.
	//
	for(i=0; i<this->NumStreams(); i++)	if(!this->GetStream(i)->Subject->GetFixedLimits())
	{
		vtkFloatArray *att = iRequiredCast<vtkFloatArray>(INFO,this->GetStream(i)->Data->GetPointData()->GetScalars());

		if(att!=0 && att->GetNumberOfComponents()>0  && att->GetNumberOfTuples()>0 && att->GetNumberOfComponents()==this->GetStream(i)->Subject->GetLimits()->GetNumVars()) // just in case
		{
			iBuffer<float> aMin, aMax;
			aMin.Extend(att->GetNumberOfComponents());
			aMax.Extend(att->GetNumberOfComponents());

			mHelper->FindRange(att->GetNumberOfComponents(),att->GetNumberOfTuples(),att->GetPointer(0),aMin,aMax);

			int j;
			for(j=0; j<att->GetNumberOfComponents(); j++)
			{
				this->GetStream(i)->Subject->GetLimits()->SetMin(j,aMin[j]);
				this->GetStream(i)->Subject->GetLimits()->SetMax(j,aMax[j]);
			}
		}
	}

	for(i=0; i<this->NumStreams(); i++)
	{
		this->FinalizePolyData(this->GetStream(i)->Data);
	}
}


void iParticleFileLoader::ShiftDataBody(vtkDataSet *data, double dx[3])
{
	int i;
	for(i=0; i<3; i++) if(fabs(dx[i]) > 1.0e-100)
	{
		this->ShiftPolyData(iRequiredCast<vtkPolyData>(INFO,data),i,dx[i]);
	}
}


void iParticleFileLoader::FinalizePolyData(vtkPolyData *data)
{
	if(data==0 || data->GetPoints()==0) return;

	int i;
	vtkIdType l, loff, ntot = data->GetNumberOfPoints();

	//
	//  Boundary condition
	//
	for(i=0; i<3; i++) this->SetDirectionPeriodic(i,this->IsBoxPeriodic());

	if(this->IsBoxPeriodic())
	{
		if(data->GetPoints()->GetDataType() == VTK_FLOAT)
		{
			float *xptrF = (float *)data->GetPoints()->GetVoidPointer(0);
			for(i=0; i<3; i++) if(this->IsDirectionPeriodic(i))
			{
				for(l=0; l<ntot; l++)
				{
					loff = i + 3*l;
					xptrF[loff] -= 2.0*floor(0.5*(1.0+xptrF[loff]));
				}
			}
		}
		else if(data->GetPoints()->GetDataType() == VTK_DOUBLE)
		{
			double *xptrD = (double *)data->GetPoints()->GetVoidPointer(0);
			for(i=0; i<3; i++) if(this->IsDirectionPeriodic(i))
			{
				for(l=0; l<ntot; l++)
				{
					loff = i + 3*l;
					xptrD[loff] -= 2.0*floor(0.5*(1.0+xptrD[loff]));
				}
			}
		}
		else
		{
			this->GetErrorStatus()->Set("Internal bug: ivalid points data type.");
		}
	}

	if(mHaveNormals)
	{
		vtkFloatArray *newNormals;
		newNormals = vtkFloatArray::New(); IERROR_ASSERT(newNormals);
		newNormals->SetNumberOfComponents(3);
		// Allocates and Sets MaxId
		newNormals->SetNumberOfTuples(ntot);
		float *p = (float *)newNormals->GetVoidPointer(0);
		if(p != 0)
		{
			for(l=0; l<ntot; l++)
			{
				p[3*l+0] = p[3*l+1] = 0.0f;
				p[3*l+2] = 1.0f;
			}
			data->GetPointData()->SetNormals(newNormals);
		}
		newNormals->Delete();
	}

	//
	//  Check for overflow
	//
	if(data->GetPointData()->GetScalars() != 0)
	{
		float *p = (float *)data->GetPointData()->GetScalars()->GetVoidPointer(0);
		int natt = data->GetPointData()->GetScalars()->GetNumberOfComponents();
		if(p != 0)
		{
			for(l=0; l<ntot; l++)
			{
				for(i=0; i<natt; i++)
				{
					if(p[natt*l+i] < -iMath::_LargeFloat)
					{
						p[natt*l+i] = -iMath::_LargeFloat;
						mOverflow = true;
					}
					if(p[natt*l+i] >  iMath::_LargeFloat)
					{
						p[natt*l+i] =  iMath::_LargeFloat;
						mOverflow = true;
					}
				}
			}
		}
	}
}


void iParticleFileLoader::ShiftPolyData(vtkPolyData *data, int d, double dx)
{
	if(data==0 || data->GetPoints()==0 || d<0 || d>2) return;

	long n = data->GetPoints()->GetNumberOfPoints();
	float *pf = (float *)data->GetPoints()->GetVoidPointer(0);
	double *pd = (double *)data->GetPoints()->GetVoidPointer(0);
		
	if(pf!=0 && pd!=0)
	{
		mHelper->ShiftData(data->GetPoints()->GetDataType()==VTK_FLOAT,d,n,mPeriodic,dx,pf,pd);
		data->Modified();
	}
}


bool iParticleFileLoader::ReadPositions(iFile &F, int inNum, int outOffset, float updateStart, float updateDuration, float *scale, float *offset)
{
	return mHelper->ReadFortranRecordWithMaskTemplate<float,float>(F,0,inNum,outOffset,updateStart,updateDuration,scale,offset);
}


bool iParticleFileLoader::ReadPositions(iFile &F, int inNum, int outOffset, float updateStart, float updateDuration, double *scale, double *offset)
{
	return mHelper->ReadFortranRecordWithMaskTemplate<double,double>(F,0,inNum,outOffset,updateStart,updateDuration,scale,offset);
}


bool iParticleFileLoader::ReadAttributes(iFile &F, int inNum, int outOffset, float updateStart, float updateDuration, float *scale)
{
	return mHelper->ReadFortranRecordWithMaskTemplate<float,float>(F,1,inNum,outOffset,updateStart,updateDuration,scale,(float *)0);
}


bool iParticleFileLoader::ReadIntAttributes(iFile &F, int inNum, int outOffset, float updateStart, float updateDuration)
{
	return mHelper->ReadFortranRecordWithMaskTemplate<float,int>(F,1,inNum,outOffset,updateStart,updateDuration,(float *)0,(float *)0);
}


void iParticleFileLoader::ConfigureStreams(int *ntot, int *natt, bool paf)
{
	long *nt = new long[this->NumStreams()]; IERROR_ASSERT(nt);

	int i;
	for(i=0; i<this->NumStreams(); i++) nt[i] = ntot[i];

	this->ConfigureStreams(nt,natt);

	delete [] nt;
}


void iParticleFileLoader::ConfigureStreams(long *ntot, int *natt, bool paf)
{
	int n;
	ParticleStream *stream;

	for(n=0; n<this->NumStreams(); n++)
	{
		stream = this->GetStream(n);

		if(stream->Data == 0)
		{
			stream->Data = vtkPolyData::New(); IERROR_ASSERT(stream->Data);
		}

		vtkIdType nsel = mIterator.CreateOneMask(n,ntot[n],stream->Included?stream->DownsampleFactor:0);
		//
		//  Determine how many extra (derived) attributes we need.
		//
		int nattIn = natt[n];
		if(stream->Extras.DensityAttribute>-2 && stream->Extras.DensityAttribute<natt[n]) natt[n]++;
		if(stream->Extras.OrderIsAttribute) natt[n]++;
		//
		//  Try to set the requested number of attributes. DataLimits will either expand to accommodate 
		//  all of them or limit the allowed number to the number of listed records.
		//
		stream->Subject->GetLimits()->AssignVars(natt[n]);
		natt[n] = stream->Subject->GetLimits()->GetNumVars();
		if(nattIn > natt[n]) nattIn = natt[n];

		//
		//  Fill in stream info
		//
		stream->NumTotal = ntot[n];
		stream->NumSelected = nsel;
		stream->NumAttributesInFile = nattIn;
		stream->NumAttributesInData = natt[n];
		stream->InFile = true;

		//
		//  Create data arrays
		//
		vtkPoints *oldPoints = stream->Data->GetPoints();
		vtkCellArray *oldVerts = stream->Data->GetVerts();
		vtkFloatArray *oldScalars = iRequiredCast<vtkFloatArray>(INFO,stream->Data->GetPointData()->GetScalars());

		int type = paf ? VTK_FLOAT : VTK_DOUBLE;
		if(oldPoints==0 || oldPoints->GetNumberOfPoints()!=nsel || oldPoints->GetDataType()!=type)
		{
			stream->Data->SetPoints(0);
			stream->Points = vtkPoints::New(type); IERROR_ASSERT(stream->Points);
			// Allocates and Sets MaxId
			stream->Points->SetNumberOfPoints(nsel);
		}
		else
		{
			stream->Points = oldPoints;
			stream->Points->Register(0);
		}

		if(oldVerts==0 || oldVerts->GetNumberOfCells()!=nsel)
		{
			stream->Data->SetVerts(0);
			stream->Verts = vtkCellArray::New(); IERROR_ASSERT(stream->Verts);
			// This allocates but does not Set Max Id
			stream->Verts->Allocate(stream->Verts->EstimateSize(nsel,1));
			vtkIdType l;
			for(l=0; l<nsel; l++)
			{
				stream->Verts->InsertNextCell(1);
				stream->Verts->InsertCellPoint(l);
			}
		}
		else
		{
			stream->Verts = oldVerts;
			stream->Verts->Register(0);
		}

		if(oldScalars==0 || oldScalars->GetNumberOfTuples()!=nsel || oldScalars->GetNumberOfComponents()!=natt[n])
		{
			stream->Data->GetPointData()->SetScalars(0);
			if(natt[n] > 0)
			{
				stream->Scalars = vtkFloatArray::New(); IERROR_ASSERT(stream->Scalars);
				stream->Scalars->SetNumberOfComponents(natt[n]);
				stream->Scalars->SetNumberOfTuples(nsel);
			}
			else stream->Scalars = 0;
		}
		else
		{
			stream->Scalars = oldScalars;
			stream->Scalars->Register(0);
		}
	}
}


void iParticleFileLoader::ReleaseStreams()
{
	int n;
	for(n=0; n<this->NumStreams(); n++)
	{
		this->GetStream(n)->Points->Delete();
		this->GetStream(n)->Points = 0;
		this->GetStream(n)->Verts->Delete();
		this->GetStream(n)->Verts = 0;
		if(this->GetStream(n)->Scalars != 0)
		{
			this->GetStream(n)->Scalars->Delete();
			this->GetStream(n)->Scalars = 0;
		}
	}
}


void iParticleFileLoader::SetErrorMessage(const iString &filename)
{
	this->GetErrorStatus()->Set(filename);
	mObserver->Finished();
}


void iParticleFileLoader::AutoScalePositions()
{
	//
	//  Scale positions automatically
	//
	vtkPoints *p = 0;
	int i;
	for(i=0; p==0 && i<this->NumStreams(); i++) p = this->GetStream(i)->Points;
	if(p == 0) return;

	switch(p->GetDataType())
	{
	case VTK_FLOAT:
		{
			mHelper->AutoScalePositions<float>();
			break;
		}
	case VTK_DOUBLE:
		{
			mHelper->AutoScalePositions<double>();
			break;
		}
	}
}


//
//  Helper class
//
iParticleHelper::iParticleHelper(iParticleFileLoader *loader) : iParallelWorker(loader->GetViewModule()->GetParallelManager())
{
	mLoader = loader;
}


void iParticleHelper::ShiftData(bool paf, int dim, long n, bool per[3], float dr, float *pf, double *pd)
{
	mPaf = paf;
	mDim = dim;
	mTot = n;
	mDr = dr;
	mFarr = pf;
	mDarr = pd;
	mPeriodic = per;

	this->ParallelExecute(1);
}


void iParticleHelper::FindRange(int dim, long tot, float *pf, float *fMin, float *fMax)
{
	mDim = dim;
	mTot = tot;
	mFarr = pf;

	mNumProcs = this->GetManager()->GetNumberOfProcessors();
	mMin.Extend(mDim*mNumProcs);
	mMax.Extend(mDim*mNumProcs);
	
	this->ParallelExecute(2);

	int i, n;
	for(n=0; n<dim; n++)
	{
		fMin[n] = mMin[n];
		fMax[n] = mMax[n];
		for(i=1; i<mNumProcs; i++)
		{
			if(fMin[n] > mMin[n+dim*i]) fMin[n] = mMin[n+dim*i];
			if(fMax[n] < mMax[n+dim*i]) fMax[n] = mMax[n+dim*i];
		}
	}
}


void iParticleHelper::AddIndexAsAttribute(int dim, int com, long tot, float *pf)
{
	mDim = dim;
	mCom = com;
	mTot = tot;
	mFarr = pf;

	this->ParallelExecute(3);
}


int iParticleHelper::ExecuteStep(int step, iParallel::ProcessorInfo &p)
{
	long l, kstp, kbeg, kend;
	int d = mDim;
	int nc = mCom;

	iParallel::SplitRange(p,mTot,kbeg,kend,kstp);

	switch(step)
	{
	case 1:
		{
			float dr = mDr;

			if(mPaf)
			{
				float *x = mFarr + 3*kbeg;
				for(l=kbeg; l<kend; l++)
				{
					if(l%1000 == 0)
					{
						if(this->IsMaster(p)) mLoader->GetObserver()->SetProgress((d+(float)(l-kbeg)/(kend-kbeg))/3.0);
						if(mLoader->GetObserver()->IsAborted()) return 2;
					}
					x[d] += 2.0*dr;
					if(mPeriodic[d])
					{
						if(x[d] >  1.0) x[d] -= 2.0;
						if(x[d] < -1.0) x[d] += 2.0;
					}
					x += 3;
				}
			}
			else
			{
				double *x = mDarr + 3*kbeg;
				for(l=kbeg; l<kend; l++)
				{
					if(l%1000 == 0)
					{
						if(this->IsMaster(p)) mLoader->GetObserver()->SetProgress((float)(l-kbeg)/(kend-kbeg));
						if(mLoader->GetObserver()->IsAborted()) return 2;
					}
					x[d] += 2.0*dr;
					if(mPeriodic[d])
					{
						if(x[d] >  1.0) x[d] -= 2.0;
						if(x[d] < -1.0) x[d] += 2.0;
					}
					x += 3;
				}
			}
			return 0;
		}
	case 2:
		{
			int j;
			float *f = mFarr + d*kbeg, *fmin = mMin + d*p.ThisProc, *fmax = mMax + d*p.ThisProc;
			for(j=0; j<d; j++) fmin[j] = fmax[j] = f[j];
			for(l=kbeg+1; l<kend; l++)
			{
				f += d;
				for(j=0; j<d; j++)
				{
					if(fmin[j] > f[j]) fmin[j] = f[j];
					if(fmax[j] < f[j]) fmax[j] = f[j];
				}
			}
			return 0;
		}
	case 3:
		{
			float *f = mFarr + mDim;
			for(l=kbeg; l<kend; l++)
			{
				if(l%1000 == 0)
				{
					if(this->IsMaster(p)) mLoader->GetObserver()->SetProgress((float)(l-kbeg)/(kend-kbeg));
					if(mLoader->GetObserver()->IsAborted()) return 2;
				}
				f[nc*l] = l;
			}
			return 0;
		}
	default:
		{
			return 1;
		}
	}
}


template<class T, class D>
bool iParticleHelper::ReadFortranRecordWithMaskTemplate(iFile &F, int dest, int inNum, int outOffset, float updateStart, float updateDuration, T *scale, T *offset)
{
	iParticleDownsampleIterator &it(this->mLoader->mIterator);

	if(this->mLoader->NumStreams() == 0)
	{
		IERROR_HIGH("Streams must be configured prior to a call to iParticleFileLoader::ReadFortranRecordWithMask.");
		return false;
	}

	if(this->mLoader->NumStreams() != it.GetNumMasks())
	{
		IERROR_HIGH("Masks must be created prior to a call to iParticleFileLoader::ReadFortranRecordWithMask.");
		return false;
	}

	//
	//  Is component valid?
	//
	if(inNum<1 || outOffset<0)
	{
		IERROR_HIGH("Invalid call to iParticleFileLoader::ReadFortranRecordWithMask.");
		return false;
	}

	int j;
	vtkDataArray *arr;
	for(j=0; j<this->mLoader->NumStreams(); j++)
	{
		if(this->mLoader->GetStream(j)->InFile)
		{
			switch(dest)
			{
			case 0:
				{
					arr = this->mLoader->GetStream(j)->Points->GetData();
					break;
				}
			case 1:
				{
					arr = this->mLoader->GetStream(j)->Scalars;
					break;
				}
			default:
				{
					arr = 0;
				}
			}
			if((arr==0 && it.GetNumSelected(j)>0) || !it.AttachBuffer(j,arr,outOffset))
			{
				IERROR_HIGH("Invalid call to iParticleFileLoader::ReadFortranRecordWithMask.");
				return false;
			}
			it.SkipMask(j,false);
		}
		else it.SkipMask(j,true);
	}

	//
	//  Read the header
	//
	long lrec1, lrec2;
	if(!this->mLoader->ReadFortranHeaderFooter(F,lrec1))
	{
		this->mLoader->GetErrorStatus()->Set("Corrupted data.");
		return false;
	}

	//
	//  Find # of components in the record
	//
	int nCompRec = lrec1/(sizeof(T)*it.GetNumGlobalTotal());
	if(lrec1 != nCompRec*sizeof(T)*it.GetNumGlobalTotal())
	{
		this->mLoader->GetErrorStatus()->Set("Corrupted data.");
		return false;
	}

	//
	// Decide how to read
	//
	vtkIdType l, lpiece1, lpiece = it.GetNumGlobalTotal()/1000;
	if(lpiece < 1000) lpiece = 1000;
	int npieces = (it.GetNumGlobalTotal()+lpiece-1)/lpiece;

	//
	//  Create tmp array
	//
	T *ptr, *ptmp;
	D *d = new D[lpiece*nCompRec];
	if(d == 0) 
	{ 
		this->mLoader->GetErrorStatus()->Set("Not enough memory to create the data.");
		return false;
	}

	//
	//  parameters for the Progress Bar
	//
	updateDuration /= npieces;

	//
	//  Read piece by piece
	//
	int i;
	it.Start();
	for(j=0; j<npieces; j++)
	{
		if(j < npieces-1)
		{
			lpiece1 = lpiece;
		}
		else
		{
			//
			//  Correct for the last record
			//
			lpiece1 = it.GetNumGlobalTotal() - j*lpiece;
		}
		if(!this->mLoader->ReadBlock(F,d,inNum*lpiece1,updateStart+j*updateDuration,updateDuration))
		{
			this->mLoader->GetErrorStatus()->Set("Corrupted data.");
			delete [] d;
			return false;
		}
		if(this->mLoader->GetObserver()->IsAborted())
		{
			this->mLoader->GetErrorStatus()->SetAbort();
			delete [] d;
			return false;
		}
		for(l=0; l<lpiece1; l++) if(it.IsSelected())
		{
			//
			//  Direct access to data
			//
			ptr = (T *)it.BufferPtr();
			for(i=0; i<inNum; i++)
			{
				ptr[i] = d[nCompRec*l+i];
			}
		}
	}
	it.Stop();

	delete [] d;

	//
	//  Read the footer
	//
	if(!this->mLoader->ReadFortranHeaderFooter(F,lrec2) || lrec1!=lrec2)
	{
		this->mLoader->GetErrorStatus()->Set("Corrupted data.");
		return false;
	}

	//
	//  Do we need to scale?
	//
	int nCompArr;
	vtkIdType nSizeArr;
	if(scale != 0)
	{
		if(offset != 0)
		{
			//
			//  Scale as positions
			//
			for(j=0; j<it.GetNumMasks(); j++)
			{
				ptr = (T *)it.GetBuffer(j);
				nSizeArr = it.GetNumSelected(j);
				nCompArr = it.GetBufferWidth(j);
				for(l=0; l<nSizeArr; l++)
				{
					ptmp = ptr + nCompArr*l;
					for(i=0; i<inNum; i++)
					{
						ptmp[i] = -1.0 + scale[i]*(ptmp[i]-offset[i]);
					}
				}
			}
		}
		else
		{
			//
			//  Just scale
			//
			for(j=0; j<it.GetNumMasks(); j++)
			{
				ptr = (T *)it.GetBuffer(j);
				nSizeArr = it.GetNumSelected(j);
				nCompArr = it.GetBufferWidth(j);
				for(l=0; l<nSizeArr; l++)
				{
					ptmp = ptr + nCompArr*l;
					for(i=0; i<inNum; i++)
					{
						ptmp[i] *= scale[i];
					}
				}
			}
		}
	}

	if(this->mLoader->GetObserver()->IsAborted())
	{
		this->mLoader->GetErrorStatus()->SetAbort();
		return false;
	}

	return true;
}


template<class T>
void iParticleHelper::AutoScalePositions()
{
	//
	//  Scale positions automatically
	//
	int i, j;
	vtkIdType l, np;
	T min[3], max[3], r = 0.0, *ptr; 

	//
	//  Find min/max
	//
	for(j=0; j<this->mLoader->NumStreams(); j++) if(this->mLoader->GetStream(j)->Points != 0)
	{
		ptr = (T *)this->mLoader->GetStream(j)->Points->GetVoidPointer(0);
		if(ptr == 0) continue;

		if(j == 0)
		{
			for(i=0; i<3; i++)
			{
				min[i] = ptr[i];
				max[i] = ptr[i];
			}
		}

		np = this->mLoader->GetStream(j)->Points->GetNumberOfPoints();
		for(l=0; l<np; l++)
		{
			for(i=0; i<3; i++)
			{
				if(ptr[3*l+i] < min[i]) min[i] = ptr[3*l+i];
				if(ptr[3*l+i] < max[i]) max[i] = ptr[3*l+i];
			}
		}

		for(i=0; i<3; i++)
		{
			if(r < (max[i]-min[i])) r = max[i] - min[i];
		}
	}

	if(!(r > 0.0)) return;

	r = 2.0/r;

	for(j=0; j<this->mLoader->NumStreams(); j++) if(this->mLoader->GetStream(j)->Points != 0)
	{
		ptr = (T *)this->mLoader->GetStream(j)->Points->GetVoidPointer(0);
		if(ptr == 0) continue;

		np = this->mLoader->GetStream(j)->Points->GetNumberOfPoints();
		for(l=0; l<np; l++)
		{
			for(i=0; i<3; i++)
			{
				ptr[i] = -1.0 + r*(ptr[i]-min[i]);
			}
		}
	}
}


//
//  Out stream
//
iParticleFileLoader::ParticleStream::ParticleStream()
{
	Included = InFile = true;
	DownsampleFactor = 1;
	NumAttributesInFile = NumAttributesInData = 0;
	NumSelected = 0;
	Data = 0;
	Points = 0;
	Verts = 0;
	Scalars = 0;
}


iParticleFileLoader::Stream* iParticleFileLoader::CreateNewStream() const
{
	return new ParticleStream;
}


iParticleFileLoader::ParticleStreamExtras::ParticleStreamExtras()
{
	DensityAttribute = -2;
	OrderIsAttribute = false;
}
