/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ivectorfieldviewsubject.h"


#include "iboundedplanesource.h"
#include "iboundeddisksource.h"
#include "iboundedspheresource.h"
#include "icolorbars.h"
#include "icommondatadistributors.h"
#include "idata.h"
#include "idatahelper.h"
#include "idatalimits.h"
#include "ierror.h"
#include "imarker.h"
#include "imarkerfamily.h"
#include "ipicker.h"
#include "ireplicatedactor.h"
#include "ivectorfieldglyphpipeline.h"
#include "ivectorfieldstreamlinepipeline.h"
#include "iviewmodule.h"
#include "iviewsubjectparallelpipeline.h"

#include <vtkCellArray.h>
#include <vtkFloatArray.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>
#include <vtkProperty.h>

//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWSUBJECT_DEFINE_TYPE(iVectorFieldViewSubject,VectorField,v);

IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,LineDir,ld,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,LineLength,ll,Float,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,LineQuality,lq,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,Method,m,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,SourceDirection,sd,Float,3);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,SourceOpacity,so,Float,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,NumberOfStreamLines,nl,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,ShowSourceObject,sso,Bool,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,SourceType,st,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,TubeRangeFactor,tr,Float,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,TubeSize,ts,Int,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,TubeVariationFactor,tv,Float,1);

IOBJECT_DEFINE_DISTANCE_KEY(iVectorFieldViewSubject,SourceSize,ss);

//
//  Inherited keys
//
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COMMON(iVectorFieldViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iVectorFieldViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_POSITION(iVectorFieldViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COLOR(iVectorFieldViewSubject,2);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_OPACITY(iVectorFieldViewSubject,2);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_PALETTE(iVectorFieldViewSubject,2);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_FIELD(iVectorFieldViewSubject);


#ifdef IOBJECT_BACKWARD_COMPATIBLE
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,AttachSourceToMarker,-am,OffsetInt,1);
IOBJECT_DEFINE_KEY(iVectorFieldViewSubject,SourcePositionToPickedPoint,-p2p,Bool,1);
IOBJECT_DEFINE_POSITION_KEY(iVectorFieldViewSubject,SourcePosition,-sx);
#endif


//
// iVectorFieldViewSubject class
//
iVectorFieldViewSubject::iVectorFieldViewSubject(iViewModule *vm, const iDataType &type, const iDataType &scalarType, const iString &name) : iFieldViewSubject(vm,type,scalarType,name,2,_ViewSubjectAlwaysShaded|_ViewSubjectHasPosition), mSourceSize(vm,true)
{
	mObjectType = _ObjectTypeVectorField;

	mMethod = _VectorMethodGlyph;
	mLineQuality = 0;
	mLineLength = 1.0;
	mLineDir = _StreamLineDirBothWays;
	mTubeSize = 1;
	mTubeRangeFactor = 10;
	mTubeVariationFactor = 0.01; 
	mColorByOffset = 0;

	mSourceShow = false;
	mSourceType = _VectorSourcePlane;
	mNumLines = 4;
	mSourceDir[0] = mSourceDir[1] = 0.0; mSourceDir[2] = 1.0;
	mSourceSize = 0.5;

	mColorBy = _VectorColorByNone;
	//
	//  Do VTK stuff
	//	
	mStreamLineSourcePoints = vtkPoints::New(VTK_DOUBLE); IERROR_ASSERT(mStreamLineSourcePoints);
	mStreamLineSourceVerts = vtkCellArray::New(); IERROR_ASSERT(mStreamLineSourceVerts);
	mStreamLineSourceNorms = vtkFloatArray::New(); IERROR_ASSERT(mStreamLineSourceNorms);
	mStreamLineSourceNorms->SetNumberOfComponents(3);

	mSourceDisk = iBoundedDiskSource::New(this); IERROR_ASSERT(mSourceDisk);
	mSourcePlane = iBoundedPlaneSource::New(this); IERROR_ASSERT(mSourcePlane);
	mSourceSphere = iBoundedSphereSource::New(this); IERROR_ASSERT(mSourceSphere);

	mSourceDisk->SetResolution(10);
	mSourcePlane->SetResolution(10);
	mSourceSphere->SetResolution(10);

	mSourceDisk->SetWithArrow(false);

	this->SetOpacity(1,0.5f);
 	mActors[1]->SetInput(mSourcePlane->GetOutput());

	this->UpdateMaterialProperties();
}


iVectorFieldViewSubject::~iVectorFieldViewSubject()
{
	for(int i=0; i<=this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex(); i++) this->GetViewModule()->GetMarkerFamily()->GetMember(i)->DetachViewSubject(this);
	mStreamLineSourcePoints->Delete();
	mStreamLineSourceVerts->Delete();
	mStreamLineSourceNorms->Delete();
	mSourceDisk->Delete();
	mSourcePlane->Delete();
	mSourceSphere->Delete();
}


void iVectorFieldViewSubject::ConfigureBody()
{
	//
	//  Create pipelines (must be created after the object is fully created)
	//	
	this->AddMainPipeline(1);
	this->AddMainPipeline(2);
	mActors[0]->SetInput(this->Pipeline()->GetOutput(0));

	vtkPolyData *pd = vtkPolyData::New(); IERROR_ASSERT(pd);
	pd->SetPoints(mStreamLineSourcePoints);
	pd->SetVerts(mStreamLineSourceVerts);
	pd->GetPointData()->SetNormals(mStreamLineSourceNorms);
	this->Pipeline(1)->SetNthInput(0,pd);
	pd->Delete();
}


void iVectorFieldViewSubject::ResetPipelineInput(vtkDataSet *input)
{
	this->Pipeline(0)->SetNthInput(0,input);
	this->Pipeline(1)->SetNthInput(1,input);
}


void iVectorFieldViewSubject::ResetPipeline()
{
	mStreamLineSourcePoints->Initialize();
	mStreamLineSourceVerts->Initialize();
	this->UpdatePipelines();
}


void iVectorFieldViewSubject::UpdatePipelines()
{
	this->Pipeline(0)->UpdateContents(iVectorFieldGlyphPipeline::_Size);
	this->Pipeline(1)->UpdateContents(iVectorFieldStreamLinePipeline::_Pipeline);
}


void iVectorFieldViewSubject::UpdateColorByExtra()
{
	if(mColorByOffset > 0)
	{
		mActors[0]->SetScalarVisibility(mColorBy != _VectorColorByNone);
		switch (mColorBy) 
		{
		case _VectorColorByMagnitude:
			{ 
				mActors[0]->ColorByArrayComponent(0,1); 
				mActors[0]->SyncWithLimits(this->GetLimits(),0);
				break;
			}
		case _VectorColorByVorticity:
			{
				mActors[0]->ColorByArrayComponent(0,2); 
				mActors[0]->SyncWithLimits(this->GetLimits(),0,2.0);
				break;
			}
		case _VectorColorByDivergence:
			{
				mActors[0]->ColorByArrayComponent(0,3); 
				mActors[0]->SyncWithLimits(this->GetLimits(),0,2.0);
				break;
			}
		}
	}
//	this->UpdatePipelines();
}


void iVectorFieldViewSubject::UpdateAfterMarkerMove()
{
	if(mSourceType == _VectorSourceMarker)
	{
		int i;
		vtkIdType l[1];
		bool reset = false;

		if(mStreamLineSourcePoints->GetNumberOfPoints() != this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex()+1)
		{
			mStreamLineSourcePoints->Initialize();
			mStreamLineSourceVerts->Initialize();
			mStreamLineSourceNorms->Initialize();
			mStreamLineSourcePoints->SetNumberOfPoints(this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex()+1);
			reset = true;
		}

		iMarker *m;
		for(i=0; i<=this->GetViewModule()->GetMarkerFamily()->GetMaxMemberIndex(); i++)
		{
			m = this->GetViewModule()->GetMarkerFamily()->GetMember(i);
			m->AttachViewSubject(this);
			mStreamLineSourcePoints->SetPoint(i,m->GetPosition());
			if(reset)
			{
				//
				//  If we are not resetting, then the vert array did not change
				//
				l[0] = i;
				mStreamLineSourceVerts->InsertNextCell(1,l);
			}
		}

		this->Pipeline(1)->GetInput()->Modified();
	}
	else
	{
		this->iFieldViewSubject::UpdateAfterMarkerMove();
	}
}


void iVectorFieldViewSubject::UpdateStreamLineSource()
{ 
	switch(mSourceType)
	{
	case _VectorSourceMarker:
		{
			this->UpdateAfterMarkerMove();
			break;
		}
	case _VectorSourceDisk:
	case _VectorSourcePlane:
	case _VectorSourceSphere:
		{
			int i, k;
			vtkIdType n, l;
			double x[3], rx, ry, rf;
			bool reset = false;
			
			n = mNumLines;
			if(n < 1) return;

			mSourceDisk->SetCenter(mPosition);
			mSourceDisk->SetNormal(mSourceDir);
			mSourceDisk->SetRadius(mSourceSize);
			mSourcePlane->SetCenter(mPosition);
			mSourcePlane->SetNormal(mSourceDir);
			mSourceSphere->SetCenter(mPosition);
			mSourceSphere->SetRadius(mSourceSize);

			if(mStreamLineSourcePoints->GetNumberOfPoints() != n)
			{
				mStreamLineSourcePoints->Initialize();
				mStreamLineSourceVerts->Initialize();
				mStreamLineSourceNorms->Initialize();
				mStreamLineSourcePoints->SetNumberOfPoints(n);
				mStreamLineSourceNorms->SetNumberOfComponents(3);
				mStreamLineSourceNorms->SetNumberOfTuples(n);
				reset = true;
			}

			l = 0;
			vtkMath::RandomSeed(123456789);
			//
			//  Find two basis vectors for the plane
			//
			float t, t1[3], t2[3], z[3];
			do
			{
				for(i=0; i<3; i++) t1[i] = 2*vtkMath::Random() - 1;
				vtkMath::Normalize(t1);
				vtkMath::Cross(mSourceDir,t1,t2);
			}
			while(t2[0]*t2[0]+t2[1]*t2[1]+t2[2]*t2[2] < 1.0e-10);
			
			t = vtkMath::Dot(mSourceDir,t1);
			for(i=0; i<3; i++) t1[i] -= t*mSourceDir[i];
			vtkMath::Normalize(t1);
			vtkMath::Normalize(t2);

			k = 0;
			rf = 1.0;
			double sz = mSourceSize;
			double pos[3];
			for(i=0; i<3; i++) pos[i] = mPosition[i];

			while(l < n)
			{
				//
				//  Pick up a random point on the source surface
				//
				switch(mSourceType)
				{
				case _VectorSourceDisk:
					{
						//
						//  Random point on the disk
						//
						rx = 2*vtkMath::Random() - 1;
						ry = 2*vtkMath::Random() - 1;
						if(rx*rx+ry*ry > 1.0)
						{
							x[0] = x[1] = x[2] = 2.0;
						}
						else
						{
							for(i=0; i<3; i++) 
							{
								x[i] = pos[i] + rf*sz*(rx*t1[i]+ry*t2[i]);
								z[i] = mSourceDir[i];
							}
						}
						break;
					}
				case _VectorSourcePlane:
					{
						//
						//  Random point on the plane
						//
						rx = 7.0*vtkMath::Random() - 3.5;
						ry = 7.0*vtkMath::Random() - 3.5;
						for(i=0; i<3; i++) 
						{
							x[i] = pos[i] + rf*(rx*t1[i]+ry*t2[i]);
							z[i] = mSourceDir[i];
						}
						break;
					}
				case _VectorSourceSphere:
					{
						//
						//  Random point on the sphere
						//
						z[2] = 2*vtkMath::Random() - 1;
						rx = sqrt(1.0-z[2]*z[2]);
						ry = 6.2831853*vtkMath::Random();
						z[0] = rx*cos(ry);
						z[1] = rx*sin(ry);
						for(i=0; i<3; i++)
						{
							x[i] = pos[i] + sz*z[i];
							if(x[i] < -1.0) x[i] += 2.0; 
							if(x[i] >  1.0) x[i] -= 2.0;
						}
						break;
					}
				}

				k++;

				if(k<100 && (x[0]<-1.0 || x[0]>1.0 || x[1]<-1.0 || x[1]>1.0 || x[2]<-1.0 || x[2]>1.0)) 
				{
					if(k > 20) rf *= 0.5;
					continue;
				}

				if(k >= 100)
				{
#ifdef I_CHECK1
					IERROR_REPORT_BUG;
#endif
					for(i=0; i<3; i++) x[i] = mPosition[i];
				}

				k = 0;
				rf = 1.0;
				mStreamLineSourcePoints->SetPoint(l,x); 
				mStreamLineSourceNorms->SetTuple(l,z); 
				if(reset) mStreamLineSourceVerts->InsertNextCell(1,&l); 
				l++;
			}
			break;
		}
	default:
		{
#ifdef I_CHECK1
			IERROR_REPORT_BUG;
#endif
		}
	}

	this->Pipeline(1)->GetInput()->Modified();
}


void iVectorFieldViewSubject::SetShowSourceObject(bool s)
{
	if(s) mActors[1]->VisibilityOn(); else mActors[1]->VisibilityOff();
	mSourceShow = s;
	this->ClearCache();
}


void iVectorFieldViewSubject::SetSourceType(int v)
{
	if(v>=0 && v<__NumVectorSources && (v<_VectorSourceMarker || (this->GetViewModule()->GetMarkerFamily()->IsVisible())))
	{
		mSourceType = v;
		switch(mSourceType)
		{
		case _VectorSourceDisk:
			{
				mActors[1]->SetInput(mSourceDisk->GetOutput());
				this->SetShowSourceObject(mSourceShow);
				break;
			}
		case _VectorSourcePlane:
			{
				mActors[1]->SetInput(mSourcePlane->GetOutput());
				this->SetShowSourceObject(mSourceShow);
				break;
			}
		case _VectorSourceSphere:
			{
				mActors[1]->SetInput(mSourceSphere->GetOutput());
				this->SetShowSourceObject(mSourceShow);
				break;
			}
		default:
			{
				this->SetShowSourceObject(false);
			}
		}
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::SetNumberOfStreamLines(int v)
{
	if(v > 0)
	{
		mNumLines = v;
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::SetSourceOpacity(float v)
{
	if(v>0.0f && !(v>1.0f))
	{
		mOpacities[1] = v;
		mActors[1]->GetProperty()->SetOpacity(v);
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::SetMethod(int v)
{ 
	if(v>=0 && v<__NumVectorMethods) 
	{
		mMethod = v;
		switch(mMethod)
		{
		case _VectorMethodGlyph:
			{
				mColorByOffset = 0;
				mActors[0]->SetInput(vtkPolyData::SafeDownCast(this->Pipeline()->GetOutput(0)));
				break;
			}
		case _VectorMethodStreamLine:
		case _VectorMethodStreamTube:
		case _VectorMethodStreamBand:
			{
				mColorByOffset = 4;
				mActors[0]->SetInput(vtkPolyData::SafeDownCast(this->Pipeline(1)->GetOutput(0)));
				break;
			}
		}
		this->SetColorBy(mColorBy);
		this->UpdatePipelines();
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::SetLineDir(int v)
{ 
	if(v>=0 && v<__NumStreamLineDirs) 
	{
		mLineDir = v;
		this->Pipeline(1)->UpdateContents(iVectorFieldStreamLinePipeline::_Direction);
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::UpdateGlyphSize()
{ 
	this->Pipeline()->UpdateContents(iVectorFieldGlyphPipeline::_Size);
}


void iVectorFieldViewSubject::SetLineLength(float v)
{ 
	if(v>0.0 && v<100.0)
	{
		mLineLength = v;
		this->Pipeline(1)->UpdateContents(iVectorFieldStreamLinePipeline::_Length);
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::UpdateGlyphSampleRate()
{ 
	this->Pipeline()->UpdateContents(iVectorFieldGlyphPipeline::_SampleRate);
}


void iVectorFieldViewSubject::SetLineQuality(int v)
{ 
	if(v>=0 && v<10) 
	{
		mLineQuality = v;
		this->Pipeline(1)->UpdateContents(iVectorFieldStreamLinePipeline::_Quality);
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::SetTubeSize(int v)
{ 
	if(v>=1 && v<=10) 
	{
		mTubeSize = v;
		this->Pipeline(1)->UpdateContents(iVectorFieldStreamLinePipeline::_TubeSize);
		mActors[0]->GetProperty()->SetLineWidth(v);
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::SetTubeRangeFactor(float q)
{ 
	if(q>1.0 && q<100.0) 
	{
		mTubeRangeFactor = q;
		this->Pipeline(1)->UpdateContents(iVectorFieldStreamLinePipeline::_TubeRangeFactor);
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::SetTubeVariationFactor(float q)
{ 
	if(q>1.0e-7 && q<1.0) 
	{
		mTubeVariationFactor = q;
		this->Pipeline(1)->UpdateContents(iVectorFieldStreamLinePipeline::_TubeVariationFactor);
		this->ClearCache();
	}
}


void iVectorFieldViewSubject::ShowBody(bool show)
{
	if(show)
	{
		this->UpdatePipelines();
		this->UpdateStreamLineSource();
		mActors[0]->VisibilityOn();
		this->ShowColorBars(true);
	} 
	else 
	{
		this->ShowColorBars(false);
		this->SetShowSourceObject(false);
		mActors[0]->VisibilityOff();
	}
}


float iVectorFieldViewSubject::GetExtraMemorySize() const
{
	float s = 0.0;
	s += mStreamLineSourcePoints->GetActualMemorySize();
	s += mStreamLineSourceVerts->GetActualMemorySize();
	return s;
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iVectorFieldViewSubject::FieldViewSubjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyMethod(),mMethod);
	this->PackValue(s,KeyTubeSize(),mTubeSize);
	this->PackValue(s,KeyLineQuality(),mLineQuality);
	this->PackValue(s,KeyLineDir(),mLineDir);
	this->PackValue(s,KeyLineLength(),mLineLength);
	this->PackValue(s,KeyTubeRangeFactor(),mTubeRangeFactor);
	this->PackValue(s,KeyTubeVariationFactor(),mTubeVariationFactor);

	this->PackValue(s,KeyShowSourceObject(),mSourceShow);
	this->PackValue(s,KeySourceOpacity(),mOpacities[1]);
	this->PackValue(s,KeySourceType(),mSourceType);
	this->PackValue(s,KeyNumberOfStreamLines(),mNumLines);
	this->PackValue(s,KeySourceDirection(),mSourceDir,3);

	this->PackValueDistance(s,KeySourceSize(),mSourceSize);
}


void iVectorFieldViewSubject::FieldViewSubjectUnPackStateBody(const iString &s)
{
	int i; bool b; float f, fv[3];

	if(this->UnPackValue(s,KeyMethod(),i)) this->SetMethod(i);
	if(this->UnPackValue(s,KeyTubeSize(),i)) this->SetTubeSize(i);
	if(this->UnPackValue(s,KeyLineQuality(),i)) this->SetLineQuality(i);
	if(this->UnPackValue(s,KeyLineDir(),i)) this->SetLineDir(i);
	if(this->UnPackValue(s,KeyLineLength(),f)) this->SetLineLength(f);
	if(this->UnPackValue(s,KeyTubeRangeFactor(),f)) this->SetTubeRangeFactor(f);
	if(this->UnPackValue(s,KeyTubeVariationFactor(),f)) this->SetTubeVariationFactor(f);

	if(this->UnPackValue(s,KeyShowSourceObject(),b)) this->SetShowSourceObject(b);
	if(this->UnPackValue(s,KeySourceOpacity(),f)) this->SetSourceOpacity(f);
	if(this->UnPackValue(s,KeySourceType(),i)) this->SetSourceType(i);
	if(this->UnPackValue(s,KeyNumberOfStreamLines(),i)) this->SetNumberOfStreamLines(i);
	for(i=0; i<3; i++) fv[i] = mSourceDir[i];
	if(this->UnPackValue(s,KeySourceDirection(),fv,3)) this->SetSourceDirection(fv);

	if(this->UnPackValueDistance(s,KeySourceSize(),mSourceSize)) this->SetSourceSize(mSourceSize);

#ifdef IOBJECT_BACKWARD_COMPATIBLE
	iObject::ReportMissingKeys(false);
 	if(this->UnPackValue(s,KeyAttachSourceToMarker(),i)) this->AttachToMarker(i);
	if(this->UnPackValuePosition(s,KeySourcePosition(),mPosition)) this->SetPosition(mPosition);
	if(this->UnPackValue(s,KeySourcePositionToPickedPoint(),b) && b) this->SetPosition(_LocationPickedPoint); 
	iObject::ReportMissingKeys(true);
#endif
}


void iVectorFieldViewSubject::UpdateAfterMarkerAdd(iMarker * )
{
	this->UpdateAfterMarkerMove();
}


void iVectorFieldViewSubject::UpdateAfterMarkerDelete(iMarker *m)
{
	this->UpdateAfterMarkerMove();
	iViewSubject::UpdateAfterMarkerDelete(m);
}


int iVectorFieldViewSubject::GetDataOffset() const
{
	return 0;
}


bool iVectorFieldViewSubject::CanBeShown() const
{
	iDataHelper *h = this->CreateDataHelper(this->GetData());
	bool ret = (h->IsThereVectorData() && this->GetDataOffset()>=0 && this->GetDataOffset()<this->GetLimits()->GetNumVars());
	delete h;
	return ret;
}


void iVectorFieldViewSubject::SetSourceSize(const iDistance &s)
{
	if(s < 1.0)
	{
		mSourceSize = s;
		this->UpdateStreamLineSource();
		this->ClearCache();
	}
}
	

void iVectorFieldViewSubject::UpdatePosition(const iPosition &)
{
	mPosition.CutToBounds();
	this->UpdateStreamLineSource();
}


void iVectorFieldViewSubject::SetSourceDirection(const float n[3])
{
	int i;
	for(i=0; i<3; i++) mSourceDir[i] = n[i];
	this->UpdateStreamLineSource();
	this->ClearCache();
}


void iVectorFieldViewSubject::FieldViewSubjectSyncWithData(const iDataSyncRequest &)
{
	this->UpdatePipelines();
}


iViewSubjectPipeline* iVectorFieldViewSubject::CreatePipeline(int id)
{
	switch(id)
	{
	case 0: return new iVectorFieldGlyphPipeline(this);
	case 1: return new iVectorFieldStreamLinePipeline(this);
	default: return 0;
	}
}


void iVectorFieldViewSubject::ConfigureMainPipeline(iViewSubjectPipeline *p, int id)
{
	iViewSubjectParallelPipeline *pp = iRequiredCast<iViewSubjectParallelPipeline>(INFO,p);

	switch(id)
	{
	case 0:
		{
			iImageDataDistributor *idd = new iImageDataDistributor(pp->GetDataManager());
			pp->GetDataManager()->AddDistributor(idd);
			pp->GetDataManager()->AddCollector(new iPolyDataCollector(pp->GetDataManager(),idd));
			break;
		}
	case 1:
		{
            pp->GetDataManager()->AddDistributor(new iPolyDataDistributor(pp->GetDataManager()));
			pp->GetDataManager()->AddCollector(new iPolyDataCollector(pp->GetDataManager()));
			break;
		}
	}
}


void iVectorFieldViewSubject::RemoveInternalDataForExtraComponents()
{
	mStreamLineSourcePoints->Initialize();
	mStreamLineSourceVerts->Initialize();
	mStreamLineSourceNorms->Initialize();
}

