// KreateCD - CD recording software for the K desktop environment
//
// 2001 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#ifndef AUDIOFILEINFO_INCLUDED
#define AUDIOFILEINFO_INCLUDED

/** 
    @short AudioFileInfo - container class that holds audio file information
    @author Alexander Feigl <Alexander.Feigl@gmx.de>

    The AudioFileInfo class ist a container class which holds all information neccessary to 
    access audio files - such as endianess , sample format, data position, file length
*/

class AudioConverter;

class AudioFileInfo {

 public:

  /** 
      <pre>
      Byte endian enum
      Endian_Little : the file is in litte endian format
      Endian_Big : the file is in big endian format
      </pre>
  */
  enum Sample_Endian{Endian_Little,Endian_Big};

  /**
     <pre>
     Sample type enum
     Samples_PCM : samples are in PCM format
     Samples_PCM_Unsigned : samples are in PCM unsigned format
     </pre>
  */
  enum Sample_Format{Samples_PCM,Samples_PCM_Unsigned,Samples_ALaw,Samples_ULaw};

  /**
     <pre>
     Sample size enum
     Samples_8Bit : sample size is 8 bit
     Samples_16Bit : sample size is 16 bit
     Samples_24Bit : sample size is 24 bit
     Samples_32Bit : sample size is 32 bit
     </pre>
  */
  enum Sample_Size{Samples_8Bit,Samples_16Bit,Samples_24Bit,Samples_32Bit};


  AudioFileInfo(void);
  ~AudioFileInfo(void);
  AudioFileInfo(const AudioFileInfo &object);
  
  void setEndian(enum Sample_Endian endi);
  void setFormat(enum Sample_Format form);
  void setSize(enum Sample_Size siz);
  void setFormatName(const char *form);
  void setFilename(const char *fn);
  void setRate(int rat);
  void setChannels(int chan);  
  void setDataPos(long pos);
  void setDataSize(long siz);
  void setDataSamples(long samp);
  void setStartPos(long pos);
  void setEndPos(long pos);
  void setBoost(float boo);
  void setBalance(float bal);
  void setMaxBoost(float mboo);
  void setConverter(AudioConverter *conv,int (*function)(int argc,char **argv)=0);
  void setNeedsRipping(bool nr);
    
  Sample_Endian endian(void);
  Sample_Format format(void);
  Sample_Size size(void);
  const char *formatName(void);
  int rate(void);
  int channels(void);
  int sampleBytes(void);
  long dataPos(void);
  long dataSize(void);
  long dataSamples(void);
  long startPos(void);
  long endPos(void);
  float boost(void);
  float balance(void);
  float maxBoost(void);
  bool needsRipping(void);
  AudioConverter *converter(void);
  int(*converterFunction(void))(int argc,char **argv);

  /**
     This method converts an audio format which cannot be used directly be 
     KreateCD
   
     @return  return status 
  */
  bool                 validateFile(void);

  /** 
     This method returns the file that has to be burned on cd. It is the
     original file for non-complex audio and the converted audio for complex.
     
     @return  file name of audio file
  */ 

  const char          *getBurnFile(void);

  /** 
     This method returns the original source of the audio file. 
     
     @return  file name of audio file
  */ 

  const char          *getSourceName(void);

  /**
     This method returns the duration (in 1/75 seconds) of the selected part of the audio file

     @return length of selected part in 1/75 seconds
  */
  int getSelectedDuration(void);

  /**
     This methods returns the size of the whole audio file (in 1/75 seconds)

     @return length of whole audio file in 1/75 seconds
  */
  int getAbsoluteDuration(void);

  /**
     This method returns whether this audio file needs to be converted to CD-R before it can be
     written.
   
     @return  the audio file needs a temporary image
  */
  int                  needImage(void);

  /**
     This method returns whether this audio file needs a temporary file before it can be
     written.
   
     @return  the audio file needs a temporary image
  */
  int                  needFile(void);

     
 private:

  void updateSampleBytes();
  void updateDataSamples();
  void resetSelection();

  /**
     sample endian
  */
  enum Sample_Endian           SampleEndian;

  /**
     sample format (PCM)
  */
  enum Sample_Format           SampleFormat;

  /**
     sample channels
  */
  int                          SampleChannels;

  /**
     size of samples
  */
  enum Sample_Size             SampleSize;

  /**
     sampling rate
  */
  long int                     SampleRate;

  /**
     nuber of bytes of one sample (for all channels)
  */
  int                          SampleBytes;

  /**
     file offset where the sample data starts
  */ 
  long int                     DataPos;

  /**
     number of bytes of the sample date body 
  */
  long int                     DataSize; 

  /**
     number of samples of the sample data body
  */
  long int                     DataSamples;

  /**
     character string of the file format name
  */
  char                         Format[16];

  /**
     file name of the audio file - the original file which was passed to
     KreateCD. This can be a mp3, wav or similar stream
  */
  char                         FileName[1024];

  /**
     file name of the audio file which has been decoded from an complex
     file fomrat
  */
  char                         DecodedFileName[1024];

  /**
     pointer to an audio file converter
  */
  class AudioConverter        *Converter;

  /**
     pointer to the converter function for forking processes
  */
  int (*ConverterFunction)(int argc,char **argv);


  /**
     sample offset where the selection starts
  */
  long int                     SelectedStart;

  /**
     sample offset where the selection ends
  */
  long int                     SelectedEnd;

  /**
     boost factor
  */
  float                        BoostFactor;

  /**
     balance
  */
  float                        Balance;

  /**
     cached maximum boost factor - this is initalized by AudioScanImage 
  */
  float                        MaxBoost;

  /**
     this AudioFileInfo contains an unripped CD track
  */
  bool                         NeedsRipping;

};


#endif
